/* lphase - linear phase filtering */

/* Mark Huckvale - University College London */

/* version 1.0 - March 1995 */

#define PROGNAME "lphase"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH LPHASE SFS1 UCL
.SH NAME
lphase -- linear phase filtering using non-recursive filter
.SH SYNOPSIS
.B lphase
(-i item) (-l lowpassedge) (-h highpassedge) (-n order) file
.SH DESCRIPTION
.I lphase
is a program to perform a linear phase filtering using a non-recursive
filter.  The filter is designed using a Hamming-windowed sinc function
of the given length, centered about the output sample (= zero phase).
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -l lowpassedge
Specify filter edge for low-pass filter.
.TP 11
.BI -h highpassedge
Specify filter edge for high-pass filter.
.TP 11
.BI -n order
Specify size of filter (default 64).
.SH INPUT ITEMS
.IP SPEECH
Any speech signal
.IP LX
Any Lx signal
.SH OUTPUT ITEMS
.IP SP
Filtered signal.
.IP LX
Filtered signal.
.SH HISTORY
.IP low
low-pass frequency edge
.IP high
high-pass frequency edge
.IP order
order of filter.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
genfilt
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"

/* manifest constants */
#define MAXORDER	1024
#define DEFORDER	65

/* filter coefficients */
float	halfilt[MAXORDER];
int	halflen=DEFORDER/2;
float	filt[MAXORDER];
int	filtlen=DEFORDER;

/* item data */
struct item_header	ipitem;
short			sbuf[10*MAXORDER];
struct item_header	opitem;
short			dbuf[10*MAXORDER];

/* forward declarations */
void	designlowpass();
void	xformhighpass();
void	xformbandpass();
void	dofilt();

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int32		it;		/* item selection */
	char		*ty;		/* item sub type */
	/* file variables */
	int32		ipit=SP_TYPE;
	char		*iptype="0";
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		ofid;
	double		lofreq=-1;
	double		hifreq=-1;
	int		srate;
	int		i,len;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:l:h:n:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Linear-phase filtering V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it == SP_TYPE) || (it == LX_TYPE)) {
					ipit = it;
					iptype = ty;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'l' :	/* low edge */
			lofreq = atof(optarg);
			break;
		case 'h' :	/* low edge */
			hifreq = atof(optarg);
			break;
		case 'n' :
			filtlen = atoi(optarg) | 1;	/* always odd */
			if ((filtlen <= 0) || (filtlen > MAXORDER))
				error("illegal filter size (max 1024)");
			halflen = filtlen/2;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-l lowpassedge) (-h highpassedge) (-n order) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL))<0)
		error("access error on '%s'",filename);

	/* locate input item */
	if (!sfsitem(fid,ipit,iptype,&ipitem))
		error("could not find input item in '%s'",filename);
	srate = (int)(0.5+1.0/ipitem.frameduration);

	/* construct filter */
	if ((lofreq<0) && (hifreq<0))
		error("no filter frequencies specified");
	else if ((lofreq>srate/2) || (hifreq>srate/2))
		error("illegal frequencies");
	else if (hifreq < 0)
		designlowpass(lofreq/srate,filtlen);
	else if (lofreq < 0) {
		designlowpass(0.5-hifreq/srate,filtlen);
		xformhighpass(filtlen);
	}
	else if (hifreq < lofreq) {
		designlowpass((lofreq-hifreq)/(2*srate),filtlen);
		xformbandpass((lofreq+hifreq)/(2*srate),filtlen);
	}
	else
		error("no support for band-stop");

	/* create output item header */
	sfsheader(&opitem,ipit,0,2,1,ipitem.frameduration,ipitem.offset,1,0,0);
	sprintf(opitem.history,"%s(%d.%02d;low=%g,high=%g,order=%d)",
		PROGNAME,
		ipitem.datatype,ipitem.subtype,
		(lofreq<0)?srate/2:lofreq,
		(hifreq<0)?0:hifreq,filtlen);

	/* open output channel */
	if ((ofid=sfschannel(filename,&opitem))<0)
		error("could not open output channel");

	/* OK do the filtering */
	halflen = filtlen/2;
	memset(sbuf,0,halflen*2);
	for (i=0;(len=sfsread(fid,i,9*MAXORDER,sbuf+halflen));i+=len) {
		if (len < 9*MAXORDER)
			memset(sbuf+halflen+len,0,halflen*2);
		len -= halflen;
		if (len <= 0) break;
		dofilt(sbuf,dbuf,len);
		if (sfswrite(ofid,len,dbuf)!=len)
			error("output write error");
		memcpy(sbuf,(sbuf+halflen)+(len-halflen),halflen*2);
		if (ttytest()) {
			printf("Frame %d/%d\r",i+len,ipitem.numframes);
			fflush(stdout);
		}
	}

	/* update */
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);

	/* that's all folks */
	exit(0);
}

#define ABS(x)	(((x)<0)?(-(x)):(x))

/* design a low pass filter */
void	designlowpass(freq,order)
double	freq;
int	order;
{
	double	w;
	int	i;

	/* generate one half of coefficients from sinc() function */
	w = 2.0 * M_PI * freq;
	halfilt[0] = (float)(w/M_PI);
	for (i=1;i<=halflen;i++)
		halfilt[i] = (float)(sin(i*w)/(i*M_PI));

	/* window with (half-)hamming window */
	for (i=0;i<=halflen;i++)
		halfilt[i] *= (float)(0.54 + 0.46 * cos(i*M_PI/halflen));

	/* copy as symmetric filter */
	for (i=0;i<order;i++)
		filt[i] = halfilt[ABS(halflen-i)];

}

/* high-pass transform */
void	xformhighpass(order)
int	order;
{
	int	i;

	for (i=1;i<=halflen;i+=2) {
		filt[halflen-i] = -filt[halflen-i];
		filt[halflen+i] = -filt[halflen+i];
	}
}

/* band-pass transform */
void	xformbandpass(cfreq,order)
double	cfreq;
int	order;
{
	double	w;
	int	i;

	w = 2.0 * M_PI * cfreq;
	filt[halflen] = 2*filt[halflen];
	for (i=1;i<=halflen;i++) {
		filt[halflen-i] = (float)(2 * filt[halflen-i] * cos(i*w));
		filt[halflen+i] = (float)(2 * filt[halflen+i] * cos(i*w));
	}
}

/* do the filtering */
void	dofilt(src,dst,len)
short	*src;
short	*dst;
int	len;
{
	register int	i,j;
	short		*sp;
	float		sum;

	for (i=0;i<len;i++) {
		sum=(float)0.0;
		sp = src+i;
		for (j=0;j<filtlen;j++)
			sum += (*sp++) * filt[j];
		*dst++ = (short)sum;
	}
}
