/* prep -- perform standard waveform preparation functions */

/* m.a. huckvale - March 1988 */

/* version 1.0 */

/* version 1.1 - May 1992
	- add -16 switch
*/
/* version 1.2 - November 1994
	- add -s scaledb option
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH PREP SFS1 UCL
.SH NAME
prep - waveform preparation
.SH SYNOPSIS
.B prep
(-a|-16|-s scaledb) (-p|-d) (-n) (-i item) file
.SH DESCRIPTION
.I prep
is a program to perform standard waveform preparation functions
of AGC, pre-emphasis and de-emphasis or inversion.
The pre-emphasis and de-emphasis coefficient is fixed at 0.96.
Multiple options work together (not "p" and "d"), order of effect is 
emphasis, agc, inversion.  One operation must be selected.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.B -a
Select automatic gain adjustment, Maximum (minimum) of waveform is set to 2047 (-2047).
.TP 11
.B -16
Select automatic gain adjustment, Maximum (minimum) of waveform is set to 16383 (-16383).
.TP 11
.BI -s scaledb
Scale waveform by supplied number of db (+ve = amplify, -ve = attenuate).
.TP 11
.B -p
Select pre-emphasis, lift spectrum by 6db/octave > 100Hz.
.TP 11
.B -d
Select de-emphasis, flatten spectrum by 6db/octave > 100Hz.
.TP 11
.B -n
Select waveform inversion.
.TP 11
.BI -i item
Select input item.
.SH INPUT ITEMS
.IP 1.xx 11
Any speech item.
.IP 2.xx 11
Any excitation item.
.SH VERSION/AUTHOR
1.2 - Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/
#define PROGNAME "prep"
#define PROGVERS "1.2"
char	*progname=PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"			/* header structures */

/* manifest constants */
#define PREEMP	0.96
#define BUFSIZE	4096

/* global data */
struct item_header item;	/* input item header data */
struct item_header newitem;	/* output item header */
short	buff[BUFSIZE];

/* operations selected */
int	agc=0;			/* agc selected */
int	maxval=2047;		/* maximum scaled value */
int	premp=0;		/* pre-emphasis selected */
int	deemp=0;		/* de-emphasis selected */
int	inv=0;			/* inversion selected */
int	doscale=0;		/* do scaling */
double	scalefac=1;

/* pre-emphasis */
void preemphasis(buff,len)
register short	*buff;
int	len;
{
	register int	i;
	register short	last;
	static short	old=0;

	for (i=0;i<len;i++,buff++) {
		last = *buff;
		*buff = (short)(*buff - PREEMP * old);
		old = last;
	}
}

/* de-emphasis */
void deemphasis(buff,len)
register short	*buff;
int	len;
{
	register int	i;
	static short	old=0;

	for (i=0;i<len;i++,buff++) {
		*buff = (short)(*buff+old*PREEMP);
		old = *buff;
	}
}

/* agc */
void autogain(buff,len,max)
register short	*buff;
int	len;
short	max;
{
	register int	i;
	register float	gain=(float)maxval/max;

	for (i=0;i<len;i++,buff++) {
		*buff = (short)(*buff * gain);
	}
}

/* scale */
void scale(buff,len,fac)
register short	*buff;
int	len;
double	fac;
{
	register int	i;

	for (i=0;i<len;i++,buff++) {
		*buff = (short)((double)*buff * fac);
	}
}

/* inversion */
void invert(buff,len)
register short	*buff;
int	len;
{
	register int	i;

	for (i=0;i<len;i++,buff++) {
		*buff = - *buff;
	}
}

/* get absolute maximum value */
int getabsmax(buff,len,max)
register short	*buff;
int		len;
short		max;
{
	register int 	i;

	for (i=0;i<len;i++,buff++) {
		if (abs(*buff) > max) max = abs(*buff);
	}
	return(max);
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int		errflg=0;	/* option error flag */
	int		c;		/* option char */
	int		it;
	char		*ty;
	int		initem=SP_TYPE;
	char		*intype="0";
	char		filename[SFSMAXFILENAME];	
					/* database file name */
	int		fid,ofid;
	int		i,len,agcmax=0;
	char		msg[80];

	/* decode switches */
	while ( (c = getopt(argc,argv,"Iapdni:1:s:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Waveform preparation functions V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'a' :	/* AGC */
			agc++;
			doscale=0;
			break;
		case '1' :	/* AGC16 */
			if (strcmp(optarg,"6")==0) {
				agc++;
				maxval=16383;
			}
			else
				error("illegal option '-1%s'",optarg);
			doscale=0;
			break;
		case 's' :	/* scale in dB */
			doscale=1;
			agc=0;
			scalefac = pow(10.0,atof(optarg)/20);
			break;
		case 'p' :	/* pre-emphasis */
			premp++;
			deemp=0;
			break;
		case 'd' :	/* de-emphasis */
			deemp++;
			premp=0;
			break;
		case 'n' :	/* invert */
			inv++;
			break;
		case 'i' :	/* item spec */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it == SP_TYPE) || (it == LX_TYPE)) {
					initem = it;
					intype = ty;
				}
				else
					error("unsuitable item specification %s",optarg);
			}
			else
				error("illegal item specification %s",optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-a|-16|-s scaledb) (-p|-d) (-n) (-i item) file\n",PROGNAME);

	/* check an option selected */
	if (!agc && !premp && !deemp && !inv && !doscale)
		error("no function selected: -a=agc, -16=agc(16bit) -s dB=scale -p=preemp -d=deemp -n=invert",NULL);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("unable to find file '%s'",filename);
		else
			error("access error on '%s'",filename);
	}

	/* locate input item */
	if (!sfsitem(fid,initem,intype,&item))
		error("unable to find input item in '%s'",filename);

	/* create output item header */
	sfsheader(&newitem,item.datatype,item.floating,
			item.datasize,item.framesize,
			item.frameduration,item.offset,
			item.windowsize,item.overlap,item.lxsync);
	sprintf(newitem.history,"%s(%d.%02d;",
			PROGNAME,
			item.datatype,item.subtype);
	if (premp) strcat(newitem.history,"pre-emphasis,");
	if (deemp) strcat(newitem.history,"de-emphasis,");
	if (agc)   strcat(newitem.history,(maxval==2047)?"agc,":"agc16,");
	if (doscale) {
		sprintf(msg,"scaledb=%g,",20.0*log10(scalefac));
		strcat(newitem.history,msg);
	}
	if (inv)   strcat(newitem.history,"invert;");
	newitem.history[strlen(newitem.history)-1] = ')';

	/* open output channel */
	if ((ofid=sfschannel(filename,&newitem)) < 0)
		error("unable to open output file",NULL);

	/* first pass for agc */
	if (agc) {
		for (i=0;(len=sfsread(fid,i,BUFSIZE,buff)) > 0;i+=len) {

			/* do pre-emphasis */
			if (premp) preemphasis(buff,len);

			/* do de-emphasis */
			if (deemp) deemphasis(buff,len);

			/* get max */
			agcmax = getabsmax(buff,len,agcmax);

		}

		/* re-set internal variables */
		i=0;
		if (premp) preemphasis(&i,1);
		if (deemp) deemphasis(&i,1);
	}

	/* process file */
	for (i=0;(len=sfsread(fid,i,BUFSIZE,buff)) > 0;i+=len) {

		/* do pre-emphasis */
		if (premp) preemphasis(buff,len);

		/* do de-emphasis */
		if (deemp) deemphasis(buff,len);

		/* do agc */
		if (agc) autogain(buff,len,agcmax);

		/* do scale */
		if (doscale) scale(buff,len,scalefac);

		/* do invert */
		if (inv) invert(buff,len);

		/* write out result of processing */
		if (sfswrite(ofid,len,buff) != len)
			error("write error on output file",NULL);
	}

	/* update file */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* ... that's all folks */
	exit(0);
}

