/* coload -- load CO item from coefficients listing produced by colist */

/* M.A.Huckvale */

/* version 1.0 - March 1990 */

/* version 1.1 - March 1996
	- fix deletion of second frame!
	- add offset switch
*/

#define PROGNAME "coload"
#define PROGVERS "1.1"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH COLOAD SFS1 UCL
.SH NAME
coload - load spectral coefficients item from text listing
.SH SYNOPSIS
.B coload (-O offset)
listfile sfsfile
.SH DESCRIPTION
.I coload
creates a new CO item in the file
.I sfsfile
from spectral coefficients listed in the text file
.I listfile.
Input format is that written by 
.I colist,
namely 1 line per frame, with a leading time specified in milliseconds.
.PP
To load parameters into the coefficient item header, use a line before the first
valid data frame with the format:
.br
	"! params=<parameters to load>"
.br
The
.I colist
program may be used to demonstrate the format.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version.
.TP 11
.BI -O offset
Give the loaded item an offset in seconds.
.SH OUTPUT ITEMS
.IP 11 11
Edited CO item.
.SH VERSION/AUTHOR
1.1 - Mark Huckvale
.SH SEE ALSO
colist
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <string.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header coitem;	/* filing system item header for new item */

#define MAXLINE	2048
char	iline[MAXLINE];

/* get number of numbers on a line */
int getnumbernum(str)
char	*str;
{
	int	cnt=0;
	char	*p;

	p=strtok(str," \t\n");
	while (p && *p && (isdigit(*p) || (*p=='-'))) {
		cnt++;
		p = strtok(NULL," \t\n");
	}
	return(cnt);
}

/* get number of coefficients */
int getnumcoeff(ip)
FILE	*ip;
{
	int	nc,ncoeff=0;

	while (fgets(iline,MAXLINE,ip)) if (isdigit(iline[0]) || (iline[0]==' ')) {
		nc = getnumbernum(iline)-1;
		if (nc > ncoeff) 
			ncoeff = nc;
	}
	rewind(ip);
	return(ncoeff);
}

/* get coefficient data from text file */
int getframe(corec,num,ip)
struct co_rec		*corec;		/* returned data */
int			num;		/* number of coefficients */
FILE			*ip;		/* input file */
{
	char		*p;
	int		i;

	/* get next data line */
	do {
		if (fgets(iline,MAXLINE,ip) == NULL) return(1);

		if (strncmp(iline,"! params=",9)==0)
			strncpy(coitem.params,strtok(iline+9,"\n"),sizeof(coitem.params));

	} while (!isdigit(iline[0]) && (iline[0] != ' '));

	/* get time */
	p = strtok(iline," \t\n");
	corec->posn = (int)(0.5+(atof(p)*0.001)/coitem.frameduration);
	corec->flag = 0;
	corec->gain = (float)1.0;
	for (i=0;i<num;i++) 
		corec->data[i]=(float)0.0;

	/* get fixed # coefficients */
	for (i=0;i<num;i++) {
		p = strtok(NULL," \t\n");
		if (p && *p && (isdigit(*p) || (*p=='-')))
			corec->data[i] = (float)atof(p);
		else
			break;
	}

	return(0);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	char		textname[SFSMAXFILENAME];	/* text file name */
	FILE		*ip;		/* text file variable */
	int		ofid;
	/* processing variables */
	struct co_rec	*corec,*lcorec;		/* input coefficient record */
	int		ncoeff;
	int		cnt=0;
	int		i;
	double		offset=0;
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"IO:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Load spectral coefficients V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'O' :	/* offset */
			offset = atof(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}

	/* check for option decoding error */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-O offset) textfile sfsfile",PROGNAME);

	/* get text filename */
	if (optind < argc)
		strcpy(textname,argv[optind]);
	else
		error("no listing file specified",NULL);
	optind++;

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open text file */
	if ((ip=fopen(textname,"r")) == NULL)
		error("unable to open text file %s",textname);

	/* get number of coefficients per frame */
	ncoeff = getnumcoeff(ip);

	/* create CO item header */
	sfsheader(&coitem,CO_TYPE,1,4,
		(sfsstruct[CO_TYPE]+ncoeff*sizeof(float))/4,
		0.0001,offset,0,0,1);
	sprintf(coitem.history,"%s(file=%s)",PROGNAME,textname);

	/* get input buffer */
	if ((lcorec=(struct co_rec *)sfsbuffer(&coitem,1))==NULL)
		error("could not get memory buffer",NULL);
	if ((corec=(struct co_rec *)sfsbuffer(&coitem,1))==NULL)
		error("could not get memory buffer",NULL);

	/* get first frame */
	if (getframe(lcorec,ncoeff,ip)==0) {

		/* open output channel */
		if ((ofid=sfschannel(filename,&coitem)) < 0)
			error("unable to create temporary file",NULL);

		/* load frames from text file */
		while (getframe(corec,ncoeff,ip) == 0) {
			lcorec->size = corec->posn - lcorec->posn;
			if (sfswrite(ofid,1,lcorec) != 1)
				error("write error on temporary file",NULL);
			lcorec->posn = corec->posn;
			lcorec->size = corec->size;
			lcorec->flag = corec->flag;
			lcorec->mix = corec->mix;
			lcorec->gain = corec->gain;
			for (i=0;i<ncoeff;i++)
				lcorec->data[i] = corec->data[i];
			cnt++;
		}

		/* do last frame */
		lcorec->size = 0;
		if (sfswrite(ofid,1,lcorec) != 1)
			error("write error on temporary file",NULL);
		cnt++;
	}

	/* update data file */
	if (cnt == 0)
		error("no legal frames found in %s",textname);
	else if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* that's all folks */
	exit(0);
}


