function spectrogram(fname,item,wtime,otype)
% SPECTROGRAM Calculates a spectrographic representation of a signal
%
%   SPECTROGRAM(fname,item) calculates and stores a wide-band
%   spectrogram from a speech item in an SFS file fname, storing
%   the spectrogram as a DISPLAY item in the file.
%
%   SPECTROGRAM(fname,item,wtime) calculates a spectrogram using
%   analysis windows of size wtime seconds (0.003=wide band, 0.020
%   =narrow-band).
%
%   SPECTROGRAM(fname,item,wtime,'coeff') calculates a spectrogram
%   as above but stores the results as a COEFF item in the file.

%
% set defaults
%
if (nargin < 4) otype='display'; end;
if (nargin < 3) wtime=0.003; end;
%
% get item from SFS file (h=header, d=data set)
%
[h,d]=sfsgetitem(fname,item);
%
% get FFT size
%
n=256;
while ((n*h.frameduration) <= wtime) n=2*n; end;
%
% get hamming window
%
len=round(wtime/h.frameduration);
w=hamming(len);
%
% perform spectrogram calculation
%
y = filter([1, -0.95], 1, d);       % pre-emphasis
[b,f,t]=specgram(y,n,1.0/h.frameduration,w);
numf=length(t);
%
% convert to dB (and rotate so time goes down rows)
%
e=20.0*log10(abs(b)'+eps);
%
% build SFS item to hold spectrogram
%
if strcmp(otype,'coeff')
    oh=sfsnewitem(11,h.frameduration,h.offset,n/2,numf);
    oh.history=sprintf('matlab/CO(%d.%02d;script=spectrogram,wtime=%g)',h.datatype,h.subtype,wtime);
    od=[round(t/h.frameduration) len*ones(numf,1) zeros(numf,1) zeros(numf,1) zeros(numf,1) e];
else
    oh=sfsnewitem(9,h.frameduration,h.offset,n/2,numf);
    oh.history=sprintf('matlab/DI(%d.%02d;script=spectrogram,wtime=%g)',h.datatype,h.subtype,wtime);
    e=max(16*(e-max(max(e))+50)/50,zeros(size(e)));
    od=[round(t/h.frameduration) len*ones(numf,1) e];
end
%
% store in SFS file
%
sfsputitem(fname,oh,od);
