/* win32raw -- DIG routines for WIN32 graphics */

/* Mark Huckvale - University College London */

/* version 1.0 - April 2000 */

/* This version is written for Microsoft Visual C/C++ V5
   It does not require use of digstart/digend, instead
   the code should be initialised using w32raw_init()
 */

#include "SFSCONFG.h"
#ifdef	EVW

#define NEWGSCALE 1
#define GREYGAMMA	1.0

#include <windows.h>
#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <time.h>
#include <string.h>
#include "dig.h"
#include "digdata.h"
#include "digtable.h"

/* local storage of parameters */
struct {
	HDC	hDC;		/* printer device context */
	int	mode;		/* system modes */
	int	xoff,yoff;	/* margin */
	int	left,top,right,bottom;	/* page offset */
	int	printing;	/* if printing:
					- rotate 90degrees clockwise to get landscape
					- use dark colours
					- use larger maxels for grey scale
				*/
	int	xdpmm;		/* x pixels per mm */
	int	ydpmm;		/* y pixels per mm */
	int	lwidth;		/* line width = 1/8mm */
	int	gwidth;		/* grey maxel width = 1/4mm */
	HPEN	hpen[16];	/* grey scale pens */
	int	rop;		/* raster operation, 0=copy, 1=xor */
	COLORREF lastcolour;
} digDEVW;

#define W32RAW_INIT	1
#define LEFT_MARGIN	10	/* margins in mm */
#define RIGHT_MARGIN	10
#define TOP_MARGIN	15
#define BOTTOM_MARGIN	10

/* VGA colours */
#define NUM_COLOUR	32
struct rgb_rec {
	int	entry;
	int	r,g,b;
} win32rgbtab[NUM_COLOUR]={
	{ 0,250,250,250},	/* 0 = background */
	{ 0,200, 50, 50},	/* 1 */
	{ 0,100, 75,100},	/* 2 */
	{ 0,100, 75, 50},	/* 3 */
	{ 0,150,200,  0},	/* 4 */
	{ 0, 25, 25,100},	/* 5 = was red - mapped to dark grey/blue */
	{ 0, 50,200,100},	/* 6 */
	{ 0, 50,100,200},	/* 7 */
	{ 0,150,150,  0},	/* 8 */
	{ 0, 50, 50, 50},	/* 9 = was white - mapped to dark grey */
	{ 0,200,100,  0},	/* 10 */
	{ 0,100,  0,100},	/* 11 */
	{ 0,  0,100,100},	/* 12 */
	{ 0, 25,100, 25},	/* 13 */
	{ 0, 50,100,200},	/* 14 */
	{ 0,  0,  0,  0},	/* 15 = foreground */
	{ 0,250,250,250},	/* 16 */
	{ 0,248,248,248},	/* 17 */
	{ 0,244,244,244},	/* 18 */
	{ 0,238,238,238},	/* 19 */
	{ 0,230,230,230},	/* 20 */
	{ 0,220,220,220},	/* 21 */
 	{ 0,208,208,208},	/* 22 */
	{ 0,194,194,194},	/* 23 */
	{ 0,178,178,178},	/* 24 */
	{ 0,160,160,160},	/* 25 */
	{ 0,140,140,140},	/* 26 */
	{ 0,118,118,118},	/* 27 */
	{ 0, 94, 94, 94},	/* 28 */
	{ 0, 68, 68, 68},	/* 29 */
	{ 0, 40, 40, 40},	/* 30 */
	{ 0, 10, 10, 10},	/* 31 */
};

/* initialisation */
int w32raw_init(HDC hDC,int left,int top,int right,int bottom,int printing)
{
	TEXTMETRIC	tm;
	double rxscale,ryscale;

	if (digDEVW.mode & W32RAW_INIT) return(0);

	digDEVW.hDC = hDC;
	digDEVW.xdpmm = GetDeviceCaps(digDEVW.hDC,LOGPIXELSX)/25;
	digDEVW.ydpmm = GetDeviceCaps(digDEVW.hDC,LOGPIXELSY)/25;
	digDEVW.printing = printing;
	digDEVW.left = left;
	digDEVW.top = top;
	digDEVW.right = right;
	digDEVW.bottom = bottom;
	if (printing) {
		digDEVW.lwidth = digDEVW.xdpmm/6;	/* 150 per inch */
		digDEVW.gwidth = digDEVW.xdpmm/3;	/* 75 per inch */
		if (digDEVW.gwidth > 6) digDEVW.gwidth=6;
		if (digDEVW.gwidth < 4) digDEVW.gwidth=4;
		digDEVW.xoff = BOTTOM_MARGIN*digDEVW.xdpmm;
		digDEVW.yoff = LEFT_MARGIN*digDEVW.ydpmm;
	}
	else {
		digDEVW.lwidth = 1;
		digDEVW.gwidth = 1;
		digDEVW.xoff = 0;
		digDEVW.yoff = bottom;
	}
	digDEVW.rop=0;

	/* switch drawing over to WIN32RAW */
	for (digdata.select=0;digdata.select < digtablesize;digdata.select++)
		if (digtable[digdata.select].devcode=='W') break;

	if (digtable[digdata.select].devcode!='W') exit(0);

	/* set primary parameters of output device */
	digdata.subtype = printing;
	GetTextMetrics(digDEVW.hDC,&tm);
	if (printing) {
		digdata.digdev.nhoriz  = bottom-top+1-(LEFT_MARGIN+RIGHT_MARGIN)*digDEVW.ydpmm;
		digdata.digdev.nvert   = right-left+1-(TOP_MARGIN+BOTTOM_MARGIN)*digDEVW.xdpmm;
	}
	else {
		digdata.digdev.nvert   = bottom-top+1;
		digdata.digdev.nhoriz  = right-left+1;
	}
#if NEWGSCALE
	digdata.digdev.greylevels = 256;
#else
	if (digDEVW.gwidth==6)
		digdata.digdev.greylevels = 32;
	else
		digdata.digdev.greylevels = 16;
#endif
	digdata.digdev.greypixwidth = digDEVW.gwidth;
	digdata.digdev.greypixheight = digDEVW.gwidth;
	digdata.digdev.aspect = 1.0;
	digdata.digdev.chwidth = tm.tmAveCharWidth;
	digdata.digdev.chheight = tm.tmHeight;

	/* move commonly used details up to old positions */
	digdata.nhoriz        = digdata.digdev.nhoriz;
	digdata.nvert         = digdata.digdev.nvert;
	digdata.aspect        = digdata.digdev.aspect;
	digdata.chwidth       = digdata.digdev.chwidth;
	digdata.chheight      = digdata.digdev.chheight;
	digdata.cheightmin    = digdata.chheight/2;

	/* fix scaling as 0,0 to 1,1 */
	rxscale = 1.0/(digdata.nhoriz - 1);
	ryscale = 1.0/(digdata.nvert - 1);
	digdata.xscale = (float)(1.0/rxscale);
	digdata.yscale = (float)(1.0/ryscale);
	digdata.horiz = (float)(rxscale * (digdata.nhoriz - 1));
	digdata.vert = (float)(ryscale * (digdata.nvert - 1));
	digdata.xorig = 0.0;
	digdata.yorig = 0.0;
	digdata.clipxmin   = digdata.clipymin = 0;
	digdata.clipxmax   = digdata.nhoriz-1;
	digdata.clipymax   = digdata.nvert-1;
	digdata.setup = 1;

    /* make 16 pens for grey scale */
    if (printing) {
		digDEVW.hpen[0] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(250,250,250));
		digDEVW.hpen[1] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(234,234,234));
		digDEVW.hpen[2] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(218,218,218));
		digDEVW.hpen[3] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(202,202,202));
		digDEVW.hpen[4] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(186,186,186));
		digDEVW.hpen[5] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(170,170,170));
		digDEVW.hpen[6] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(154,154,154));
		digDEVW.hpen[7] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(138,138,138));
		digDEVW.hpen[8] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(122,122,122));
		digDEVW.hpen[9] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(106,106,106));
		digDEVW.hpen[10] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(90,90,90));
		digDEVW.hpen[11] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(74,74,74));
		digDEVW.hpen[12] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(58,58,58));
		digDEVW.hpen[13] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(42,42,42));
		digDEVW.hpen[14] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(26,26,26));
		digDEVW.hpen[15] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(10,10,10));
	}
	else {
		digDEVW.hpen[0] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(250,250,250));
		digDEVW.hpen[1] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(248,248,248));
		digDEVW.hpen[2] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(244,244,244));
		digDEVW.hpen[3] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(238,238,238));
		digDEVW.hpen[4] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(230,230,230));
		digDEVW.hpen[5] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(220,220,220));
		digDEVW.hpen[6] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(208,208,208));
		digDEVW.hpen[7] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(194,194,194));
		digDEVW.hpen[8] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(178,178,178));
		digDEVW.hpen[9] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(160,160,160));
		digDEVW.hpen[10] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(140,140,140));
		digDEVW.hpen[11] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(118,118,118));
		digDEVW.hpen[12] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(94,94,94));
		digDEVW.hpen[13] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(68,68,68));
		digDEVW.hpen[14] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(40,40,40));
		digDEVW.hpen[15] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(10,10,10));
	}

	digDEVW.mode |= W32RAW_INIT;

	return(0);
}

float w32raw_chheight()
{
	return(digdata.chheight / digdata.yscale);
}
float w32raw_chwidth()
{
	return(digdata.chwidth / digdata.xscale);
}
int w32raw_textlen(char *str)
{
	TEXTMETRIC	tm;
	SIZE	sz;
	GetTextMetrics(digDEVW.hDC,&tm);
	GetTextExtentPoint32(digDEVW.hDC,str,strlen(str),&sz);
	return(sz.cx);
}

/* forward declarations */
static void w32raw_pen_colour();
static void w32raw_brush_colour();

/* get graphics parameters */
int w32raw_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{
	return(0);
}

/* open output */
int w32raw_open()
{
	return(0);
}

/* close file */
void w32raw_close()
{
	int	i;
	if (digDEVW.mode) {
		for (i=0;i<16;i++) DeleteObject(digDEVW.hpen[i]);
	}
	digDEVW.mode=0;
}

void w32raw_inter()
{
	w32raw_close();
}

/* clear screen */
int w32raw_clear()
{
	return(0);
}

/* set up pen colour from bundle */
static void w32raw_pen_colour(bundle)
int	bundle;
{
	static int oldbundle = -1;
	static int oldrop = -1;
	HPEN	pen,oldpen;
	LOGBRUSH lb;
	int	dash=0,width=0;

	if ((bundle==oldbundle) && (digDEVW.rop==oldrop)) return;
	oldbundle = bundle;

    if (bundle < 35)
		dash = PS_SOLID;
    else if (bundle < 50)
		dash = PS_DASH;
    else if (bundle < 65)
		dash = PS_DOT;
    else if (bundle < 80)
		dash = PS_DASHDOT;
    else
		dash = PS_DASHDOTDOT;

	/* bundles are 0=background, 1-100 various colours/line styles, >100 greyscale */
	if (bundle==0) {
		pen = GetStockObject(WHITE_PEN);
		digDEVW.lastcolour = RGB(255,255,255);
	}
	else if (bundle < 5) {
		digDEVW.lastcolour=RGB(win32rgbtab[bundle].r,win32rgbtab[bundle].g,win32rgbtab[bundle].b);
		pen = CreatePen(dash,2*digDEVW.lwidth,digDEVW.lastcolour);
	}
	else if (bundle >= 100) {
		digDEVW.lastcolour=RGB(win32rgbtab[bundle-84].r,win32rgbtab[bundle-84].g,win32rgbtab[bundle-84].b);
		pen = CreatePen(dash,digDEVW.lwidth,digDEVW.lastcolour);
	}
	else {
		bundle = bundle % 15;
		if (bundle==0) {
			pen = GetStockObject(BLACK_PEN);
			digDEVW.lastcolour=RGB(0,0,0);
		}
		else {
			digDEVW.lastcolour = RGB(win32rgbtab[bundle].r,win32rgbtab[bundle].g,win32rgbtab[bundle].b);
			lb.lbStyle = BS_SOLID;
			lb.lbColor = digDEVW.lastcolour;
			lb.lbHatch = 0;
			pen = ExtCreatePen(PS_GEOMETRIC|dash,digDEVW.lwidth,&lb,0,NULL);
		}
	}

	oldpen = SelectObject(digDEVW.hDC,pen);

	if (digDEVW.rop!=oldrop) {
		if (digDEVW.rop)
			SetROP2(digDEVW.hDC,R2_XORPEN);
		else
			SetROP2(digDEVW.hDC,R2_COPYPEN);
		oldrop = digDEVW.rop;
	}

	DeleteObject(oldpen);
}

/* set up brush colour from bundle */
static void w32raw_brush_colour(bundle)
int	bundle;
{
	static int oldbundle = -1;
	HBRUSH	brush,oldbrush;

	if (bundle==oldbundle) return;
	oldbundle = bundle;

	brush = (bundle==0) ? GetStockObject(WHITE_BRUSH) : GetStockObject(BLACK_BRUSH);

	oldbrush = SelectObject(digDEVW.hDC,brush);

	DeleteObject(oldbrush);
}

/* change line/text raster operation */
int w32raw_rasterop(mode)
int	mode;
{
	int	ret;

	/* set new raster op 0=write, 1=xor */
	ret=digDEVW.rop;
	digDEVW.rop=mode;
	return(ret);
}

/* plot poly-line */
void w32raw_polyline(bundle,buff,len)
int bundle;
short *buff;
int	len;
{
	int     x,y;

	/* set up plot colour */
	w32raw_pen_colour(bundle);

	if (digDEVW.printing) {
		/* move to first point */
		y = digDEVW.xoff + *buff++;
		x = digDEVW.yoff + *buff++;
		len -= (int32)2;
		MoveToEx(digDEVW.hDC,x,y,NULL);

		/* plot lines */
		while (len > (int32)0) {
			y = digDEVW.xoff + *buff++;
			x = digDEVW.yoff + *buff++;
			len -= (int32)2;
			LineTo(digDEVW.hDC, x, y);
		}
		SetPixel(digDEVW.hDC,x,y,digDEVW.lastcolour);
	}
	else {
		/* move to first point */
		x = digDEVW.xoff + *buff++;
		y = digDEVW.yoff - *buff++;
		len-=(int32)2;
		MoveToEx(digDEVW.hDC,x,y,NULL);

		/* plot lines */
		while (len > (int32)0) {
			x = digDEVW.xoff + *buff++;
			y = digDEVW.yoff - *buff++;
			len -= (int32)2;
			LineTo(digDEVW.hDC, x, y);
		}
		SetPixel(digDEVW.hDC,x,y,digDEVW.lastcolour);
	}
}

/* draw text */
void w32raw_text(bundle,x,y,str)
int bundle;
int x;
int y;
char *str;
{
	WCHAR	wstr[1024];

	/* set up text colour */
	if (bundle==0)
		SetTextColor(digDEVW.hDC,RGB(255,255,255));
	else if (bundle < 5)
		SetTextColor(digDEVW.hDC,RGB(win32rgbtab[bundle].r,win32rgbtab[bundle].g,win32rgbtab[bundle].b));
	else if (bundle >= 100)
		SetTextColor(digDEVW.hDC,RGB(win32rgbtab[bundle-84].r,win32rgbtab[bundle-84].g,win32rgbtab[bundle-84].b));
	else {
		bundle = bundle % 15;
		if (bundle==0)
			SetTextColor(digDEVW.hDC,RGB(0,0,0));
		else
			SetTextColor(digDEVW.hDC,RGB(win32rgbtab[bundle].r,win32rgbtab[bundle].g,win32rgbtab[bundle].b));
	}

	/* draw string */
	SetBkMode(digDEVW.hDC,TRANSPARENT);
	MultiByteToWideChar(CP_UTF8,0,str,-1,wstr,1024);
	if (digDEVW.printing)
		TextOutW(digDEVW.hDC,digDEVW.yoff+y+digdata.chheight,digDEVW.xoff+x,wstr,wcslen(wstr));
	else
		TextOutW(digDEVW.hDC,digDEVW.xoff+x,digDEVW.yoff-y-digdata.chheight,wstr,wcslen(wstr));
	SetBkMode(digDEVW.hDC,OPAQUE);

}

/* draw a filled rectangle */
void w32raw_fillrect(bundle,x1,y1,x2,y2)
int bundle;
int x1;
int y1;
int x2;
int y2;
{
	w32raw_pen_colour(bundle);
	w32raw_brush_colour(bundle);

	if (digDEVW.printing)
		Rectangle(digDEVW.hDC,digDEVW.yoff+y1,digDEVW.xoff+x1,digDEVW.yoff+y2,digDEVW.xoff+x2);
	else
		Rectangle(digDEVW.hDC,digDEVW.xoff+x1,digDEVW.yoff-y1,digDEVW.xoff+x2,digDEVW.yoff-y2);

}

/* gamma correction */
double gamma(double val,double gamma)
{
	return(pow(val,gamma));
}

/* grey-level display */
void w32raw_gscale(int x,int y,int height,unsigned char *buff)
{
	HANDLE		oldobj,pold;
	int		i,lasty,j,t;
	unsigned char 	*p,lastc;
	int		pw=digDEVW.gwidth;
	LOGPEN	logpen;

	if (digDEVW.printing) {
#if NEWGSCALE
		BITMAPINFO *lpBitsInfo;
		RGBQUAD		*pRGB;
		unsigned char *pBits;

		lpBitsInfo=(BITMAPINFO *)calloc(1,sizeof(BITMAPINFOHEADER)+digdata.digdev.greylevels*sizeof(RGBQUAD)+height);
		pBits = (unsigned char *)lpBitsInfo+sizeof(BITMAPINFOHEADER)+digdata.digdev.greylevels*sizeof(RGBQUAD);

		lpBitsInfo->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
		lpBitsInfo->bmiHeader.biWidth = height;
		lpBitsInfo->bmiHeader.biHeight = 1;
		lpBitsInfo->bmiHeader.biPlanes = 1;
		lpBitsInfo->bmiHeader.biBitCount = 8;
		lpBitsInfo->bmiHeader.biCompression = BI_RGB;
		lpBitsInfo->bmiHeader.biSizeImage  = height;
		lpBitsInfo->bmiHeader.biXPelsPerMeter = 300;
		lpBitsInfo->bmiHeader.biYPelsPerMeter = 300;
		lpBitsInfo->bmiHeader.biClrUsed = digdata.digdev.greylevels;
		lpBitsInfo->bmiHeader.biClrImportant = digdata.digdev.greylevels;

		pRGB=lpBitsInfo->bmiColors;

		for (i=0;i<digdata.digdev.greylevels;i++) {
			pRGB[i].rgbBlue = pRGB[i].rgbGreen = pRGB[i].rgbRed = (int)(digdata.digdev.greylevels*gamma((255-i)/(float)(digdata.digdev.greylevels),GREYGAMMA));
		}

		/* invert buffer */
		for (i=0;i<height;i++) pBits[i]=buff[height-1-i];

		StretchDIBits(digDEVW.hDC,
			digDEVW.xoff+y,
			digDEVW.yoff+x,
			digDEVW.gwidth*height,
			digDEVW.gwidth,
			0,
			0,
			height,
			1,
			pBits,
			lpBitsInfo,
			DIB_RGB_COLORS,
			SRCCOPY);

		free(lpBitsInfo);
#else
		HPEN mypen[16];

		mypen[0] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(250,250,250));
		mypen[1] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(234,234,234));
		mypen[2] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(218,218,218));
		mypen[3] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(202,202,202));
		mypen[4] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(186,186,186));
		mypen[5] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(170,170,170));
		mypen[6] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(154,154,154));
		mypen[7] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(138,138,138));
		mypen[8] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(122,122,122));
		mypen[9] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(106,106,106));
		mypen[10] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(90,90,90));
		mypen[11] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(74,74,74));
		mypen[12] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(58,58,58));
		mypen[13] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(42,42,42));
		mypen[14] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(26,26,26));
		mypen[15] = CreatePen(PS_SOLID,digDEVW.gwidth,RGB(10,10,10));

		/* get ready */
		if ((oldobj = SelectObject(digDEVW.hDC,mypen[0]))==NULL) {
			error("SelectObject(1) fails!");
		}
		x += digDEVW.yoff+digDEVW.gwidth/2;

		/* draw coloured lines matching greys */
		p = buff;
		lastc = *p;
		lasty = digDEVW.xoff + y + pw*height - 1;
		MoveToEx(digDEVW.hDC,lasty,x,NULL);
		for (i = 1; i < height; i++, p++) {
			if (*p != lastc) {
				if ((lastc<0)||(lastc>15)) {
					error("Bad pen number in grey scale()");
				}
				if ((pold=SelectObject(digDEVW.hDC,mypen[lastc]))==NULL) {
					error("SelectObject(2) fails!");
				}
				for (j=0;j<16;j++) if (pold==mypen[j]) break;
				if (j==16) {
					error("Select Object returns other than pen!");
				}
				LineTo(digDEVW.hDC,lasty-pw*i,x);
				lastc = *p;
			}
		}
		if ((lastc<0)||(lastc>15)) {
			error("Bad pen number in grey scale()");
		}
		if (SelectObject(digDEVW.hDC,mypen[lastc])==NULL) {
			/* error("SelectObject(3) fails!"); */
		}
		LineTo(digDEVW.hDC,lasty-pw*(height-1),x);
		SelectObject(digDEVW.hDC,oldobj);

		for (i=0;i<16;i++) DeleteObject(mypen[i]);
#endif
	}
	else {
#if NEWGSCALE

#if 1
		BITMAPINFO *lpBitsInfo;
		RGBQUAD		*pRGB;
		unsigned char *pBits;

		lpBitsInfo=(BITMAPINFO *)calloc(1,sizeof(BITMAPINFOHEADER)+digdata.digdev.greylevels*sizeof(RGBQUAD)+4*height);
		pBits = (unsigned char *)lpBitsInfo+sizeof(BITMAPINFOHEADER)+digdata.digdev.greylevels*sizeof(RGBQUAD);

		lpBitsInfo->bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
		lpBitsInfo->bmiHeader.biWidth = 1;
		lpBitsInfo->bmiHeader.biHeight = height;
		lpBitsInfo->bmiHeader.biPlanes = 1;
		lpBitsInfo->bmiHeader.biBitCount = 8;
		lpBitsInfo->bmiHeader.biCompression = BI_RGB;
		lpBitsInfo->bmiHeader.biSizeImage  = height;
		lpBitsInfo->bmiHeader.biXPelsPerMeter = 300;
		lpBitsInfo->bmiHeader.biYPelsPerMeter = 300;
		lpBitsInfo->bmiHeader.biClrUsed = digdata.digdev.greylevels;
		lpBitsInfo->bmiHeader.biClrImportant = digdata.digdev.greylevels;

		pRGB=lpBitsInfo->bmiColors;

		for (i=0;i<digdata.digdev.greylevels;i++) {
			pRGB[i].rgbBlue = pRGB[i].rgbGreen = pRGB[i].rgbRed = (int)(digdata.digdev.greylevels*gamma((255-i)/(float)(digdata.digdev.greylevels),GREYGAMMA));
		}

		/* invert buffer */
		for (i=0;i<height;i++) pBits[4*i]=buff[height-1-i];

		StretchDIBits(digDEVW.hDC,
			digDEVW.xoff+x+1,
			digDEVW.yoff-y-height+1,
			1,
			height,
			0,
			0,
			1,
			height,
			pBits,
			lpBitsInfo,
			DIB_RGB_COLORS,
			SRCCOPY);

		free(lpBitsInfo);

#else
		/* slow but safe version */
		for (i=0;i<height;i++)
			SetPixel(digDEVW.hDC,digDEVW.xoff+x,digDEVW.yoff-y-height+1+i,RGB(255-buff[i],255-buff[i],255-buff[i]));

#endif

#else
		/* get ready */
		oldobj = SelectObject(digDEVW.hDC,digDEVW.hpen[0]);
		x += digDEVW.xoff+1;

		/* draw coloured lines matching greys */
		p = buff;
		lastc = *p;
		lasty = digDEVW.yoff - y - pw*height +1;
		MoveToEx(digDEVW.hDC,x,lasty,NULL);
		for (i = 1; i < height; i++, p++) {
			if (*p != lastc) {
				SelectObject(digDEVW.hDC,digDEVW.hpen[lastc]);
				LineTo(digDEVW.hDC,x,lasty+pw*i);
				lastc = *p;
			}
		}
		SelectObject(digDEVW.hDC,digDEVW.hpen[lastc]);
		LineTo(digDEVW.hDC,x,lasty+pw*(height-1));
		GetObject(digDEVW.hpen[lastc],sizeof(LOGPEN),&logpen);
		SetPixel(digDEVW.hDC,x,lasty+pw*(height-1),logpen.lopnColor);
		SelectObject(digDEVW.hDC,oldobj);
#endif
	}
}

/* wait for key routine - NULL don't wait */
void w32raw_pause()
{
}
#endif
