/* elliptic - elliptic filter design and implementation */

/* Mark Huckvale - University College London */

/* based on:

	CEPHES library
	Moshier, Methods and Programs for Mathematical
	Functions, Prentice-Hall, 1989.

*/

/* version 1.0 - June 2013 */

#define DEBUG 1

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#define _USE_MATH_DEFINES
#include <math.h>
#include <malloc.h>
#include <memory.h>
#include "filter.h"

double C1 = 1.3862943611198906188E0; /* log(4) */
double MACHEP =  1.11022302462515654042E-16;   /* 2**-53 */
double MAXNUM =  1.79769313486231570815E308;    /* 2**1024*(1-MACHEP) */
double PIO2   =  1.57079632679489661923;       /* pi/2 */

typedef struct { double r; double i; } cmplx;

static void cdiv( cmplx *a, cmplx *b, cmplx *c )
{
	double y, p, q, w;

	y = a->r * a->r  +  a->i * a->i;
	p = b->r * a->r  +  b->i * a->i;
	q = b->i * a->r  -  b->r * a->i;

	if( y < 1.0 ) {
		w = MAXNUM * y;
		if( (fabs(p) > w) || (fabs(q) > w) || (y == 0.0) ) {
			c->r = MAXNUM;
			c->i = MAXNUM;
#if DEBUG
			fprintf(stderr,"math error in cdiv\n");
#endif
			return;
		}
	}
	c->r = p/y;
	c->i = q/y;
}


/* complete elliptic integral of the second kind */
static double ellipke(double m)
{
	double	a0 = 1;
	double	b0 = sqrt(1-m);
	double	s0 = m;
	double	i1 = 0;
	double	mm = 1;
	double	a1,b1,c1,w1;

	while (mm > 1.0E-10) {
		a1 = (a0+b0)/2;
		b1 = sqrt(a0*b0);
		c1 = (a0-b0)/2;
		i1 = i1 + 1;
		w1 = pow(2.0,i1)*c1*c1;
		mm = w1;
		s0 = s0 + w1;
		a0 = a1;
		b0 = b1;
	}
	return M_PI/(2*a1);
}

/* find the minimum order elliptic filter */
static int findelliporder(double WA,double rp,double rs)
{
	double epsilon = sqrt(pow(10.0,(0.1*rp))-1);
	double k1 = epsilon/sqrt(pow(10.0,(0.1*rs))-1);
	double k = 1/WA;
	double capk11 = ellipke(k*k);
	double capk12 = ellipke(1-k*k);
	double capk21 = ellipke(k1*k1);
	double capk22 = ellipke(1-k1*k1);

#if DEBUG
	printf("WA=%g rp=%g rs=%g epsilon=%g k1=%g k=%g cap=%g,%g,%g,%g\n",WA,rp,rs,epsilon,k1,k,capk11,capk12,capk21,capk22);
#endif
	return (int)ceil((capk11*capk22)/(capk12*capk21));
}

/* polynomial evaluation */
static double polevl( double x, double *coef, int N )
{
	double *p=coef;
	int		i=N;
	double ans;

	ans = *p++;
	do
		ans = ans * x  +  *p++;
	while( --i );

	return( ans );
}
static double p1evl( double x, double *coef, int N )
{
	double *p=coef;
	int		i=N-1;
	double ans;

	ans = x + *p++;
	do
		ans = ans * x  + *p++;
	while( --i );

	return( ans );
}

/* Complete elliptic integral of the first kind */
static double ellpk(double x)
{
static double P[] =
{
 1.37982864606273237150E-4,
 2.28025724005875567385E-3,
 7.97404013220415179367E-3,
 9.85821379021226008714E-3,
 6.87489687449949877925E-3,
 6.18901033637687613229E-3,
 8.79078273952743772254E-3,
 1.49380448916805252718E-2,
 3.08851465246711995998E-2,
 9.65735902811690126535E-2,
 1.38629436111989062502E0
};

static double Q[] =
{
 2.94078955048598507511E-5,
 9.14184723865917226571E-4,
 5.94058303753167793257E-3,
 1.54850516649762399335E-2,
 2.39089602715924892727E-2,
 3.01204715227604046988E-2,
 3.73774314173823228969E-2,
 4.88280347570998239232E-2,
 7.03124996963957469739E-2,
 1.24999999999870820058E-1,
 4.99999999999999999821E-1
};


	if( (x < 0.0) || (x > 1.0) ) {
		fprintf(stderr,"math error in ellpk\n");
		return( 0.0 );
	}

	if( x > MACHEP ) {
		return( polevl(x,P,10) - log(x) * polevl(x,Q,10) );
	}
	else {
		if( x == 0.0 ) {
#if DEBUG
			fprintf(stderr,"math error in ellpk\n");
#endif
			return( MAXNUM );
		}
		else {
			return( C1 - 0.5 * log(x) );
		}
	}
}

/* Complete elliptic integral of the second kind */
static double ellpe(double x)
{
static double P[] = {
  1.53552577301013293365E-4,
  2.50888492163602060990E-3,
  8.68786816565889628429E-3,
  1.07350949056076193403E-2,
  7.77395492516787092951E-3,
  7.58395289413514708519E-3,
  1.15688436810574127319E-2,
  2.18317996015557253103E-2,
  5.68051945617860553470E-2,
  4.43147180560990850618E-1,
  1.00000000000000000299E0
};
static double Q[] = {
  3.27954898576485872656E-5,
  1.00962792679356715133E-3,
  6.50609489976927491433E-3,
  1.68862163993311317300E-2,
  2.61769742454493659583E-2,
  3.34833904888224918614E-2,
  4.27180926518931511717E-2,
  5.85936634471101055642E-2,
  9.37499997197644278445E-2,
  2.49999999999888314361E-1
};

	if( (x <= 0.0) || (x > 1.0) ) {
		if( x == 0.0 ) return( 1.0 );
#if DEBUG
		fprintf(stderr,"math error in ellpe\n");
#endif
		return( 0.0 );
	}
	return( polevl(x,P,10) - log(x) * (x * polevl(x,Q,9)) );
}

/* Incomplete elliptic integral of the first kind */
static double ellik( double phi, double m )
{
	double a, b, c, e, temp, t, K;
	int d, mod, sign, npio2;

	if( m == 0.0 ) return( phi );

	a = 1.0 - m;
	if( a == 0.0 ) {
		if( fabs(phi) >= PIO2 ) {
#ifdef DEBUG
			fprintf(stderr,"math error in ellik\n");
#endif
			return( MAXNUM );
		}
		return(  log(  tan( (PIO2 + phi)/2.0 )  )   );
	}
	npio2 = (int)floor( phi/PIO2 );
	if( npio2 & 1 ) npio2 += 1;
	if( npio2 ) {
		K = ellpk( a );
		phi = phi - npio2 * PIO2;
	}
	else
		K = 0.0;
	if( phi < 0.0 ) {
		phi = -phi;
		sign = -1;
	}
	else
		sign = 0;
	b = sqrt(a);
	t = tan( phi );
	if( fabs(t) > 10.0 ) {
		/* Transform the amplitude */
		e = 1.0/(b*t);
		/* ... but avoid multiple recursions.  */
		if( fabs(e) < 10.0 ) {
			e = atan(e);
			if( npio2 == 0 ) K = ellpk( a );
			temp = K - ellik( e, m );
			goto done;
		}
	}
	a = 1.0;
	c = sqrt(m);
	d = 1;
	mod = 0;

	while( fabs(c/a) > MACHEP ) {
		temp = b/a;
		phi = phi + atan(t*temp) + mod * M_PI;
		mod = (int)((phi + PIO2)/M_PI);
		t = t * ( 1.0 + temp )/( 1.0 - temp * t * t );
		c = ( a - b )/2.0;
		temp = sqrt( a * b );
		a = ( a + b )/2.0;
		b = temp;
		d += d;
	}

	temp = (atan(t) + mod * M_PI)/(d * a);

done:
	if( sign < 0 )
		temp = -temp;
	temp += npio2 * K;
	return( temp );
}

/* Jacobian Elliptic Function */
static int ellpj( double u, double m, double *sn, double *cn, double *dn, double *ph )
{
	double ai, b, phi, t, twon;
	double a[9], c[9];
	int i;

	/* Check for special cases */
	if( m < 0.0 || m > 1.0 ) {
#if DEBUG
		fprintf(stderr,"math error in ellpj\n" );
#endif
		*sn = 0.0;
		*cn = 0.0;
		*ph = 0.0;
		*dn = 0.0;
		return(-1);
	}
	if( m < 1.0e-9 ) {
		t = sin(u);
		b = cos(u);
		ai = 0.25 * m * (u - t*b);
		*sn = t - ai*b;
		*cn = b + ai*t;
		*ph = u - ai;
		*dn = 1.0 - 0.5*m*t*t;
		return(0);
	}
	if( m >= 0.9999999999 ) {
		ai = 0.25 * (1.0-m);
		b = cosh(u);
		t = tanh(u);
		phi = 1.0/b;
		twon = b * sinh(u);
		*sn = t + ai * (twon - u)/(b*b);
		*ph = 2.0*atan(exp(u)) - PIO2 + ai*(twon - u)/b;
		ai *= t * phi;
		*cn = phi - ai * (twon - u);
		*dn = phi + ai * (twon + u);
		return(0);
	}

	/*	A. G. M. scale		*/
	a[0] = 1.0;
	b = sqrt(1.0 - m);
	c[0] = sqrt(m);
	twon = 1.0;
	i = 0;

	while( fabs(c[i]/a[i]) > MACHEP ) {
		if( i > 7 ) {
#if DEBUG
			fprintf(stderr,"math error in ellpj\n" );
#endif
			goto done;
		}
		ai = a[i];
		++i;
		c[i] = ( ai - b )/2.0;
		t = sqrt( ai * b );
		a[i] = ( ai + b )/2.0;
		b = t;
		twon *= 2.0;
	}

done:
	/* backward recurrence */
	phi = twon * a[i] * u;
	do {
		t = c[i] * sin(phi) / a[i];
		b = phi;
		phi = (asin(t) + phi)/2.0;
	} while( --i );

	*sn = sin(phi);
	t = cos(phi);
	*cn = t;
	*dn = t/cos(phi-b);
	*ph = phi;
	return(0);
}

/* Find parameter corresponding to given nome by expansion in theta functions */
static double cay(double q)
{
	double a=1, b=1, p=q, r=1;
	double t1, t2;

	do {
		r *= p;
		a += 2.0 * r;
		t1 = fabs( r/a );

		r *= p;
		b += r;
		p *= q;
		t2 = fabs( r/b );
		if ( t2 > t1 ) t1 = t2;
	} while( t1 > MACHEP );

	a = b/a;
	a = 4.0 * sqrt(q) * a * a;	/* see above formulas, solved for m */
	return(a);
}

#define ARRSIZ	100
double	zs[ARRSIZ];

static int ellipdesign(int n,double Rp, double Rs, double Wp, int ftype, double *bcoeff, double *acoeff)
{
	double	dbfac=10.0/log(10.0);
	int		rn=n;
	double	dbr=Rp;
	double	eps = exp( dbr/dbfac );
	double	scale=1.0;
	double	fs=2,fnyq=1;
	double	f1=0.0;
	double	f2=Wp;
	double	dbd=-Rs;
	double	a,b,bw,ang,cang,c,cgam;
	double	m1,m1p,Kk1,Kpk1,q,k,wr,f3;
	double	sang,cbp;
	double	aa[ARRSIZ];
	int		i,j;
	double	wc,m,Kk,Kpk,r,phi,u,sn,cn,dn;
	int		np,nz,nt,lr,ii,ir;
	double	sn1,cn1,dn1,phi1;
	cmplx z[ARRSIZ];
	cmplx rc, cnum, cden;
	double C;
	int	nc,jt,icnt,zord;
	cmplx lin[2];
	double	pp[ARRSIZ],y[ARRSIZ];
	int		jj,jh;
	double	pn,an,gain;

	if((n & 1) == 0) scale = sqrt( eps );
	eps = sqrt( eps - 1.0 );

	if (ftype==2) {
		// high pass
		bw = f2;
		a = fnyq;
	}
	else {
		bw = f2 - f1;
		a = f2;
	}

	/* Frequency correspondence for bilinear transformation
	 *
	 *  Wanalog = tan( 2 pi Fdigital T / 2 )
	 *
	 * where T = 1/fs
	 */
	ang = bw * M_PI / fs;
	cang = cos( ang );
	c = sin(ang) / cang; /* Wanalog */
	cgam = cos( (a+f1) * M_PI / fs ) / cang;

	/* calculate band edge from db down */
	a = exp( -dbd/dbfac );

	m1 = eps/sqrt( a - 1.0 );

	m1 *= m1;
	m1p = 1.0 - m1;
	Kk1 = ellpk( m1p );
	Kpk1 = ellpk( m1 );

	q = exp( -M_PI * Kpk1 / (rn * Kk1) );
	k = cay(q);
	if (ftype==2)
		wr = k;
	else
		wr = 1.0/k;

	if ((ftype==1)||(ftype==2)) {
		f3 = atan( c * wr ) * fs / M_PI;
	}
	else {
		a = c * wr;
		a *= a;
		b = a * (1.0 - cgam * cgam) + a * a;
		b = (cgam + sqrt(b))/(1.0 + a);
		f3 = (M_PI/2.0 - asin(b)) * fs / (2.0*M_PI);
	}

	ang = f3 * M_PI / fs;
	cang = cos(ang);
	sang = sin(ang);

	wr = sang/(cang*c);

	if( ftype==2 )
		wr = 1.0/wr;
	if( wr < 0.0 )
		wr = -wr;
	y[0] = 1.0;
	y[1] = wr;
	cbp = wr;

	if( ftype==2 )
		y[1] = 1.0/y[1];

	for( i=1; i<=2; i++ ) {
		aa[i] = atan( c * y[i-1] ) * fs / M_PI ;
	}

#if DEBUG
	printf( "pass band %.9E\n", aa[1] );
	printf( "stop band %.9E\n", aa[2] );
#endif

	/* find locations in lambda plane */

	wc = 1.0;
	k = wc/wr;
	m = k * k;
	Kk = ellpk( 1.0 - m );
	Kpk = ellpk( m );
	q = exp( -M_PI * rn * Kpk / Kk );	/* the nome of k1 */
	m1 = cay(q); /* see below */
	/* Note m1 = eps / sqrt( A*A - 1.0 ) */
	a = eps/m1;
	a =  a * a + 1;
	a = 10.0 * log(a) / log(10.0);
#if DEBUG
	printf( "dbdown %.9E\n", a );
#endif
	a = 180.0 * asin( k ) / M_PI;
	b = 1.0/(1.0 + eps*eps);
	b = sqrt( 1.0 - b );
#if DEBUG
	printf( "theta %.9E, rho %.9E\n", a, b );
#endif
	m1 *= m1;
	m1p = 1.0 - m1;
	Kk1 = ellpk( m1p );
	Kpk1 = ellpk( m1 );
	r = Kpk1 * Kk / (Kk1 * Kpk);
#if DEBUG
	printf( "consistency check: n= %.14E\n", r );
#endif

	/*   -1
	 * sn   j/eps\m  =  j ellik( atan(1/eps), m )
	 */
	b = 1.0/eps;
	phi = atan( b );
	u = ellik( phi, m1p );
#if DEBUG
	printf( "phi %.7e m %.7e u %.7e\n", phi, m1p, u );
#endif
	/* consistency check on inverse sn */
	ellpj( u, m1p, &sn, &cn, &dn, &phi );
	a = sn/cn;
#if DEBUG
	printf( "consistency check: sn/cn = %.9E = %.9E = 1/eps\n", a, b );
#endif
	u = u * Kk / (rn * Kk1);	/* or, u = u * Kpk / Kpk1 */

	/* calculate s plane poles and zeros, normalized to wc = 1 */
	for( i=0; i<ARRSIZ; i++ ) zs[i] = 0.0;
	np = (n+1)/2;
	nz = 0;

	nz = n/2;
	ellpj( u, 1.0-m, &sn1, &cn1, &dn1, &phi1 );
	for( i=0; i<ARRSIZ; i++ ) zs[i] = 0.0;
	for( i=0; i<nz; i++ ) {	/* zeros */
		a = n - 1 - i - i;
		b = (Kk * a) / rn;
		ellpj( b, m, &sn, &cn, &dn, &phi );
		lr = 2*np + 2*i;
		zs[ lr ] = 0.0;
		a = wc/(k*sn);	/* k = sqrt(m) */
		zs[ lr + 1 ] = a;
	}
	for( i=0; i<np; i++ ) {	/* poles */
		a = n - 1 - i - i;
		b = a * Kk / rn;
		ellpj( b, m, &sn, &cn, &dn, &phi );
		r = k * sn * sn1;
		b = cn1*cn1 + r*r;
		a = -wc*cn*dn*sn1*cn1/b;
		lr = i + i;
		zs[lr] = a;
		b = wc*sn*dn1/b;
		zs[lr+1] = b;
	}
	if( ftype==2 ) {
		nt = np + nz;
		for( j=0; j<nt; j++ ) {
			ir = j + j;
			ii = ir + 1;
			b = zs[ir]*zs[ir] + zs[ii]*zs[ii];
			zs[ir] = zs[ir] / b;
			zs[ii] = zs[ii] / b;
		}
		while( np > nz ) {
			ir = ii + 1;
			ii = ir + 1;
			nz += 1;
			zs[ir] = 0.0;
			zs[ii] = 0.0;
		}
	}

#if DEBUG
	printf( "s plane poles:\n" );
	j = 0;
	for( i=0; i<np+nz; i++ ) {
		a = zs[j];
		++j;
		b = zs[j];
		++j;
		printf( "%.9E %.9E\n", a, b );
		if( i == np-1 ) printf( "s plane zeros:\n" );
	}
#endif

	/* convert s plane poles and zeros to the z plane */

	C = c;

	for( i=0; i<ARRSIZ; i++ ) {
		z[i].r = 0.0;
		z[i].i = 0.0;
	}

	nc = np;
	jt = -1;
	ii = -1;

	for( icnt=0; icnt<2; icnt++ ) {
		/* The maps from s plane to z plane */
		do {
			ir = ii + 1;
			ii = ir + 1;
			rc.r = zs[ir];
			rc.i = zs[ii];

			cnum.r = 1 + C * rc.r;
			cnum.i = C * rc.i;
			cden.r = 1 - C * rc.r;
			cden.i = -C * rc.i;
			jt += 1;
			cdiv( &cden, &cnum, &z[jt] );
			if( rc.i != 0.0 ) {
				/* fill in complex conjugate root */
				jt += 1;
				z[jt].r = z[jt-1 ].r;
				z[jt].i = -z[jt-1 ].i;
			}
		} while( --nc > 0 );

		if( icnt == 0 ) {
			zord = jt+1;
			if( nz <= 0 ) break;
		}
		nc = nz;
	} /* end for() loop */

	lin[1].r = 1.0;
	lin[1].i = 0.0;

	while( 2*zord - 1 > jt ) {
		jt += 1;
		z[jt].r = -1.0; /* zero at Nyquist frequency */
		z[jt].i = 0.0;
	}
#if DEBUG
	printf( "order = %d\n", zord );
#endif

	/* Expand the poles and zeros into numerator and
	 * denominator polynomials
	 */
	for( icnt=0; icnt<2; icnt++ ) {
		for( j=0; j<ARRSIZ; j++ ) {
			pp[j] = 0.0;
			y[j] = 0.0;
		}
		pp[0] = 1.0;
		for( j=0; j<zord; j++ ) {
			jj = j;
			if( icnt ) jj += zord;
			a = z[jj].r;
			b = z[jj].i;
			for( i=0; i<=j; i++ ) {
				jh = j - i;
				pp[jh+1] = pp[jh+1] - a * pp[jh] + b * y[jh];
				y[jh+1] =  y[jh+1]  - b * pp[jh] - a * y[jh];
			}
		}
		if( icnt == 0 ) {
			for( j=0; j<=zord; j++ ) aa[j] = pp[j];
		}
	}

	/* Scale factors of the pole and zero polynomials */
	a = 1.0;
	if (ftype==1) {
		pn = 1.0;
		an = 1.0;
		for( j=1; j<=zord; j++ ) {
			pn = a * pn + pp[j];
			an = a * an + aa[j];
		}
	}
	else if (ftype==2) {
		a = -1.0;
		pn = 1.0;
		an = 1.0;
		for( j=1; j<=zord; j++ ) {
			pn = a * pn + pp[j];
			an = a * an + aa[j];
		}
	}

	gain = an/(pn*scale);
#if DEBUG
	printf( "constant gain factor %23.13E\n", gain );
#endif
	for( j=0; j<=zord; j++ ) pp[j] = gain * pp[j];

#if DEBUG
	printf( "z plane Denominator      Numerator\n" );
	for( j=0; j<=zord; j++ ) {
		printf( "%2d %17.9E %17.9E\n", j, aa[j], pp[j] );
	}
#endif

	for (j=0;j<=zord;j++) {
		acoeff[j]=aa[j];
		bcoeff[j]=pp[j];
	}

	return zord;
}

/*****************************************************************************************/
/* public interface */
/*****************************************************************************************/

/* find the order for elliptic filter */
/* wp,ws=normalised pass & stop freq */
/* rp,rs=response pass and stop (dB) */
/* ftype=1 (low), 2(high) */
int filter_elliptic_order(int ftype,double fpass,double fstop,double rpass,double rstop,double srate)
{
	/* prewarp */
	double	wp=fpass/(srate/2);
	double	ws=fstop/(srate/2);
	double	WP=tan(M_PI*wp/2);
	double	WS=tan(M_PI*ws/2);
	double	WA;
	int		order=0;

	if (ftype==FILTER_LOW_PASS) {
		WA=WS/WP;
		order=findelliporder(WA,rpass,rstop);
	}
	else if (ftype==FILTER_HIGH_PASS) {
		WA=WP/WS;
		order=findelliporder(WA,rpass,rstop);
	}
	else {
		fprintf(stderr,"Filter type not supported\n");
		return(0);
	}

	return(order);
}

/* design an elliptic filter given band edges and ripple size */
FILTER * filter_elliptic_design(int ftype,int forder,double fpass,double rpass,double rstop,double srate)
{
	FILTER	*filter;
	int		i;

	if ((ftype!=FILTER_LOW_PASS)&&(ftype!=FILTER_HIGH_PASS)) {
		fprintf(stderr,"Filter type not supported\n");
		return(NULL);
	}

	/* build FILTER structure */
	if ((filter=(FILTER *)calloc(1,sizeof(FILTER)))==NULL) {
		fprintf(stderr,"filter: out of memory\n");
		return(NULL);
	}
	filter->nsection = 1;
	if ((filter->section=(FILTER_SECTION *)calloc(filter->nsection,sizeof(FILTER_SECTION)))==NULL) {
		fprintf(stderr,"filter: out of memory\n");
		return(NULL);
	}
	for (i=0;i<filter->nsection;i++) {
		filter->section[i].ncoeff = forder+1;
		if ((filter->section[i].acoeff=(double *)calloc(forder+1,sizeof(double)))==NULL) {
			fprintf(stderr,"filter: out of memory\n");
			return(NULL);
		}
		if ((filter->section[i].bcoeff=(double *)calloc(forder+1,sizeof(double)))==NULL) {
			fprintf(stderr,"filter: out of memory\n");
			return(NULL);
		}
		if ((filter->section[i].memory=(double *)calloc(forder+2,sizeof(double)))==NULL) {
			fprintf(stderr,"filter: out of memory\n");
			return(NULL);
		}
	}

	ellipdesign(forder,rpass,rstop,fpass/(srate/2), ftype, filter->section[0].acoeff, filter->section[0].bcoeff);

	return filter;
}


#if EMO

int main(int argc,char *argv[])
{
	double fp,fs;
	double rp,rs,srate;
	int		ftype,order,i;
	FILTER	*filt;

	if (argc!=6) {
		fprintf(stderr,"usage: elliptic FPASS FSTOP RPASS RSTOP SRATE\n");
		exit(1);
	}

	fp=atof(argv[1]);
	fs=atof(argv[2]);
	rp=atof(argv[3]);
	rs=atof(argv[4]);
	srate=atof(argv[5]);

	if (fp<fs) ftype=FILTER_LOW_PASS; else ftype=FILTER_HIGH_PASS;

	order=filter_elliptic_order(ftype,fp,fs,rp,rs,srate);

	printf("Elliptic filter order=%d\n",order);

	filt = filter_elliptic_design(ftype,order,fp,rp,rs,srate);

	printf("Coefficients:\n  Numerator\t  Denominator\n");
	for (i=0;i<=order;i++)
		printf("%13.8f\t%13.8f\n",filt->section[0].acoeff[i],filt->section[0].bcoeff[i]);

	printf("Impulse response:\n");
	printf("%g,",filter_sample(filt,1.0));
	for (i=0;i<99;i++) printf("%g,",filter_sample(filt,0.0));

	exit(0);
}

#endif
