/* lsp - function to support LP coding using Line Spectral Pairs */

/* Mark Huckvale - University College London */

/* version 1.0 - Novermber 2006 */

#include <stdio.h>
#include <math.h>
#include "complex.h"
#include "lsys.h"
#include "lsp.h"

#ifndef PI
#define	PI	3.14159265358979323846
#define PI2	6.28318530717958647692
#endif

#define MAXORDER	50

/* Chebyshev polynomial evaluation */
static double chebpoly(double x, double *c, int order)
{
	register int	i;
	double		  b[3];

	b[1] = b[2] = 0.0;
	for(i=order; i>0; i--){
		b[0] = 2.0 * x * b[1] - b[2] + c[i];
		b[2] = b[1];
		b[1] = b[0];
	}
	b[0] = x * b[1] - b[2] + c[0];

	return(b[0]);
}


/* Signal to LPC coefficients (autocorrelation method) */
double lsp_auto_lpc(double *x,int n,double *pc,int order)
{
	double	r[MAXORDER+1];		/* autocorrelations */
	double	ai,aj,akk;		/* temporary coefficient values */
	double	sum;			/* accumulator */
	double	pe;				/* predictor error */
	int	i,k;			/* loop counters */

	/* check and zero */
	if (order > MAXORDER) order = MAXORDER;
	for (i=0;i<=order;i++) pc[i]=0;

	/* compute autocorrelations */
	for (i=0;i<=order;i++) {
		sum = 0;
		for (k=0;k<(n-i);k++) sum += x[k] * x[k+i];
		r[i] = sum;
	}

	/* check something to do */
	if (r[0] == 0) return(0.0);

	/* calculate predictor */
	pe = r[0];
	pc[0] = 1.0;
	for (k=1;k<=order;k++) {
		sum = 0;
		for (i=1;i<=k;i++) sum -= pc[k-i] * r[i];
		akk = sum/pe;

		/* new predictor coefficients */
		pc[k] = akk;
		for (i=1;i<=(k/2);i++) {
			ai = pc[i];
			aj = pc[k-i];
			pc[i] = ai + akk * aj;
			pc[k-i] = aj + akk * ai;
		}

		/* new prediction error */
		pe = pe * (1.0 - akk*akk);
		if (pe <= 0.0) return(0.0);
	}

	return(pe);
}

/* convert predictor coefficients to LSP */
#define numsp 512
#define maxitr 8
#define eps 1.0e-6

int lsp_from_lpc(double *lpc, double *lsp, int order)
{
	register int	i;
	register double *p1, *p2;
	int			 mh1, mh2, mh, mm, itr, flag_odd;
	double		  delta, x0, x1, g0, g1, x, y;
	double		  acos(), fabs(), chebpoly();
	double   c1[MAXORDER+1], c2[MAXORDER+1];

	/* get search accuracy */
	delta = 1.0 / (double) numsp;

	/* check for odd order */
	flag_odd = 0;
	if(order % 2 == 0)
		mh1 = mh2 = order / 2;
	else {
		mh1 = (order+1) / 2;
		mh2 = (order-1) / 2;
		flag_odd = 1;
	}

	/* calculate symmetric and antisymmetric polynomials */
	p1 = lpc + 1;
	p2 = lpc + order;
	c1[mh1] = c2[mh2] = 1.0;
	if (flag_odd) {
		c2[mh2+1] = 0.0;
		for (i=mh2-1; i>=0; i--) {
			c1[i+1] = *p1   + *p2;
			c2[i]   = *p1++ - *p2-- + c2[i+2];
		}
		c1[0] = *p1 + *p2;
	}
	else {
		for(i=mh1-1; i>=0; i--) {
			c1[i] = *p1   + *p2   - c1[i+1];
			c2[i] = *p1++ - *p2-- + c2[i+1];
		}
	}
	c1[0] *= 0.5;
	c2[0] *= 0.5;

	/* root search */
	p1 = c1;
	mh = mh1;
	g0 = chebpoly(1.0, p1, mh);

	mm = 0;
	for(x=1.0-delta; x>-delta-1.0; x-=delta) {

		g1 = chebpoly(x, p1, mh);

		if (g0*g1 <= 0.0) {
			x0 = x + delta;
			x1 = x;

			itr = 0;
			do {
				x = (x0 + x1) / 2.0;
				y = chebpoly(x, p1, mh);

				if (y*g0 < 0.0) {
					x1 = x;
					g1 = y;
				}
				else {
					x0 = x;
					g0 = y;
				}

				itr++;
			} while ((fabs(y) > eps) && (itr < maxitr));

			x = (g1*x0 - g0*x1) / (g1 - g0);
			lsp[mm] = acos(x) / PI2;

			mm++;
			if(mm == order) return(0);

			if (p1 == c1) {
				p1 = c2;
				mh = mh2;
			}
			else {
				p1 = c1;
				mh = mh1;
			}

			g1 = chebpoly(x, p1, mh);
		}
		g0 = g1;
	}
	return(-1);
}

/* convert LSP back to predictor coefficients */
void lsp_to_lpc(double *lsp, double *pc, int order)
{
	int		i, k, mh1, mh2, flag_odd;
	double	xx, xf, xff;
	double	f[MAXORDER+1];
	double	p[MAXORDER+1];
	double	q[MAXORDER+1];
	double	a0[MAXORDER+1];
	double	a1[MAXORDER+1];
	double	a2[MAXORDER+1];
	double	b0[MAXORDER+1];
	double	b1[MAXORDER+1];
	double	b2[MAXORDER+1];

	/* check for odd order */
    flag_odd = 0;
    if (order % 2 == 0)
        mh1 = mh2 = order / 2;
    else {
		mh1 = (order + 1) / 2;
		mh2 = (order - 1) / 2;
		flag_odd = 1;
    }

	/* set up arrays */
    for (i=0;i<order;i++) f[i]=lsp[i];
	for (i=0;i<=mh1;i++) {
		a0[i]=a1[i]=a2[i]=0;
	}
	for (i=0;i<=mh2;i++) {
		b0[i]=b1[i]=b2[i]=0;
	}

    /* lsp filter parameters */
    for (i=k=0; i<mh1; i++,k+=2)
		p[i] = -2.0 * cos(PI2 * f[k]);
    for (i=k=0; i<mh2; i++,k+=2)
		q[i] = -2.0 * cos(PI2 * f[k+1]);

    /* impulse response of analysis filter */
    xx = 1.0;
    xf = xff = 0.0;
	for (k=0; k<=order; k++) {
		if (flag_odd) {
			a0[0] = xx;
			b0[0] = xx - xff;
			xff = xf;
			xf  = xx;
		}
		else {
			a0[0] = xx + xf;
			b0[0] = xx - xf;
			xf = xx;
		}

		for (i=0; i<mh1; i++) {
			a0[i+1] = a0[i] + p[i] * a1[i] + a2[i];
			a2[i] = a1[i]; a1[i] = a0[i];
		}
		for (i=0; i<mh2; i++) {
			b0[i+1] = b0[i] + q[i] * b1[i] + b2[i];
			b2[i] = b1[i]; b1[i] = b0[i];
		}

		if (k != 0)
			pc[k-1] = -0.5 * (a0[mh1] + b0[mh2]);

		xx = 0.0;
    }

    for (i=order-1; i>=0; i--)
		pc[i+1] = -pc[i];
    pc[0] = 1.0;
}

/* calculate residual given predictor coefficients */
void lsp_residual_lpc(double *x,int n,double *pc,int order,double *y)
{
	int	i,j;
	double	pred;

	for (i=0;i<n;i++) {
		pred=0;
		for (j=1;j<=order;j++)
			if (i-j >= 0)
				pred -= pc[j] * x[i-j];
		y[i]=x[i]-pred;
	}
}

/* calculate signal given predictor coefficients and residual */
void lsp_synth_lpc(double *x,int n,double *pc,int order,double *y)
{
	int	i,j;
	double	pred;

	for (i=0;i<n;i++) {
		pred=x[i];
		for (j=1;j<=order;j++)
			if (i-j >= 0)
				pred -= pc[j] * y[i-j];
		y[i]=pred;
	}
}

/* calculate signal given predictor coefficients and residual + perceptual filter */
void lsp_synth_lpc_perceptual(double *x,int n,double *pc,int order,double *y,double factor)
{
	int	i,j;
	double	pred,f;
	double	state[MAXORDER+1];

	memset(state,0,sizeof(state));

	for (i=0;i<n;i++) {
		pred=x[i];
		for (j=1;j<=order;j++)
			if (i-j >= 0)
				pred -= pc[j] * y[i-j];
		y[i]=pred;
	}

	for (i=0;i<n;i++) {
		/* shift state */
		for (j=order;j>0;j--) state[j]=state[j-1];
		state[0]=y[i];
		/* apply denominator */
		for (j=1;j<=order;j++)
			state[0] -= pc[j] * state[j];
		/* apply numerator */
		pred = state[0];
		f=factor;
		for (j=1;j<=order;j++) {
			pred += f * pc[j] * state[j];
			f *= factor;
		}
		y[i]=pred;
	}

}

/* package up analysis */
void lsp_analysis(double *x,int n,double *lsp,int order,double *y)
{
	double	pc[MAXORDER+1];

	lsp_auto_lpc(x,n,pc,order);
	lsp_from_lpc(pc,lsp,order);
	lsp_residual_lpc(x,n,pc,order,y);
}

/* package up synthesis */
void lsp_synthesis(double *x,int n,double *lsp,int order,double *y)
{
	double pc[MAXORDER+1];

	lsp_to_lpc(lsp,pc,order);
	lsp_synth_lpc(x,n,pc,order,y);
}

#ifdef EMO
#define WINDOWSIZE 32
#define NCOEFF 4
#define C1	0.16
#define C2	-0.12
#define C3	0.08
#define C4	-0.04
main()
{
	double	iwin[WINDOWSIZE+1];
	double	owin[WINDOWSIZE+1];
	double	swin[WINDOWSIZE+1];
	double	pc[NCOEFF+1];
	double	lsp[NCOEFF+1];
	double	res;
	int	i;

	/* make a test signal from recursive filter */
	iwin[0] = 0.0;
	iwin[1] = 0.0;
	iwin[2] = 0.0;
	iwin[3] = 0.0;
	iwin[4] = 0.75;
	for (i=5;i<WINDOWSIZE;i++)
		iwin[i] = -C1 * iwin[i-1] - C2 * iwin[i-2] -
				C3 * iwin[i-3] - C4 * iwin[i-4];
	printf("Input=");
	for (i=0;i<WINDOWSIZE;i++) printf("%g ",iwin[i]);
	printf("\n");

	/* do LPC analysis */
	res=lsp_auto_lpc(iwin,WINDOWSIZE,pc,NCOEFF);
	printf("Residual Power=%g\n",res);
	printf("Predictor coefficients:\n");
	for (i=0;i<=NCOEFF;i++) printf("%d = %g\n",i,pc[i]);

	/* get residual */
	lsp_residual_lpc(iwin,WINDOWSIZE,pc,NCOEFF,owin);
	printf("Residual=");
	for (i=0;i<WINDOWSIZE;i++) printf("%g ",owin[i]);
	printf("\n");

	/* get resynthesis */
	lsp_synth_lpc(owin,WINDOWSIZE,pc,NCOEFF,swin);
	printf("Synthesis=");
	for (i=0;i<WINDOWSIZE;i++) printf("%g ",swin[i]);
	printf("\n");

	/* do LSP conversion */
	lsp_from_lpc(pc,lsp,NCOEFF);
	printf("Line spectral pairs:\n");
	for (i=0;i<NCOEFF;i++) printf("%d = %g\n",i,lsp[i]);

	/* convert back to PC */
	lsp_to_lpc(lsp,pc,NCOEFF);
	printf("Predictor coefficients:\n");
	for (i=0;i<=NCOEFF;i++) printf("%d = %g\n",i,pc[i]);

	/* get resynthesis */
	lsp_synth_lpc(owin,WINDOWSIZE,pc,NCOEFF,swin);
	printf("Synthesis=");
	for (i=0;i<WINDOWSIZE;i++) printf("%g ",swin[i]);
	printf("\n");

	exit(0);
}
#endif
