/* root.c -- find roots of polynomial */

/*
 * (c) 1994 Mark Huckvale
 * Search Systems Ltd
 * 52-54 High Holborn
 * London WC1E 6BT
 * Tel. 071-430 1685
 */

/*
 * Adapted from laguer() and zroots() in 'Numerical recipes in C'
 */

#include <stdio.h>
#include <math.h>
#include "complex.h"
#include "root.h"

/* find single root */
void Laguerre(ap,m,r)
COMPLEX *ap;
int m;
COMPLEX *r;
{
	COMPLEX rlast;
	int	j,iter;
	double	err,abx;
	COMPLEX sq,h,gp,gm,g2,g,bp,d,dx,f;

	iter = 0;
	do {
		rlast = *r;
		bp = ap[m];
		err = CMag(bp);
		f = CMake(0.0,0.0);
		d = f;
		abx = CMag(*r);
		/* compute value of polynomial & derivatives */
		for (j=m-1;j>=0;j--) {
			f = CAdd(CMult(*r,f),d);
			d = CAdd(CMult(*r,d),bp);
			bp = CAdd(CMult(*r,bp),ap[j]);
			err = CMag(bp)+abx*err;
		}
		/* if polynomial = zero then already at root */
		err = err * ROUND_ERROR;
		if (CMag(bp) > err) {
			/* no, iterate using Laguerre's formula */
			g = CDiv(d,bp);
			g2 = CMult(g,g);
			h = CSub(g2,CScale(CDiv(f,bp),2.0));
			sq = CSqrt(CScale(CSub(CScale(h,m*1.0),g2),m-1.0));
			gp = CAdd(g,sq);
			gm = CSub(g,sq);
			if (CMag(gp) < CMag(gm)) gp = gm;
			dx = CDiv(CMake(m*1.0,0.0),gp);
			*r = CSub(*r,dx);
		}
		iter++;
	} while (!((iter==100) || (CMag(bp)<=err) || ((r->re == rlast.re) && (r->im == rlast.im))));
		/* terminating condition for iteration */
}

/* find all roots */
void AllRoots(ap,m,roots)
COMPLEX *ap;
int m;
COMPLEX *roots;
{
	int	k,j,i;
	COMPLEX	x,bp,c;
	COMPLEX ad[MAXPOLY];

	for (j=0;j<=m;j++) ad[j] = ap[j];

	for (j=m;j>=1;j--) {
		/* find root */
		x = CMake(0.0,0.0);
		Laguerre(ad,j,&x);
		if (fabs(x.im) <= (IM_RANGE*fabs(x.re))) 
			x.im = 0.0;
		roots[j] = x;

		/* deflation */
		bp = ad[j];
		for (k=j-1;k>=0;k--) {
			c = ad[k];
			ad[k] = bp;
			bp = CAdd(CMult(x,bp),c);
		}
	}

	/* sort into increasing root.real */
	for (j=2;j<=m;j++) {
		x = roots[j];
		i = j;
		while ((i > 1) && (x.re < roots[i-1].re)) {
			roots[i] = roots[i-1];
			i = i - 1;
		}
		roots[i] = x;
	}
}

#ifdef EMO
COMPLEX	poly[20];
COMPLEX roots[20];
main()
{
	double	val;
	int	i;
	int	m;

	printf("Enter Size: "); scanf("%d",&m);

	printf("Enter Coeffs (x^0 x^1 x^2 ...) : ");
	for (i=0;i<=m;i++) {
		scanf("%lf",&val);
		poly[i] = CMake(val,0.0);
	}
	
	printf("Input=");
	for (i=0;i<=m;i++)
		CPrint(poly[i]);
	printf("\n");	
	
	AllRoots(poly,m,roots);

	printf("Roots=");
	for (i=0;i<=m;i++)
		CPrint(roots[i]);
	printf("\n");	
}
#endif
