/* anmbrola -- list an annotation item and Fx item in MBROLA .pho format */

/* M.A.Huckvale - August 1998 */

/* version 1.0 */

/*-------------------------------------------------------------------*/
/**MAN
.TH ANMBROLA SFS1 UCL
.SH NAME
anmbrola -- list an annotation item and Fx item in MBROLA .pho format
.SH SYNOPSIS
.B anmbrola
(-i item) (-s) file
.SH DESCRIPTION
.I anmbrola
dumps the contents of an annotation item and a fundamental frequency
item onto the standard output in a form suitable for replaying with
MBROLA.  It assumes that the annotation items are segment labels in
a format compatible with an MBROLA database.  The output may also be
use as input to repros (SFS1) the prosody mapping program.
.PP
.B Options
.TP 11
.B -I
Identify the program version.
.TP 11
.BI -i item
Select input item.
.TP 11
.B -s
Do not add silences if absent from file.
.SH INPUT ITEMS
.IP AN 11
Any annotation item
.IP FX 11
Any fundamental frequency item
.SH VERSION/AUTHOR
1.0 - Mark Huckvale
.SH BUGS
*/
/*---------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "anmbrola"
#define PROGVERS "1.0"
char	*progname = PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
char			filename[SFSMAXFILENAME];	/* dbase file name */
struct item_header 	anitem;		/* annotation item header  */
struct an_rec		*an;		/* annotation record */
char			*antype="0";	/* annotation type */
struct item_header	fxitem;		/* fx item */
short			*fx;		/* fx data */
char			*fxtype="0";	/* fx type */
int		dosilence=1;

int issilence(char *label)
{
	return((strcmp(label,"_")==0)||(strcmp(label,"/")==0));
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;			/* option index */
	extern char	*optarg;		/* option argument ptr */
	int		errflg = 0;		/* option error flag */
	int32		it;			/* item type */
	char		*ty="0";		/* sub-type = last */
	int		c;			/* option switch */

	/* processing variables */
	int		i;
	int		fid;			/* input file descriptor */
	double		t,stim,etim;
	int		idx;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:s")) != EOF ) switch (c) {
	case 'I' :	/* Identify */
		fprintf(stderr,"%s: List AN, FX items in MBROLA format V%s\n",PROGNAME,PROGVERS);
		exit(0);
		break;
	case 'i' :	/* input specification */
		if (itspec(optarg,&it,&ty)==0) {
			if (it==AN_TYPE)
				antype=ty;
			else if (it==FX_TYPE)
				fxtype=ty;
			else
				error("unsuitable item specification: '%s'",optarg);
		}
		else
			error("illegal item specification: '%s'",optarg);
		break;
	case 's':	/* suppress addition of silences */
		dosilence=0;
		break;
	case '?' :	/* unknown */
		errflg++;
	}

	/* check command line */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) file",PROGNAME);

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open data file */
	if ((fid=sfsopen(filename,"r",NULL)) < 0) {
		if (fid==-1)
			error("cannot find file %s",filename);
		else
			error("access error on %s",filename);
	}

	/* locate input annotation item */
	if (!sfsitem(fid,AN_TYPE,antype,&anitem))
		error("unable to find input annotations in '%s'",filename);

	/* get buffer */
	if ((an=(struct an_rec *)sfsbuffer(&anitem,anitem.numframes))==NULL)
		error("could not get memory buffer",NULL);

	/* read in all annotations */
	if (sfsread(fid,0,anitem.numframes,an)!=anitem.numframes)
		error("read error on input");

	/* locate input Fx item */
	if (!sfsitem(fid,FX_TYPE,fxtype,&fxitem))
		error("unable to find input Fx in '%s'",filename);

	/* get buffer */
	if ((fx=(short *)sfsbuffer(&fxitem,fxitem.numframes))==NULL)
		error("could not get memory buffer",NULL);

	/* read in Fx */
	if (sfsread(fid,0,fxitem.numframes,fx)!=fxitem.numframes)
		error("read error on input");

	/* print title */
	printf("; Produced by: %s %s-i%d.%02d -i%d.%02d %s\n",
		PROGNAME,
		(dosilence==0)?"-s ":"",
		anitem.datatype,anitem.subtype,
		fxitem.datatype,fxitem.subtype,
		filename);

	/* print opening silence */
	if (dosilence) {
		if (!issilence(an[0].label))
			printf("_ 100\n");
		else if (an[0].posn!=0)
			printf("_ %d\n",(int)(1000*(anitem.offset+an[0].posn*anitem.frameduration)));
	}

	/* print annotations */
	for (i=0;i<anitem.numframes;i++) {
		stim = anitem.offset + an[i].posn*anitem.frameduration;
		if (i<(anitem.numframes-1))
			etim = anitem.offset + an[i+1].posn*anitem.frameduration;
		else
			etim = stim + an[i].size*anitem.frameduration;
		if (etim > stim) {
			printf("%s %d",an[i].label,(int)(1000.0*(etim-stim)));
			for (t=stim;t<etim;t+=0.02) {
				idx = (int)((t-fxitem.offset)/fxitem.frameduration);
				if ((0 <= idx) && (idx < fxitem.numframes)) {
					if (fx[idx]>0)
						printf(" %d %d",
							(int)(100*(t-stim)/(etim-stim)),
							fx[idx]);
				}
			}
			printf("\n");
		}
	}

	/* print closing silence */
	if (dosilence) {
		if (!issilence(an[anitem.numframes-1].label))
			printf("_ 100\n");
	}

	/* that's all folks */
	sfsclose(fid);
	exit(0);

}

