/* filtbank -- general purpose filterbank analysis */

/* M.A.Huckvale - University College London */

/* version 2.0 - September 1998 */

#define PROGNAME "filtbank"
#define PROGVERS "2.1"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH FILTBANK UCL1 UCL SFS
.SH NAME
filt - general purpose filterbank analysis
.SH SYNOPSIS
.B genfilt
(-i item) (-n numfilt) (-o order) (-r rate) (-3) file
.SH DESCRIPTION
.I filterbank
is a program to design and apply a general linear filterbank
to a signal.
In normal model
.I filtbank
uses N bandpass filters linearly spaced of equal size
with matching 3dB points between 50Hz and (Fs/2)-50Hz.
In Third Octave mode (-3 switch)
.I filtbank
uses N constant Q filters of one-third octave spacing
starting at 100Hz.
Note that fewer than N third-octave filters may be
possible.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -n numfilt
Select number of filters >= 2.  Default 16.
.TP 11
.BI -o order
Select filter order.  Default 4. Maximum is 20.
.TP 11
.BI -r rate
Output frame rate required.  Energies are rectified then low-pass filtered
at half this rate before sampling.
.TP 11
.B -3
Use third-octave filters.
.SH INPUT ITEMS
.IP SP
Any speech item
.IP LX
(optional) Any Lx item
.SH OUTPUT ITEMS
.IP CO
Filterbank energies.
.SH HISTORY
.IP numfilt=
number of filters
.IP order=
order of filters
.IP rate=
output frame rate
.IP type=
third-octave
.SH VERSION/AUTHOR
.IP 2.1
Mark Huckvale
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include "sfs.h"
#include "filter.h"

/* global variables */
struct item_header spitem;
#define SPBUFSIZE 4096
short sp[SPBUFSIZE];
struct item_header coitem;
struct co_rec *co;

/* single channel */
struct channel_rec {
	FILTER	*filter;
	FILTER	*smooth;
};

/* bank of filters */
struct channel_rec *fbank;
int	numfilt=16;
int	filtorder=2;
double	outrate=100.0;
int		dothird=0;
struct third_filter {
	double	cf;
	double	lf;
	double	hf;
} thirdfilt[100];

double mylog10(double val)
{
	if (val < 1.0E-5)
		return(-5);
	else
		return(log10(val));
}

/* generate third octave specs */
void genthird()
{
	double	two3=pow(2.0,1.0/3.0);
	double	two6=pow(2.0,1.0/6.0);
	int		i;
	double	f;

	f=1000;
	for (i=10;i>=0;i--) {
		thirdfilt[i].cf = f;
		thirdfilt[i].lf = f/two6;
		thirdfilt[i].hf = f*two6;
		f = f/two3;
	}
	f=1000;
	for (i=10;i<100;i++) {
		thirdfilt[i].cf = f;
		thirdfilt[i].lf = f/two6;
		thirdfilt[i].hf = f*two6;
		f = f*two3;
	}

//	printf("3rd octave filters:\n");
//	for (i=0;i<20;i++) printf("%g\t%g\t%g\n",thirdfilt[i].lf,thirdfilt[i].cf,thirdfilt[i].hf);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it=SP_TYPE;	/* item selection */
	char		*ty="0";	/* item sub type */
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		ofid;		/* output file descriptor */

	/* processing variables */
	int		half_freq;
	int		len,pos;
	int		outsamp;
	double		ival,oval,sval;
	int		samp,frame;
	int		i,j;
	double		chan_freq,smooth_freq;
	char		mbuf[32];

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:o:n:r:3")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: General-purpose filtering V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it == SP_TYPE) || (it== LX_TYPE))
					/* ok */;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'n' :	/* number of filters */
			numfilt = atoi(optarg);
			if (numfilt < 2)
				error("filterbank must have at least two filters");
			break;
		case 'o' :	/* order of filters */
			filtorder = atoi(optarg);
			break;
		case 'r' :	/* output rate */
			outrate = atof(optarg);
			break;
		case '3':	/* third octave */
			dothird++;
			genthird();
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-n numfilt) (-o order) (-r rate) (-3) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* find input item */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on '%s'",filename);
	if (!sfsitem(fid,it,ty,&spitem))
		error("could not find input item in '%s'",filename);

	/* check filter freq */
	if (dothird) {
		double fmax = 0.88*(0.5/spitem.frameduration);
		while (thirdfilt[numfilt-1].cf > fmax) numfilt--;
	}

	/* get memory for filter bank */
	fbank = (struct channel_rec *)calloc(numfilt,sizeof(struct channel_rec));
	if (fbank==NULL)
		error("out of memory");

	/* design filter bank */
	half_freq = (int)(0.5+0.5/spitem.frameduration);
	chan_freq = (half_freq-200)/numfilt;
	outsamp = (int)(0.5+1.0/(spitem.frameduration*outrate));
	outrate = 1.0/(spitem.frameduration*outsamp);
	smooth_freq = outrate/2;

#if 0
	if ((fbank[0].filter = filter_design(FILTER_LOW_PASS,2*filtorder,chan_freq,chan_freq,1.0/spitem.frameduration))==NULL)
		error("could not design filter");
	if ((fbank[0].smooth = filter_design(FILTER_LOW_PASS,filtorder,smooth_freq,smooth_freq,1.0/spitem.frameduration))==NULL)
		error("could not design filter");

	for (i=1;i<numfilt-1;i++ ) {
		if ((fbank[i].filter = filter_design(FILTER_BAND_PASS,filtorder,i*chan_freq,(i+1)*chan_freq,1.0/spitem.frameduration))==NULL)
			error("could not design filter");
		if ((fbank[i].smooth = filter_design(FILTER_LOW_PASS,filtorder,smooth_freq,smooth_freq,1.0/spitem.frameduration))==NULL)
			error("could not design filter");
	}

	if ((fbank[numfilt-1].filter = filter_design(FILTER_HIGH_PASS,2*filtorder,half_freq-chan_freq,half_freq-chan_freq,1.0/spitem.frameduration))==NULL)
		error("could not design filter");
	if ((fbank[numfilt-1].smooth = filter_design(FILTER_LOW_PASS,filtorder,smooth_freq,smooth_freq,1.0/spitem.frameduration))==NULL)
		error("could not design filter");
#endif

	if (dothird) {
		for (i=0;i<numfilt;i++ ) {
			printf("Channel %d from %g to %g\n",i,thirdfilt[i].lf,thirdfilt[i].hf);
			if ((fbank[i].filter = filter_design(FILTER_BAND_PASS,filtorder,thirdfilt[i].lf,thirdfilt[i].hf,1.0/spitem.frameduration))==NULL)
				error("could not design filter");
			if ((fbank[i].smooth = filter_design(FILTER_LOW_PASS,filtorder,smooth_freq,smooth_freq,1.0/spitem.frameduration))==NULL)
				error("could not design filter");
		}
	}
	else {
		for (i=0;i<numfilt;i++ ) {
			printf("Channel %d from %g to %g\n",i,100+i*chan_freq,100+(i+1)*chan_freq);
			if ((fbank[i].filter = filter_design(FILTER_BAND_PASS,filtorder,100+i*chan_freq,100+(i+1)*chan_freq,1.0/spitem.frameduration))==NULL)
				error("could not design filter");
			if ((fbank[i].smooth = filter_design(FILTER_LOW_PASS,filtorder,smooth_freq,smooth_freq,1.0/spitem.frameduration))==NULL)
				error("could not design filter");
		}
	}

	/* create output item header */
	sfsheader(&coitem,CO_TYPE,1,4,5+numfilt,1.0/outrate,spitem.offset,1,0,0);
	sprintf(coitem.history,"%s(%d.%02d;numfilt=%d,order=%d,rate=%g,type=%s)",
		PROGNAME,
		spitem.datatype,
		spitem.subtype,
		numfilt,filtorder,outrate,(dothird)?"3rdoctave":"linear");
	if (dothird) {
		sprintf(coitem.params,"htktype=7,labels=%d",(int)thirdfilt[0].cf);
		for (i=1;i<numfilt;i++) {
			sprintf(mbuf,"|%d",(int)thirdfilt[i].cf);
			strncat(coitem.params,mbuf,sizeof(coitem.params));
		}
	}
	else if (numfilt <= 16) {
		sprintf(coitem.params,"htktype=7,labels=%d",(int)(100+chan_freq/2));
		for (i=1;i<numfilt;i++) {
			sprintf(mbuf,"|%d",(int)(100+(i+0.5)*chan_freq));
			strncat(coitem.params,mbuf,sizeof(coitem.params));
		}
	}
	else if (numfilt <= 32) {
		sprintf(coitem.params,"htktype=7,labels=x2|%d",(int)(100+chan_freq/2));
		for (i=3;i<numfilt;i+=2) {
			sprintf(mbuf,"|%d",(int)(100+(i+0.5)*chan_freq));
			strncat(coitem.params,mbuf,sizeof(coitem.params));
		}
	}
	else {
		sprintf(coitem.params,"htktype=7,labels=x10|%d",(int)(100+chan_freq/2));
		for (i=10;i<numfilt;i+=10) {
			sprintf(mbuf,"|%d",(int)(100+(i+0.5)*chan_freq));
			strncat(coitem.params,mbuf,sizeof(coitem.params));
		}
	}

	/* open output channel */
	if ((ofid=sfschannel(filename,&coitem))<0)
		error("could not open output channel");

	/* get output frame */
	co = (struct co_rec *)sfsbuffer(&coitem,1);

	/* process the signal */
	pos=0;
	samp=0;
	frame=0;
	while ((len=sfsread(fid,pos,SPBUFSIZE,sp)) > 0) {
		for (i=0;i<len;i++) {
			ival = (float)(sp[i]);
			samp++;
			for (j=0;j<numfilt;j++) {
				oval = filter_sample(fbank[j].filter,ival);
				if (oval < 0) oval = -oval;
				sval = filter_sample(fbank[j].smooth,oval);
				if (samp==outsamp)
					co->data[j] = (float)(20.0*mylog10(sval));
			}
			if (samp==outsamp) {
				co->posn = frame;
				co->size = 1;
				co->flag = 0;
				co->mix = 0;
				co->gain = 0;
				if (sfswrite(ofid,1,co)!=1)
					error("write error on output");
				samp=0;
				frame++;
			}
		}
		pos += len;
		if (ttytest()) {
			printf("%d/%d samples\r",pos+1,spitem.numframes);
			fflush(stdout);
		}
	}
	if (ttytest()) {
		printf("                        \r");
		fflush(stdout);
	}

	/* update and exit */
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);
	exit(0);
}


