/* repros -- change speaking rate and/or pitch (advanced version) */

/* M.A.Huckvale - University College London */

/* version 1.0 - August 1998
	- from repitch */

#define IAG

#define PROGNAME "repros"
#define PROGVERS "1.1"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH REPROS 1 SFS UCL
.SH NAME
repros - change speaking rate and or pitch of speech (advanced)
.SH SYNOPSIS
.B repros
(-I) (-i item) (-m MBROLAcontrolfile) (-v) (-j) (-O txoffset) (-P txpercent) file
.SH DESCRIPTION
.I repros
is a program to modify the pitch and duration of an utterance.
It uses the PSOLA algorithm and requires a set of pitch-epoch
annotations.  These can be generated from a Laryngograph signal
using Lx->Tx conversion, followed by Tx->An conversion.
.PP
Control data for the change in prosody is specified by a text
file in MBROLA format.  This file consists of a series of
lines, one per annotated segment in the target file.  Each line
contains: segment label, new segment duration in ms, new fx
contour specified as series of pairs: % position through segment,
fx value in Hz. The segment label '_' matches initial or final
silence in the file.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -m MBROLAcontolfile
Specify MBROLA format control file for prosody change.
.TP 11
.BI -O txoffset
Shift TX with respect to speech (0.001s is typical value).  Default 0.
.TP 11
.BI -P txpercent
Reduce pitch windows to this %age of available time.  Deafult 100.
.TP 11
.B -j
Add pitch jitter to hide artifacts.
.TP 11
.B -v
Verbose mode.
.SH INPUT ITEMS
.IP SP 11
Speech item
.IP TX 11
Pitch epochs.
.IP AN 11
Segment labels.
.SH OUTPUT ITEMS
.IP SP 11
Prosody changed speech
.SH HISTORY
.IP MBROLA=
MBROLA format control file.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
respeed, repitch
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include "sfs.h"
#define MAX(x,y) (((x)>(y))?(x):(y))
#define ABS(x) (((x)>0)?(x):-(x))

/* global defines */
#define MINFX		50.0		/* below this is unvoiced */
#define MAXSTEP		0.016		/* max window step = 16ms */
#define MAXJITTER	0.0003		/* maximum pitch jitter */

/* window record */
struct window_rec {
	int	ssamp;		/* start sample in signal */
	int	lsize;		/* number of samples in left half */
	int	rsize;		/* number of samples in right half */
	int	fx;		/* voicing flag */
	int	sil;		/* silence flag */
};

/* segment record */
struct segment_rec {
	char	*label;		/* segment name */
	double	ostime;		/* original start time */
	double	odur;		/* original duration */
	double	nstime;		/* new start time */
	double	ndur;		/* new duration */
};

/* global data */
struct item_header	spitem;		/* input speech */
struct item_header	txitem;		/* input pitch epochs */
int			*tx;
struct item_header	anitem;		/* input segment labels */
struct an_rec		*an;
short			*ofx;		/* output fx contour */
int			ofxcount;
struct window_rec	*wtab;		/* table of windows */
int			wcount;
struct segment_rec	*stab;		/* table of segments */
int			scount;
struct item_header	opitem;		/* output speech */
short			*isp,*osp;	/* input/output signal */
float			*risp,*rosp;	/* real input/output signal */
float			*wisp,*wosp;	/* real input/output window */
int			bufsize;	/* size of internal buffers */

char			mfilename[SFSMAXFILENAME]; /* control file name */
int			verbose=0;	/* dump debugging info */
int			jitter=0;
int			njitter;
double			txoffset=0;
double			txpercent=1.0;

/* save string in memory */
char *strsave(char *str)
{
	char *ptr=malloc(strlen(str)+1);
	if (ptr==NULL)
		error("out of memory");
	strcpy(ptr,str);
	return(ptr);
}

int issilence(char *label)
{
	return((strcmp(label,"_")==0)||(strcmp(label,"/")==0));
}

int complabel(char *lab1,char *lab2)
{
	if (issilence(lab1)&&issilence(lab2))
		return(0);
	else
		return(strcmp(lab1,lab2));
}

/* load MBROLA control file */
void loadmbrola(char *fname)
{
	FILE	*ip;
	char	line[1024];
	char	label[128];
	char	*p;
	int	dur;
	double	totdur=0;
	int	i,j,idx;
	double	t,pc;
	int	fx;
	int	f1,f2;

	if ((ip=fopen(fname,"r"))==NULL)
		error("could not find '%s'",fname);

	/* find total duration */
	while (fgets(line,1024,ip)) if (line[0]!=';') {
		sscanf(line,"%s %d",label,&dur);
		if (dur==0)
			error("zero length segment in '%s'",fname);
		totdur += dur/1000.0;
	}
#ifdef IAG
	printf("totdur=%g\n",totdur);
#endif
	rewind(ip);

	/* allocate buffer for output Fx */
	ofxcount = (int)(0.5+(0.5+totdur)/0.01);
	if ((ofx=(short *)calloc(ofxcount,sizeof(short)))==NULL)
		error("out of memory");

	/* run through control lines, relating to segments on file */
	stab[0].nstime=0.0;
	for (i=0;fgets(line,1024,ip);) if (line[0]!=';') {
		if (i>0) stab[i].nstime = stab[i-1].nstime+stab[i-1].ndur;
		p = strtok(line," \t\n");
		if (complabel(p,stab[i].label)!=0) {
			if ((i<scount) && issilence(stab[i].label)) {
				i++;
				stab[i].nstime = stab[i-1].nstime+stab[i-1].ndur;
			}
		}
		if ((i < scount) && (complabel(p,stab[i].label)==0)) {
			p = strtok(NULL," \t\n");
			stab[i].ndur = atoi(p)/1000.0;
		}
		else {
			fprintf(stderr,"Line %d. Target:%s Control-File:%s\n",i,stab[i].label,p);
			error("segment mismatch between control file and target file");
		}

		/* store FX values */
		p = strtok(NULL," \t\n");
		while (p && isdigit(*p)) {
			pc = atoi(p)/100.0;
			p = strtok(NULL," \t\n");
			if (p && isdigit(*p)) {
				fx = atoi(p);
				t = stab[i].nstime + pc*stab[i].ndur;
				idx = (int)(t/0.01);
				if ((0 <= idx) && (idx < ofxcount))
					ofx[idx]=fx;
			}
			p = strtok(NULL," \t\n");
		}
		i++;
	}

	/* now join up Fx values with linear interpolation */
	idx=0;
	f1=-1;
	for (i=0;i<ofxcount;i++) {
		f2 = ofx[i];
		if (f2 > 0) {
			if (f1<0) f1=f2;
			for (j=idx;j<i;j++)
				ofx[j] = f1 + (j-idx)*(f2-f1)/(i-idx);
			f1 = f2;
			idx = i;
		}
	}
	for (j=idx;j<ofxcount;j++) ofx[j] = f1;

	fclose(ip);
}

/* find a window to match a time */
int findwindowrec(int samp,struct window_rec *wr)
{
	int	i;
	int	min=20000;
	int	idx=0;
	int	dif;

	if (samp > wtab[wcount-1].ssamp+wtab[wcount-1].lsize+wtab[wcount-1].rsize) return(0);
	for (i=0;i<wcount;i++) {
		dif = wtab[i].ssamp + wtab[i].lsize - samp;
		if (dif < 0) dif = -dif;
		if (dif < min) {
			min = dif;
			idx = i;
		}
	}
	*wr = wtab[idx];
	return(1);
}

/* half window signal */
void halfwindow(short *buf,float *obuf,float *wbuf,int len)
{
	double 	w;
	int	i;

	w = M_PI/(len+1);
	for (i=1;i<=len;i++,buf++,wbuf++,obuf++) {
		*wbuf = (float)(0.5 + 0.5*cos(i*w));
		*obuf = (float)*buf * *wbuf;
	}
}

/* window signal */
void fullwindow(short *buf,float *obuf,float *wbuf,int size1,int size2)
{
	double 	w;
	int	i;

	w = 2.0*M_PI/(2*size1+1);
	for (i=0;i<size1;i++,buf++,wbuf++,obuf++) {
		*wbuf = (float)(0.5 - 0.5*cos((i+1)*w));
		*obuf = (float)*buf * *wbuf;
	}
	w = 2.0*M_PI/(2*size2+1);
	for (i=0;i<size2;i++,buf++,wbuf++,obuf++) {
		*wbuf = (float)(0.5 - 0.5*cos((i+1+size2)*w));
		*obuf = (float)*buf * *wbuf;
	}
}

/* overlap-add two signals */
int	overlapadd(float *s1buf,float *w1buf,int s1len,float *s2buf,float *w2buf,int s2off)
{
	int	i;

	s2buf += s2off;
	w2buf += s2off;
	for (i=0;i<s1len;i++,s1buf++,w1buf++,s2buf++,w2buf++) {
		*s2buf += *s1buf;
		*w2buf += *w1buf;
	}
	return(s2off+s1len);
}

/* re-synthesize */
void outsynth(short *obuf,float *sbuf,float *wbuf,int slen)
{
	int	i;

	for (i=0;i<slen;i++,obuf++,sbuf++,wbuf++) {
		if (*wbuf==0.0) {
/*			fprintf(stderr,"zero window\n"); */
			*obuf = 32767;
		}
		else
			*obuf = (short)(*sbuf / *wbuf);
	}
}

/* find details of next operation */
int findwindow(double ntime,struct window_rec *wrec,double *newrate,double *newpitch)
{
	int	i;
	double	otime;
	double	pc;

	/* search segment table */
	for (i=0;i<scount;i++)
		if ((stab[i].nstime <= ntime) && (ntime < (stab[i].nstime+stab[i].ndur)))
			break;
	if (i==scount) return(0);

	/* find equivalent input time */
	pc = (ntime-stab[i].nstime)/stab[i].ndur;
	otime = stab[i].ostime + pc*stab[i].odur;
#ifdef IAG
printf("Out-time=%g In-time=%g\n",ntime,otime);
#endif
	/* find correct window */
	findwindowrec((int)(otime/spitem.frameduration),wrec);

	/* voiceless windows can go anywhere */
	if (wrec->fx==0) {
		wrec->ssamp = (int)(otime/spitem.frameduration)-wrec->lsize;
		if (wrec->ssamp < 0) wrec->ssamp=0;
	}

	/* force silence to be manufactured */
//	if (issilence(stab[i].label)) {
//		wrec->sil = 1;
//		wrec->fx = 0;
//		wrec->lsize = wrec->rsize = (int)(stab[i].ndur/(2*spitem.frameduration));
//	}
//	else
		wrec->sil = 0;

	/* return prosody parameters */
	*newrate = stab[i].odur/stab[i].ndur;
#if 0
	if (*newrate < 0.5)
		*newrate = 0.5;
	else if (*newrate > 2.0)
		*newrate = 2.0;
#endif
	i = (int)(ntime/0.01);
	if ((0 <= i) && (i < ofxcount))
		*newpitch = ofx[i]/(1.0/(wrec->lsize*spitem.frameduration));
	else
		*newpitch=1;
#if 0
	if (*newpitch < 0.5)
		*newpitch = 0.5;
	else if (*newpitch > 2.0)
		*newpitch = 2.0;
#endif

	return(1);
}

/* align window with output */
int	align(wbuf,wlen,sbuf,slen,omin,omax,otarg)
short	*wbuf;
int	wlen;
short	*sbuf;
int	slen;
int	omin;
int	omax;
int	otarg;
{
	int	i,j;
	float	sum;
	int	offset;
	float	peakval;
	float	ac[2048];
	int	currdist;

	if (omax > slen) omax=slen;
	if (omax > 2045) omax=2045;	/* to fit ac[], OK for 48kHz/32ms */
	if (omax > wlen) omax=wlen-1;
	if (omin >= wlen)
		return(slen-wlen/2);
	offset = (slen<wlen/2) ? 0 : wlen/2;

	for (i=omin-2;i<=omax+2;i++) {
		sum = (float)0.0;
		for (j=0;j<i;j++)
			sum += (float)wbuf[i-j-1] * (float)sbuf[slen-j-1];
		ac[i] = sum/i;
	}

	/* smooth */
	peakval=(ac[omin-2]+ac[omin-1]+ac[omin]+ac[omin+1]+ac[omin+2])/5;
	for (i=omin+1;i<=omax;i++) {
		ac[i] = (ac[i-2]+ac[i-1]+ac[i]+ac[i+1]+ac[i+2])/5;
		if (ac[i]>peakval) peakval=ac[i];
	}

	/* search for peak close to target */
	peakval *= (float)0.75;
	currdist=omax;
	offset=otarg;
	if (verbose==2) printf("Align: min=%d max=%d target=%d peaks at ",omin,omax,otarg);
	for (i=omin;i<=omax;i++) if ((ac[i-1] < ac[i]) && (ac[i] > ac[i+1])) {
		if ((verbose==2) && (ac[i]>peakval))
			printf("%d(%g),",i,ac[i]);
		if ((ac[i]>peakval) && (ABS(otarg-i)<currdist)) {
			currdist = ABS(otarg-i);
			offset = i;
		}
	}
	if (verbose==2) printf(" chosen=%d\n",offset);

	return(offset);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char		*ty;		/* item sub type */
	char		*sptype="0";
	char		*txtype="0";
	char		*antype="0";
	int		epoch=AN_TYPE;
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		ofid;
	double		curtime;
	int		cursamp,newsamp,lastsamp;
	int		framelen,offset;
	struct window_rec wrec;
	double		t0,t1,t2;
	int		i;
	double		newrate,newpitch;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:m:vjO:P:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Change pitch and duration V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype=ty;
				else if (it == TX_TYPE)
					txtype=ty;
				else if (it == AN_TYPE)
					antype=ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'm' :	/* MBROLA control file */
			strcpy(mfilename,optarg);
			break;
		case 'j' :	/* jitter */
			jitter++;
			break;
		case 'v' :	/* verbose */
			verbose++;
			break;
		case 'O' :	/* tx offset */
			txoffset = atof(optarg);
			if ((txoffset < -0.01)||(txoffset > 0.01))
				error("tx offset too large");
			break;
		case 'P' :	/* txpercent */
			txpercent = atof(optarg)/100.0;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-m control_file) (-j) (-v) (-O txoffset) (-P txpercent) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	if (mfilename[0]==0)
		error("no control file specified");

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL))<0)
		error("access error on '%s'",filename);

	/* locate and read Tx */
	if (!sfsitem(fid,TX_TYPE,txtype,&txitem))
		error("unable to find input TX item in '%s'",filename);
	if ((tx = (int *)sfsbuffer(&txitem,txitem.numframes+1))==NULL)
		error("could not get memory");
	if (sfsread(fid,0,txitem.numframes,tx)!=txitem.numframes)
		error("read error on input");

	/* locate and read annotations */
	if (!sfsitem(fid,AN_TYPE,antype,&anitem))
		error("unable to find input AN item in '%s'",filename);
	if ((an = (struct an_rec *)sfsbuffer(&anitem,anitem.numframes))==NULL)
		error("could not get memory");
	if (sfsread(fid,0,anitem.numframes,an)!=anitem.numframes)
		error("read error on input");
	for (i=1;i<anitem.numframes;i++) {
		an[i-1].size = an[i].posn-an[i-1].posn;
	}

	/* locate input speech */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("unable to find input SP item in '%s'",filename);
	an[anitem.numframes-1].size = (int)(spitem.numframes*spitem.frameduration/anitem.frameduration)-an[anitem.numframes-1].posn;

	/* check for zero length tx */
	if (txitem.numframes==0) {
		tx[0] = (int)((spitem.numframes*spitem.frameduration)/txitem.frameduration);
		txitem.numframes=1;
	}

	/* count number of windows we shall need */
	wcount=0;
	t0=0;
	t1=txoffset+tx[0]*txitem.frameduration;
	for (i=1;i<=txitem.numframes;i++) {
		if (i==txitem.numframes)
			t2 = spitem.numframes*spitem.frameduration;
		else
			t2 = t1 + tx[i]*txitem.frameduration;
		wcount++;
		while ((t2-t0) > (2.0/MINFX)) {
			wcount++;
			t0 += MAXSTEP;
		}
		t0 = t1;
		t1 = t2;
	}
#ifdef IAG
	printf("wcount=%d\n",wcount);
#endif

	/* make window records from Tx */
	wtab = (struct window_rec *)calloc(wcount+100,sizeof(struct window_rec));
	wcount=0;
	t0=0;
	t1=txoffset+tx[0]*txitem.frameduration;
	for (i=1;i<=txitem.numframes;i++) {
		if (i==txitem.numframes)
			t2 = spitem.numframes*spitem.frameduration;
		else
			t2 = t1 + tx[i]*txitem.frameduration;

		wtab[wcount].ssamp = (int)(t0/spitem.frameduration);

		if (((t1-t0) <= MINFX)&&((t2-t1) <= MINFX)) {
			/* two voiced cycles */
			wtab[wcount].lsize = (int)(txpercent*(t1-t0)/spitem.frameduration);
			wtab[wcount].rsize = (int)(txpercent*(t2-t1)/spitem.frameduration);
			wtab[wcount].fx = 1;
			wcount++;
		}
		else if (((t1-t0) <= MINFX)&&((t2-t1) > MINFX)) {
			/* voiced then unvoiced cycles */
			wtab[wcount].lsize = (int)(txpercent*(t1-t0)/spitem.frameduration);
			if ((t2-t1) < MAXSTEP)
				wtab[wcount].rsize = (int)((t2-t1)/spitem.frameduration);
			else
				wtab[wcount].rsize = (int)(MAXSTEP/spitem.frameduration);
			wtab[wcount].fx = 1;
			wcount++;
		}
		else if ((t2-t0) < (2.0*MAXSTEP)) {
			/* unvoiced cycle */
			wtab[wcount].lsize = (int)((t1-t0)/(spitem.frameduration));
			wtab[wcount].rsize = (int)((t2-t1)/(spitem.frameduration));
			wtab[wcount].fx = 0;
			wcount++;
		}
		else {
			/* multiple unvoiced cycle */
			wtab[wcount].lsize = (int)(MAXSTEP/spitem.frameduration);
			wtab[wcount].rsize = (int)(MAXSTEP/spitem.frameduration);
			wtab[wcount].fx = 0;
			wcount++;
			t0 += MAXSTEP;
			while ((t2-t0) >= (2.0*MAXSTEP)) {
				wtab[wcount].ssamp = (int)(t0/spitem.frameduration);
				wtab[wcount].lsize = (int)(MAXSTEP/spitem.frameduration);
				wtab[wcount].rsize = (int)(MAXSTEP/spitem.frameduration);
				wtab[wcount].fx = 0;
				wcount++;
				t0 += MAXSTEP;
			}
			wtab[wcount].ssamp = (int)(t0/spitem.frameduration);
			wtab[wcount].lsize = (int)(MAXSTEP/spitem.frameduration);
			if ((t2-t1) < MAXSTEP)
				wtab[wcount].rsize = (int)((t2-t1)/spitem.frameduration);
			else
				wtab[wcount].rsize = (int)(MAXSTEP/spitem.frameduration);
			wtab[wcount].fx = 0;
			wcount++;
		}
		t0 = t1;
		t1 = t2;
	}
#ifdef IAG
	printf("Window table:\n");
	for (i=0;i<wcount;i++)
		printf("%4d. p=%5d s=%5d/%5d f=%d\n",
			i,wtab[i].ssamp,wtab[i].lsize,wtab[i].rsize,wtab[i].fx);
#endif

	/* load annotations */
	scount = anitem.numframes+2;
	if ((stab=(struct segment_rec *)calloc(scount,sizeof(struct segment_rec)))==NULL)
		error("out of memory");
	if (!issilence(an[0].label)) {
		stab[0].label = strsave("_");
		stab[0].ostime = 0.0;
		stab[0].ndur = stab[0].odur = an[0].posn*anitem.frameduration;
		scount=1;
	}
	else
		scount=0;
	for (i=0;i<anitem.numframes;i++) {
		stab[scount].label = strsave(an[i].label);
		stab[scount].ostime = an[i].posn*anitem.frameduration;
		stab[scount].ndur = stab[scount].odur = an[i].size*anitem.frameduration;
		scount++;
	}
	if (!issilence(an[anitem.numframes-1].label)) {
		stab[scount].label = strsave("_");
		stab[scount].ostime = stab[scount-1].ostime+stab[scount-1].odur;
		stab[scount].ndur = stab[scount].odur = 0.0;
		scount++;
	}

	/* load control file */
	loadmbrola(mfilename);

#ifdef IAG
printf("Segment table:\n");
for (i=0;i<scount;i++)
	printf("%d. %s %g %g %g %g\n",
		i,stab[i].label,stab[i].ostime,stab[i].odur,
		stab[i].nstime,stab[i].ndur);
#endif
#ifdef IAG
printf("Fx contour:\n");
for (i=0;i<ofxcount;i++) {
	printf("%3d ",ofx[i]);
	if ((i%10)==9) printf("\n");
}
printf("\n");
#endif
	/* buffer size about 1 seconds worth */
	bufsize = (int)(1.0/spitem.frameduration);

	/* get buffers */
	if ((isp=(short *)sfsbuffer(&spitem,bufsize))==NULL)
		error("could not get memory buffer");
	if ((osp=(short *)sfsbuffer(&spitem,bufsize))==NULL)
		error("could not get memory buffer");
	if ((risp=(float *)calloc(bufsize,sizeof(float)))==NULL)
		error("could not get memory buffer");
	if ((rosp=(float *)calloc(bufsize,sizeof(float)))==NULL)
		error("could not get memory buffer");
	if ((wisp=(float *)calloc(bufsize,sizeof(float)))==NULL)
		error("could not get memory buffer");
	if ((wosp=(float *)calloc(bufsize,sizeof(float)))==NULL)
		error("could not get memory buffer");

	/* make output header */
	sfsheader(&opitem,SP_TYPE,0,2,1,spitem.frameduration,spitem.offset,1,0,0);
	sprintf(opitem.history,"%s(%d.%02d,%d.%02d,%d.%02d;MBROLA=%s%s,txoffset=%g,txpercent=%g)",
			PROGNAME,
			spitem.datatype,spitem.subtype,
			txitem.datatype,txitem.subtype,
			anitem.datatype,anitem.subtype,
			mfilename,(jitter)?",jitter":"",txoffset,100*txpercent);

	/* get output channel */
	if ((ofid=sfschannel(filename,&opitem))<0)
		error("could not open output channel to '%s'",filename);

	/* pre-charge buffer with half a window */
	framelen = sfsread(fid,wtab[0].ssamp,wtab[0].lsize,isp);
	halfwindow(isp,rosp,wosp,framelen);
	lastsamp=0;		/* centre of last window */
	cursamp=framelen;	/* end of last window */
	curtime=framelen*spitem.frameduration;	/* time at end of last window */

	njitter = (int)(0.5+(MAXJITTER/spitem.frameduration));

	/* processing loop */
	while (findwindow(curtime,&wrec,&newrate,&newpitch)) {
#ifdef IAG
printf("Window: out=%g ssamp=%d lsize=%d rsize=%d fx=%d newrate=%g newpitch=%g\n",
	curtime,wrec.ssamp,wrec.lsize,wrec.rsize,wrec.fx,newrate,newpitch);
#endif
		/* load signal */
		framelen = wrec.lsize+wrec.rsize;
		if (wrec.sil) {
			/* need silence */
			for (i=0;i<framelen;i++) isp[i]=0;
		}
		else if (sfsread(fid,wrec.ssamp,framelen,isp)!=framelen) break;

		/* calculate overlap */
		if (wrec.fx==0)
			offset = (int)(wrec.lsize/newrate);
		else
			offset = (int)(wrec.lsize/newpitch);

		/* do synchronisation if required */
		if (jitter) {
			offset = align(isp,framelen,osp,cursamp,offset-njitter,offset+njitter,offset);
		}

		/* check for irregularities */
		if (offset < (wrec.lsize/16)) offset = wrec.lsize/16;
		if (offset > ((31*wrec.lsize)/16)) offset = (31*wrec.lsize)/16;
printf("lastsamp=%d offset=%d lsize=%d\n",lastsamp,offset,wrec.lsize);
		if ((lastsamp+offset-wrec.lsize) > cursamp)
			offset = cursamp-lastsamp+wrec.lsize;
printf("offset=%d\n",offset);

		/* window signal */
		fullwindow(isp,risp,wisp,wrec.lsize,wrec.rsize);

		/* add in */
		if ((lastsamp+offset-wrec.lsize+framelen) >= bufsize) {
			fprintf(stderr,"? had to adjust framelen from %d => %d samples\n",
				framelen,bufsize - (lastsamp+offset-wrec.lsize));
			framelen = bufsize - (lastsamp+offset-wrec.lsize);
		}
		newsamp = overlapadd(risp,wisp,framelen,rosp,wosp,lastsamp+offset-wrec.lsize);

		if (verbose)
			printf("Out %7.4f-%.4f\tIn %7.4f-%.4f\tPeriod %6.4fs.\n",
				curtime-offset*spitem.frameduration,
				curtime+(newsamp-cursamp)*spitem.frameduration,
				wrec.ssamp*spitem.frameduration,
				(wrec.ssamp+framelen)*spitem.frameduration,
				offset*spitem.frameduration);

		/* create 'dummy' output */
		outsynth(osp,rosp,wosp,newsamp);
printf("curtime=%g newsamp=%d cursamp=%d\n",curtime,newsamp,cursamp);
		curtime += (newsamp-cursamp)*spitem.frameduration;
		cursamp = newsamp;
		lastsamp += offset;

		/* purge output from time to time */
		if (cursamp >= (3*bufsize/4)) {
			outsynth(osp,rosp,wosp,bufsize/2);
			/* write out first half */
			sfswrite(ofid,bufsize/2,osp);
			memcpy((char *)osp,(char *)(osp+bufsize/2),bufsize);
			memcpy((char *)rosp,(char *)(rosp+bufsize/2),bufsize*2);
			memcpy((char *)wosp,(char *)(wosp+bufsize/2),bufsize*2);
			memset((char *)(osp+bufsize/2),0,bufsize);
			memset((char *)(rosp+bufsize/2),0,bufsize*2);
			memset((char *)(wosp+bufsize/2),0,bufsize*2);
			cursamp -= bufsize/2;
			lastsamp -= bufsize/2;
		}
	}

	/* write last buffer */
	outsynth(osp,rosp,wosp,cursamp);
	sfswrite(ofid,cursamp,osp);

	/* that's all folks */
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);
	exit(0);
}
