/* soft - "C" implementation of JSRU software formant synthesizer */

/* history:
       - JSRU report 1016 Holmes & Rye
       - "C" version "dbsoft" Bishop & Davies
       - SFS version Huckvale
       - Extended version Judd
       - Extended OP version Breen
*/

/*-------------------------------------------------------------------------*/
/**MAN
.TH SOFT SFS1 UCL
.SH NAME
soft - JSRU parallel-formant speech synthesizer
.SH SYNOPSIS
.B soft
(-I) (-x|-X|-r) (-i item) (-p parfil) (-g glotfil) (-l) (-f female) (-e extended formants) (-t stat_table) file
.SH DESCRIPTION
.I soft
is a software implementation of a parallel-formant speech
synthesizer.  It is a new "C"-language implementation of the FORTRAN
software synthesizer of JSRU report 1016.
.PP
An SY item in the source file
.I file
is converted to an SP item and written back to
.I file.
.PP
.I Options
and their meanings are:
.TP
.B -I
Identify program name and version number, then exit.
.TP 11
.BI -p paramfile
Use the parameter file
.I paramfile
instead of the default parameter file.  More information on the
parameter file may be found in
.B parfil(5).
.TP
.BI -g glottalfile
Read the glottal pulse shape and glottal area from the file
.I glottalfile
instead of the parameter file.  The format for a glottal file is an
ASCII file containing two lists of integers separated by spaces.  The
first list has 72 entries in the range -127 to +127 and represents the
glottal pulse shape.  The second list has 18 entries in the range
0 to 15 and represents the glottal area.
.TP
.B -x
Write the excitation waveform and glottal area to the database file
.I dbasefile.
The excitation and glottal area waveforms are written as LX items.
.TP
.B -r
Read the excitation waveform and glottal area from the database file
.I dbasefile
which must contain two LX items.  The first is assumed to be the excitation,
the last the glottal area.
Selection of input waveforms can be over-ridden by item selections
for LX: the first for excitation, the second for glottal area.
.TP
.BI -i item
Select input item number.
.TP
.B -l
List the parameters and tables configuring the synthesizer to standard output and exit.
.TP
.B -X
Generate the excitation and glottal area only.  No speech item
is generated. The excitation item and the glottal area
item are written back to the database file
.I dbasefile.
.TP
.B -f
With this flag included the AHF formant is placed at 4200 Hz.
.TP
.B -e
This flag includes formants at 3500Hz 4200Hz 5200Hz.
.TP
.B -t stat_table
Use OP stat table file (produced by inlx(1) or statlx(1))
.SH FILES
SFSBASE/data/nparfil - default parameter file.
.SH VERSION/AUTHORS
.TP
dbsoft
1.0 - Mark Bishop and Paul Davies
.br
2.1 - Mark Huckvale
.TP
soft
1.0s - Mark Huckvale
.br
1.1s - Mike Judd
.br
1.2s - Andrew Breen
.SH SEE ALSO
JSRU report 1016 and "dbsoft" manual (Bishop & Davies)
.SH BUGS
The output sampling frequency (srate) cannot be changed (from 20kHz) without changing internal filters.
*/
/*------------------------------------------------------------------------*/

/* program name and version */
#define PROGNAME "soft"
#define PROGVERS "1.2s"
char *progname=PROGNAME;

/* other header files */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard io library */
#include <stdlib.h>
#include <unistd.h>
#include <math.h>		/* maths library */
#include <string.h>		/* string library */
#include <time.h>
#include "sfs.h"		/* speech filing system headers */

/* define boolean type */
typedef char boolean;
#define TRUE  1
#define FALSE 0

/* parameter re-ordering for SY item */
#define FN	0
#define F1	1
#define F2	2
#define F3	3
#define ALF	4
#define A1	5
#define A2	6
#define A3	7
#define AHF	8
#define V	9
#define FX	10
#define MS	11

/* MJ additions */
#define AHFF	12
#define	AF	13

/* macro definitions */
#define	RANGE(lo,val,hi)	(((val)<(lo))?(lo):(((val)>(hi))?(hi):(val)))
#define MIN(x,y)		(((x)<(y))?(x):(y))
#define MAX(x,y)		(((x)>(y))?(x):(y))
#define MOD(x,y)		((x)-(y)*(int)((x)/(y)))

/* boolean variables specifying synthesizer functions
 * from command line switches
 * speech     - TRUE  -> generate and write speech
 * 		FALSE -> don't generate speech
 * print_flag - TRUE  -> print synthesizer parameters
 * readex     - TRUE  -> read excitation from database file
 * 		FALSE -> generate excitation
 * writex     - TRUE  -> write generated excitation to database file
 *
 MJ additions
 * female     - TRUE  -> use "female" AHF
		FALSE -> use "male"   AHF
 * extend     - TRUE  -> use 2 additional HF channels - 6kHz O/P
		FALSE -> as JSRU report -4.5KHz O/P
 */

boolean speech;
boolean print_flag;
boolean	readex;
boolean writex;
/* AB addition */
boolean table;
/* MJ additions */
boolean female;
boolean extend;

/* filenames */
char	dbfilename[SFSMAXFILENAME];		/* data file name */
char	gltname[SFSMAXFILENAME];		/* glottal pulse shape file */
char	paramname[SFSMAXFILENAME];		/* parameter file */
/* AB addition */
char	table_name[SFSMAXFILENAME];		/* stats table file */

/* input file descriptors */
int	dbfid;			/* data file */
FILE	*glt_ptr;		/* glottal pulse file */
int	gifid;			/* glottal area input */
int	xifid;			/* excitation input */
FILE	*param_ptr;		/* parameter file */

/* output file descriptors */
int	ofid;			/* speech output */
int	xofid;			/* excitation output */
int	gofid;			/* glottal area output */

/* item selection variables */
char	*sytype;		/* default: last */
char	*extype;		/* defualt: first */
char	*gatype;		/* default: last */

/* item headers */
struct item_header syitem,exitem,gaitem;

/* SY item input arrays and pointers to current and last frame */
short	ctrl1[16], ctrl2[16], *old_ptr, *new_ptr;

/* waveform arrays */
float	formant[200];		/* input and output of formant resonators */
float	uvox[200];		/* unvoiced excitation */
float	vox[200];		/* voiced excitation */
float	output[200];		/* output of synthesizer */


float	glarea[51];  		/* 1 frames worth of glottal area function,
			    		due to voicing component only.
			    	*/
float 	glamix [51];		/* 1 frames worth of glottal area function,
			    		including equivalent unvoiced component.
			    	*/
int	ictlst[51];  		/* start position of subframes of excitation */
int	ictlfn[51];  		/* end position of subframes of excitation   */
float	srate;			/* Sampling rate (hz)	*/
float	frate;			/* Frame rate    (hz)	*/
float	cratef;			/* glottal function sampling rate */
float 	cratex;			/* excitation sampling rate */
int	nc;			/* No. of control values/frame = (lf-1)/lc+1  */
int	lc;			/* No. of samples/control value		 */
int	lf;			/* No. of samples/frame	= srate/frate  */
int	ifrecl;			/* limit of frequency change without
			    		setting the formant control clamping flag.
			    	*/
int	iampcl;			/* limit of amplitude change without
			    		setting the formant control clamping flag.
			    	*/
int	ipocl;			/* fundamental period clamping limit */
float	gouv;			/* glottal area for unvoiced conditions */
int	length;
float	xscale;  		/* maximum value for output,
			    		suitable for output through D/A
			    	*/
int	isqthr;  		/* frequency maintenance threshold across frames */
int	lsqthr;  		/* as isqthr for silent formants */
/* MJ change */
float	ffq[9][51];		/* Array holding formant frequencies
			    		in costab position
			   	*/
float	costab[257];	 	/* 256 values of cos(2*pi*ffq/srate) */
float	exptab[33];		/* 32  values of exp(2*pi*bandw/srate) */
/* MJ change */
float	gainct [9];		/* formant fixed gain constants	*/
float	flflu  [9];		/* formant l.f. limits in costab positions */
float	finlu  [9];		/* formant frequency steps in costab positions */
float	ffqmod [9];  		/* formant frequency modifications in costab positions */
float	forbdw[9];		/* formant fixed bandwidths in exptab positions */
float	fbwmod[9];		/* formant bandwidth modifications in exptab positions */
float	fbdwna[9];		/* formant bandwidth modifications for nasals */
float	b[9];			/* b and c are delay elements in formant generators */
float	c[9];

float	fqzero;
float	bwzero;
float	dnasal;			/* degree of nasality */
float	fotab[64];		/* 63 values of excitation pitch */
float	amptab[64];		/* 63 values of formant amplitude */
/* AB addition */
float	pltab[64];		/* 63 values of pl lengths in sec. */
/* MJ change */
float	flflim[8];		/* 6  formant l.f. limits in hertz */
float	fincrm[8];		/* 6  formant frequency steps in hertz */
float	bdwtab[8];		/* 6  formant fixed bandwidths (Hz) */
float	bdwmod[8];		/* 6  formant bandwidth modifications (Hz) */
float	bdwnas[8];		/* 6  formant bandwidth modifications for nasal */
float	fqmod[8];		/* 6  formant frequency modifications (Hz) */
float	dvoffs[8];		/* offsets for voiced / unvoiced excitation mixing */
float	fmgain[8];		/* 6 individual gains */

int	ictlof[16];		/* control offsets */
float	grpl; 			/* glottal pulse length gradient */
float	conts; 			/* contant component of glottal pulse length. */
float	foscal, fobase; 	/* values for calculating excitation pitch table */

float	fqlow, fqstep;		/* formant generator cosine table frequencies */
float 	bwlow, bwstep; 		/* formant generator bandwidth table frequencies */
float 	dbstep;			/* number of dB between each control value */
float 	aoffdn;			/* amplitude, frequency and scale for nasality */
float 	foffdn;
float 	scaldn;
float 	expat[72];		/* excitation pattern */
float	gapat[18];		/* glottal area pattern */
float	po[51];			/* period and pulse length control arrays for excitation */
float 	pl[51];
float	pov;			/* saved values of po and pl */
float	plv;
float	posn;			/* current excitation sample position */
int  	npatt;			/* current excitation pattern index */
int	lcx;			/* length of subframe of excitation */
int	lc4;			/* length of glottal control subframe */
int	lfx;			/* length of frame of excitation */
int	ncx; 			/* number of subframes in one frame */
float	usrate;
/* MJ change */
float	gain[8][50];		/* internal controls for amplitude for each channel */
float	voxmix[8];		/* voiced / unvoiced mixing ratio for each channel */


float	pwscl;			/* scale factor for unvoiced excitation */
int	nsil;			/* duration of silence to be replaced by stops */

boolean	uvtov;			/* unvoiced to voiced transition flag */
boolean impexc;			/* single impulse excitation flag (stop) */
boolean clampf;			/* frequency clamping flag */
boolean clampx;			/* excitation clamping flag */
boolean alfcon;			/* enables link between channels ALF channel and FN (params) */
boolean vfr;			/* fully voiced frame */
boolean uvfr;			/* fully unvoiced frame */
boolean uvlfr;			/* last frame unvoiced */

int	ixctl;			/* current control number for excitation */
float	sgap;			/* gap between excitation samples */
float	pgap;			/* gap between excitation pulses */

/* overload check on output */
int	overload;		/* =0 none, =1 correctable, =2 severe */

/* forward declarations */
#ifdef __STDC__
void openoutputs(void);
void write_flshort(int fid,float *buff);
void read_params(void);
void pritab(void);
void make_tables(void);
void read_float (FILE *fp, int n, float *array);
void read_int(FILE *fp, int n, int *array);
void decode_frame(int fno);
void squeak_removal(void);
void controls(void);
void po_filt(float *arrin, float *arrout);
void pl_filt(float *arrin, float *arrout);
void amp_con_filt(int nchan,float *arrin,float *arrout);
void fre_con_filt(int nchan,float *arrin,float *arrout);
void excite(int fno);
void gnoise(void);
void gen_excite(float *store);
void adjust(void);
void excite_filt(float *xin,float *xout);
void mix(int nchan);
void arzero(float *array,int len);
void form_gen(int nchan);
void output_filter(float *wave);
void eoutput_filter(float *wave);
void mixing_filt(int nchan, float *arrin, float *arrout);
void make_op_table(char *name);
int maximum(int *stat_array,int po,int dimension);
#else
void openoutputs();
void write_flshort();
void read_params();
void pritab();
void make_tables();
void read_float();
void read_int();
void decode_frame();
void squeak_removal();
void controls();
void po_filt();
void pl_filt();
void amp_con_filt();
void fre_con_filt();
void excite();
void gnoise();
void gen_excite();
void adjust();
void excite_filt();
void mix();
void arzero();
void form_gen();
void output_filter();
void eoutput_filter();
void mixing_filt();
void make_op_table();
int maximum();
#endif

/* main program */
void main (argc,argv)
int    argc;
char   *argv[];
{
       /* option decoding */
       extern char     *optarg;
       extern int      optind;
       int	     c,errflg=0,first=1;
       int	     it;
       char	    *ty;
       /* processing control */
       int	     frame_no,nchan;

       /* set up default parameter file name */
       strcpy(paramname,sfsbase());
       strcat(paramname,"/data/nparfil");

       /* set up default operation modes */
       speech = TRUE;
       print_flag = FALSE;
       readex = FALSE;
       writex = FALSE;

/* MJ addition */
       female = FALSE;
       extend = FALSE;
/* AB addition */
       table = FALSE;

       /* set up default input item selections */
       sytype="0";	     /* default: last */
       extype="*";	     /* defualt: first */
       gatype="0";	     /* default: last */

       /* decode switches */
       while ((c = getopt (argc, argv, "Ii:p:xg:t:rlXef")) != EOF) switch (c) {

	       case 'I' :      /* identify */
		       fprintf (stderr, "%s : JSRU/Holmes formant synthesizer V%s\n", PROGNAME, PROGVERS);
		       exit (0);
		       break;
/* AB additions */
	       case 't' :      /* specify OP stats table */
		       strcpy(table_name,optarg);
		       table = TRUE;
		       break;
/* MJ additions */
	       case 'f' :      /* use female AHF */
		       female = TRUE;
		       break;

	       case 'e' :      /* use 2 additional channels */
		       extend = TRUE;
		       break;


	       case 'r' :      /* read excitation from file */
		       readex = TRUE;
		       break;

	       case 'x' :      /* write excitation to file */
		       writex = TRUE;
		       break;

	       case 'X' :      /* only write excitation to file - no speech */
		       writex = TRUE;
		       speech = FALSE;
		       break;

	       case 'g' :      /* read glottal pulse shape from specified file */
		       strcpy (gltname, optarg);
		       break;

	       case 'p' :      /* specify alternative parameter file */
		       strcpy (paramname, optarg);
		       break;

	       case 'l' :      /* list parameters */
		       print_flag = TRUE;
		       break;

	       case 'i' :      /* specific sub-item */
		       if (itspec (optarg, &it, &ty) == 0) {
			       if (it == SY_TYPE)
				       sytype = ty;
			       else if (it==LX_TYPE) {
				       if (!first)
					       gatype=ty;
				       else {
					       extype=ty;
					       first=0;
				       }
			       }
			       else
				       error ("unsuitable item specifier %s",optarg);
			  }
			  else
			       error ("illegal item specifier %s", optarg);
			  break;

	       case '?' : /* unknown flag */
			  errflg ++;

       }

       /* check for option decoding error or missing filename */
       if (errflg || (argc < 2))
	       error ("usage : %s (-I) (-i item) (-p paramfile) (-g glotfile) (-t stats table) (-r|-x|-X|-f|-e) (-l) file",PROGNAME);

       /* check if only parameters required */
       if (print_flag) {
	       read_params();
	       pritab();
	       exit(0);
       }

       /* get filename */
       if (optind < argc)
	       strcpy(dbfilename, sfsfile(argv[optind]));
       else
	       error ("no database file specified", NULL);

       /* check option selections */
       if (readex && writex)
	       error ("can't read and write excitation in same run", NULL);

       /* open data file */
       if ((dbfid=sfsopen(dbfilename,"w",NULL)) < 0) {
	       if (dbfid==-1)
		       error("cannot find file %s",dbfilename);
	       else
		       error("access error on %s",dbfilename);
       }

       /* locate input control data */
       if (!sfsitem(dbfid,SY_TYPE,sytype,&syitem))
	       error ("cannot find input SY item in %s", dbfilename);

       /* if excitation input, find and check LX items */
       if (readex) {
	       xifid = sfsdup(dbfid);
	       gifid = sfsdup(dbfid);
	       if (!sfsitem(xifid,LX_TYPE,extype,&exitem))
		       error("cannot find input LX type in %s",dbfilename);
	       if (!sfsitem(gifid,LX_TYPE,gatype,&gaitem))
		       error("cannot find input LX type in %s",dbfilename);
	       if (exitem.subtype==gaitem.subtype)
		       error("only found single LX item in %s",dbfilename);

	       if ((exitem.numframes < syitem.numframes * 200)
		  || (gaitem.numframes < syitem.numframes * 50))
	       error ("%s contains wrong amount of excitation", dbfilename);
	       if (fabs(exitem.frameduration-5.0e-5)>0.1e-5)
		       error("incorrect sampling frequency on excitation",NULL);
	       if (fabs(gaitem.frameduration-2.0e-4)>0.1e-4)
		       error("incorrect sampling frequency on glottal area",NULL);
       }

       /* read synthesizer parameter file */
       read_params();

       /* open output channels */
       openoutputs();

       /* make constant data tables */
       make_tables ();

       /* process input data frame by frame */
       for (frame_no = 0; frame_no < syitem.numframes; frame_no ++) {

	       /* read in one frame */
	       decode_frame(frame_no);

	       /* check slew rates */
	       squeak_removal();

	       /* set synthesizer controls */
	       controls();

	       /* generate excitation */
	       excite(frame_no);

	       /* produce speech waveform if required */
	       if (speech) {
		       /* clear speech buffer */
		       arzero (output, 200);

		       /* generate each channel in turn */
		       for (nchan = 0; nchan <= 7; nchan ++) {

			       /* get excitation mixture */
			       mix (nchan);

			       /* generate formant */
			       form_gen (nchan);

			       /* add result to signal */
			       mixing_filt (nchan, formant, output);
		       }

		       /* perform output filtering */
				/* MJ change */
		       if (extend) eoutput_filter (output);
		       else       output_filter (output);

		       /* write speech to output */
		       write_flshort(ofid,output);

	       }
	       /* report progress */
	       if (ttytest()) {
		       printf ("\rFrame %d/%d",frame_no+1,syitem.numframes);
		       fflush (stdout);
	       }
       }
       if (ttytest()) {
	       printf ("\rFrame %d/%d\n",frame_no,syitem.numframes);
	       fflush (stdout);
       }

       /* check on overload */
       if (overload==1)
	       fprintf(stderr,"%s: warning output close to maximum\n",PROGNAME);
       else if (overload==2)
	       fprintf(stderr,"%s: severe overload detected on output\n",PROGNAME);

       /* update file */
       if (!sfsupdate(dbfilename))
	       error("update error on %s",dbfilename);

       /* that's all folks */
	exit(0);

}

/* open output channels */
void openoutputs()
{
       struct item_header      newitem;

       /* set up excitation and glottal area item headers if necessary */
       if (writex) {
	       /* create output excitation item header */
	       sfsheader(&newitem,LX_TYPE,0,2,1,5.0e-05,syitem.offset,1,0,0);
	       /* make history field for excitation item */
		if(table) {
	       		sprintf(newitem.history,"%s/EX(%d.%02d;parfil=%s,table=%s)",
		       PROGNAME,
		       syitem.datatype,syitem.subtype,
		       paramname,table_name);
		}
		else {
	       		sprintf(newitem.history,"%s/EX(%d.%02d;parfil=%s)",
		       PROGNAME,
		       syitem.datatype,syitem.subtype,
		       paramname);
		}

	       /* open output channel for excitation */
	       if ((xofid=sfschannel(dbfilename,&newitem)) < 0)
		       error("cannot create temporary file",NULL);

	       /* create output glottal area header */
	       sfsheader(&newitem,LX_TYPE,0,2,1,2.0e-04,syitem.offset,1,0,0);
	       /* make history field for glottal area item */
	       if(table) {
	       	  sprintf(newitem.history,"%s/GA(%d.%02d;parfil=%s,table=%s)",
	       	       PROGNAME,
		       syitem.datatype,syitem.subtype,
		       paramname,table_name);
	       }
	       else {
		  sprintf(newitem.history,"%s/GA(%d.%02d;parfil=%s)",
		       PROGNAME,
		       syitem.datatype,syitem.subtype,
		       paramname);
	       }

	       /* open output channel for glottal area */
	       if ((gofid=sfschannel(dbfilename,&newitem)) < 0)
		       error("cannot create temporary file",NULL);
       }

       /* open output channel for speech item */
       if (speech) {
	       /* set up item header for speech */
	       sfsheader(&newitem,SP_TYPE,0,2,1,5.0e-05,syitem.offset,1,0,0);
	       /* set up history field for speech */
	       if (readex) {
		if( !female && !extend ) {
		  if(table) {
		       sprintf(newitem.history,"%s/SP(%d.%02d,%d.%02d,%d.%02d;parfil=%s,table=%s)",
			       PROGNAME,
			       syitem.datatype,syitem.subtype,
			       exitem.datatype,exitem.subtype,
			       gaitem.datatype,gaitem.subtype,
			       paramname,table_name);
		  }
		  else {
		       sprintf(newitem.history,"%s/SP(%d.%02d,%d.%02d,%d.%02d;parfil=%s)",
			       PROGNAME,
			       syitem.datatype,syitem.subtype,
			       exitem.datatype,exitem.subtype,
			       gaitem.datatype,gaitem.subtype,
			       paramname);
		  }
		}
		else if( female && !extend ) {
		  if(table) {
		       sprintf(newitem.history,"%s/SP(%d.%02d,%d.%02d,%d.%02d;parfil=%s,tabl=%s,female)",
			       PROGNAME,
			       syitem.datatype,syitem.subtype,
			       exitem.datatype,exitem.subtype,
			       gaitem.datatype,gaitem.subtype,
			       paramname,table_name);
		  }
		  else {
		       sprintf(newitem.history,"%s/SP(%d.%02d,%d.%02d,%d.%02d;parfil=%s,female)",
			       PROGNAME,
			       syitem.datatype,syitem.subtype,
			       exitem.datatype,exitem.subtype,
			       gaitem.datatype,gaitem.subtype,
			       paramname);
		  }
		}
		else {
		  if(table) {
		       sprintf(newitem.history,"%s/SP(%d.%02d,%d.%02d,%d.%02d;parfil=%s,table=%s,extended)",
			       PROGNAME,
			       syitem.datatype,syitem.subtype,
			       exitem.datatype,exitem.subtype,
			       gaitem.datatype,gaitem.subtype,
			       paramname,table_name);
		  }
		  else {
		       sprintf(newitem.history,"%s/SP(%d.%02d,%d.%02d,%d.%02d;parfil=%s,extended)",
			       PROGNAME,
			       syitem.datatype,syitem.subtype,
			       exitem.datatype,exitem.subtype,
			       gaitem.datatype,gaitem.subtype,
			       paramname);
		  }
		}
	       }
	       else {
		if( !female && !extend ) {
		  if(table) {
		       sprintf(newitem.history,"%s/SP(%d.%02d;parfil=%s,table=%s)",
			       PROGNAME,
			       syitem.datatype,syitem.subtype,
			       paramname,table_name);
		  }
		  else {
		       sprintf(newitem.history,"%s/SP(%d.%02d;parfil=%s)",
			       PROGNAME,
			       syitem.datatype,syitem.subtype,
			       paramname);
		  }
		}
		else if( female && !extend ) {
		  if(table) {
		       sprintf(newitem.history,"%s/SP(%d.%02d;parfil=%s,table=%s,female)",
			       PROGNAME,
			       syitem.datatype,syitem.subtype,
			       paramname,table_name);
		  }
		  else {
		       sprintf(newitem.history,"%s/SP(%d.%02d;parfil=%s,female)",
			       PROGNAME,
			       syitem.datatype,syitem.subtype,
			       paramname);
		  }
		}
		else {
		  if(table) {
		       sprintf(newitem.history,"%s/SP(%d.%02d;parfil=%s,table=%s,extended)",
			       PROGNAME,
			       syitem.datatype,syitem.subtype,
			       paramname,table_name);
		   }
		   else {
		       sprintf(newitem.history,"%s/SP(%d.%02d;parfil=%s,extended)",
			       PROGNAME,
			       syitem.datatype,syitem.subtype,
			       paramname);
		  }
		}
	       }
	       /* open output channel for speech */
	       if ((ofid=sfschannel(dbfilename,&newitem)) < 0)
		       error("ccannot open temporary file",NULL);
       }

}


/*
 * write_flshort ()
 * converts a float array to shorts and writes to the file
 * pointed to by fid
 */

void write_flshort(fid,buff)
int    fid;
float  *buff;
{
       register int    i;
       short	   out[200];

       /* copy floats to shorts */
       for (i = 0; i < lf; i++) {
	       if ((*buff >= -4095.0) && (*buff <= 4095.0)) {
		       out[i] = (short) (*buff * 8);
		       if (fabs(*buff) > 2048) overload=1;
	       }
	       else {
		       out[i] = 0;
		       overload=2;
	       }
	       buff++;
       }

       /* write output record */
       if (sfswrite(fid,lf,out) != lf)
	       error("write error on temporary file",NULL);

}

/* read synthesizer parameters from definition file */
void read_params ()
{
	float   foi, sil, gclf;
	int     i, nfin, nstt, flag;
	int     iexpat[72], igapat [18];
	int     correct_lcx;

	/* open the parameter file */
	if ((param_ptr = fopen(paramname, "r")) == NULL)
		error( "can't open parameter file '%s'",paramname);

	/* read in general parameters */
	if (fscanf(param_ptr,"%f srate%f frate%f cratef%f cratex",
		&srate,&frate,&cratef,&cratex) != 4)
		error("error reading general variables from '%s'",paramname);

	if (fscanf(param_ptr,"%f fqlow%f fqstep%f bwlow%f bwstep%f foscal%f fobase%f dbstep%f aoffdn%f foffdn%f scaldn%f conts%f grpl",
		&fqlow,&fqstep,&bwlow,&bwstep,
		&foscal,&fobase,&dbstep,&aoffdn,
		&foffdn,&scaldn,&conts,&grpl) !=12)
		error("error reading general variables from '%s'",paramname);

	if (fscanf(param_ptr,"%f xscale%f foi%f gclf%f sil%f gouv",
		&xscale,&foi,&gclf,&sil,&gouv) !=5)
		error("error reading general variables from '%s'",paramname);

	if (fscanf(param_ptr,"%d isqthr%d lsqthr",&isqthr,&lsqthr) != 2)
		error("error reading general variables from '%s'",paramname);

	/* control parameter offsets */
	fscanf(param_ptr," ictlof ");
/* MJ change */
	for (i = FN ; i <= AF ; i++ ) {
		if (fscanf(param_ptr,"%d",&ictlof[i]) != 1)
			error("error reading control signal offsets from '%s'",paramname);
	}

	fscanf(param_ptr," flflim ");
/* MJ changes */
	read_float( param_ptr, 8, flflim); /* reads in 8 floats to array flflim */
	fscanf(param_ptr," fincrm ");
	read_float( param_ptr, 8, fincrm);
	fscanf(param_ptr," bdwtab ");
	read_float( param_ptr, 8, bdwtab);
	fscanf(param_ptr," bdwmod ");
	read_float( param_ptr, 8, bdwmod);
	fscanf(param_ptr," bdwnas ");
	read_float( param_ptr, 8, bdwnas);
	fscanf(param_ptr," fqmod ");
	read_float( param_ptr, 8, fqmod );
	fscanf(param_ptr," dvoffs ");
	read_float( param_ptr, 8, dvoffs);
	fscanf(param_ptr," fmgain ");
	read_float( param_ptr, 8, fmgain);

	/* smoothing filter clamping limits */

	if (fscanf(param_ptr,"%d ifrecl%d iampcl%d ipocl",
		&ifrecl,&iampcl,&ipocl) != 3)
		error("error reading smoothing filter clamping limits from '%s'", paramname);

	/* operation control flag */
	if ( fscanf(param_ptr,"%d alfcon",&flag)!= 1)
		error("error reading operation control flag (alfcon) from '%s' ", paramname);

	alfcon = (boolean) flag;

	/* get glottal data from paramfile or user-file */
	if (*gltname != '\0' ) {
		/* open glottal pulse file */
		if ((glt_ptr = fopen (gltname, "r")) == NULL)
			error("cannot open glottal file '%s'",gltname);

		/* read glottal data from glottal file */
		read_int(glt_ptr,72,iexpat);
		read_int(glt_ptr,18,igapat);

		fclose(glt_ptr);
	}
	else {
		/* read glottal data from parameter file */
		fscanf(param_ptr," expat ");
		read_int(param_ptr,72,iexpat);
		fscanf(param_ptr," gapat ");
		read_int(param_ptr,18,igapat);
	}

	/* scale glottal pulse */
	for (i = 0; i < 72; i++)
		expat [i] = iexpat [i] * xscale / 127.0;
	for (i = 0; i < 18; i++)
		gapat [i] = igapat [i] / 15.0;

	/* close parameter file */
	fclose(param_ptr);

	/* check general parameters */
	if ((int)(srate+0.5) != 20000)
		error("Main sampling rate must be 20000", NULL);
	if (frate <= 0.0)
		error("Frame rate < 0", NULL);
	if (cratef <= 0.0)
		error("Formant smoothed control rate < 0.0", NULL);
	if (cratex <= 0.0)
		error("Excitation smoothed control rate < 0.0", NULL);
	if (cratef < frate)
		error("Formant control rate < frame rate", NULL);
	if (cratex < frate)
		error("Excitation control rate < frame rate", NULL);

	/* Check integer number of samples per frame or frames not too long */
	lf = (int)(srate / frate + 0.5 );
	if (MOD(srate,frate) != 0.0 )
		fprintf(stderr,"%s: warning - non integer number of samples per frame\nMain sampling rate adjusted to %f Hz\n",PROGNAME,lf * frate);
	srate = frate * lf;
	if (lf > 200)
		error ("frame rate too low: frame > 200 samples", NULL);

	/* check integer number of samples per control */
	lc = (int)(srate / cratef + 0.5);

	if (MOD(srate,cratef ) != 0.0 )
		fprintf(stderr,"%s: warning - Formant smoothed control rate not submultiple of main rate\nControl rate adjusted to %f Hz\n",PROGNAME,srate/lc);

	correct_lcx = (int)(4.0 * srate / cratex + 0.5);
	if (MOD(4*srate, cratex) != 0.0)
		fprintf(stderr,"%s: warning - Excitation smoothed control rate not a submultiple of higher excitation sampling rate, adjusted to %f Hz\n",PROGNAME,4*srate / correct_lcx);

	cratef = srate /lc;
	cratex = 4.0 * srate / correct_lcx ;

	/* check control rates not too high */
	if ( cratef / frate > 50.0 )
		error (">50 smoothed formant controls per frame", NULL);
	if (cratex / frate > 50.0)
		error (">50 smoothed excitation controls per frame", NULL);

	/* calculate dependent parameters */
	lc = (int)(srate  / cratef );
	lf = (int)(srate  / frate + 0.5);
	foi = MAX(0.0 , foi);
	pwscl = xscale * sqrt( foi/srate);
	nsil  = (int)(sil * srate / 1000.0);
	nsil  = MIN(lf,nsil);

	if (dbstep != 0.0 )
		aoffdn /= dbstep;

	if (fincrm [0] != 0.0)
		foffdn = (foffdn - flflim [0] ) / fincrm [0];

	conts /= 1000.0;

	nc = (lf - 1)/lc + 1;

	nfin = 0;
	for (i = 0 ; i < nc ; i++ ) {
		nstt = nfin + 1;
		nfin += lc;
		if (nfin > lf) nfin = lf;
		ictlst[i] = nstt-1;
		ictlfn[i] = nfin-1;
	}

	fobase = fabs (fobase);
	foscal = fabs (foscal);
	usrate = 4.0 * srate;
	lcx    = (int)(usrate / cratex);
	lfx    = lf * 4;
	ncx    = (lfx - 1)/ lcx + 1;
	lc4    = lc * 4;
	pov    = 0.01;
	plv    = 0.007;

	gainct [0] = gclf;
	gainct [1] = 1.0;
	gainct [2] = 1.0;
	gainct [3] = 1.0;
	gainct [4] = 1.0;
	gainct [5] = gclf;

/* MJ additions */
	gainct [6] = 1.0;
	gainct [7] = 1.0;


 }

 /*
  * pritab ()
  *
  * Routine prints out lookup tables from make_tables on standard output.
  *
  */

void pritab ()
{
	int     i;
	char    *today, *ctime ();
	time_t    seconds;

	/* get today's date */
	time (&seconds);
	today = ctime (& seconds);

	/* print parameter information */
	printf ("Synthesizer Parameters		  %s",today);
	printf ("----------------------------------------------------------------\n");
	printf ("File				    = %s\n", paramname);
	printf ("Sampling rate			   = %g\n", srate);
	printf ("Frame rate			      = %g\n", frate);
	printf ("Frame length			    = %d\n", lf);
	printf ("Frame segment			   = %d\n", lc);
	printf ("Number of controls per frame	    = %d\n", nc);
	printf ("Excitation sampling rate		= %g\n", usrate);
	printf ("Excitation frame length		 = %d\n", lfx);
	printf ("Excitation frame segment		= %d\n", lcx);
	printf ("Number of excitation controls per frame = %d\n", ncx);
	printf ("Excitation maximum amplification	= %g\n", xscale);
	printf ("Unvoiced / voiced amplification ratio   = %g\n", pwscl / xscale);
	printf ("Constant component of glottal pulse len.= %g\n", conts);
	printf ("Slope component of glottal pulse length = %g\n", grpl);
	printf ("Formant Frequency clamping limit	= %d\n", ifrecl);
	printf ("Formant Amplitude clamping limit	= %d\n", iampcl);
	printf ("Excitation period clamping limit	= %d\n", ipocl);
	printf ("Squeak remover threshold		= %d\n", isqthr);
	printf ("\n");
	printf ("Formant Parameters:\n");
	printf ("-------------------\n");

	printf ("#\tFlflim\tFincrm\tBdwtab\tBdwmod\tBdwnas\tFqmod\tFmgain\tVoffset\n");
	printf ("-----------------------------------------------------------------------\n");

/* MJ change */
	for (i = 0; i <= 7; i++)
		printf ("%d\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\t%.1f\n",
			i, flflim [i], fincrm [i], bdwtab [i],
			bdwmod [i], bdwnas [i], fqmod [i],
			fmgain [i], dvoffs [i]);

	printf("\n");
 }


 /* make_tables ()
  * This subroutine produces lookup tables 'costab' , 'exptab' from which
  * feedback constants in the formant generator 'form_gen' are
  * calculated , also lookups to which all the control signal arrays are
  * referred.
  *
  */

#define pi 3.1415927

void make_tables ()
{
	int	     i;
	float	   fscale, bscale;

	for (i = 0 ; i <= 255 ; i++)
		costab [i] = cos( (double) (2.0 * pi * (fqlow + fqstep * i) / srate) );

	for (i = 0 ; i<= 31 ; i++)
		exptab [i] = exp( (double) ( -pi * (bwlow + bwstep * i)/ srate ) );

	fscale  = 0.0;
	bscale  = 0.0;
	if ( fqstep != 0.0 )
		fscale = 1.0 / fqstep;
	if ( bwstep != 0.0 )
		bscale = 1.0 / bwstep;

	bwzero = bscale * bwlow - 1.5;
	fqzero = fscale * fqlow - 1.5;

/* MJ change */
	for (i = 0; i <= 7 ; i++) {
		gainct [i]  = gainct [i] * fmgain[i];
		/* scale formant frequencies and bandwidths to table steps */
		forbdw [i]  = bdwtab [i] * bscale - bwzero;
		fbwmod [i]  = bdwmod [i] * bscale ;
		fbdwna [i]  = bdwnas [i] * bscale ;
		flflu  [i]  = flflim [i] * fscale - fqzero;
		finlu  [i]  = fincrm [i] * fscale ;
		ffqmod [i]  = fqmod  [i] * fscale ;
	}

	/* scale fundamental frequency and amplitudes */
	for (i = 0 ; i <= 63 ; i++) {
		fotab [i]   = foscal * pow (fobase,(i+1)/16.0);
		amptab[i]   = pow ( 10.0 , (float)(i-63) * dbstep / 20.0);
	}
	amptab [1] = 0.0;
	amptab [2] = 0.0;

	/* AB addition */
	/* make pl table */

	if(table)
		make_op_table(table_name);

	/* initialise frame pointers and initial "last frame" */
	old_ptr = ctrl1;
	new_ptr = ctrl2;

/* MJ change */
	for (i = FN; i <= AF; i++)
		old_ptr[i] = -63;

 }

 /* read in array of floats */
void read_float (fp, n, array)
 FILE	   *fp;
 int	    n;
 float	  *array;
 {
	while (n-- > 0)
		if (fscanf (fp, "%f", array++) != 1)
			error("error reading '%s'",paramname);
 }

 /* read in array of integers */
void read_int( fp, n, array)
 FILE	   *fp;
 int	    n;
 int	    *array;
 {
	while (n-- > 0)
		if (fscanf (fp, "%d", array++) != 1)
			error("error reading file",NULL);
 }


void decode_frame(fno)
int fno;
{
	short   *temp_ptr;
	short   frame[100];
	double  fxval,log();

	/* get record at frame 'fno' */
	if (sfsread(dbfid,fno,1,frame) != 1)
		     error("read error on SY input",NULL);

	/* swap current encoded frame with last */
	temp_ptr = old_ptr;
	old_ptr = new_ptr;
	new_ptr = temp_ptr;

	/* encode current frame */
	*temp_ptr++ = (frame [15] - 90) / 5;   /* FN */
	*temp_ptr++ = (frame [3] - 100) / 25;  /* F1 */
	*temp_ptr++ = (frame [6] - 500) / 50;  /* F2 */
	*temp_ptr++ = (frame [9] - 1300) / 50; /* F3 */
	*temp_ptr++ = frame [16] / 10;	 /* ALF */
	*temp_ptr++ = frame [4] / 10;	  /* A1 */
	*temp_ptr++ = frame [7] / 10;	  /* A2 */
	*temp_ptr++ = frame [10] / 10;	 /* A3 */

/* MJ change - control signal for original HF channel (AHF) */
	if ( female && !extend ) *temp_ptr++ = 0;  /* "female" option */
	else
	*temp_ptr++ = frame [13] / 10;	     /* "male" option */

	*temp_ptr++ = frame [18] / 4;	  /* V */
	fxval = frame[0];		    /* FX */

	*temp_ptr++ = fxval == 0.0 ? 0 : (short)(23.09 * log(fxval) - 75.37);

	/* Mark/Space ratio (fixed) */
	*temp_ptr++ = 32;


/* MJ additions - these are control signals for the new HF channels */


	/* option to use 2 additional HF channels */
/* **IMPORTANT**  this option assumes that frame [22] and frame [25] hold amplitude values for 2 additional HF channels */
	if ( extend ) {
	     *temp_ptr++ = frame [22] / 10;
	     *temp_ptr   = frame [25] / 10;
	}

	/*  normal "female" option */
	if ( female && !extend ) {
	     *temp_ptr++ = frame [13] / 10;
	     *temp_ptr   = 0;
	}

	/*  normal "male" option (default) */
	if ( !female && !extend ) {
	     *temp_ptr++ = 0;
	     *temp_ptr   = 0;
	}
}

/*
 * squeak_removal ()
 *
 * removes rapid jumps from incoming synthesizer control parameters
 */

void squeak_removal ()
{
	int     i;

	for (i = ALF; i <= A3; i++) {
		if ((new_ptr [i] <= (old_ptr [i] - isqthr))
			|| ((new_ptr [i] == 1) && (old_ptr [i] > lsqthr))) {
			new_ptr [i-4] = old_ptr [i-4];
		}
	}


}

/*
 * controls ()
 * Calculates synthesizer controls from control data
 * and sets flags depending on control conditions
 */

void controls ()
{
	float	   povar,plvar,vcvar;
	int	     i, j, channel, ind;
	float	   any[51];
	float	   x, y;

	if (new_ptr[FX]>63) new_ptr[FX]=63;
	if(table)
		plvar = pltab[new_ptr[FX]];
	else
		plvar = conts + grpl / fotab [(new_ptr [MS] - 32) / 2 + new_ptr [FX]];
	povar = 1.0 / fotab[new_ptr[FX]];
	vcvar = (float) (new_ptr [V])/2.0-21.0;

	/* calculate controls for each formant */
	for (channel = FN; channel <= F3; channel++) {
		if (channel == FN)
			ind = 5;
		else
			ind = channel-1;

		voxmix [ind] = sqrt (RANGE(0.0, (dvoffs [ind] + vcvar) / 10.0, 1.0));
		for (j = 0; j < nc; j++) {
			ffq [ind][j] = flflu [ind] + (float) (new_ptr [channel] - 1) * finlu [ind];


			gain [ind][j] = amptab [new_ptr [channel+4]];
		}
	}

	/* F4  and FHFM */
	for (ind = 3; ind <= 4; ind ++) {

/*MJ changes  */
		 for (j = 0; j < nc; j++)
			 gain [ind][j] = amptab [new_ptr [AHF]];

		for (j = 0; j < nc; j++) {

			ffq [ind][j] = flflu [ind];

		}


		voxmix [ind] = sqrt (RANGE(0.0, (dvoffs [ind] + vcvar) / 10.0, 1.0));
	}

	for (j = 0; j < nc; j++) {

			gain [3][j] = 0.0;  /* turn F4 off, always use FHFM */

	 }

	 /* FHFF AND FF */
	 for (ind = 6; ind <=7; ind++) {

		  for (j = 0; j < nc; j++ )
			 gain [ind][j] = amptab [new_ptr [ind + 6] ];

		  for (j = 0; j < nc; j++ ) {

			 ffq [ind][j] = flflu [ind];


		  }

		  voxmix [ind] = sqrt (RANGE(0.0, (dvoffs [ind] + vcvar) / 10.0, 1.0));

	}


	for (j = 0; j < ncx; j++) {
		po [j] = povar;
		pl [j] = plvar;
	}

	/* calculate degree of nasality */
	x = MAX(0.0, ((float) new_ptr [ALF] - (float) new_ptr [A1] - aoffdn));
	y = MAX(0.0, (float) ((float) new_ptr [F1]  - foffdn));
	dnasal = MIN(1.0, x * y / scaldn);

	/* set control flags */
	/* clamp formant controls - due to frequency jumps */
	clampf = FALSE;

	for (j = FN; j <= F3; j++)
	       clampf = clampf || (abs (new_ptr [j] - old_ptr [j]) >= ifrecl);

	/* clamp formant controls - due to amplitude jumps */
	for (j = ALF; j <= AHF; j++)
		clampf = clampf || (abs (new_ptr [j] - old_ptr [j]) >= iampcl);

 /* MJ addition */
	for (j = AHFF; j <= AF; j++)
		clampf = clampf || (abs (new_ptr [j] - old_ptr [j]) >= iampcl);

	/* excitation clamping */

	clampx = abs (new_ptr [FX] - old_ptr [FX]) >= ipocl;

	/* completely voiced or unvoiced frame */
	uvlfr = uvfr;
	uvfr = TRUE;
	vfr = TRUE;


/* MJ change */
	for (i=0; i<=7; i++) {
		uvfr = uvfr && (voxmix [i] == 0.0); /* totally unvoiced */
		vfr = vfr && (voxmix [i] == 1.0);   /* totally   voiced */
	}

	/* unvoiced to voiced transition */
	uvtov = uvlfr && (~uvfr);

	/* unvoiced / voiced single impulse */
/* MJ note: I have not changed this criterion */
	impexc = FALSE;
	j = 0;
	for (i = ALF; i <= AHF; i++) {
		if ((new_ptr[i] - old_ptr[i]) >= iampcl)
			j++;
	}
	if (j>=2)
		impexc = TRUE;


	/* smooth synthesizer controls */
	if (clampf) {
		/* clamp formant controls */
		for (i = 1; i <= 20; i++) {

 /* MJ changes */
			for (j = 0; j<= 7; j++) {

				amp_con_filt (j, &gain[j][0], any );
				if ( (j <= 2)  ||  (j==5)  )
					fre_con_filt (j, &ffq[j][0], any );
			}
		}
	}

/* MJ changes */

	else

	/* control signal smoothing filters */
	for (j = 0; j<= 7; j++) {

		amp_con_filt (j, &gain[j][0], &gain[j][0] );

		if ( (j <= 2)  ||  (j==5) )

		       fre_con_filt (j, &ffq[j][0], &ffq[j][0] );

	}

	if (clampx) {
		/* clamp excitation controls */
		for (i=1; i <= 20; i++) {
			po_filt (po, any );
			pl_filt (pl, any );
		}
	}

/* MJ change */
	else {

	po_filt (po, po );
	pl_filt (pl, pl );

	}


}

void po_filt ( arrin, arrout )
float   *arrin, *arrout;
{
/**FILTER					*/
/* Excitation control smoothing filters	  */
/* Both filters contain only a double real pole. */
/* Sampling rate 2 kHz			   */
/* Gain factor   1 at 0 Hz		       */
/* Poles (-14.0, 0) (-14.0, 0)		   */

	static float    x1 = 0.0, x2 = 0.0;
	register float  h;
	register int    i;

	for ( i = 0; i < ncx; i++ ) {
		h = *arrin++ * 0.00185156 + 1.9139418 * x1 - 0.9157933 * x2;
		*arrout++ = h;

		x2 = x1;
		x1 = h;
	}


}

void pl_filt ( arrin, arrout )
float   *arrin, *arrout;
{
/**FILTER					*/
/* Excitation control smoothing filters	  */
/* Both filters contain only a double real pole. */
/* Sampling rate 2 kHz			   */
/* Gain factor   1 at 0 Hz		       */
/* Poles (-14.0, 0) (-14.0, 0)		   */

	static float    x1 = 0.0, x2 = 0.0;
	register float  h;
	register int    i;

	for ( i = 0; i < ncx; i++ ) {
		h = *arrin++ * 0.00185156 + 1.9139418 * x1 - 0.9157933 * x2;
		*arrout++ = h;
		x2 = x1;
		x1 = h;
	}


}

/**FILTER				   */
/* formant control signal smoothing filters */
/* all filters contain only a double	*/
/* real pole				*/
/* sampling rate 5 kHz		      */
/* gain factor 1.0 at 0 Hz		  */
/* poles (-47.0, 0 ), (-47.0, 0 )	   */

void amp_con_filt ( nchan, arrin, arrout )
int nchan;
float *arrin, *arrout;
{

/* MJ changes */

	static float    last1[8];
	static float    last2[8];

	register float  h, x1, x2;
	register int    i;

	x1 = last1 [ nchan ];
	x2 = last2 [ nchan ];

	for ( i = 0; i < nc; i++ ) {
		h = *arrin++ * 0.00328928 + 1.88529678 * x1 - 0.888586 * x2;
		*arrout++ = h;
		x2 = x1;
		x1 = h;
	}

	last1 [ nchan ] = x1;
	last2 [ nchan ] = x2;


}

/**FILTER				   */
/* formant control signal smoothing filters */
/* all filters contain only a double	*/
/* real pole				*/
/* sampling rate 5 kHz		      */
/* gain factor 1.0 at 0 Hz		  */
/* poles (-47.0, 0 ), (-47.0, 0 )	   */

void fre_con_filt ( nchan, arrin, arrout )
int     nchan;
float   *arrin, *arrout;
{

/* MJ changes */

	static float    last1[8];
	static float    last2[8];

	register float  h, x1, x2;
	register int    i;

	x1 = last1 [ nchan ];
	x2 = last2 [ nchan ];

	for ( i = 0; i <= nc; i++ ) {
		h = *arrin++ * 0.00328928 + 1.88529678 * x1 - 0.888586 * x2;
		*arrout++ = h;
		x2 = x1;
		x1 = h;
	}

	last1 [ nchan ] = x1;
	last2 [ nchan ] = x2;


}

 /*
  * excite ()
  *
  * This subroutine prepares one frame of voiced excitation in array
  * 'store' at a sampling rate of four times the main rate, prior to
  * l.p. filtering and down sampling at the main rate.
  * A pulse shape read from array 'expat' is distributed throughout
  * 'store' with period 'po' and pulse length 'pl' (in seconds).
  * Common variables 'npatt' and 'posn' are used to remember 'expat'
  * sample number and actual (real number) associated 'store' position
  * at frame boundaries.
  *
  */

void excite (fno)
 int    fno;		    /* frame number */
 {
	int     i, j;
	float   store [801]; /* temporary array to store pre-downsampled excitation */
	float   any [801];   /* output of clamping if required */
	short   ga_io [201], vox_io [201];

	/* generate unvoiced excitation if required */
	if (! vfr) {
		arzero (uvox, lf);
		gnoise ();
	}

	if (readex) {
		if (sfsread(xifid,fno*lf,lf,vox_io) != lf)
			error("read error on input LX item",NULL);
		if (sfsread(gifid,fno*nc,nc,ga_io) != nc)
			error("read error on input LX item",NULL);
		if ( !uvfr ) {
			for ( i = 0; i < lf; i++)
				vox [i] = (float) vox_io [i] ;
			for ( i = 0; i < nc; i++)
				glarea [i] = 1/2000.0 * (float) ga_io [i];
		}
	}
	else {
		arzero ( store, lfx);
		arzero ( glarea, nc);
		if ( !uvfr ) {
			gen_excite ( store );

			/* filter excitation */
			excite_filt ( store, store );

			/* down sample to main rate */
			i = 0;
			for ( j = 4; j <= lfx; j += 4 )
				vox [ i++ ] = store [j-1];
		}
		else {
			if ( ! uvlfr ) {
				for ( i = 1; i <= 6; i++)
					excite_filt (store, any);
				arzero (vox, lf);
			}
		}

		if ( writex ) {
			for( i = 0; i < lf; i++)
				vox_io [ i ] = (short) vox [i];
			for ( i = 0; i < nc; i++)
				ga_io [ i ] = (short) (2000.0 * glarea [i] );
			if (sfswrite(xofid,lf,vox_io) != lf)
				error("write error on temporary file",NULL);
			if (sfswrite(gofid,nc,ga_io) != nc)
				error("write error on temporary file",NULL);
		}
	}
 }

 /*
  * gnoise ()
  *
  * provides one frame of unvoiced excitation held in array 'uvox' with the
  * same power of the voiced excitation.
  */

void gnoise ()
 {
	register int   sum;
	register int   i, j;
	int	    istart;

	istart = 0;
	if (impexc)
		istart = nsil;

	if (istart != lf) {
		for (i = istart; i < lf; i++) {
			sum = 0;
			for ( j = 0; j <= 11; j++)
				sum += rand() & 0x7FFF;
			uvox [i] = (((float) sum) / 32768.0 - 6.0 ) * pwscl;
		}
	}

	if (impexc)
		uvox [9 * nsil / 10] = pwscl * sqrt ((double) nsil);

 }


 /*
  * gen_excite ()
  * routine generates one frame of excitation in array 'store'
  */

void gen_excite ( store )
 float store [];
 {
	float   sample;
	int     nctl, ip1, ip2, i, j, iga, nseg;

	ixctl = 0;
	if (posn < 0.0)
		error ("From subroutine excite - negative value for sample position", NULL);

	if (uvtov) {
		posn = lfx / 4.0 ;
		npatt = 0;
	}
	if (posn < 1.0) {
		sgap = usrate * pl [0] / 72.0 ;
		pgap = usrate * ( po [0] - pl [0] );
	}
	else {
		adjust();
	}

	while (TRUE) {
		nctl = ((int) (posn - 0.5 ) / lc4) ;
		for ( i = npatt ; i <= 71 ; i++ ) {
			nseg = ( (int) (posn - 0.5 ) / lc4 );
			if ( nctl < nc ) {
				if (nseg >= nc )
					nseg = nc - 1;
				iga = i / 4 ;
				for ( j = nctl ; j <= nseg ; j++ )
					glarea [j] = gapat [iga];
				nctl = nseg ;
			}
			sample = expat [i];
			ip1 =  (int) posn ;
			ip2 = ip1 + 1 ;
			if (ip1 >= lfx + 1) {
				npatt= i;
				posn -= (float) (lfx);
				return;
			}
			if (ip1 > 0 ) {
				store [ip1-1] += (ip2 - posn) * sample ;
			}
			if (ip1 > lfx) {
				npatt = i;
				posn -= (float) (lfx);
				return;
			}
			store [ip2-1] += (posn - ip1) * sample;
			posn += sgap ;
			adjust ();
		}
		npatt = 0;
		posn += pgap;
		adjust ();
	}
 }

 /* Routine called from excite to adjust current expat sample position
  * in array store and update sample gap (sgap) and pulse gap (pgap),
  * if period (po) and pulse length (pl) changed during frame (at posn= dist)
  */

void adjust()
 {
	boolean space;
	float   dist , ponew , plnew , tgnew , tgv , diff ;

	tgv = pov - plv ;
	space = ( npatt == 0 );

	while ( TRUE ) {
		dist = (float) (ixctl * lcx ) + 0.5 ;
		if ( posn <= dist ) {
			return ;
		}
		if ( ixctl >= ncx ) {
			return ;
		}
		ponew = po [ixctl];
		plnew = pl [ixctl];
		ixctl++;

		tgnew = (ponew-plnew) ;
		while ( tgnew <= 0.0 ) {
			tgnew += ponew ;
		}

		diff = posn - dist ;

		if ( space ) {
			posn = dist + diff * tgnew / tgv ;
		}
		else {
			posn = dist + diff * plnew / plv ;
		}

		plv = plnew ;
		pov = ponew ;
		tgv = tgnew ;
		sgap = usrate * plv / 72.0;
		pgap = usrate * tgv ;
	}
 }

 /* filter excitation waveform */
void excite_filt ( xin, xout )
 float  *xin;
 float  *xout;
 {
 /**FILTER
  * Voiced excitation low pass filter.
  * Sixth order LP elliptic,
  * cut off 4.5 kHz with a partial integrator.
  *
  * Sampling rate 80 kHz
  * Gain factor   3.9 at 2 kHz
  * poles: (-8,0) (-221.7,4535) (-816.4,3702.4) (-1544,1546.9)
  * zeros: (-640,0) (0,6134.5) (0,7657.2) (0,16820)
  */
	static	  float st1, st2, st3, st4, st5, st6, st7;
	register	float x1, x2, x3, x4, x5, x6, x7;
	register	float h, y;
	register	int i;

	x1 = st1;
	x2 = st2;
	x3 = st3;
	x4 = st4;
	x5 = st5;
	x6 = st6;
	x7 = st7;

	for (i = lfx; i; i--) {

		h = *xin++ * 0.00818187 + 1.8421164 * x1 - 0.9657748 * x2;
		y = h - 1.7723222 * x1 + x2;
		x2 = x1;
		x1 = h;

		h = y + 1.7970375 * x3 - 0.8796424 * x4;
		y = h - 1.6490942 * x3 + x4;
		x4 = x3;
		x3 = h;

		h = y + 1.7585387 * x5 - 0.7846395 * x6;
		y = h - 0.4943363 * x5 + x6;
		x6 = x5;
		x5 = h;

		h = y + 0.9993719 * x7;
		*xout++ = h - 0.9509769 * x7;
		x7 = h;
	}

	st1 = x1;
	st2 = x2;
	st3 = x3;
	st4 = x4;
	st5 = x5;
	st6 = x6;
	st7 = x7;
 }

 /* mix
  *
  *  A subroutine to mix one frame of voiced excitation held in array 'vox'
  * with one frame of unvoiced excitation held in array 'uvox' according to
  * the degree of voicing control 'voxmix [n]' for the particular formant.
  * The glottal function in array 'glarea' and the constant unvoiced glottal
  * area 'gouv', are mixed in the same way.
  *
  */

void mix(nchan)
 int    nchan;
 {
	float	   v, g, gv, hrel, gouvrl ;
	int	     i, j;
	static float    temp [200];

	v = voxmix [ nchan ];
	if ( v == 1.0 ) { /* fully voiced */
		for ( i = 0 ; i < nc ; i++ ) {
			g = gain [nchan] [i] ; /* get gain */
			glamix [i] = glarea [i]; /* get glottal area due to voicing */
			for ( j = ictlst [i] ; j <= ictlfn [i] ; j++ ) {
				formant [j] = g * vox [j];
			}
		}
	}
	else if (v != 0.0) { /* mixed excitation */
		if ( v > 1.0 ) {
			error("Degree of voicing control > 1.0", NULL);
		}
		hrel = sqrt( (double) (1.0 / (v*v) - 1.0) );
					/* relative unvoiced excitation */
		gouvrl = gouv * hrel;     /* relative unvoiced glottal area */
		for ( i = 0 ; i < nc ; i++ ) {
			gv = gain [nchan] [i] * v;
			glamix [i] = v * (glarea [i] + gouvrl ) ;
						/* glottal area mixing */
			for ( j = ictlst [i] ; j <= ictlfn [i] ; j++ ) {
				/* excitation mixing */
				formant [j] = gv * (vox [j] + hrel * uvox [j]);
			}
		}
	}
	else {
		for ( i = 0; i < nc; i++ ) {
			g = gain [nchan] [i] ;
			glamix [i] = gouv;
			for ( j = ictlst [i] ; j <= ictlfn [i] ; j++ ) {
				formant [j] = g * uvox [j];
			}
		}
	}

	if (alfcon) {
		if (nchan == 0) {
			for ( i = 0 ; i < lf ; i++ ) {
				temp [i] = formant [i];
			}
		}
		else {
			if (nchan == 5) {
				for ( i = 0 ; i < lf ; i++ ) {
					formant [i] -= temp [i];
				}
			}
		}
	}
 }

/*
 * arzero (array, len)
 *
 * Routine initialises a real array to zeros.
 */

void arzero (array, len)
float   *array;
int     len;
{
	register int i;

	for (i = 0; i < len; i++) *array++ = 0.0;



}

/* This subroutine implements a recursive 2 pole digital filter
 * representing a basic formant . Frequency is controlled by data
 * in array ffq and bandwidth by rule. Both are modified pitch
 * synchronously depending on values in glottal area arrays 'glarea' and 'glamix'
 * i/p , o/p is array work
 */

void form_gen(nchan)
int     nchan;
{

	float	   bandw, bwmod, freq_mod, gainc, glarbw, glarfq, r;
	register float  alpha, beta, bn, cn, gf;
	boolean	 condsc;
	register int    i,j,end;

	/* remember previous state */
	bn = b [ nchan ] ;
	cn = c [ nchan ] ;

	/* set filter parameters   */
	bandw = forbdw [ nchan ] + dnasal * fbdwna [ nchan ] ;
	bwmod  = fbwmod [ nchan ] ;
	freq_mod  = ffqmod [ nchan ] ;
	gainc  = gainct [ nchan ] ;
	gf     = gainc ;
	condsc = (nchan == 0) || (nchan == 5);


	/* set filter coefficients */
	for ( i = 0 ; i < nc ; i++ ) {
		glarbw = glamix [ i ] ;
		glarfq = glarea [ i ] ;
		r      = exptab [ RANGE(1,(int) (bandw + glarbw * bwmod),32) - 1 ];
		alpha  = 2.0 * r * costab [ RANGE(1,(int) (ffq [nchan] [ i ] + glarfq * freq_mod ) , 256 ) - 1];

		beta   = - r * r ;

		if ( condsc ) {
			gf = gainc * ( 1.0 - alpha - beta ) ;

		}

		/* run filter for duration of constant coefficients */
		end = ictlfn [ i ];
		for ( j = ictlst [ i ] ; j <= end ; j++ ) {
			formant [j] = gf * formant [j] + alpha * bn + beta * cn ;
			cn = bn ;
			bn = formant [j];
		}


	}

	/* retain last state information */
	b[nchan] = bn ;
	c[nchan] = cn ;



}

 /*
  * output_filter ()
  * performs the final filtering on the output
  */

void output_filter (wave)
 float  *wave;
 {
 /**FILTER
  * output weighting filter
  * Eighth order elliptic low pass filter with partial integrator.
  * Sampling rate 20 kHz
  * Gain factor   1 at 0 kHz
  * Poles: (-640,0) (-91.6,4555.1) (-368.0,4351.2) (-998.5,3716.7)
  *	(-2256.9,1837.9)
  * zeros: (0,5030.9) (0,5325.8) (0,6216.1) (0,8381.2)
  */

	static float    last1, last2, last3, last4, last5;
	static float    last6, last7, last8, last9;
	register float  x1, x2, x3, x4, x5, x6, x7, x8, x9;
	register float  y, h;
	register int    i;

	x1 = last1;
	x2 = last2;
	x3 = last3;
	x4 = last4;
	x5 = last5;
	x6 = last6;
	x7 = last7;
	x8 = last8;
	x9 = last9;

	for (i = 0; i < lf; i++) {
		h = *wave * 0.00382461 + 0.2707258 * x1 - 0.9440709 * x2;
		y = h + 0.0194147 * x1 + x2;
		x2 = x1;
		x1 = h;

		h = y + 0.3606370 * x3 - 0.7935639 * x4;
		y = h + 0.2043489 * x3 + x4;
		x4 = x3;
		x3 = h;

		h = y + 0.5733842 * x5 - 0.5339911 * x6;
		y = h + 0.7456452 * x5 + x6;
		x6 = x5;
		x5 = h;

		h = y + 0.8246887 * x7 - 0.2421853 * x8;
		y = h + 1.7468922 * x7 + x8;
		x8 = x7;
		x7 = h;

		h = y + 0.8178618 * x9;
		*wave++ = h;
		x9 = h;
	}

	last1 = x1;
	last2 = x2;
	last3 = x3;
	last4 = x4;
	last5 = x5;
	last6 = x6;
	last7 = x7;
	last8 = x8;
	last9 = x9;

 }


/* MJ addition
 * eoutput_filter ()
 * performs the final filtering on the output
 */
void eoutput_filter (wave)
float   *wave;
{
/**FILTER
 * output weighting filter
 * Eighth order modified elliptic 6kHz low pass filter with partial integrator.
 * Sampling rate 20 kHz
 * Gain factor   1 at 0 kHz
 * Poles: (-640,0) (-120,5967.4) (-482.1,5700.3) (-1308.1,4869.0)
 *	(-2956.6,2407.7)
 * zeros: (0,6590.7) (0,6977.0) (0,8143.4)
 * single zeros: (0,10000) (-8040,10000)
 */

	static float    last1, last2, last3, last4, last5;
	static float    last6, last7, last8, last9;
	register float  x1, x2, x3, x4, x5, x6, x7, x8, x9;
	register float  y, h;
	register int    i;

	x1 = last1;
	x2 = last2;
	x3 = last3;
	x4 = last4;
	x5 = last5;
	x6 = last6;
	x7 = last7;
	x8 = last8;
	x9 = last9;

	for (i = 0; i < lf; i++) {
		h = *wave * 0.0104645 - 0.5763775 * x1 - 0.9273741 * x2;
		y = h + 1.6693344 * x1 + x2;
		x2 = x1;
		x1 = h;

		h = y - 0.3751269 * x3 - 0.7386638 * x4;
		y = h + 1.1638485 * x3 + x4;
		x4 = x3;
		x3 = h;

		h = y + 0.0545575 * x5 - 0.4395941 * x6;
		y = h + 0.9583827 * x5 + x6;
		x6 = x5;
		x5 = h;

		h = y + 0.5745904 * x7 - 0.1560332 * x8;
		y = h +  x7;
		x8 = x7;
		x7 = h;

		h = y + 0.8178618 * x9;
		*wave++ = h + 0.08 * x9;
		x9 = h;
	}

	last1 = x1;
	last2 = x2;
	last3 = x3;
	last4 = x4;
	last5 = x5;
	last6 = x6;
	last7 = x7;
	last8 = x8;
	last9 = x9;



}


/*
 * mixing_filt ()
 * implements individual spectrum weighting filters for each formant.
 * Mixes the output together.
 * Input array 'arrin' , output array 'arrout'.
 */

void mixing_filt (nchan, arrin, arrout )
int     nchan;
float   *arrin, *arrout;
{
	static float f1_last1, f1_last2;
	static float f2_last1;
	static float f3_last1;
	static float f4_last1;
	static float f5_last1, f5_last2, f5_last3, f5_last4, f5_last5, f5_last6;
/* MJ additions */
	static float f6_last1, f6_last2, f6_last3, f6_last4, f6_last5, f6_last6;
	static float f7_last1, f7_last2, f7_last3, f7_last4, f7_last5, f7_last6;


	register float  x1, x2;
	register float  y, h;
	register int    i;
	register float  x3, x4, x5, x6;

	switch ( nchan ) {

	case 0: /*  F1 - a real zero and phase shifting network */
		/*  Sampling rate 20 kHz			*/
		/*  Gain factor   0.141 at 0 Hz		 */
		/*  poles (-270,0)			      */
		/*  zeros (-640,0) (270,0)		      */

		x1 = f1_last1;
		x2 = f1_last2;

		for ( i = 0; i < lf; i++ ) {
			h = *arrin++ * 0.711181 + 0.9186749 * x1;
			*arrout++ += h - 1.906832 * x1 + 0.8902625 * x2;
			x2 = x1;
			x1 = h;
		}

		f1_last1 = x1;
		f1_last2 = x2;
		break;

	case 1: /* F2			   */
		/* Sampling rate 20 Khz	 */
		/* Gain factor   1 at 5 kHz     */
		/* 1 zero at (0, 0)	     */

		x1 = f2_last1;

		for( i = 0; i < lf; i++) {
			h  = *arrin++ * 0.707107;
			*arrout++ += h - x1;
			x1 = h;
		}

		f2_last1 = x1;

		break;

	case 2: /* F3			   */
		/* Sampling rate 20 Khz	 */
		/* Gain factor   1 at 5 kHz     */
		/* 1 zero at (0, 0)	     */

		x1 = f3_last1;

		for( i = 0; i < lf; i++) {
			h  = *arrin++ * 0.707107;
			*arrout++ += h - x1;
			x1 = h;
		}

		f3_last1 = x1;

		break;

	case 3: /* FHFV */
		/* Sampling rate 20 kHz */
		/* Gain factor   1 at 5 kHz */
/* MJ change */
		/* 1 zero at (0, 0) */

		x1 = f4_last1;

/* MJ changes */

		for ( i = 0; i < lf; i++ ) {

			h = *arrin++ * 0.707107;
			*arrout++ += h - x1;
			x1 = h;
		}

		f4_last1 = x1;

		break;

	case 4: /* FHFM - a real zero, formant pole cancelation
		 * and replace with bandpass.
		 * Sampling rate 20 kHz
		 * Gain factor   1 at 3.5 kHz
		 * poles : (-115,3500) (-43,3410) (-43,3590)
		 * zeros : (0,0) (-155,3500)
		 */
		x1 = f5_last1;
		x2 = f5_last2;
		x3 = f5_last3;
		x4 = f5_last4;
		x5 = f5_last5;
		x6 = f5_last6;

		for ( i = 0; i < lf; i++) {
			h = *arrin++ * 0.00218111 + 0.9451422 * x1 - 0.9733440 * x2;
			y = h - 0.8648264 * x1 + 0.9072027 * x2;
			x2 = x1;
			x1 = h;

			h = y + 0.8757627 * x3 - 0.9302921 * x4;
			y = h - x3;
			x4 = x3;
			x3 = h;

			h =  y + 0.8457371 * x5 - 0.9733440 * x6;
			*arrout++ += h;
			x6 = x5;
			x5 = h;
		}

		f5_last1 = x1;
		f5_last2 = x2;
		f5_last3 = x3;
		f5_last4 = x4;
		f5_last5 = x5;
		f5_last6 = x6;

		break;

	case 5: /* FN a null filter with gain .141 at 0 Hz */
		for ( i = 0; i < lf ; i++) {
			*arrout++ += 0.141 *  *arrin++;
		}
		break;


/* MJ addition */
	case 6: /* FHFF - a real zero, formant pole cancelation
		 * and replace with bandpass.
		 * Sampling rate 20 kHz
		 * Gain factor   1 at 4.2 kHz
		 * poles : (-170,4200) (-64,4070) (-64,4330)
		 * zeros : (0,0) (-155,4200)
		 */
		x1 = f6_last1;
		x2 = f6_last2;
		x3 = f6_last3;
		x4 = f6_last4;
		x5 = f6_last5;
		x6 = f6_last6;

		for ( i = 0; i < lf; i++) {
			h = *arrin++ * 0.00664962 + 0.5645916 * x1 - 0.9605854 * x2;
			y = h - 0.4737403 * x1 + 0.9072027 * x2;
			x2 = x1;
			x1 = h;

			h = y + 0.4715131 * x3 - 0.8986927 * x4;
			y = h - x3;
			x4 = x3;
			x3 = h;

			h =  y + 0.4095539 * x5 - 0.9605854 * x6;
			*arrout++ += h;
			x6 = x5;
			x5 = h;
		}

		f6_last1 = x1;
		f6_last2 = x2;
		f6_last3 = x3;
		f6_last4 = x4;
		f6_last5 = x5;
		f6_last6 = x6;

		break;


/* MJ addition */
	case 7: /* FF - a real zero, formant pole cancelation
		 * and replace with bandpass.
		 * Sampling rate 20 kHz
		 * Gain factor   1 at 5.2 kHz
		 * poles : (-800,5200) (-300,4600) (-300,5800)
		 * zeros : (0,0) (-155,5200)
		 */
		x1 = f7_last1;
		x2 = f7_last2;
		x3 = f7_last3;
		x4 = f7_last4;
		x5 = f7_last5;
		x6 = f7_last6;

		for ( i = 0; i < lf; i++) {
			h = *arrin++ * 0.424178 + 0.2281208 * x1 - 0.8282042 * x2;
			y = h + 0.1196124 * x1 + 0.9072027 * x2;
			x2 = x1;
			x1 = h;

			h = y - 0.0976729 * x3 - 0.6049226 * x4;
			y = h - x3;
			x4 = x3;
			x3 = h;

			h =  y - 0.4526441 * x5 - 0.8282042 * x6;
			*arrout++ += h;
			x6 = x5;
			x5 = h;
		}

		f7_last1 = x1;
		f7_last2 = x2;
		f7_last3 = x3;
		f7_last4 = x4;
		f7_last5 = x5;
		f7_last6 = x6;

		break;


	}




}

/* This procedure takes as input a stats table produced by statlx(1) or
   inlx(1) and finds the most probable OP for a given FX */

struct st_dhr
{
	int dimension;
	float maxfreq;
	float minfreq;
	float frameduration;
}stat_header; 		/* structure to data header */

void make_op_table(table_name)
char *table_name;
{
FILE	*tid;		/* file id */
int	*stat_array;	/* pointer to data table */
int	tabl_siz;	/* dimension of table */
int	i;		/* count variable */
int	min, max;	/* max and min table values */

	/* open table file for reading */

	if((tid = fopen(table_name,"r")) == NULL)
		error("cannot open %s",table_name);

	/* read in header for table */
	if(fread(&stat_header,sizeof(struct st_dhr),1,tid) != 1)
		error("error in reading header %s",table_name);

	/* input array  */
	tabl_siz = stat_header.dimension*stat_header.dimension;
	if((stat_array = (int*) calloc(tabl_siz,sizeof(int))) == NULL)
		error("cannot calloc enough space ",NULL);
	if(fread(stat_array,tabl_siz*sizeof(int),1,tid) != 1)
		error("error reading in %s",table_name);


	/* check stat_table covers required freqency range */
	min = (int)(23.09 * log((double)stat_header.minfreq) - 75.37);
	if(min < 0) min = 0;
	max = (int)(23.09 * log((double)stat_header.maxfreq) - 75.37);
	if(max > 63) max = 64;
	/* create table */
	for(i=min;i<max;i++)
		pltab[i] = stat_header.frameduration*maximum(stat_array,
			   (int)((1.0/fotab[i])/stat_header.frameduration),
			   stat_header.dimension);

	/* close file */

	fclose(tid);
}

/* This subroutine searches table for max value for particular periode and
   returns No. of samples in OP */

int maximum(stat_array,po,dimension)
int	*stat_array;	/* pointer to data table */
int	po;		/* periode in samples */
int	dimension;
{
int	maxpl;	/* contents of pl bin */
int	pl;	/* OP in samples */
int	i;	/* count variable */

	maxpl = 0;
	pl = 0;
	for(i=0;i<dimension;i++)
		if(maxpl < stat_array[i*dimension+po] ) {
			maxpl = stat_array[i*dimension+po];
			pl = i;
		}

	return(pl);
}
