/* spaned - Speech editing controlled by Annotations */

/* M.A.Huckvale - University College London */

/* version 1.0 - May 1995 */
/* version 1.1 - May 1995
	- add -c check flag
*/
/* version 1.2 - July 2000
	- add ability to copy lx and annotation items too
*/
/* version 1.2b - Novermber 2007
	- bug when overlap lengths approach segment lengths
	- PROBLEM not really resolved due to structure of code
*/

#define PROGNAME "spaned"
#define PROGVERS "1.2b"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH SPANED SFS1 UCL
.SH NAME
spaned - speech editing controlled by annotations
.SH SYNOPSIS
.B spaned
(-w overlapwidth) (-c) control-file outfile
.SH DESCRIPTION
.I spaned
is a program to concatenate a number of speech segments
from different files into a single speech signal controlled by times
or annotations listed in a control file.
The control file consists of one line per speech section to process
giving (i) the filename, (ii) the items to copy (separated by '/'),
(iii) the annotation item,
(iv) the start annotation or time, (v) the end annotation or time.  If the
item numbers are not given, the last appropriate item in the file
is chosen.  If no start time or annotation is given then it defaults
to the start of the item.  If no end time or annotation is given it defaults
to the end of the annotated region if a start annotation is supplied,
or otherwise to the end of the item.  Items should be separated by commas.
To prevent clicks, signals are slightly overlapped and weighted by a
raised cosine window.
Speech items must match in sampling rate.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -w overlap-width
Specify time for overlaps at joins.  Default 0.025s.
.TP 11
.B -c
Check mode.  Access all the files and annotations but don't do the copying.
.SH INPUT ITEMS
.IP SP 11
Speech items
.IP AN 11
Annotation items
.SH OUTPUT ITEMS
.IP SP 11
Concatenated Speech.
.SH HISTORY
.IP file=control
Control file name.
.SH VERSION/AUTHOR
.IP 1.2
Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include <string.h>
#include <ctype.h>
#include "sfs.h"

#define XBUFSIZE	48000

/* global data */
struct item_header ispitem;
struct item_header ilxitem;
struct item_header ianitem;
struct item_header anitem;
struct an_rec	*an;
struct item_header ospitem;
struct item_header olxitem;
struct item_header oanitem;

char	cfilename[SFSMAXFILENAME];
char	fname[SFSMAXFILENAME];
char	*isptype;
char	*ilxtype;
char	*iantype;
char	*antype;
char	astart[800];
char	astop[800];

double	overlaptime=0.025;
int	spoverlapsamp;
int	lxoverlapsamp;
short	*spoverlap;
short	*lxoverlap;
int	spoverlapload=0;
int	lxoverlapload=0;
double	*inweight;
double	*outweight;
int	checkonly=0;

#define SKIPSPACE(p) while (isspace(*p)) p++
#define ISFIELD(p) (!((p=='\0') || (p==',') || isspace(p)))

int	copysp();
int	flushsp();

/* read a line from control file */
int	getcontrol(ip,fname,isptype,ilxtype,iantype,antype,start,stop)
FILE	*ip;
char	*fname;
char	**isptype;
char	**ilxtype;
char	**iantype;
char	**antype;
char	*start;
char	*stop;
{
	char		line[1024];
	char		*p,*d,*ty;
	static char	*zero="0",*nul="";
	static char	it1[80],it2[80];
	int32		it;

	strcpy(fname,"");
	*isptype=nul;
	*ilxtype=nul;
	*iantype=nul;
	*antype=zero;
	strcpy(start,"");
	strcpy(stop,"");

	do {
		if (!fgets(line,1024,ip)) return(0);
	} while ((line[0]=='\n') || (line[0]=='#'));

	/* get filename */
	p = line;
	SKIPSPACE(p);
	if (*p=='"') {
		while (*p && (*p!=',') && (*p!='"'))
			*fname++ = *p++;
		if (*p=='"') p++;
	}
	else if (*p=='\'') {
		while (*p && (*p!=',') && (*p!='\''))
			*fname++ = *p++;
		if (*p=='\'') p++;
	}
	else while (ISFIELD(*p))
		*fname++ = *p++;
	*fname = '\0';
	SKIPSPACE(p);
	if (!*p) {
		*isptype=zero;
		return(1);
	}
	p++;

	/* get speech, lx & an item to copy */
	d = it1;
	SKIPSPACE(p);
	while (ISFIELD(*p))
		*d++ = *p++;
	*d = '\0';
	if (it1[0]) {
		d = strtok(it1,"/");
		while (d && *d) {
			if (itspec(d,&it,&ty)==0) {
				if (it==SP_TYPE)
					*isptype = ty;
				else if (it==LX_TYPE)
					*ilxtype = ty;
				else if (it==AN_TYPE)
					*iantype = ty;
				else
					error("bad item specification in control file");
			}
			else
				error("bad item specification in control file");
			d = strtok(NULL,"/");
		}
	}
	SKIPSPACE(p);
	if (!*p) return(1);
	p++;

	/* get annotation item */
	d = it2;
	SKIPSPACE(p);
	while (ISFIELD(*p))
		*d++ = *p++;
	*d = '\0';
	if (it2[0]) {
		if ((itspec(it2,&it,antype)!=0) || (it!=AN_TYPE))
			error("bad AN item specification in control file");
	}
	SKIPSPACE(p);
	if (!*p) return(1);
	p++;

	/* get start time */
	SKIPSPACE(p);
	while (ISFIELD(*p))
		*start++ = *p++;
	*start = '\0';
	SKIPSPACE(p);
	if (!*p) return(1);
	p++;

	/* get stop time */
	SKIPSPACE(p);
	while (ISFIELD(*p))
		*stop++ = *p++;
	*stop = '\0';

	if ((*isptype==nul) && (*ilxtype==nul) && (*iantype==nul)) *isptype=zero;

	return(1);
}

/* find time from annotation */
double	findtime(str)
char	*str;
{
	int	i;
	if (!an) return(-1.0);
	for (i=0;i<anitem.numframes;i++)
		if (strcmp(str,an[i].label)==0)
			return(anitem.offset + an[i].posn*anitem.frameduration);
	return(-1.0);
}
double	findend(str)
char	*str;
{
	int	i;
	if (!an) return(-1.0);
	for (i=0;i<anitem.numframes;i++)
		if (strcmp(str,an[i].label)==0)
			return(anitem.offset + (an[i].posn+an[i].size)*anitem.frameduration);
	return(-1.0);
}

/* calculate overlap weighting */
void	calcoverweights(int overlap)
{
	int	i;
	double	w;

	if (overlap==0) return;

	inweight = (double *)calloc(overlap,sizeof(double));
	outweight = (double *)calloc(overlap,sizeof(double));

	w = 4.0*atan(1.0)/overlap;
	for (i=0;i<overlap;i++) {
		inweight[i] = 0.5-0.5*cos(i*w);
		outweight[i] = 1 - inweight[i];
	}
}

/* copy signal */
short	spxferbuff[XBUFSIZE];
int	copysp(ifd,ofd,start,stop)
int	ifd,ofd;
int	start,stop;
{
	int	i;
	int	len;
	int	cnt=0;
	double	val;

	if (ofd < 0) return(0);

//printf("spoverlapload=%d, spoverlapsamp=%d, start=%d, stop=%d\n",spoverlapload,spoverlapsamp,start,stop);

	if (spoverlapload) {
		/* process overlap with last segment */
		len = sfsread(ifd,start,spoverlapsamp,spxferbuff);
		start += len;

		for (i=0;i<len;i++) {
			val = inweight[i]*(double)spxferbuff[i]+outweight[i]*(double)spoverlap[i];
			spxferbuff[i] = (short)val;
		}
		for (;i<spoverlapsamp;i++)
			spxferbuff[i] = (short)(outweight[i]*spoverlap[i]);

		cnt += sfswrite(ofd,spoverlapsamp,spxferbuff);
// printf("sfswrite(%d)\n",spoverlapsamp);
	}
	while (start < (stop-spoverlapsamp)) {
		len = sfsread(ifd,start,(((stop-spoverlapsamp)-start)>XBUFSIZE)?XBUFSIZE:((stop-spoverlapsamp)-start),spxferbuff);
		if (sfswrite(ofd,len,spxferbuff)!=len)
			error("output write error");
// printf("sfswrite(%d)\n",len);
		start += len;
		cnt += len;
	}
	/* hold over last portion */
	if (spoverlapsamp > 0) {
		len = sfsread(ifd,start,spoverlapsamp,spoverlap);
		spoverlapload=len;
	}
	return(cnt);
}

/* flush overlap buffer */
int	flushsp(ofd)
int	ofd;
{
	if (ofd < 0)
		return(0);
//printf("spoverlapload=%d, spoverlapsamp=%d\n",spoverlapload,spoverlapsamp);
	if (spoverlapload) {
//printf("sfswrite(%d)\n",spoverlapsamp);
		return(sfswrite(ofd,spoverlapload,spoverlap));
	}
	else
		return(0);
}

/* copy signal */
short	lxxferbuff[XBUFSIZE];
int	copylx(ifd,ofd,start,stop)
int	ifd,ofd;
int	start,stop;
{
	int	i;
	int	len;
	int	cnt=0;

	if (ofd < 0) return(0);

	if (lxoverlapload) {
		/* process overlap with last segment */
		len = sfsread(ifd,start,lxoverlapsamp,lxxferbuff);
		start += len;
		for (i=0;i<len;i++)
			lxxferbuff[i] = (short)(inweight[i]*lxxferbuff[i]+outweight[i]*lxoverlap[i]);
		for (;i<lxoverlapsamp;i++)
			lxxferbuff[i] = (short)(outweight[i]*lxoverlap[i]);
		cnt += sfswrite(ofd,lxoverlapsamp,lxxferbuff);
	}
	while (start < (stop-lxoverlapsamp)) {
		len = sfsread(ifd,start,(((stop-lxoverlapsamp)-start)>XBUFSIZE)?XBUFSIZE:((stop-lxoverlapsamp)-start),lxxferbuff);
		if (sfswrite(ofd,len,lxxferbuff)!=len)
			error("output write error");
		start += len;
		cnt += len;
	}
	/* hold over last portion */
	if (lxoverlapsamp > 0) {
		len = sfsread(ifd,start,lxoverlapsamp,lxoverlap);
		lxoverlapload=1;
	}
	return(cnt);
}

/* flush overlap buffer */
int	flushlx(ofd)
int	ofd;
{
	if (ofd < 0)
		return(0);
	if (lxoverlapload)
		return(sfswrite(ofd,lxoverlapsamp,lxoverlap));
	else
		return(0);
}

/* copy annotations */
void	copyan(ifid,ofid,stime,etime,offset)
int	ifid;
int	ofid;
double	stime;
double	etime;
double	offset;
{
	int	i;
	double	t;
	struct an_rec	*ian = (struct an_rec *)sfsbuffer(&ianitem,1);
	struct an_rec	*oan = (struct an_rec *)sfsbuffer(&oanitem,1);

	for (i=0;i<ianitem.numframes;i++) {
		sfsread(ifid,i,1,ian);
		t = ianitem.offset + ian->posn*ianitem.frameduration;
		if (((stime-0.001) <= t) && (t < (etime-0.001))) {
			oan->posn = (int)((offset+t-stime)/oanitem.frameduration);
			oan->size = 0;
			strcpy(oan->label,ian->label);
			sfswrite(ofid,1,oan);
		}
	}
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		sfid= -1;	/* output file descriptor */
	int		lfid= -1;	/* output file descriptor */
	int		afid= -1;	/* output file descriptor */
	FILE		*ip;
	int		splen,totsamp=0;
	int		inrate,outrate;
	double		stime,etime;
	int		ssamp=0,esamp=0;
	int		i;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Iw:c")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Speech editing using annotations V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'w' :
			overlaptime = atof(optarg);
			if ((overlaptime < 0.0) || (overlaptime > 1.0))
				error("illegal overlap time");
			break;
		case 'c' :	/* check mode */
			checkonly=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<3))
		error("usage: %s (-I) (-w overlapwidth) (-c) controlfile sfsfile",PROGNAME);

	/* get control file name */
	if (optind < argc)
		strcpy(cfilename,argv[optind]);
	else
		error("no control file specified");
	optind++;

	/* get SFS filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open control file */
	if ((ip=fopen(cfilename,"r"))==NULL)
		error("could not open '%s'",cfilename);

	/* get control items */
	while (getcontrol(ip,fname,&isptype,&ilxtype,&iantype,&antype,astart,astop)) {

/*
fprintf(stderr,"fname='%s' sptype='%s' lxtype='%s' oantype='%s' antype='%s' start='%s' stop='%s'\n",
	fname,isptype,ilxtype,iantype,antype,astart,astop);
*/
		/* open file */
		if ((fid=sfsopen(fname,"r",NULL)) < 0)
			error("could not open '%s'",fname);

		/* read in annotations */
		if (sfsitem(fid,AN_TYPE,antype,&anitem)) {
			an = (struct an_rec *)sfsbuffer(&anitem,anitem.numframes);
			sfsread(fid,0,anitem.numframes,an);
		}
		else
			an = NULL;

		/* locate input items */
		if (*isptype) {
			if (!sfsitem(fid,SP_TYPE,isptype,&ispitem))
				error("could not find speech item in '%s'",fname);
			if (ospitem.frameduration!=0) {
				outrate = (int)(0.5 + 0.1/ospitem.frameduration);
				inrate = (int)(0.5 + 0.1/ispitem.frameduration);
				if (inrate != outrate)
					error("sample rate doesn't match in '%s'",fname);
			}
			else {
				spoverlapsamp = (int)(overlaptime/ispitem.frameduration);
				if (spoverlapsamp > XBUFSIZE)
					error("overlap too big");
				spoverlap = (short *)sfsbuffer(&ispitem,spoverlapsamp+4096);
				calcoverweights(spoverlapsamp);
			}
		}
		if (*ilxtype) {
			if (!sfsitem(fid,LX_TYPE,ilxtype,&ilxitem))
				error("could not find speech item in '%s'",fname);
			if (olxitem.frameduration!=0) {
				outrate = (int)(0.5 + 0.1/olxitem.frameduration);
				inrate = (int)(0.5 + 0.1/ilxitem.frameduration);
				if (inrate != outrate)
					error("sample rate doesn't match in '%s'",fname);
			}
			else {
				lxoverlapsamp = (int)(overlaptime/ilxitem.frameduration);
				if (lxoverlapsamp > XBUFSIZE)
					error("overlap too big");
				lxoverlap = (short *)sfsbuffer(&ilxitem,lxoverlapsamp);
				calcoverweights(lxoverlapsamp);
			}
		}
		if (*iantype) {
			if (!sfsitem(fid,AN_TYPE,iantype,&ianitem))
				error("could not find annotation item in '%s'",fname);
		}

		/* if first time through, open output channel */
		if (!checkonly && *isptype && (sfid < 0)) {
			sfsheader(&ospitem,SP_TYPE,0,2,1,
					ispitem.frameduration,0.0,1,0,0);
			sprintf(ospitem.history,"%s(file=%s,overlap=%g)",
				PROGNAME,cfilename,overlaptime);
			if ((sfid=sfschannel(filename,&ospitem)) < 0)
				error("could not open output channel to '%s'",filename);
		}
		if (!checkonly && *ilxtype && (lfid < 0)) {
			sfsheader(&olxitem,LX_TYPE,0,2,1,
					ilxitem.frameduration,0.0,1,0,0);
			sprintf(olxitem.history,"%s(file=%s,overlap=%g)",
				PROGNAME,cfilename,overlaptime);
			if ((lfid=sfschannel(filename,&olxitem)) < 0)
				error("could not open output channel to '%s'",filename);
		}
		if (!checkonly && *iantype && (afid < 0)) {
			sfsheader(&oanitem,AN_TYPE,-1,1,-1,
					ianitem.frameduration,0.0,0,0,1);
			sprintf(oanitem.history,"%s(file=%s,overlap=%g)",
				PROGNAME,cfilename,overlaptime);
			if ((afid=sfschannel(filename,&oanitem)) < 0)
				error("could not open output channel to '%s'",filename);
		}

		/* determine region to copy */
		if (astart[0]=='\0')
			ssamp = 0;
		else if (isdigit(astart[0]))
			ssamp = (int)((atof(astart)-ispitem.offset)/ispitem.frameduration);
		else if ((stime = findtime(astart)) < 0)
			error("could not find annotation '%s' in '%s'",astart,fname);
		else
			ssamp = (int)((stime-ispitem.offset)/ispitem.frameduration);
		if (ssamp < 0) ssamp=0;
		if (ssamp > ispitem.numframes) ssamp = ispitem.numframes;

		if (astop[0]=='\0') {
			if ((astart[0]!='\0') && !isdigit(astart[0])) {
				etime = findend(astart);
				esamp = (int)((etime-ispitem.offset)/ispitem.frameduration);
			}
			else
				esamp = ispitem.numframes;
		}
		else if (isdigit(astop[0]))
			esamp = (int)((atof(astop)-ispitem.offset)/ispitem.frameduration);
		else if ((etime = findtime(astop)) < 0)
			error("could not find annotation '%s' in '%s'",astop,fname);
		else
			esamp = (int)((etime-ispitem.offset)/ispitem.frameduration);
		if (esamp < 0) esamp=0;
		if (esamp > ispitem.numframes) esamp = ispitem.numframes;

fprintf(stderr,"copying samples %d to %d from %s\n",ssamp,esamp,fname);

		/* copy items */
		if (*isptype) {
			if (!sfsitem(fid,SP_TYPE,isptype,&ispitem))
				error("could not find speech item in '%s'",fname);
			splen = copysp(fid,sfid,ssamp,esamp);
		}
		if (*ilxtype) {
			if (!sfsitem(fid,LX_TYPE,ilxtype,&ilxitem))
				error("could not find speech item in '%s'",fname);
			copylx(fid,lfid,ssamp,esamp);
		}
		if (*iantype) {
			if (!sfsitem(fid,AN_TYPE,iantype,&ianitem))
				error("could not find annotation item in '%s'",fname);
			copyan(fid,afid,ssamp*ispitem.frameduration,esamp*ispitem.frameduration,totsamp*ispitem.frameduration);
		}
		totsamp += splen;

		if (an) free(an);
		sfsclose(fid);
	}
	if (sfid>0) totsamp += flushsp(sfid);
	if (lfid>0) flushlx(lfid);

	if (!checkonly) {
		if (!sfsupdate(filename))
			error("update error on '%s'",filename);
	}

fprintf(stderr,"%d samples written to '%s'\n",totsamp,filename);

	/* that's all folks */
	exit(0);
}
