/* vtx -- estimate tx from lx by Voiscope method */

/* m.a. huckvale - march 1988 */

/* version 1.1 - December 1993
	- add variable threshold instead of fixed
*/
/* version 1.2 - March 1994
	- add filters for 11025, 22050, 44100 and 48000
*/
/* version 1.3 - January 1997
	- add interpolation for microsecond Tx
*/
/* version 2.0 - December 2006
	- rewrite using zero-phase filtering
*/

#undef IAG

#define PROGNAME "vtx"
#define PROGVERS "2.0"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH VTX UCL1 SFS
.SH NAME
vtx - convert lx to tx by Voiscope method
.SH SYNOPSIS
.B vtx
(-i item) (-H high-pass) (-L low-pass) (-C time-const) (-T threshold) (-o) file
.SH DESCRIPTION
.I vtx
estimates the location of pitch periods from a Layngograph waveform by the
method employed in the Voiscope.  This consists of: (i) filtering, (ii) agc, and
(iii) peak following. Configuration of the filtering and the time constant of the
peak following can be set by program options.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -H high-pass
Set high-pass frequency edge. Default=80Hz.
.TP 11
.BI -L high-pass
Set low-pass frequency edge. Default=2000Hz.
.TP 11
.BI -C time-const
Set time-constant of peak follower. Default=0.0025s.
.TP 11
.BI -T Peak detection threshold
Set threshold for peak detection. Default=750.
.TP 11
.B -o
Output glottal opening points.  Default: closings.
.SH INPUT ITEMS
.IP 2.xx 11
Laryngograph waveform.
.SH OUTPUT ITEMS
.IP 3 11
Voiscope tx
.SH VERSION/AUTHOR
2.0 - Mark Huckvale.
.SH BUGS
Works best if Lx is right way up (up = more current flow).
.SH SEE ALSO
HQtx
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"			/* header structures */
#include "filter.h"

/* global data */
struct item_header 	lxitem;		/* item header data */
short			*lx;		/* speech data buffer (1 x 20ms window) */
struct item_header 	opitem;		/* output item header (DIAG) */
struct item_header 	txitem;		/* item header data */
int			sampfreq;	/* current sampling frequency */

/* TX estimation control */
#define THRESH	750
#define MAXFREQ	600
double hifreq=80;
double lofreq=2000;
double tconst=0.0025;
int		thresh=THRESH;

/* reverse a signal */
void reverse(short *s,int len)
{
	int		i;
	short	t;

	for (i=0;i<len/2;i++) {
		t=s[i];
		s[i]=s[len-i-1];
		s[len-i-1]=t;
	}
}

#define MAXCLICK	1000

/* click zapper */
void clickdetect(short *lx,int numf,double sdur)
{
	int	wlen=(int)(0.1/sdur);		/* 100ms */
	int	i,j,diff;
	double omega=2.0*M_PI/(wlen-1);
	int	clicktab[MAXCLICK];
	int	nclick=0;

	for (i=wlen/2;(i<numf-wlen/2)&&(nclick<MAXCLICK);i++) {
		diff = (int)lx[i+1]-(int)lx[i];
		if ((lx[i] < -32000)||(lx[i] > 32000)||(diff < -32000)||(diff > 32000)) {
			clicktab[nclick++]=i;
			i += wlen/4;
		}
	}
	if (nclick>0) {
		fprintf(stderr,"Removing %d clicks\n",nclick);
		for (i=0;i<nclick;i++) {
			for (j=0;j<wlen;j++)
				lx[clicktab[i]+j-wlen/2] = (short)(lx[clicktab[i]+j-wlen/2]*(0.5+0.5*cos(omega*j)));
		}
	}
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int		errflg=0;	/* option error flag */
	int		c;		/* option char */
	int32		it;
	char		*ty;
	char		*lxtype = "0";	/* input sp sub-type = last */
	char		filename[SFSMAXFILENAME];
					/* database file name */
	int		fid,ofid;	/* file descriptors */
	double	maxfreq;
	int		i,j;
	int			follow=0;	/* is peak following */
	double		fraction;	/* exponential decay factor */
	double		flast=0;	/* last known follower value */
	double		fthis;		/* current follower value */
	double 		tlast=0;	/* time of last period */
	double		tthis;		/* time of this period */
	short		llast;		/* last known lx value */
	short		lthis;		/* current lx value */
	int			tx;		/* calculated tx value */
	FILTER		*flt;
	short		min,max,lmax;
	double		sum,sumsq,mean,stddev,val;
	int			cnt;
	int			cumtx=0;
	int			doopen=0;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:H:L:C:T:o")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Tx by Voiscope method V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item spec */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == LX_TYPE)
					lxtype=ty;
				else
					error("unsuitable item specification %s",optarg);
			}
			else
				error("illegal item specification %s",optarg);
			break;
		case 'H' :	/* high-pass edge */
			hifreq=atof(optarg);
			break;
		case 'L' :	/* low-pass edge */
			lofreq=atof(optarg);
			break;
		case 'C' :	/* time constant of peak follower */
			tconst=atof(optarg);
			break;
		case 'T' :	/* peak detection threshold */
			thresh=atoi(optarg);
			break;
		case 'o' :	/* do openings */
			doopen=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-H high-pass) (-L low-pass) (-C time-const) (-T threshold) (-o) file\n",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("'%s': file not found",filename);
		else
			error("access error on '%s'",filename);
	}

	/* locate input lx item */
	if (!sfsitem(fid,LX_TYPE,lxtype,&lxitem))
		error("cannot find input LX item in '%s'",filename);

	/* get buffer */
	if ((lx=(short *)sfsbuffer(&lxitem,lxitem.numframes))==NULL)
		error("could not get buffer for LX in %s",filename);

	/* read signal */
	if (sfsread(fid,0,lxitem.numframes,lx)!=lxitem.numframes)
		error("failed to read LX item in %s",filename);

	/* check filter parameters */
	maxfreq = 0.5/lxitem.frameduration;
	if (lofreq > 0.999*maxfreq) lofreq=0.999*maxfreq;
	if (hifreq < 0.001*maxfreq) hifreq=0.001*maxfreq;


	/* click detector */
	clickdetect(lx,lxitem.numframes,lxitem.frameduration);

#ifdef IAG
	sfsheader(&opitem,LX_TYPE,0,2,1,lxitem.frameduration,lxitem.offset,1,0,0);
	sprintf(opitem.history,"%s/LX(%d.%02d;clickdetect)",
		PROGNAME,lxitem.datatype,lxitem.subtype,
		hifreq,lofreq,tconst);
	ofid=sfschannel(filename,&opitem);
	sfswrite(ofid,lxitem.numframes,lx);
#endif

	/* filter signal backwards and forwards */
	if (ttytest()) {
		fprintf(stderr,"Filtering -");
		fflush(stderr);
	}

//	flt=filter_design(FILTER_HIGH_PASS,4,hifreq,0.5/lxitem.frameduration,1.0/lxitem.frameduration);
	flt=filter_design(FILTER_BAND_PASS,4,hifreq,lofreq,1.0/lxitem.frameduration);
	reverse(lx,lxitem.numframes);
	filter_signal(flt,lx,lx,lxitem.numframes);
	reverse(lx,lxitem.numframes);
	filter_clear(flt);
	filter_signal(flt,lx,lx,lxitem.numframes);

	/* normalise signal */
	min=32767;
	max=-32767;
	for (i=0;i<lxitem.numframes;i++) {
		if (lx[i] < min) min=lx[i];
		if (lx[i] > max) max=lx[i];
	}
	sum=0;
	for (i=0;i<lxitem.numframes;i++) {
		lx[i] = (short)(-30000.0+(60000.0*(lx[i]-min))/(max-min));
		sum = sum + lx[i];
	}
	mean = sum / lxitem.numframes;
	sumsq=0;
	cnt=0;
	for (i=0;i<lxitem.numframes;i++) {
		val = lx[i]-mean;
		if ((val > 1000)||(val <-1000)) {
			sumsq += val*val;
			cnt++;
		}
	}
	stddev = sqrt(sumsq/cnt);
	for (i=0;i<lxitem.numframes;i++) {
		val=(lx[i]-mean)*4096/stddev;
		if (val > 32767)
			lx[i]=32767;
		else if (val < -32767)
			lx[i]=-32767;
		else
			lx[i]=(short)val;
	}

#ifdef IAG
	sfsheader(&opitem,LX_TYPE,0,2,1,lxitem.frameduration,lxitem.offset,1,0,0);
	sprintf(opitem.history,"%s/LX(%d.%02d;hifreq=%g,lofreq=%g,tconst=%g)",
		PROGNAME,lxitem.datatype,lxitem.subtype,
		hifreq,lofreq,tconst);
	ofid=sfschannel(filename,&opitem);
	sfswrite(ofid,lxitem.numframes,lx);
#endif

	/* generate output item header */
	sfsheader(&txitem,TX_TYPE,0,4,1,1.0E-6,lxitem.offset,0,0,1);
	if ((hifreq!=40)||(lofreq!=4000)||(tconst!=0.002))
		sprintf(txitem.history,"%s(%d.%02d;hifreq=%g,lofreq=%g,tconst=%g,thresh=%d%s)",
			PROGNAME,lxitem.datatype,lxitem.subtype,
			hifreq,lofreq,tconst,thresh,
			(doopen)?",open":"");
	else if (doopen)
		sprintf(txitem.history,"%s(%d.%02d;open)",
			PROGNAME,lxitem.datatype,lxitem.subtype);
	else
		sprintf(txitem.history,"%s(%d.%02d)",
			PROGNAME,lxitem.datatype,lxitem.subtype);

	/* open output channel */
	if ((ofid=sfschannel(filename,&txitem)) < 0)
		error("cannot open temporary file",NULL);

	/* get decay factor per sample */
	fraction = exp(log(0.7)/(tconst/lxitem.frameduration));

	/* peak following */
	if (ttytest()) {
		fprintf(stderr," Processing -");
		fflush(stderr);
	}
	tlast=0;
	flast=fthis=0;
	follow=0;
	for (i=0;i<lxitem.numframes;i++) {
		lthis = lx[i];
		if (lthis < fthis) {
			/* waveform below peak follower - wait */
			flast = fthis;
			fthis *= fraction;
			if (fthis < thresh) { fthis=thresh; follow=0; }
			if (lthis > lmax) lmax=lthis;
			if (lthis < (thresh+0.5*(fthis-thresh))) follow=0;
			if (lthis < thresh) {
				follow=0;
				if (fthis <= thresh) lmax=2*thresh+1;
			}

		}
		else if (follow) {
			/* still following signal - continue */
			flast = fthis;
			fthis = lthis;
			if (lthis > lmax) lmax=lthis;
			follow=1;
		}
		else {
			/* at transitional point */
			if (doopen) {
				j=i;
				while ((j<lxitem.numframes-1)&&(lx[j+1]>0)) j++;
				tthis = j*lxitem.frameduration;
				tx = (int)(0.5+1000000.0*(tthis-tlast));
			}
			else {
				tthis = i*lxitem.frameduration +
					(flast-llast)/(((lthis-llast)-(fthis-flast))/lxitem.frameduration);
				/* linear interpolation to get exact time */
//printf("%.3f lmax=%d llast=%d lthis=%d flast=%g fthis=%g tlast=%g tthis=%g\n",
//	i*lxitem.frameduration,
//	lmax,llast,lthis,flast,fthis,tlast,tthis);
				tx = (int)(0.5+1000000.0*(tthis-tlast));
			}
			if ((tx > 1000000/MAXFREQ)&&(lmax > 2*thresh)&&(lthis>2*thresh)) {
				sfswrite(ofid,1,&tx);
				cumtx += tx;
				tlast = cumtx*txitem.frameduration;
				lmax=lthis;
				follow=1;
			}
			flast = fthis;
			fthis = lthis;
		}
		lx[i]=(short)fthis;
		llast = lthis;
	}

	if (tlast < lxitem.numframes*lxitem.frameduration) {
		tx = (int)(0.5+1000000.0*(lxitem.numframes*lxitem.frameduration-tlast));
		sfswrite(ofid,1,&tx);
	}

	/* and update file */
	if (ttytest()) {
		fprintf(stderr," Saving");
		fflush(stderr);
	}



	if (!sfsupdate(filename))
		error("backup error on '%s'",filename);
	if (ttytest()) {
		fprintf(stderr,"\n");
		fflush(stderr);
	}

	/* ... that's all folks */
	exit(0);
}

