/* anload -- load an annotation item from a text description of times and labels */

/* M.A.Huckvale - December 1987 */

/* version 1.0 */
/* version 1.1 - February 1989
	- fix lxsync in output AN
*/
/* version 1.2 - June 1989
	- TIMIT database option, annotations with sample ranges
*/
/* version 1.3 - March 1992
	- add -h HTK format load option
*/
/* version 1.4 - October 1998
	- allow spaces in labels
	- add -e ESPS format load option
*/
/* version 1.5 - February 2000
	- add -m MBROLA format load option
	- add -M millisecond format load option
*/
/* version 1.6 - March 2000
	- add loading of Fx into mbrola option
*/
/* version 1.7 - February 2002
	- attempt to support sample format with single times automatically
*/

/* version 1.8 - June 2004
	- add support for SAM label files
	- add support for HTK master label files
	- add support for command-line loading
*/

/* version 1.9 - August 2010
	- add support for Praat text grid
*/

/*-------------------------------------------------------------------*/
/**MAN
.TH ANLOAD SFS1 UCL
.SH NAME
anload -- load an annotation item from text description
.SH SYNOPSIS
.B anload
(-f freq) (-t type) (-s|-h|-e|-m|-M|-S|-p) (-H htk.mlf) -T transcription | textfile file
.SH DESCRIPTION
.I anload
constructs a new annotation item in the given file from text
descriptions of annotations in a text file.
The default format of the text file is one annotation per line.  In the
normal mode this should have two
entries: firstly the time in seconds, secondly the text label (not
containing spaces).  In the "sample" mode (-s switch), each line
should contain three entries: the starting sample number, the final sample number,
and the annotation.  Ensure that the sampling frequency is correct in the
sample mode.
.PP
.B Options
.TP 11
.B -I
Identify the program version.
.TP 11
.BI -f freq
Use a sampling interval based on 1/freq seconds for sampling the time of each
annotation. Default 10000.
.TP 11
.BI -t type
Place a "type" indicator in the created item header, with the argument
.I type.
.TP 11
.B -s
Sample mode.  Text lines contain sample numbers or sample ranges, not times.
For example: "1000 label" or "1000 2000 label".
Ensure sampling frequency is supplied with the -f switch.
.TP 11
.B -h
HTK format load.  Equivalent to switches: -s -f 10000000.
.TP 11
.B -e
ESPS compatible label file format load.
.TP 11
.B -m
MBROLA .PHO file compatible label file format load.
.TP 11
.B -M
Millisecond format.  Lines contain individual durations in milliseconds (not times)
and then label.
.TP 11
.B -S
SAM format label files.  Label lines consist of "LBA: <start>, <mid-point>, <end>, <label>"
(or with LBB) with extra-long labels continuing with lines labelled "EXT: <label>".
.TP 11
.B -p
Praat text grid files.  If multiple tiers are present these are loaded into seperate annotation items
.TP 11
.BI -H htk.mlf
Load the annotations from the supplied HTK format master label file.  In this case the textfile
name refers to the section of the master label file to load.  Any scores are stripped
during loading.
.TP 11
.BI -T transcription
Load the annotation as a single label at time 0 using the supplied argument.
.SH OUTPUT ITEMS
.IP AN 11
Loaded annotation item
.SH VERSION/AUTHOR
1.9 - Mark Huckvale
.SH BUGS
Except in sample mode, annotation sizes are calculated automatically, the last annotation
has length zero.
*/
/*---------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "anload"
#define PROGVERS "1.9"
char	*progname = PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard i-o routines */
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header 	anitem;			/* annotation item header  */
struct an_rec		*an;			/* annotation record */
struct item_header	fxitem;			/* Fx item header */
short			*fx;			/* Fx data */
int			fxcnt;
char		textname[SFSMAXFILENAME];	/* text file name */
char		filename[SFSMAXFILENAME];	/* dbase file name */
char		mlfname[SFSMAXFILENAME];	/* HTK MLF name */
char		cltrans[1024];		/* command line transcription */
double		fdur=1.0E-4;		/* default sample duration */
double		durscale=1.0;			/* duration scaling */
char		*antype;		/* annotation type */
int			sampmode=0;		/* sample mode */
int			esps=0;			/* ESPS mode */
int			mbrola=0;		/* MBROLA mode */
int			millisec=0;		/* millisecond mode */
int			sam=0;			/* SAM mode */
int			samp1;			/* one sample per line not two */
int			samp0;			/* zero samples per line not two */
int			dopraat=0;

/* process fx entry in MBROLA file */
void processfx(char *line,double stime,double etime)
{
	char	*p;
	double	percent,fxval;
	int	idx;

	/* skip first two fields */
	p = strtok(line," \t\r\n");
	p = strtok(NULL," \t\r\n");
	p = strtok(NULL," \t\r\n");
	while (p && *p && isdigit(*p)) {
		/* first value is percentage through segment */
		percent = atof(p);
		/* second value is fx value */
		p = strtok(NULL," \t\r\n");
		if (p && *p && isdigit(*p)) {
			fxval = atof(p);
			idx = (int)((stime + (etime-stime)*percent/100)/0.01);
			if ((0<=idx)&&(idx<fxcnt))
				fx[idx] = (short)fxval;
		}
		p = strtok(NULL," \t\r\n");
	}
}

/* interpolate fx contour */
void interpfx()
{
	int	i,j,ifx=0;
	double	fx1=0,fx2=0,fxslope;

	/* find first fx value */
	for (i=0;i<fxcnt;i++)
		if (fx[i]!=0) {
			fx1=fx[i];
			break;
		}

	/* do interpolations */
	for (i=0;i<fxcnt;i++) {
		if (fx[i]!=0) {
			fx2 = fx[i];
			fxslope = (fx2-fx1)/(i-ifx);
			for (j=ifx;j<=i;j++)
				fx[j] = (short)(fx1 + fxslope*(j-ifx));
			fx1 = fx[i];
			ifx = i;
		}
	}

	/* finish off */
	for (j=ifx;j<fxcnt;j++)
		fx[j] = (short)fx1;
}

/* get sampling rate from SAM file */
void getsamrate(FILE *ip)
{
	char	line[1024];
	char	*p;

	/* most SAM recordings were at 20000 samp/sec */
	fdur=1.0/20000;

	/* scan file for SAM: entry */
	while (fgets(line,1024,ip)) {
		if (strncmp(line,"SAM",3)==0) {
			p = strtok(line," :\t");
			p = strtok(NULL," \t\r\n");
			if (p && isdigit(*p)) {
				fdur = 1.0/atof(p);
				break;
			}
		}
	}
	rewind(ip);
}

/* match filename in MLF file to text name */
int mlfmatch(char *mlfname,char *texname)
{
	int	i;
	char	mname[SFSMAXFILENAME];
	char	tname[SFSMAXFILENAME];
	char	*pm,*pt;

	for (i=0;mlfname[i];i++) {
		if (isupper(mlfname[i]))
			mname[i]=tolower(mlfname[i]);
		else if (mlfname[i]=='\\')
			mname[i]='/';
		else
			mname[i]=mlfname[i];
	}
	mname[i]='\0';
	for (i=0;textname[i];i++) {
		if (isupper(texname[i]))
			tname[i]=tolower(textname[i]);
		else if (texname[i]=='\\')
			tname[i]='/';
		else
			tname[i]=texname[i];
	}
	tname[i]='\0';

	if (strcmp(mname,tname)==0) return(1);

	pm=strrchr(mname,'/');
	pt=strrchr(tname,'/');
	if (pm && pt && strcmp(pm,pt)) return(0);
	if (pm && pt && (mname[0]=='*')) return(1);
	return(0);
}

/* what strncat should have been ... */
void mystrncat(char *dst,char *src,int max)
{
	int	i;

	for (i=0;(i<max)&&(dst[i]!='\0');i++) /* skip */;
	if ((i<max)&&(dst[i]=='\0')) {
		for (;(i<max-1)&&(*src!='\0');i++) dst[i]=*src++;
		dst[i]='\0';
	}
}

/* get audio length */
double getaudiolength(char *fname)
{
	int	fid;
	struct item_header spitem;

	if ((fid=sfsopen(fname,"r",NULL))<0)
		return(0.0);

	if (!sfsitem(fid,SP_TYPE,"0",&spitem)) {
		sfsclose(fid);
		return(0.0);
	}
	else {
		sfsclose(fid);
		return(spitem.numframes*spitem.frameduration);
	}
}

/* Praat Text Grid */
struct label_rec {
	double	start;
	double	stop;
	char	*text;
};
struct tier_rec {
	char	*title;
	double	start;
	double	stop;
	struct label_rec *label;
	int	nlabel;
};
struct grid_rec {
	struct tier_rec *tier;
	int	ntier;
	double	start;
	double	stop;
};

char *strsave(char *str)
{
	char	*ptr=malloc(strlen(str)+1);
	strcpy(ptr,str);
	return(ptr);
}

int gettoken(FILE *ip,char *field,char *value)
{
	int	c;
	int	state=0;

	while ((c=getc(ip))!=EOF) {
		if (!isspace(c)) {
			ungetc(c,ip);
			break;
		}
	}
	while (c=getc(ip)) {
		if (c==EOF) {
			*field='\0';
			*value='\0';
			return(0);
		}
		switch (state) {
		case 0:	/* loading field */
			if ((c=='\n')||(c==':')||(c=='=')||(c=='[')||(c=='?')) {
				ungetc(c,ip);
				state=1;
			}
			else if (!isspace(c))
				*field++ = c;
			break;
		case 1:	/* loading value */
			if (c=='"') {
				state=2;
			}
			else if (c=='[') {
				state=3;
			}
			else if (c=='=') {
				/* skip */;
			}
			else if ((c=='\n')||(c==':')) {
				*field='\0';
				*value='\0';
				return(1);
			}
			else if (!isspace(c))
				*value++ = c;
			break;
		case 2:	/* loading string */
			if ((c=='"')||(c=='\n')) {
				*field='\0';
				*value='\0';
				return(1);
			}
			else
				*value++ = c;
			break;
		case 3:	/* loading index */
			if ((c==']')||(c=='\n')) {
				*field='\0';
				*value='\0';
				return(1);
			}
			else
				*value++ = c;
			break;
		}
	}

	*field='\0';
	*value='\0';
	return(1);
}

struct grid_rec *loadtextgrid(FILE *ip)
{
	static struct grid_rec grid;
	int	idx,lab;
	char	field[256];
	char	value[256];

	gettoken(ip,field,value);
	if (strcmp(field,"Filetype")||strcmp(value,"ooTextFile")) {
		error("%s is not a text grid\n",textname);
	}
	gettoken(ip,field,value);
	if (strcmp(field,"Objectclass")||strcmp(value,"TextGrid")) {
		error("%s is not a text grid\n",textname);
	}

	while (gettoken(ip,field,value)) {
restarttier:
		if (strcmp(field,"xmin")==0)
			grid.start = atof(value);
		else if (strcmp(field,"xmax")==0)
			grid.stop = atof(value);
		else if (strcmp(field,"size")==0) {
			grid.ntier=atoi(value);
			grid.tier=(struct tier_rec *)calloc(grid.ntier,sizeof(struct tier_rec));
		}
		else if (strcmp(field,"item")==0) {
			idx = atoi(value);
			if ((1<=idx)&&(idx<=grid.ntier)) {
				idx--;
				while (gettoken(ip,field,value)) {
restartlabel:
					if (strcmp(field,"item")==0)
						goto restarttier;
					else if (strcmp(field,"class")==0) {
						if (strcmp(value,"IntervalTier")) {
							error("unexpected value for 'class'\n");
						}
					}
					else if (strcmp(field,"name")==0)
						grid.tier[idx].title = strsave(value);
					else if (strcmp(field,"xmin")==0)
						grid.tier[idx].start = atof(value);
					else if (strcmp(field,"xmax")==0)
						grid.tier[idx].stop = atof(value);
					else if (strcmp(field,"size")==0) {
						grid.tier[idx].nlabel = atoi(value);
						grid.tier[idx].label = (struct label_rec *)calloc(grid.tier[idx].nlabel,sizeof(struct label_rec));
					}
					else if (strcmp(field,"intervals")==0) {
						lab=atoi(value);
						if ((1<=lab)&&(lab<=grid.tier[idx].nlabel)) {
							lab--;
							while (gettoken(ip,field,value)) {
								if (strcmp(field,"item")==0)
									goto restarttier;
								else if (strcmp(field,"intervals")==0)
									goto restartlabel;
								else if (strcmp(field,"xmin")==0)
									grid.tier[idx].label[lab].start = atof(value);
								else if (strcmp(field,"xmax")==0)
									grid.tier[idx].label[lab].stop = atof(value);
								else if (strcmp(field,"text")==0)
									grid.tier[idx].label[lab].text = strsave(value);
							}
						}
					}
				}
			}
		}
		else {
			/* skip */;
		}
	}

	return(&grid);
}



/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;			/* option index */
	extern char	*optarg;		/* option argument ptr */
	int		errflg = 0;		/* option error flag */
	int		c;			/* option switch */

	/* processing variables */
	int		i,first;
	FILE		*ip;			/* input text file */
	int		ofid,ofid2;			/* output file descriptors */
	double		tim,l1;
	char		line[512],lab[256];
	int		s1,s2;
	double	ds1,ds2;
	double		totdur;
	char		*p;
	int			lastsamp=0,endsamp=0;
	struct grid_rec	*pgrid;
	int			tier;

	/* decode switches */
	while ( (c = getopt(argc,argv,"If:t:shemMSH:T:p")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Load AN item from text V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'f' :	/* select sampling frequency */
			i = atoi(optarg);
			if (i<=0)
				error("bad sampling frequency: %s",optarg);
			else
				fdur = 1.0/i;
			break;
		case 't' :	/* annotation type */
			antype=optarg;
			break;
		case 's' :	/* sample mode */
			esps=0;
			mbrola=0;
			millisec=0;
			sampmode=1;
			sam=0;
			break;
		case 'H' :	/* load from HTK MLF */
			strcpy(mlfname,optarg);
			/* fall through ... */
		case 'h' :	/* HTK format load */
			esps=0;
			mbrola=0;
			millisec=0;
			sampmode=1;
			fdur = 1.0E-5;
			durscale=0.01;
			sam=0;
			break;
		case 'e' :	/* ESPS mode */
			esps=1;
			mbrola=0;
			millisec=0;
			sampmode=0;
			sam=0;
			break;
		case 'm' :	/* MBROLA mode */
			esps=0;
			mbrola=1;
			millisec=0;
			sampmode=0;
			sam=0;
			break;
		case 'M' :	/* millisecond timing */
			esps=0;
			mbrola=0;
			millisec=1;
			sampmode=0;
			sam=0;
			break;
		case 'S' :	/* SAM mode */
			esps=0;
			mbrola=0;
			millisec=0;
			sampmode=0;
			sam=1;
			break;
		case 'p' :	/* Praat Text Grid */
			dopraat=1;
			fdur=1.0E-5;
			break;
		case 'T' :	/* command line */
			strncpy(cltrans,optarg,sizeof(cltrans));
			esps=0;
			mbrola=0;
			millisec=0;
			sampmode=0;
			sam=0;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check command line */
	if (errflg || (argc<3))
		error("usage: %s (-I) (-f freq) (-t type) (-s|-h|-e|-m|-M|-S|-p) (-H htk.mlf) -T transcription|textfile file",PROGNAME);

	if (cltrans[0]=='\0') {
		/* get text file name */
		if (optind < argc)
			strcpy(textname,argv[optind]);
		else
			error("no text file specified",NULL);
		optind++;
	}

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open text file */
	if (cltrans[0]!='\0') {
	}
	else if (mlfname[0]) {
		if ((ip=fopen(mlfname,"r"))==NULL)
			error("cannot open %s",mlfname);
		if (!fgets(line,sizeof(line),ip) || strncmp(line,"#!MLF!#",7))
			error("read error on '%s'",mlfname);
		/* seek to appropriate part */
		while (fgets(line,sizeof(line),ip)) {
			if (line[0]=='"') {
				/* name of file */
				p=strtok(line+1,"\"\r\n");
				if (p && mlfmatch(p,textname)) break;
			}
		}
		if (feof(ip))
			error("could not find '%s' in '%s'",textname,mlfname);
	}
	else if ((ip=fopen(textname,"r"))==NULL)
		error("cannot open %s",textname);

	/* if Praat, load the grid */
	if (dopraat) {
		pgrid=loadtextgrid(ip);
		if (pgrid->ntier==0)
			error("no tiers loaded from '%s'\n",textname);
		printf("Loaded %d tiers from %s\n",pgrid->ntier,textname);
	}

	/* if SAM file get sampling rate */
	if (sam) getsamrate(ip);

	/* open data file */
	if ((ofid=sfsopen(filename,"w",NULL)) < 0) {
		if (ofid==-1)
			error("cannot find file %s",filename);
		else
			error("access error on %s",filename);
	}
	sfsclose(ofid);

	/* create output item header */
	sfsheader(&anitem,AN_TYPE,-1,1,-1,fdur,0.0,0,0,1);
	if (cltrans[0]) {
		strncpy(textname,cltrans,100);
		strcpy(textname+50,"...");
		if (antype)
			sprintf(anitem.history,"%s(trans=%s,type=%s)",PROGNAME,textname,antype);
		else
			sprintf(anitem.history,"%s(trans=%s)",PROGNAME,textname);
	}
	else if (antype)
		sprintf(anitem.history,"%s(file=%s,type=%s)",PROGNAME,textname,antype);
	else if (dopraat)
		sprintf(anitem.history,"%s(file=%s,type=%s)",PROGNAME,textname,pgrid->tier[0].title);
	else
		sprintf(anitem.history,"%s(file=%s)",PROGNAME,textname);

	/* open output channel */
	if ((ofid=sfschannel(filename,&anitem)) < 0)
		error("unable to open temporary file",NULL);

	/* get output record buffer */
	if ((an=(struct an_rec *)sfsbuffer(&anitem,1))==NULL)
		error("could not get memory buffer",NULL);
	an->posn=0;

	/* skip any ESPS header */
	if (esps) {
		while (fgets(line,512,ip) != NULL)
			if (line[0]=='#') break;
	}

	/* find overall duration of MBROLA file */
	if (mbrola) {
		totdur=0;
		while (fgets(line,512,ip) != NULL) {
			if (line[0]==';') continue;
			if (sscanf(line,"%s %lf",lab,&l1)<2) break;
			totdur += l1*0.001;
		}
		rewind(ip);
		sfsheader(&fxitem,FX_TYPE,0,2,1,0.01,0.0,1,0,0);
		sprintf(fxitem.history,"%s/FX(file=%s)",PROGNAME,textname);
		fxcnt = (int)(0.5+totdur/0.01);
		fx = (short *)calloc(fxcnt,sizeof(short));
		ofid2 = sfschannel(filename,&fxitem);
	}

	if (cltrans[0]) {
		an->posn=0;
		an->size=(int)(getaudiolength(filename)/fdur);
		strncpy(an->label,cltrans,MAXANLABEL-1);
		an->label[MAXANLABEL]='\0';
		if (sfswrite(ofid,1,an) != 1)
			error("write error on output file",NULL);
	}
	else if (dopraat) {
		for (tier=0;tier<pgrid->ntier;tier++) {
			for (i=0;i<pgrid->tier[tier].nlabel;i++) {
				an->posn = (int)(0.5+pgrid->tier[tier].label[i].start/fdur);
				an->size = (int)(0.5+pgrid->tier[tier].label[i].stop/fdur)-an->posn;
				strncpy(an->label,pgrid->tier[tier].label[i].text,MAXANLABEL-1);
				if (sfswrite(ofid,1,an) != 1)
					error("write error on output file",NULL);
			}
			if ((tier+1)<pgrid->ntier) {
				sfsheader(&anitem,AN_TYPE,-1,1,-1,fdur,0.0,0,0,1);
				sprintf(anitem.history,"%s(file=%s,type=%s)",PROGNAME,textname,pgrid->tier[tier+1].title);
				if ((ofid=sfschannel(filename,&anitem)) < 0)
					error("unable to open temporary file",NULL);
			}
		}
	}
	else {
		/* process input file line by line */
		first=1;
		tim=0;
		while (fgets(line,512,ip) != NULL) {
			if (mlfname[0]&&(strcmp(line,".\n")==0)) break;
			if (sam) {
				if ((strncmp(line,"LBA",3)==0)||(strncmp(line,"LBB",3)==0)) {
					if (!first) {
						if (sfswrite(ofid,1,an) != 1)
							error("write error on output file",NULL);
					}
					p = strtok(line," :\t");
					p = strtok(NULL," ,");
					if (p && isdigit(*p)) {
						an->posn=atoi(p);
						p = strtok(NULL,",");
						p = strtok(NULL," ,");
						if (p && isdigit(*p)) {
							lastsamp=atoi(p);
							an->size=lastsamp-an->posn;
							p = strtok(NULL,"\r\n");
							while (p && isspace(*p)) p++;
							if (p && *p) {
								strncpy(an->label,p,MAXANLABEL-1);
							}
						}
					}
					first=0;
				}
				else if (strncmp(line,"LBR",3)==0) {
					if (!first) {
						if (sfswrite(ofid,1,an) != 1)
							error("write error on output file",NULL);
					}
					p = strtok(line," :\t");
					p = strtok(NULL," ,");
					if (p && isdigit(*p)) {
						an->posn=atoi(p);
						p = strtok(NULL," ,");
						if (p && isdigit(*p)) {
							lastsamp=atoi(p);
							an->size=lastsamp-an->posn;
							p = strtok(NULL," ,");	/* skip gain */
							p = strtok(NULL," ,");	/* skip min */
							p = strtok(NULL," ,");	/* skip max */
							p = strtok(NULL,"\r\n");
							while (p && isspace(*p)) p++;
							if (p && *p) {
								strncpy(an->label,p,MAXANLABEL-1);
							}
						}
					}
					first=0;
				}
				else if (strncmp(line,"EXT",3)==0) {
					p = strtok(line," :\t");
					p = strtok(NULL,"\r\n");
					while (isspace(*p)) p++;
					if (p && *p) {
						mystrncat(an->label," ",MAXANLABEL-1);
						mystrncat(an->label,p,MAXANLABEL-1);
					}
					first=0;
				}
				else if (strncmp(line,"END",3)==0) {
					p = strtok(line," :\t");
					p = strtok(NULL," \t\r\n");
					if (p && isdigit(*p)) endsamp=atoi(p);
				}
				else if (strncmp(line,"ELF",3)==0) {
					if (!first) {
						if (sfswrite(ofid,1,an) != 1)
							error("write error on output file",NULL);
					}
					/* put in closing annotation */
					an->posn=lastsamp;
					if (endsamp)
						an->size = endsamp-lastsamp;
					else
						an->size=0;
					strcpy(an->label,"/");
					if (sfswrite(ofid,1,an) != 1)
						error("write error on output file",NULL);
				}
			}
			else if (esps) {
				if (first)
					an->posn = 0;
				else
					an->posn = (int)(tim/fdur);
				if (sscanf(line,"%lf %d %[^\n]\n",&tim,&s1,lab) < 3) break;
				an->size = (int)(tim/fdur - an->posn);
				strncpy(an->label,lab,MAXANLABEL-1);
				if (sfswrite(ofid,1,an) != 1)
					error("write error on output file",NULL);
				first=0;
			}
			else if (mbrola) {
				if (line[0]==';') continue;
				if (first)
					an->posn = 0;
				else
					an->posn = (int)(tim/fdur);
				if (sscanf(line,"%s %lf",lab,&l1)<2) break;
				an->size = (int)((0.001*l1)/fdur);
				strncpy(an->label,lab,MAXANLABEL-1);
				if (sfswrite(ofid,1,an) != 1)
					error("write error on output file",NULL);
				processfx(line,tim,tim+0.001*l1);
				tim += 0.001*l1;
				first=0;
			}
			else if (sampmode) {
				if (samp0) {
					s1=0;
					if (sscanf(line,"%[^\n]\n",lab) < 1)
						break;
				}
				else if (samp1) {
					if (sscanf(line,"%lf %[^\n]\n",&ds1,lab) < 2)
						break;
					s1 = (int)(0.5+ds1*durscale);
				}
				else {
					if (sscanf(line,"%lf %lf %[^\n]\n",&ds1,&ds2,lab) < 3) {
						if (first && (sscanf(line,"%lf %[^\n]\n",&ds1,lab)==2)) {
							ds2=0;
							samp1=1;
						}
						else if (first && sscanf(line,"%[^\n]\n",lab)==1) {
							ds1=0;
							ds2=0;
							samp0=1;
						}
						else
							break;
					}
					s1 = (int)(0.5+ds1*durscale);
					s2 = (int)(0.5+ds2*durscale);
				}
				if (!first && (s1 < an->posn))
					error("annotations not in order in '%s'",textname);
				if (mlfname[0]) strtok(lab," ");
				if (samp0) {
					an->posn = 0;
					an->size = 0;
					if (!first) mystrncat(an->label," ",MAXANLABEL-1);
					mystrncat(an->label,lab,MAXANLABEL-1);
				}
				else if (samp1) {
					if (!first) {
						an->size = s1 - an->posn;
						if (sfswrite(ofid,1,an) != 1)
							error("write error on output file",NULL);
					}
					an->posn = s1;
					strncpy(an->label,lab,MAXANLABEL-1);
				}
				else {
					an->posn = s1;
					an->size = s2-s1;
					strncpy(an->label,lab,MAXANLABEL-1);
					if (sfswrite(ofid,1,an) != 1)
						error("write error on output file",NULL);
				}
				first=0;
			}
			else if (millisec) {
				if (first)
					an->posn = 0;
				else
					an->posn = (int)(tim/fdur);
				if (sscanf(line,"%lf %[^\n]\n",&l1,lab) < 2) break;
				an->size = (int)(0.001*l1/fdur);
				strncpy(an->label,lab,MAXANLABEL-1);
				if (sfswrite(ofid,1,an) != 1)
					error("write error on output file",NULL);
				tim += 0.001*l1;
				first=0;
			}
			else {
				if (sscanf(line,"%lf %[^\n]\n",&tim,lab) < 2) break;
				if (!first) {
					an->size = (int)(tim/fdur - an->posn);
					if (sfswrite(ofid,1,an) != 1)
						error("write error on output file",NULL);
				}
				an->posn = (int)(tim/fdur);
				strncpy(an->label,lab,MAXANLABEL-1);
				first=0;
			}
		}
		if (!first && !sam && ((!sampmode && !esps && !mbrola && !millisec)||(samp1==1))) {
			an->size = 0;
			if (sfswrite(ofid,1,an) != 1)
				error("write error on output file",NULL);
		}
		if (mbrola) {
			interpfx();
			sfswrite(ofid2,fxcnt,fx);
		}
	}

	/* update data file */
	if (!sfsupdate(filename))
		error("backup error on %s",filename);

	/* that's all folks */
	exit(0);

}
