/* Es - interactive device independent item display */

/* M.A.Huckvale - April 1988 */

/* version 1.0
	- from Ds version 1.2
*/
/* version 2.0 - June 1988
	- with annotation facility
*/
/* version 2.1 - July 1988
	- minor changes to scrolling
	- annotation hit accuracy reduced
*/
/* version 2.2 - November 1988
	- pick up last annotation item of given type
*/
/* version 2.3 - February 1989
	- add hardware synthesis replay
*/
/* version 2.4 - January 1990
	- faster grey-scale display
	- faster loading of fixed-frame dataset sections
	- move annotation facility in label mode
*/
/* version 2.5 - August 1990
	- 16-bit signal replay conversion
*/
/* version 2.6 - January 1991
	- add VU item display
*/
/* version 2.7 - March 1991
	- fix failure to display off end of data set
*/
/* version 2.8 - May 1991
	- add UG item display
*/
/* version 2.9 - February 1992
	- changes for SPARC
		- change syscall to syscallbuff
		- add -H as alias to -16
*/
/* version 3.0 - May 1993
	- add window resizing for X11
	- add support for new SFSDAC routines
*/
/* version 3.1 - August 1993
	- module re-organisation/rationalisation
	- add spectrogram display
	- add tree mode
	- add copy and link
	- add file sub-menu
*/
/* version 3.2 - April 1995
	- add play and other-play commands
	- add goto command
	- add annotate end command
*/
/* version 3.3 - November 1995
	- add measurement display option
*/
/* version 3.4 - April 1997
	- mouse on play button plays top waveform
	- add <> control on left cursor
	- add RETURN command on annotation move
*/
/* version 3.5 - August 1999
	- improved control of spectrogram displays
*/

/* program name and version */
#define	PROGNAME "Es"
#define PROGVERS "3.5"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH ES SFS1 UCL SFS
.SH NAME
Es - interactive device-independent data display
.SH SYNOPSIS
.B Es
(-i item) (-a item) (-s start) (-e end) (-f item lo hi)
(-l labelname) (-12|-0) (-g spectitem) (-t) (-m) file
.SH DESCRIPTION
.I Es
is a program to display portions of data sets in a file using device-independent graphics.
Mouse interaction allows selective expansion of regions of the display, and replay of speech items.
Options allow for the selection of items and the initial display of part of the timescale.
Default operation is to display all 
items in file.
.I Es
may also be used to add/edit annotations using the '-l' option.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select first item matching item specification.
.TP 11
.BI -a type
Select all items matching item specification.
.TP 11
.BI -g spitem
Select a speech item for display as a spectrogram using on-line FFT.
.TP 11
.BI -G opt
Control of spectrogram display.  Argument is made from these letters:
'S' = display speech signal, 'W' = display wideband, 'N' = display narrowband.
This applies to all speech items displayed.
.TP 11
.B -t
Start up with file processing tree display for item selection.
.TP 11
.BI -s start
Specify a start time (in seconds) other than 0.0.
.TP 11
.BI -e end
Specify an end time (in seconds) other than the length of the first speech 
item in the file.  If no speech item found, default timescale is 1 second.
.TP 11
.BI -f "item lo hi"
Fix all displays of a given type to the y-range lo..hi.
.TP 11
.BI -l labelname
Select labelling mode: an annotation editing facility.  
The annotations to be edited must be given a name (labelname)
that is stored in the history for the annotation data set.  If any 
existing annotation items in the file
have the same name (i.e. with "type=<name>" in the history field), 
the last is used as the initial contents of the 
annotation data set.
.TP 11
.B -12
Specify that all SP items otherwise 'unmarked' will be treated as
12-bits (-2048..2047).  The marking is shown by a parameter field
entry of "bits=12" or "bits=16".  Default: all unmarked items are treated as
containing 16-bit data.
.TP 11
.B -m
Select measurement mode.  Cursor position is displayed in time and frequency
in area normally used for bottom timescale.
.TP 11
.B -0
Disable replay.
.SH MOUSE COMMANDS
.SS LEFT
Set "start" marker.  For accurate placement, use '<' and '>' keys in
combination with mouse.
.SS RIGHT
Set "stop" marker.
.SS MIDDLE
Replay selected segment.  Appropriate speech item chosen automatically.  In labelling mode, the
region to be replayed depends on the mouse location; normally the region between cursors is
replayed wherever the mouse is positioned.  SY items are replayed through hardware
synthesizer or by software emulation of the SRU synthesizer (see hwsynth(3)).  To move an annotation in labelling mode, click the middle button
over the annotation in the annotation edit window and then the annotation will jump to
the position indicated by the next click of the left or right buttons (or on hit of RETURN key).
To deselect, click the middle button outside the annotation edit window.
.SH MAIN MENU
.SS QUIT
(Any button, 'q', 'x', ESCAPE) Quit program.
.SS FILE
(Any button, 'f') Display file sub-menu.
.SS PLAY
(Any button, 'p') Play region of signal between cursors.  Or 'o' plays region to left
cursor, and '[' plays region from right cursor.  The speech item selected for replay is
the one under the last known mouse position - note that this means that the 'play'
button cannot be selected by the mouse!
.SS MOVE
(Left button, 'l') Scroll left. "stop" marker sets new right margin if present.
(Right button, 'r') Scroll right. "start" marker sets new left margin if present.
(Middle button, 'u') Undo previous "zoom".
('t') Go to top level of zoom.
.SS ZOOM
(Any button, 'z') Zoom in to selected region.  Or 'g' to goto a particular time
in the file; you are prompted to enter the time.
.SS ANNO
(Any button, 'a'.  Labelling mode only) Add an 
annotation at the current position of the left cursor.  Type the name of the annotation
and press [RETURN].  To replace an existing annotation, position the cursor on top of it, key 'a', the 
new annotation label and [RETURN].  To delete an existing annotation, 
position the cursor on top of it, key 'a' and type [RETURN] only.  Or 'b' to
input an annotation at the right cursor.
.SH FILE MENU
.SS MAIN
(Any button, 'q', 'x', 'm', ESCAPE or RETURN) Return to Main menu.
.SS HARD
(Any button, 'h' or 'p') Print screen ("Hardcopy").
.SS TREE
(Any button, 't') Display file history tree to reselect items for display.
.SS COPY
(Any button, 'c') Copy displayed section of displayed items to a new file.
(Currently only some items types may be copied).
.SS LINK
(Any button, 'l') Link displayed section of displayed items to a new file.
(Currently only some items types may be linked).
.SH TREE MENU
.SS OK
(Any button, 'q', 'x', 'o', ESCAPE, RETURN) Accept selections and return to Es display.
.SS HARD
(Any button, 'p', 'h') Print hard copy of file history tree.  Not yet implemented.
.SS ALL
(Any button, 'a') Select all items.
.SS ZERO
(Any button, 'z') Clear selection of items.
.SH INPUT ITEMS
.IP 1.xx 11
Any speech item.  Displayed as waveform with '-a' or '-i' switches, as
spectrogram with '-g' switch.
.IP 2.xx 11
Any excitation item.  Displayed as waveform.
.IP 3.xx 11
Any period data item.  Displayed at period markers.
.IP 4.xx 11
Any fundamental frequency item.  Displayed as fx trace on linear scale.
.IP 5.xx 11
Any annotation item.  Displayed as markers.
.IP 7.xx 11
Any synthesizer control data item.  Displayed in graphical form.
.IP 9.xx 11
Fixed frame Display data in SFS format.  Displayed as grey scale.
.IP 10.xx 11
Voice/Unvoiced markers. Voicing displayed as "V" annotations.
.IP 11.xx 11
Any coefficient data item.  Displayed as spectra.
.IP 12.xx 11
Any raw formant estimates item.  Displayed as numbered points.
.IP 16.xx 11
Any parameter track item.  Displayed as trace.
.PP
Other items are merely noted in the display.
.SH OUTPUT ITEMS
.IP 5 11
Edited annotations.
.SH ENVIRONMENT
The terminal type is determined from the environment variable GTERM
if present, which is used to configure the graphics using the
information in $SFSBASE/data/digmap.  If GTERM is not set, then the
TERM variable is used.  The default printer is selected from the
environment variable GPRINT, which is used to configure the printed
graphics using $SFSBASE/data/digmap.  If GPRINT is not set, it is
assumed to have the value 'printer'.  The replay device type is
determined from the environment variable DAC, which is used to
configure the replay according to a table built in to the software
(sfsdac.c).  If DAC is not set, no replay is available.
.SH VERSION/AUTHOR
3.3 - Mark Huckvale
.SH SEE ALSO
Ds(SFS1) dig(SFS3) dig(SFS1)
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard i-o routines */
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <signal.h>
#include "sfs.h"		/* database filing system structures */
#include "dig.h"
#include "digdata.h"
#include "digitem.h"
#include "sfstree.h"
#include "es.h"

/* manifest constants */
#define MAXSELECT	20	/* maximum number of item selections */
#define MAXDISP		20	/* maximum number of displayed items */
#define MAXZOOM		10	/* maximum depth of zoom stack */
#define MAXANNOT	3000	/* maximum # annotations */
#define ANDUR		0.00005	/* annotation precision (seconds) */
#define MAX(x,y) (((x)>(y))?(x):(y))

/* spectrogram display modes */
#define SGRAM_WAVE	1
#define SGRAM_WIDE	2
#define SGRAM_NARROW	4

/* item selection data */
struct itemsel_rec itab[MAXSELECT];		/* normal items */
int	itcnt=0;
struct itemsel_rec citab[MAXSELECT];		/* FFT displays */
int	citcnt=0;
struct itemsel_rec nitab[MAXSELECT];		/* narrowband FFT displays */
int	nitcnt=0;

/* list of items */
struct itemno_rec itemlist[MAXITEM];		/* normal items */
int	itlistno;
struct itemno_rec citemlist[MAXITEM];		/* FFT displays */
int	citlistno;
struct itemno_rec nitemlist[MAXITEM];		/* narrowband FFT displays */
int	nitlistno;

/* item display table */
int	sizes[MAXDATATYPE+1]={	0,3,3,1,5,1,0,7,7,7,
				1,5,7,0,0,0,5,0,0,0,
				0,0,0,0,0,3};
int	inum[MAXDATATYPE+1];

/* item display axis fixing */
struct {
	int	fix;
	double	lo,hi;
} fixtab[MAXDATATYPE+1];

/* time stack */
struct {
	double	s,e;
} tstack[MAXZOOM];
int	ts=0;

/* replay control */
struct {
	float	yb,yt;
	int	datatype;
	char	itemtype[8];
	short	*buff;
	short	link;
	int	numframes;
	double	offset,sfreq;
	int	nbits;
	double	lo,hi;
} reptab[MAXDISP];

/* data variables */
struct main_header head;
char	    filename[SFSMAXFILENAME];
char	    cfilename[SFSMAXFILENAME];
int		fid;	/* file descriptor */
double		tsize,fixfrac=0,varfrac=0,totime=0;
int		numdisp=0;
char		title[256];
double		starttime= -1.0,endtime= -1.0;
char		syscallbuff[1024],syscallbuff2[256];
int		scale12=0;		/* 12-bit scaling enable */
int		canplay=1;		/* replay checking */
int		dotree=0;		/* initial display of tree */
int		doprint=0;		/* send direct to printer */
char		printtitle[128];	/* title for print */
int		domeasure=0;		/* do measurement */
int		gmode=0;

/* annotation control variables */
int			labelmode=0;	/* labelling mode selected */
char			*labtitle;	/* labels title */
char			anhist[256];	/* history field of input item */
struct item_header	anitem;		/* annotation item header */
struct an_rec		*antab;		/* table of annotations */
struct an_rec		*antemp;	/* temporary annotation record */
int			ancount;	/* # annotations in buffer */
int			anchange;	/* changes made to annotations */
int			angran;		/* annotation granularity */

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding variables */
	extern int      optind;
	extern char     *optarg;
	int	     errflg=0;
	int	     c;
	double		len,atof();
	/* file variables */
	struct item_header item;
	/* matching variables */
	int	     i,j,comm;
	char		ans[10],messg[128];
	int		ofid;
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:a:g:n:s:e:f:l:1:tpT:0mG:")) != EOF) switch (c) {
		case 'I' :      /* Identify */
			fprintf(stderr,"%s: Interactive display of SFS file V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&itab[itcnt].it,&itab[itcnt].match) != 0)
				error("bad 'i' option : %s",optarg);
			if (itcnt < MAXSELECT)
				itcnt++;
			break;
		case 'a' :	/* generic item */
			if (itspec(optarg,&itab[itcnt].it,&itab[itcnt].match) != 0)
				error("bad 'a' option : %s",optarg);
			if (strcmp(itab[itcnt].match,"0")==0)
				itab[itcnt].match = "*";
			itab[itcnt].all=1;
			if (itcnt < MAXSELECT)
				itcnt++;
			break;
		case 'g' :	/* specific item for FFT display */
			if (itspec(optarg,&citab[citcnt].it,&citab[citcnt].match) != 0)
				error("bad 'g' option : %s",optarg);
			if (citab[citcnt].it != SP_TYPE)
				error("-g switch on speech items only");
			if (citcnt < MAXSELECT)
				citcnt++;
			break;
		case 'n' :	/* specific item for narrow FFT display */
			if (itspec(optarg,&nitab[nitcnt].it,&nitab[nitcnt].match) != 0)
				error("bad 'n' option : %s",optarg);
			if (nitab[nitcnt].it != SP_TYPE)
				error("-n switch on speech items only");
			if (nitcnt < MAXSELECT)
				nitcnt++;
			break;
		case 's' :	/* start time */
			starttime=atof(optarg);
			break;
		case 'e' :	/* end time */
			endtime=atof(optarg);
			break;
		case 'f' :	/* fix scales */
			if (itspec(optarg,&itab[itcnt].it,&itab[itcnt].match) != 0)
				error("bad 'f' option : %s",optarg);
			fixtab[itab[itcnt].it].fix++;
			fixtab[itab[itcnt].it].lo=atof(argv[optind++]);
			fixtab[itab[itcnt].it].hi=atof(argv[optind++]);
			break;
		case 'l' :	/* labelling mode */
			labelmode++;
			labtitle=optarg;
			break;
		case '1' :	/* scale to 12 bits */
			if (*optarg=='2')
				scale12++;
			else
				error("bad '-1' option, '-12' intended?");
			break;
		case '0' :
			canplay=0;
			break;
		case 't' :	/* initial display of tree */
			dotree++;
			break;
		case 'p' :	/* print display */
			doprint++;
			break;
		case 'T' :	/* print title */
			strncpy(printtitle,optarg,sizeof(printtitle)-1);
			break;
		case 'm' :	/* measurement */
			domeasure++;
			break;
		case 'G' :	/* spectrogram display modes */
			if (strchr(optarg,'S')||strchr(optarg,'s')) gmode |= SGRAM_WAVE;
			if (strchr(optarg,'W')||strchr(optarg,'w')) gmode |= SGRAM_WIDE;
			if (strchr(optarg,'N')||strchr(optarg,'n')) gmode |= SGRAM_NARROW;
			break;			
		case '?' :      /* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-a item) (-g spitem) (-n spitem) (-s start) (-e end) (-f item lo hi) (-l labelname) (-12|-0) (-t) (-m) (-p) (-T printtitle) (-G sgopt) file",argv[0]);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open datafile */
	if ((fid = sfsopen(filename,(labelmode)?"w":"r",&head)) < 0) {
		if (fid == -1)
			error("cannot open %s",filename);
		else
			error("access error on %s",filename);
	}

	/* get annotation buffer */
	if (labelmode) {
		if ((antab=(struct an_rec *)calloc(MAXANNOT,sizeof(struct an_rec)))==NULL)
			error("unable to get memory for annotation buffer",NULL);
		sfsheader(&anitem,AN_TYPE,-1,1,-1,ANDUR,0.0,0,0,1);
		sprintf(anitem.history,"%s/AN(type=%s)",PROGNAME,labtitle);
		antemp = (struct an_rec *) sfsbuffer(&anitem,1);
	}

	/* convert item selections into item numbers */
	if (dotree && (itcnt==0))
		itlistno = 0;
	else if ((citcnt>0) && (itcnt==0))
		itlistno = 0;
	else
		itlistno = esitem(fid,itab,itcnt,itemlist);
	if (citcnt)
		citlistno = esitem(fid,citab,citcnt,citemlist);
	else
		citlistno = 0;
	if (nitcnt)
		nitlistno = esitem(fid,nitab,nitcnt,nitemlist);
	else
		nitlistno = 0;


	/* add in any required spectrograms */
	if (gmode&(SGRAM_WIDE|SGRAM_NARROW)) {
		for (i=0;i<itlistno;i++) {
			if (itemlist[i].datatype==SP_TYPE) {
				if (gmode & SGRAM_WIDE) {
					citemlist[citlistno].datatype=itemlist[i].datatype;
					citemlist[citlistno].subtype=itemlist[i].subtype;
					citlistno++;
				}
				if (gmode & SGRAM_NARROW) {
					nitemlist[nitlistno].datatype=itemlist[i].datatype;
					nitemlist[nitlistno].subtype=itemlist[i].subtype;
					nitlistno++;
				}
			}
		}
		/* but delete any spectrogram only's */
		if (!(gmode&SGRAM_WAVE)) {
			for (i=0;i<itlistno;i++) {
				if (itemlist[i].datatype==SP_TYPE) {
					for (j=i+1;j<itlistno;j++)
						itemlist[j-1] = itemlist[j];
					itlistno--;
					i--;
				}
			}
		}
	}


	/* open graphics */
	if (doprint && !dotree)
		digstart(DIG_DEFAULT_PRINTER,NULL,0);
	else 
		digstart(DIG_DEFAULT_TERM,NULL,0);

	/* display tree if required */
	if (dotree) {
dotreedisplay:
		/* edit itemlist on tree */
		itlistno = estree(filename,itemlist,itlistno,0);
		if (doprint) {
			digquit(0);
			digstart(DIG_DEFAULT_PRINTER,NULL,0);
		}
	}
	else
		/* build index */
		sfstree(filename);
	
	/* do pass to count items */
	memset((char *)inum,0,sizeof(inum));
	sfsnextitem(fid,NULL);
	while (sfsnextitem(fid,&item)) {
		if ((item.datatype==SP_TYPE) && (totime==0.0))
			totime=item.numframes*item.frameduration;
		if (labelmode && (item.datatype==AN_TYPE)) {
			if (strcmp(params(item.history,"type",""),labtitle)==0) 
				/* keep history to find it later */
				sprintf(anhist,"%02d",item.subtype);
				/* old: strcpy(anhist,item.history); */
		}
		if (onitemlist(itemlist,itlistno,item.datatype,item.subtype))
			inum[item.datatype]++;
		if ((item.datatype==SP_TYPE) &&
		    onitemlist(citemlist,citlistno,item.datatype,item.subtype))
			inum[DI_TYPE]++;
		if ((item.datatype==SP_TYPE) &&
		    onitemlist(nitemlist,nitlistno,item.datatype,item.subtype))
			inum[DI_TYPE]++;
	}
	if (totime==0.0) totime=1.0;
	if (starttime < 0.0) starttime=0.0;
	if (endtime < 0.0) endtime=totime;
	if (endtime <= starttime) endtime=starttime+1.0;

	/* load input annotations if required */
	if (labelmode && anhist[0] && !doprint) {
		sfsitem(fid,AN_TYPE,anhist,&item);
		preloadan(fid,&item);
	}

	/* set up for display */
	digscale((float)1.0,(float)1.0,0);
	if (!doprint) digclearscreen();
	signal(SIGINT,breakhandler);

	/* count size data sets to display */
	tsize = 2 * digdata.chheight / digdata.yscale;
	if (labelmode && !doprint)
		fixfrac = 5*tsize;
	else
		fixfrac = 3*tsize;
	varfrac = 0;
	numdisp = 0;
	for (i=1;i<=MAXDATATYPE;i++) {
		fixfrac += inum[i] * tsize;
		varfrac += inum[i] * sizes[i];
		numdisp += inum[i];
	}
	if (numdisp==0) {
		digend();
		error("nothing to display",NULL);
	}
	else if (numdisp >= MAXDISP) {
		warning("Too many items to display");
		goto dotreedisplay;
	}

	/* check amount of data against screen size */
	if (fixfrac > 1.0) {
		warning("screen/window too small");
		goto dotreedisplay;		
	}
	varfrac = (1.0 - fixfrac)/varfrac;

	/* do graphics */
	dograph();
	if (doprint) {
		/* that's all folks if printing */
		sfsclose(fid);
		digquit(0);
		exit(0);
	}

	/* do interaction */
	tstack[0].s = starttime;
	tstack[0].e = endtime;
	tstack[1].s = starttime;
	tstack[1].e = endtime;
	ts=1;
	while ((comm = digitmouse(242220,(float)0.0,(float)0.0,(float)1.0,(float)1.0,&starttime,&endtime)) != ES_QUIT) {
		switch (comm) {
		case ES_UP:		/* up */
			if (ts==1) {
				tstack[1].s = tstack[0].s;
				tstack[1].e = tstack[0].e;
			}
			else if (ts > 1) ts--;
			break;
		case ES_DOWN:		/* down */
			if (ts < 10) ts++;
			tstack[ts].s = starttime;
			tstack[ts].e = endtime;
			break;
		case ES_LEFT:		/* left */
			if (ts==1) {
				tstack[ts+1]=tstack[ts];
				ts++;
			}
			len = tstack[ts].e - tstack[ts].s;
			if (endtime < tstack[ts].e)
				tstack[ts].s = MAX(0.0,endtime-len);
			else
				tstack[ts].s = MAX(0.0,starttime-len);
			tstack[ts].e = tstack[ts].s + len;
			break;
		case ES_RIGHT:		/* right */
			if (ts==1) {
				tstack[ts+1]=tstack[ts];
				ts++;
			}
			len = tstack[ts].e - tstack[ts].s;
			if (starttime > tstack[ts].s)
				tstack[ts].s = starttime;
			else 
				tstack[ts].s = endtime;
			tstack[ts].e = tstack[ts].s + len;
			break;
		case ES_PRINT:		/* print */
			printtitle[0]='\0';
			if (getprompt("Print title : ",printtitle)) {
#ifdef WIN32
				sprintf(syscallbuff,"start /min %s",argv[0]);
#else
				strcpy(syscallbuff,argv[0]);
#endif
				for (i=0;i<itlistno;i++) {
					sprintf(messg," -i%d.%02d",
						itemlist[i].datatype,
						itemlist[i].subtype);
					strcat(syscallbuff,messg);
				}
				for (i=0;i<citlistno;i++) {
					sprintf(messg," -g%d.%02d",
						citemlist[i].datatype,
						citemlist[i].subtype);
					strcat(syscallbuff,messg);
				}
				for (i=0;i<nitlistno;i++) {
					sprintf(messg," -n%d.%02d",
						nitemlist[i].datatype,
						nitemlist[i].subtype);
					strcat(syscallbuff,messg);
				}
				if (printtitle[0])
#ifdef WIN32
					sprintf(syscallbuff2," -s%g -e%g -p -T \"%s\" \"%s\"",
						starttime,endtime,printtitle,filename);
#else
#ifdef _MSC_VER
					sprintf(syscallbuff2," -s%g -e%g -p -T \"%s\" %s",
						starttime,endtime,printtitle,filename);
#else
					sprintf(syscallbuff2," -s%g -e%g -p -T '%s' %s",
						starttime,endtime,printtitle,filename);
#endif
#endif
				else
					sprintf(syscallbuff2," -s%g -e%g -p %s",
						starttime,endtime,filename);
				strcat(syscallbuff,syscallbuff2);
#ifndef _MSC_VER
				strcat(syscallbuff," &");
#endif
				digprompt("Wait ..");
				if (system(syscallbuff)!=0)
					warning("Error in print request");
				else
					sleep(2);
			}
			clear_top_line();
			title_display();
#ifdef _MSC_VER
goto dorepaint;
#endif
			continue;
		case ES_TOP:		/* top */
			tstack[1].s = tstack[0].s;
			tstack[1].e = tstack[0].e;
			ts=1;
			break;
		case ES_TREE:			/* display file tree */
dotreeagain:
			itlistno = estree(filename,itemlist,itlistno,0);
			/* check if all */
			if (itlistno==0)
				itlistno = esitem(fid,itab,0,itemlist);
			/* do pass to count items */
			memset((char *)inum,0,sizeof(inum));
			sfsnextitem(fid,NULL);
			while (sfsnextitem(fid,&item)) {
				if (onitemlist(itemlist,itlistno,item.datatype,item.subtype))
					inum[item.datatype]++;
				if ((item.datatype==SP_TYPE) &&
				    onitemlist(citemlist,citlistno,item.datatype,item.subtype))
					inum[DI_TYPE]++;
				if ((item.datatype==SP_TYPE) &&
				    onitemlist(nitemlist,nitlistno,item.datatype,item.subtype))
					inum[DI_TYPE]++;
			}
			/* fall through to repaint */
		case ES_REPAINT:		/* repaint/resize */
dorepaint:
			digscale((float)1.0,(float)1.0,0);
			digclearscreen();
			tsize = 2 * digdata.chheight / digdata.yscale;
			if (labelmode)
				fixfrac = 5*tsize;
			else
				fixfrac = 3*tsize;
			varfrac = 0;
			numdisp = 0;
			for (i=1;i<=MAXDATATYPE;i++) {
				fixfrac += inum[i] * tsize;
				varfrac += inum[i] * sizes[i];
				numdisp += inum[i];
			}
			varfrac = (1.0 - fixfrac)/varfrac;
			if (numdisp >= MAXDISP) {
				warning("Too many items to display");
				goto dotreeagain;
			}
			break;
		case ES_COPY:			/* copy part of display */
		case ES_LINK:			/* link part of display */
			strcpy(cfilename,filename);
			if (getprompt("Filename : ",cfilename)) {
				switch (escheckfile(cfilename)) {
				case ES_NOEXIST:
					do {
						sprintf(messg,"Create '%s' (y/n) ? ",cfilename);
						digprompt(messg);
						ans[0] = digwait();
					} while ((ans[0]!='y') && (ans[0]!='n'));
					if (ans[0]!='y')
						break;
					if ((ofid = sfsopen(cfilename,"c",&head)) < 0) {
						warning("could not create file");
						break;
					}
					sfsclose(ofid);
					/* fall through to ok */
				case ES_OK:
					digprompt("Wait ..");
					if (comm==ES_COPY) {
						i = escopyitem(filename,fid,itemlist,itlistno,
							tstack[ts].s,tstack[ts].e,
							cfilename);
						sprintf(messg,"%d items copied to '%s'",i,cfilename);
					}
					else {
						i = eslinkitem(filename,fid,itemlist,itlistno,
							tstack[ts].s,tstack[ts].e,
							cfilename);
						sprintf(messg,"%d items linked to '%s'",i,cfilename);
					}
					warning(messg);
					break;
				case ES_NOWRITE:
					warning("File is not writable");
					break;
				case ES_NOTSFS:
					warning("File exists and is not SFS");
					break;
				}
			}
			clear_top_line();
			title_display();
			continue;
		}
		starttime = tstack[ts].s;
		endtime = tstack[ts].e;
		dograph();
	}

	/* save annotations */
	if (labelmode && anchange && (ancount > 0)) {
		sortan(1);
		digclearscreen();
		do {
			digprompt("Es: save annotations ? (y/n) : ");
			ans[0] = digwait();
		} while ((ans[0]!='y') && (ans[0]!='n'));
		if (ans[0]=='y') {
			digprompt("Please wait ..");
			putitem(filename,&anitem,ancount,antab);
		}
	}

	/* that's all folks */
	if (canplay==2) dac_close(0);
	sfsclose(fid); 
	digquit(14); 
	exit(0);
}

void breakhandler(dummy)
int dummy;
{
	if (canplay==2) dac_close(0);
	digclearscreen();
	digquit(14);
	if (labelmode)
		error("Interrupt.  Annotations not saved.",NULL);
	else
		error("Interrupt.",NULL);
}

int getprompt(prompt,fname)
char	*prompt;
char	*fname;
{
	char	but;
	float	x,y;
	int	pxm,pym;	

	clear_top_line();
	digprompt(prompt);
	diginitmouse((float)0.5,(float)0.5);
	do {
		diggetmousep(fname,&but,&pxm,&pym);

		/* convert to real co-ordinates */
		x = (pxm - digdata.xorig) / digdata.xscale;
		y = (pym - digdata.yorig) / digdata.yscale;

	} while (*fname=='\0');
	digkillmouse();
	if ((*fname=='\n') || (*fname=='\r')) {
		*fname='\0';
		return(0);
	}
	else {
		if (fname[strlen(fname)-1]=='\n')
			fname[strlen(fname)-1] = '\0';
		if (fname[strlen(fname)-1]=='\r')
			fname[strlen(fname)-1] = '\0';
		return(1);
	}
}

void warning(messg)
char	*messg;
{
	clear_top_line();
	digprompt(messg);
	sleep(2);
	clear_top_line();
}

void clear_top_line()
{
	int	oldmode;
	oldmode = diglinemode(0);
	digfillbox(0,(float)0.0,(float)(1.0-tsize),(float)1.0,(float)1.1);
	digline(24,(float)0.0,(float)(1.0-tsize),(float)1.0,(float)(1.0-tsize));
	digflush();
	diglinemode(oldmode);
}

char *logname()
{
	static char *UNKNOWN="unknown";
	char *p,*getenv();

	if ((p=getenv("LOGNAME"))==NULL)
		return(UNKNOWN);
	else
		return(p);
}

void title_display()
{
	int	oldmode;
	oldmode = diglinemode(0);
	/* display filename, speaker and token */
	sprintf(title,"file=%s speaker=%s token=%s",filename,head.speaker,head.token);
	digtext(24,(float)0.0,(float)(1.0-2*tsize/3),title);
	if (doprint && printtitle[0]) {
		sprintf(title,"title=%s",printtitle);
		digtext(24,(float)1.0-digtextlen(title),(float)(1.0-2*tsize/3),title);
	}
	digflush();
	diglinemode(oldmode);
}

void displaymeasure(str)
char *str;
{
	digitemNUM(242420,(float)0.0,(float)0.0,(float)1.0,(float)tsize,NULL,str,starttime,endtime,3);
}

#ifdef __STDC__
double findvalue(float y)
#else
double findvalue(y)
float	y;
#endif
{
	int	i,d;

	/* determine item to replay */
	d = -1;
	for (i=0;i<numdisp;i++) if ((y>=reptab[i].yb) && (y<=reptab[i].yt)) {
		d=i;
		break;
	}
	if (d==-1) return(0.0);
	if (reptab[d].lo==reptab[d].hi) return(reptab[d].lo);
	return(reptab[d].lo + (y-reptab[d].yb)*(reptab[d].hi-reptab[d].lo)/(reptab[d].yt-reptab[d].yb));
}

void dograph()
{
	int	count=0,ent;
	float	yt;
	struct item_header item;
	double	oldoffset;
	int	oldnumframes;

	/* display filename, speaker and token */
	title_display();

	/* display top timescale */
	digitemtime(24,(float)0.0,(float)(1.0-2*tsize),(float)1.0,(float)(1.0-tsize),starttime,endtime,1);

	/* process data items */
	sfsnextitem(fid,NULL);	/* reposition */
	yt = (float)(1.0 - 2*tsize);
	while ((count < numdisp) && sfsnextitem(fid,&item)) {
		oldnumframes = item.numframes;
		oldoffset = item.offset;
		if (onitemlist(itemlist,itlistno,item.datatype,item.subtype)) {
			reptab[count].yt=yt;
			display(fid,&item,yt-tsize-sizes[item.datatype]*varfrac,yt,starttime,endtime);
			yt -= (float)(tsize + sizes[item.datatype]*varfrac);
			reptab[count].yb=yt;
			reptab[count].datatype=SP_TYPE;
			reptab[count].lo = digitab.lo;
			reptab[count].hi = digitab.hi;
			if (item.datatype==SY_TYPE) {
				reptab[count].datatype=SY_TYPE;
				sprintf(reptab[count].itemtype,"%d",item.subtype);
			}
			else if ((ent=findspeech(item.datatype,item.subtype)) >= 0)
				sprintf(reptab[count].itemtype,"%d",tree[ent].subtype);
			else
				reptab[count].itemtype[0]='\0';
			if (reptab[count].buff) {
				if (!reptab[count].link) free(reptab[count].buff);
				reptab[count].buff=NULL;
				reptab[count].link=0;
			}
			count++;
		}
		if ((item.datatype==SP_TYPE) &&
		     onitemlist(citemlist,citlistno,item.datatype,item.subtype)) {
			reptab[count].yt=yt;
			item.numframes = oldnumframes;	/* re-read data - rubbish I know .. */
			item.offset = oldoffset;
			displayFFT(fid,&item,yt-tsize-sizes[DI_TYPE]*varfrac,yt,starttime,endtime,0);
			yt -= (float)(tsize + sizes[DI_TYPE]*varfrac);
			reptab[count].yb=yt;
			reptab[count].datatype=SP_TYPE;
			reptab[count].lo = digitab.lo;
			reptab[count].hi = digitab.hi;
			if ((ent=findspeech(item.datatype,item.subtype)) >= 0)
				sprintf(reptab[count].itemtype,"%d",tree[ent].subtype);
			else
				reptab[count].itemtype[0]='\0';
			if (reptab[count].buff) {
				if (!reptab[count].link) free(reptab[count].buff);
				reptab[count].buff=NULL;
				reptab[count].link=0;
			}
			count++;
		}
		if ((item.datatype==SP_TYPE) &&
		     onitemlist(nitemlist,nitlistno,item.datatype,item.subtype)) {
			reptab[count].yt=yt;
			item.numframes = oldnumframes;	/* re-read data - rubbish I know .. */
			item.offset = oldoffset;
			displayFFT(fid,&item,yt-tsize-sizes[DI_TYPE]*varfrac,yt,starttime,endtime,1);
			yt -= (float)(tsize + sizes[DI_TYPE]*varfrac);
			reptab[count].yb=yt;
			reptab[count].datatype=SP_TYPE;
			reptab[count].lo = digitab.lo;
			reptab[count].hi = digitab.hi;
			if ((ent=findspeech(item.datatype,item.subtype)) >= 0)
				sprintf(reptab[count].itemtype,"%d",tree[ent].subtype);
			else
				reptab[count].itemtype[0]='\0';
			if (reptab[count].buff) {
				if (!reptab[count].link) free(reptab[count].buff);
				reptab[count].buff=NULL;
				reptab[count].link=0;
			}
			count++;
		}
	}

	/* calculate annotation granularity */
	angran = (int)(4.0*((endtime-starttime)/(digitab.ixr-digitab.ixl-digitab.ixoff))/ANDUR);

	/* display annotation edit window */
	if (labelmode) {
		dispan((float)tsize,(float)(3*tsize),starttime,endtime,1);
	}

	if (domeasure) {
		/* put up measurement status box */
		displaymeasure("Measurement Box");
	}
	else {
		/* put up second timescale */
		digitemtime(24,(float)0.0,(float)0.0,(float)1.0,(float)tsize,starttime,endtime,1);
	}


}

/* display an item */
void display(fid,item,yb,yt,t1,t2)
int		fid;
struct item_header *item;
double		yb,yt;
double		t1,t2;
{
	char	*buff;
	int	flags;

	/* check if axes need fixing */
	if (fixtab[item->datatype].fix) {
		digitab.hi=fixtab[item->datatype].hi;
		digitab.lo=fixtab[item->datatype].lo;
		flags=DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL | DIGITEMFIX;
	}
	else
		flags=DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL;
	digitab.title=NULL;

	/* load data from file */
	getdata(fid,item,t1,t2,&buff);
	if (buff==NULL)
		error("could not get buffer for item",NULL);

	/* display it */
	switch (item->datatype) {
	case SP_TYPE:
			digitemSP(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(short *)buff,t1,t2,flags);
			break;
	case LX_TYPE:
			digitemLX(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(short *)buff,t1,t2,flags);
			break;
	case TX_TYPE:
			digitemTX(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(int32 *)buff,t1,t2,flags);
			break;
	case FX_TYPE:
			digitemFX(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(short *)buff,t1,t2,flags);
			break;
	case AN_TYPE:
			digitemAN(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(struct an_rec *)buff,t1,t2,flags);
			break;
	case SY_TYPE:
			digitemSY(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(short *)buff,t1,t2,flags);
			break;
	case WD_TYPE:
			digitemWD(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(struct wd_rec *)buff,t1,t2,flags);
			break;
	case DI_TYPE:
			digitemDI(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(struct di_rec *)buff,t1,t2,flags);
			break;
	case CO_TYPE:
			digitemCO(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(struct co_rec *)buff,t1,t2,flags);
			break;
	case VU_TYPE:
			digitemVU(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(struct vu_rec *)buff,t1,t2,flags);
			break;
	case FM_TYPE:
			digitemFM(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(struct fm_rec *)buff,t1,t2,flags);
			break;
	case TR_TYPE:
			digitemTR(242405,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(float *)buff,t1,t2,flags | DIGITEMTOPHAT);
			break;
	case UG_TYPE:
			digitemUG(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(short *)buff,t1,t2,flags);
			break;
	default:
			digitemNULL(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(void *)buff,t1,t2,flags);
			break;
	}

	/* free data */
	free(buff);
}

/* display a speech item with FFT */
void displayFFT(fid,item,yb,yt,t1,t2,narrow)
int		fid;
struct item_header *item;
double		yb,yt;
double		t1,t2;
int		narrow;
{
	char	*buff;
	int	flags;

	/* check if axes need fixing */
	if (fixtab[item->datatype].fix) {
		digitab.hi=fixtab[item->datatype].hi;
		digitab.lo=fixtab[item->datatype].lo;
		flags=DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL | DIGITEMFIX;
	}
	else
		flags=DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL;
	if (narrow) flags |= DIGITEMFFTNARROW;
	digitab.title=NULL;

	/* load data from file */
	getdata(fid,item,t1,t2,&buff);
	if (buff==NULL)
		error("could not get buffer for item",NULL);

	/* display it */
	digitemFFT(242420,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(short *)buff,t1,t2,flags);

	/* free data */
	free(buff);
}

/* convert item selections into item numbers */
int esitem(fid,itab,itcnt,itemlist)
int	fid;
struct itemsel_rec *itab;
int		itcnt;
struct itemno_rec *itemlist;
{
	struct item_header item;
	char		*exphist;
	int		cnt=0;

	sfsnextitem(fid,NULL);
	while (sfsnextitem(fid,&item)) {
		exphist = lab_store(&item);
		itrequired(itab,itcnt,&item,exphist,0);
	}
	sfsnextitem(fid,NULL);
	while (sfsnextitem(fid,&item)) {
		exphist = lab_store(&item);
		if (((itcnt==0) && (item.datatype > 0)) || itrequired(itab,itcnt,&item,exphist,1)) {
			itemlist[cnt].datatype = item.datatype;
			itemlist[cnt].subtype = item.subtype;
			cnt++;
		}
	}
	return(cnt);
}

/* check if item required */
int itrequired(itab,itcnt,item,hist,flag)
struct itemsel_rec	*itab;
int			itcnt;
struct item_header 	*item;
char			*hist;
int			flag;
{
	int	i,ty,req=0;

	for (i=0;i<itcnt;i++) if ((itab[i].it==0) || (itab[i].it==item->datatype)) {
		if (itab[i].ty==item->subtype)
			req++;
		else if (!itab[i].match)
			continue;
		else if (isdigit(itab[i].match[0]) || (itab[i].match[0]=='-')) {
			ty=atoi(itab[i].match);
			if ((ty==-1) || (ty==item->subtype)) {
				if (itab[i].all==0) itab[i].ty=item->subtype;
				req++;
			}
			else if ((ty==0) && (flag==0)) {
				itab[i].it=abs(item->datatype);
				itab[i].ty=item->subtype;
			}
		}
		else if (itab[i].ty==0) {
			if (histmatch(itab[i].match,hist,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL)==1) {
				if (itab[i].all==0) itab[i].ty=item->subtype;
				req++;
			}
		}
	}
	return(req);
}

/* check if item on list */
int onitemlist(list,cnt,it,ty)
struct itemno_rec *list;
int	cnt;
int	it;
int	ty;
{
	int	i;
	for (i=0;i<cnt;i++)
		if ((list[i].datatype==it) &&
		    (list[i].subtype==ty)) {
		    	return(1);
		    }
	return(0);
}

/* load data from file */
void getdata(fid,item,t1,t2,buff)
int		fid;
struct item_header *item;
double		t1,t2;
char		**buff;
{
	int	start,stop,len;
	char	itemno[12];
	double	fdur;

	sprintf(itemno,"%d.%02d",item->datatype,item->subtype);

	if ((item->lxsync==0) && (item->datatype != TX_TYPE) && (item->datatype != AN_TYPE)) {
		if (sfsstruct[item->datatype]==0)
			/* unstructured item */
			fdur = item->frameduration;
		else
			/* fixed window structured */
			fdur = (item->windowsize-item->overlap)*item->frameduration;

		/* load part we need +/- 10ms for safety */
		start = (int)((t1-item->offset-0.01)/fdur);
		stop = (int)((t2-item->offset+0.01)/fdur);
		if (start < 0) start=0;
		if (stop < 0) stop=0;
		if (start > item->numframes) start=item->numframes;
		if (stop > item->numframes) stop=item->numframes;
		len = stop - start;
		if (len > 0) {
			if ((*buff=sfsbuffer(item,len))==NULL)
				error("could not get buffer for item '%s'",itemno);
			if (sfsread(fid,start,len,*buff) != len)
				error("read error on input file",NULL);
			if (sfsstruct[item->datatype]==0)
				item->offset += start*item->frameduration;
			item->numframes = len;
		}
		else {
			/* off end of data set */
			if ((*buff=sfsbuffer(item,1))==NULL)
				error("could not get buffer for item '%s'",itemno);
			if (sfsstruct[item->datatype]==0)
				item->offset += start*item->frameduration;
			item->numframes = 0;
		}
	}
	else {
		/* variable frame structured item or TX */
		/* -- load all of it */
		if ((*buff=sfsbuffer(item,item->numframes))==NULL)
			error("could not get buffer for item '%s'",itemno);
		if (sfsread(fid,0,item->numframes,*buff) != item->numframes)
			error("read error on input file",NULL);
	}
}

int findspeech(it,ty)
int	it,ty;
{
	int	ent=0;
	while ((ent < treelen) && ((tree[ent].datatype!=it) || (tree[ent].subtype!=ty))) ent++;
	return(findsp(ent));
}

int findsp(ent)
int	ent;
{
	int	i,ent2;
	if (tree[ent].datatype==SP_TYPE)
		return(ent);
	else for (i=0;i<tree[ent].numlink;i++) {
		ent2=findsp(tree[ent].link[i]);
		if (ent2 >= 0) return(ent2);
	}
	return(-1);
}

#ifdef __STDC__
void repspeech(float y,double s,double e,double t1,double t2)
#else
void repspeech(y,s,e,t1,t2)
float	y;
double	s,e;
double	t1,t2;
#endif
{
	int	i,d,bstart,numf;
	struct item_header item;
	short		*sytemp=NULL;
	static int	dacopen=0;

	if (canplay && !dacopen) {
		canplay = (dac_open(NULL) < 0) ? 0 : 2;
		dacopen=1;
	}

	/* determine item to replay */
	d = -1;
	for (i=0;i<numdisp;i++) if ((y>=reptab[i].yb) && (y<=reptab[i].yt)) {
		d=i;
		break;
	}
	if ((d==-1) || (reptab[d].itemtype[0]=='\0')) {
		fprintf(stderr,"\007");
		fflush(stderr);
		return;
	}

	/* load data into replay buffer */
	if (reptab[d].buff==NULL) {

		if (reptab[d].datatype==SP_TYPE) {

			/* locate item */
			sfsitem(fid,SP_TYPE,reptab[d].itemtype,&item);

			/* load it */
			getdata(fid,&item,s,e,(char **)&reptab[d].buff);

			/* init record */
			reptab[d].numframes = item.numframes;
			reptab[d].sfreq = 1.0/item.frameduration;
			reptab[d].offset = item.offset;
			reptab[d].nbits = (int)param(item.params,"bits",(scale12)?12.0:16.0);
	
			/* no need to load other copies of this data */
			for (i=0;i<numdisp;i++) {
				if ((i!=d) && (reptab[i].datatype==SP_TYPE) && (strcmp(reptab[i].itemtype,reptab[d].itemtype)==0)) {
					/* link other dispays to this buffer */
					reptab[i].buff   = reptab[d].buff;
					reptab[i].numframes = reptab[d].numframes;
					reptab[i].sfreq  = reptab[d].sfreq;
					reptab[i].offset = reptab[d].offset;
					reptab[i].nbits = reptab[d].nbits;
					reptab[i].link   = 1;
				}
			}
		}
		else {
			/* load SY data */
			sfsitem(fid,SY_TYPE,reptab[d].itemtype,&item);

			/* load it */
			getdata(fid,&item,s,e,(char **)&sytemp);

			/* get replay buffer */
			if ((reptab[d].buff=(short *)calloc(item.numframes+1,32))==NULL)
				error("could not get SY buffer",NULL);

			/* encode SY data */
			hwsynthcode(sytemp,(unsigned short *)reptab[d].buff);
			for (i=0;i<item.numframes;i++)
				hwsynthcode(sytemp+i*item.framesize,
					    (unsigned short *)reptab[d].buff+16*i+16);

			/* free up old SY data */
			free(sytemp);

			/* store buffer data */
			reptab[d].numframes = item.numframes;
			reptab[d].sfreq = 1.0/item.frameduration;
			reptab[d].offset = item.offset;
		}
	}

	/* get portion of buffer to replay */
	bstart = (int)((t1-reptab[d].offset)*reptab[d].sfreq);
	numf = (int)((t2-t1)*reptab[d].sfreq);

	if (bstart < 0) {
		numf += bstart;
		bstart=0;
	}
	if ((bstart+numf) > reptab[d].numframes) numf=reptab[d].numframes-bstart;
	if (numf > 0) {
		if (reptab[d].datatype==SP_TYPE) {
			dac_playback(&reptab[d].buff[bstart],
				numf,
				reptab[d].sfreq,reptab[d].nbits,1,1);
		}
		else
			hwsynth(&reptab[d].buff[bstart*16],numf+1);
	}

	return;
}

/* invert box on signal being replayed */
#ifdef __STDC__
void invbox(float y,float x1,float x2)
#else
void invbox(y,x1,x2)
float	y,x1,x2;
#endif
{
	int	i,d;

	/* find item position */
	d = -1;
	for (i=0;i<numdisp;i++) if ((y>=reptab[i].yb) && (y<=reptab[i].yt)) {
		d=i;
		break;
	}
	if ((d==-1) || (reptab[d].itemtype[0]=='\0')) {
		return;
	}

	/* invert box */
	digfillbox(24,x1,reptab[d].yb,x2,reptab[d].yt);
	digflush();
}

/* annotation control routines */

/* preloadan -- load up internal annotations buffer */
void preloadan(fid,item)
int	fid;
struct item_header	*item;
{
	int	i;

	/* read input data set */
	for (i=0;sfsread(fid,i,1,antemp);i++) {
		antab[i].posn=(int)(0.5+(antemp->posn*item->frameduration+item->offset)/ANDUR);
		antab[i].size=(int)(0.5+(antemp->size*item->frameduration)/ANDUR);
		antab[i].label=(char *)malloc(strlen(antemp->label)+1);
		strcpy(antab[i].label,antemp->label);
	}
	ancount=i;

	/* sort annotations */
	sortan(1);
}

/* sort annotation table */
void sortan(full)
int	full;	/* full sort */
{
	register int	i,j;

	/* pass 1 - sort by starttime */
	if (full || (ancount < 2))
		i=1;
	else
		i=ancount-1;
	for (;i<ancount;i++) {
		*antemp = antab[i];
		j=i;
		while ((j>0) && (antemp->posn < antab[j-1].posn)) {
			antab[j]=antab[j-1];
			j--;
		}
		antab[j] = *antemp;
	}

	/* check for duplicates and empty annotations */
	if (ancount > 0) {
		for (i=0,j=0;j<ancount;j++) {
			if (j==ancount-1) {		
				/* last one */
				if (antab[j].label[0]!='\0') {
					antab[i]=antab[j];
					i++;
				}
			}
			else if (antab[j].posn == antab[j+1].posn) {
				/* delete earlier duplicate */
				free(antab[j].label);
				anchange++;
			}
			else if (antab[j].label[0]!='\0') {
				antab[i]=antab[j];
				i++;
			}
		}
		ancount=i;
	}

	/* set up 'size' fields */
	if (full) {
		for (i=0;i<ancount-1;i++) 
			antab[i].size = antab[i+1].posn-antab[i].posn;
	}
}

/* display contents of annotation buffer */
#ifdef __STDC__
void dispan(float yb,float yt,double starttime,double endtime,int redo)
#else
void dispan(yb,yt,starttime,endtime,redo)
float		yb,yt;		/* box location */
double		starttime,endtime;	/* box timing */
int		redo;		/* clear box before draw */
#endif
{
	int	flags;

	if (redo)
		flags = DIGITEMBOX | DIGITEMTITLE | DIGITEMGAP;
	else
		flags = DIGITEMBOX | DIGITEMTITLE | DIGITEMOVERLAY | DIGITEMGAP;

	digitab.title=labtitle;
	anitem.numframes=ancount;
	digitemAN(212122,(float)0.0,(float)yb,(float)1.0,(float)yt,&anitem,antab,starttime,endtime,flags);
	digflush();
}

/* add an annotation to the database */
void addan(name,atime,stime,etime)
char	*name;
double	atime,stime,etime;
{
	int	posn;
	int	i,owrite = -1;

	/* get location of annotation */
	posn = (int)(0.5 + atime/ANDUR);

	/* check against existing table */
	for (i=0;i<ancount;i++) {
		if (abs(antab[i].posn-posn) <= angran) {
			owrite=i;
			break;
		}
	}

	/* if overwriting, replace and rewrite screen */
	if (owrite >= 0) {
		antab[owrite].posn = posn;
		antab[owrite].size = (int)(0.5+etime/ANDUR) - posn;
		antab[owrite].label = (char *)realloc(antab[owrite].label,strlen(name)+1);
		strcpy(antab[owrite].label,name);
		anchange++;
		sortan(1);
		dispan((float)tsize,(float)(3*tsize),stime,etime,1);
	}
	else {
		/* add annotation and update screen */
		antab[ancount].posn = posn;
		antab[ancount].size=(int)(0.5+etime/ANDUR) - posn;
		antab[ancount].label=(char *)malloc(strlen(name)+1);
		strcpy(antab[ancount].label,name);
		if (ancount < MAXANNOT) ancount++;
		anchange++;
		sortan(0);
		dispan((float)tsize,(float)(3*tsize),stime,etime,0);
	}
}

/* select an annotation for moving */
int selan(atime,stime,etime)
double	atime,stime,etime;
{
	int	posn;
	int	i,entry = -1;

	/* get location of annotation */
	posn = (int)(0.5 + atime/ANDUR);

	/* check against existing table */
	for (i=0;i<ancount;i++) {
		if (abs(antab[i].posn-posn) <= angran) {
			entry=i;
			break;
		}
	}
	return(entry);
}

void movan(entry,atime,stime,etime)
int	entry;
double	atime,stime,etime;
{
	int	posn;

	/* get location of annotation */
	posn = (int)(0.5 + atime/ANDUR);

	/* move annotation and update screen */
	antab[entry].posn = posn;
	antab[entry].size=(int)(0.5+etime/ANDUR) - posn;
	anchange++;
	sortan(1);
	dispan((float)tsize,(float)(3*tsize),stime,etime,1);
}
