/* estree -- DIG graphic representation of file structure */

/* M.A.Huckvale - University College London */

/* version 1.0 - August 1993
	- from old 'tree.c' code
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH ESTREE SFS3 UCL SFS
.SH NAME
estree - display/interact with file structure tree
.SH SYNOPSIS
.nf
int estree(filename,itemlist,curitems,flags)
char	*filename;
struct itemno_rec *itemlist;
int	curitems;
int	flags;
.fi
.SH DESCRIPTION
.I estree
is a routine to display the internal file structure of an SFS file
as a graphic structure using DIG graphics.  DIG graphics must have been
opened prior to the call, but the mouse must not be initialised.
the routine accepts a number of item numbers as input and returns a list
of selected items.  The return value is the number of selected items
if zero or positive, or -1 if aborted.  If the flags parameter is set
to 1 then the picture is produced without interaction.
.SH MOUSE BUTTONS
.TP 11
.B left
Select. Select/Deselect items. In top region of display requests re-draw of screen.
.SH MOUSE BOXES
.TP 11
.B exit
Return with current selections.  Keys: q,x,[RETURN].
.TP 11
.B hard
Send current graphic to printer.  Keys: h.
.TP 11
.B all
Select all items.
.TP 11
.B zero
Clear all selections.
.SH GRAPHICS
Each item is described using the item number and the first 8 characters of
the program name in the item history.  Deleted items have a cross-bar.  Selected
items are displayed in inverse video.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include <stdio.h>
#include <ctype.h>
#include <fcntl.h>
#include <math.h>
#include "sfs.h"
#include "sfstree.h"
#include "dig.h"
#include "digdata.h"
#include "es.h"

/* mouse commands */
#define COM_NULL	-1
#define COM_QUIT	0	/* index into menu_names[] */
#define	COM_HARD	1
#define	COM_ALL		2
#define COM_ZERO	3
#define COM_SELECT	4	/* select box */
#define COM_REDRAW	5	/* redraw screen */

/* mouse menu */
#define MENUSIZE	4
static char	*menu_names[MENUSIZE]={"OK","hard","all","zero"};
static char	*menu_keys[MENUSIZE]={"XxqQOo\r\n","hHpP","aAeE","ZzCc"};

/* display structure */
#define ITXMAX	10
#define ITYMAX	10
static struct dtab_rec {
	int	entry;
	float	x;
	float	y;
	float	cy;		/* calculated y co-ordinate */
	int	ncy;		/* number in cy summation */
	int	select;
} dtab[ITXMAX][ITYMAX];
static int	dcnt[ITXMAX];
static int	xent[MAXTREELEN];
static int	yent[MAXTREELEN];
static int	dlevel=0;
static float	bheight,bwidth,cheight,cwidth;	/* graphics constants */
static int	plen;
static float	xpos=(float)0.5,ypos=(float)0.5;			/* stored mouse x,y */
static int	button;

/* global data */
static int	interact=1;			/* interactive */
extern char	*sfslabel[];
static struct main_header head;

/* prototypes */
#ifdef __STDC__
static void treedisplay(char *);
static void drawbox(float,float,int,int,char *);
static int sortlevel(int);
static void drawprompt(char *);
static void selectbox(int,int);
static void initselect(struct itemno_rec *,int);
static int getmousecom(int *,int *);
static void buttonup(void);
#else
static void treedisplay();
static void drawbox();
static int sortlevel();
static void drawprompt();
static void selectbox();
static void initselect();
static int getmousecom();
static void buttonup();
#endif
/* display file structure */
int	estree(filename,itemlist,itemcount,flags)
char	*filename;
struct itemno_rec *itemlist;
int	itemcount;
int	flags;
{
	int		i,j,k;
	int		entry;
	int		fid;
	int		com,row,col;

	if (flags) interact=0;

        /* open datafile to get header */
        if ((fid = sfsopen(filename,"r",&head)) < 0) {
                if (fid == -1)
                        error("cannot open %s",filename);
                else
                        error("access error on %s",filename);
        }
	sfsclose(fid);

        /* get tree */
        if (!sfstree(filename))
		error("cannot open %s",filename);

	/* set up scaling */
	digclearscreen();
	digscale((float)1.0,(float)(1.0+3.5*digdata.chheight/digdata.yscale),0);
	digorigin((float)0.0,(float)(1.5*digdata.chheight/digdata.yscale));

	/* clear data structures */
	memset((char *)dtab,0,sizeof(dtab));
	memset((char *)dcnt,0,sizeof(dcnt));
	memset((char *)xent,0,sizeof(xent));
	memset((char *)yent,0,sizeof(yent));
	dlevel = 0;

	/* display tree */
	treedisplay(filename);

	/* interactive mode */
	if (interact) {
		/* make initial selections */
		initselect(itemlist,itemcount);

		/* display mouse boxes */
		esmenu_display(28,menu_names,menu_keys,MENUSIZE);
		
		/* start up mouse */
		drawprompt("Select box with mouse");
		diginitmouse(xpos,ypos);

		/* process commands */
		while ((com=getmousecom(&col,&row)) != COM_QUIT) switch (com) {
		case COM_HARD:
			drawprompt("Sorry, printing is not yet supported!");
			break;
		case COM_ALL:
			for (i=0;i<dlevel;i++)
				for (j=0;j<dcnt[i];j++)
					if (!dtab[i][j].select)
						selectbox(i,j);
			break;
		case COM_ZERO:
			for (i=0;i<dlevel;i++)
				for (j=0;j<dcnt[i];j++)
					if (dtab[i][j].select)
						selectbox(i,j);
			break;
		case COM_SELECT:
			selectbox(col,row);
			drawprompt(tree[dtab[col][row].entry].history);
			break;
		case COM_REDRAW:
			digkillmouse();
			treedisplay(filename);
			esmenu_display(16,menu_names,menu_keys,MENUSIZE);
			diginitmouse(xpos,ypos);
		default:
			break;
		}
		buttonup();
		digkillmouse();
	}

	/* return selections */
	k=0;
	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) if (dtab[i][j].select) {
			entry = dtab[i][j].entry;
			itemlist[k].datatype = tree[entry].datatype;
			itemlist[k].subtype = tree[entry].subtype;
			if (k < MAXITEM-1) k++;
		}
	}

	/* that's all folks */
	return(k);
}

/* display item relation tree */
static void treedisplay(filename)
char	*filename;
{
	int	i,j,k,nassign=0;
	int	entry,lentry;
	float	gap,minygap,ytop;
	char	title[256];
	int	dcntmax=0;
	int	rep;
	float	chg;

	/* assign items to level in tree */
	dlevel=0;
	for (i=0;(dlevel<ITXMAX) && (nassign<treelen);i++) {
		dcnt[dlevel]=0;
		for (j=0;j<treelen;j++) {
			if (tree[j].level==i) {
				dtab[dlevel][dcnt[dlevel]].entry=j;
				dtab[dlevel][dcnt[dlevel]].select=0;
				xent[j]=dlevel;
				yent[j]=dcnt[dlevel];
				dcnt[dlevel]++;
				nassign++;
				if (dcnt[dlevel] == ITYMAX) {
					if (dlevel < ITXMAX) {
						dlevel++;
						dcnt[dlevel]=0;
					}
					else
						break;
				}
			}
		}
		dlevel++;
	}

	/* allow space for title */
	cheight = (float)(digdata.chheight/digdata.yscale);
	cwidth  = (float)(digdata.chwidth/digdata.xscale);
	bheight = (float)(1.25 * cheight);
	bwidth  = (float)(4.5 * cwidth);
	gap = (float)((1.0-2*dlevel*bwidth)/(dlevel+1));
	if (gap < 0.0) gap=(float)0.0;
	plen = (int)(1.0/cwidth);	/* max char on line */

	/* plot title */
	digclearscreen();
	sprintf(title,"file=%s speaker=%s token=%s",filename,head.speaker,head.token);
	title[plen]='\0';
	digprompt(title);
	digbox(20,(float)0.0,(float)0.0,(float)1.0,(float)1.0);
	digclippush();
	digclip((float)0.0,(float)0.0,(float)1.0,(float)1.0);

	/* get max dcnt */
	for (i=0;i<dlevel;i++)
		if (dcnt[i] > dcntmax) dcntmax = dcnt[i];
	minygap = (float)(1.0/((float)dcntmax+1.0));

	/* calculate x,y co-ordinates */
	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) {
			dtab[i][j].x = (float)((gap+bwidth) + i*(gap + 2*bwidth));
			dtab[i][j].y = (float)(1.0 - ((float)(j+1))/(dcnt[i]+1));
		}
	}

	/* relax to best positions */
	chg=(float)1.0;
	for (rep=0;(rep<30) && (chg>0.001);rep++) {
		chg = (float)0.0;

		/* 1. process left to right */
		for (i=0;i<dlevel;i++) {

			/* relate to previous nodes */
			for (j=0;j<dcnt[i];j++) {
				dtab[i][j].cy = (float)0.0;
				dtab[i][j].ncy = 0;
				entry = dtab[i][j].entry;
				for (k=0;k<tree[entry].numlink;k++) {
					lentry = tree[entry].link[k];
					/* average backward links */
					dtab[i][j].cy += dtab[xent[lentry]][yent[lentry]].cy;
					dtab[i][j].ncy++;
				}
				if (dtab[i][j].ncy==0)
					dtab[i][j].cy = dtab[i][j].y;
				else
					dtab[i][j].cy /= dtab[i][j].ncy;
			}

			/* sort on calculated y co-ordinates */
			chg += (float)sortlevel(i);

			/* push apart */
			ytop = (float)(1.0 - minygap);
			for (j=0;j<dcnt[i];j++) {
				if (dtab[i][j].cy > ytop)
					dtab[i][j].cy = ytop;
				ytop = dtab[i][j].cy - minygap;
			}
			ytop = minygap;
			for (j=dcnt[i]-1;j>=0;j--) {
				if (dtab[i][j].cy < ytop)
					dtab[i][j].cy = ytop;
				ytop = dtab[i][j].cy + minygap;
			}
		}

		/* calculate new co-ordinates */
		for (i=0;i<dlevel;i++) {
			/* reassign positions */
			for (j=0;j<dcnt[i];j++) {
				chg += (float)(fabs(dtab[i][j].y-dtab[i][j].cy));
				dtab[i][j].y = dtab[i][j].cy;
			}
		}

		/* 2. process right to left */
		for (i=dlevel-1;i>=0;i--) {

			/* relate to future nodes */
			for (j=0;j<dcnt[i];j++) {
				dtab[i][j].cy = (float)0.0;
				dtab[i][j].ncy = 0;
				for (entry=0;entry<treelen;entry++) {
					for (k=0;k<tree[entry].numlink;k++) {
						if (tree[entry].link[k]==dtab[i][j].entry) {
							/* average forward links */
							dtab[i][j].cy += dtab[xent[entry]][yent[entry]].cy;
							dtab[i][j].ncy++;
						}
					}
				}
				if (dtab[i][j].ncy==0)
					dtab[i][j].cy = dtab[i][j].y;
				else
					dtab[i][j].cy /= dtab[i][j].ncy;
			}

			/* sort on calculated y co-ordinates */
			chg += (float)sortlevel(i);

			/* push apart */
			ytop = (float)(1.0 - minygap);
			for (j=0;j<dcnt[i];j++) {
				if (dtab[i][j].cy > ytop)
					dtab[i][j].cy = ytop;
				ytop = dtab[i][j].cy - minygap;
			}
			ytop = minygap;
			for (j=dcnt[i]-1;j>=0;j--) {
				if (dtab[i][j].cy < ytop)
					dtab[i][j].cy = ytop;
				ytop = dtab[i][j].cy + minygap;
			}
		}

		/* calculate new co-ordinates */
		for (i=0;i<dlevel;i++) {
			/* reassign positions */
			for (j=0;j<dcnt[i];j++) {
				chg += (float)(fabs(dtab[i][j].y-dtab[i][j].cy));
				dtab[i][j].y = dtab[i][j].cy;
			}
		}

#ifdef EMO
		sprintf(messg,"%2d) chg=%g",rep,chg);
		drawprompt(messg);
		sleep(1);
#endif
	}

	/* plot lines */
	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) {
			entry = dtab[i][j].entry;
			for (k=0;k<tree[entry].numlink;k++) {
				lentry = tree[entry].link[k];
				digline(27,dtab[xent[lentry]][yent[lentry]].x+bwidth,
					   dtab[xent[lentry]][yent[lentry]].y,
					   dtab[i][j].x-bwidth,dtab[i][j].y);
			}
		}
	}

	/* plot itemnos */
	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) {
			entry = dtab[i][j].entry;
			drawbox(dtab[i][j].x,dtab[i][j].y,
				tree[entry].datatype,
				tree[entry].subtype,
				tree[entry].history);
		}
	}

	digflush();
	digclippop();
}

/* sort boxes within a level by y co-ord */
static int sortlevel(level)
int	level;
{
	int	i,j,entry,chg=0;
	struct dtab_rec dhold;

	/* insertion sort */
	for (i=1;i<dcnt[level];i++) {
		j = i;
		dhold = dtab[level][j];
		while ((j > 0) && (dtab[level][j-1].cy < dhold.cy)) {
			dtab[level][j] = dtab[level][j-1];
			chg++;
			j--;
		}
		dtab[level][j] = dhold;
	}
	/* reassign links */
	for (i=0;i<dcnt[level];i++) {
		entry = dtab[level][i].entry;
		yent[entry] = i;
	}
	return(chg);
}

/* draw item box */
#ifdef __STDC__
static void drawbox(float x,float y,int it,int ty,char *hist)
#else
static void drawbox(x,y,it,ty,hist)
float	x,y;
int	it,ty;
char	*hist;
#endif
{
	char	iname[8];
	char	ilabel[32];
	int	llen;
	float	tlen;
	int	i;

	/* get text for box */
	sprintf(iname,"%s.%02d",sfslabel[abs(it)],ty);
	llen = strcspn(hist," (/");
	for (i=0;(i<llen) && (i<8);i++) ilabel[i]=hist[i];
	ilabel[i]='\0';
	tlen = digtextlen(ilabel);

	/* clear box */
	digflush();
	digfillbox(0,x-bwidth,y-bheight,x+bwidth,y+bheight);

	/* write outline if undeleted item */
	if (it > 0) {
		digbox(31,x-bwidth,y-bheight,x+bwidth,y+bheight);
	}
	else {
		digbox(29,x-bwidth,y-bheight,x+bwidth,y+bheight);
		digline(29,x-bwidth,y+bheight,x+bwidth,y-bheight);
	}

	/* write item number */
	digtext(24,(float)(x-2.0*cwidth),(float)y,iname);

	/* write item label */
	digtext(24,(float)(x-tlen/2),(float)(y-1.125*cheight),ilabel);

}

/* select/deselect box */
static void selectbox(col,row)
int	col,row;
{
	if (tree[dtab[col][row].entry].datatype > 0) {
		dtab[col][row].select = !dtab[col][row].select;
		diglinemode(1);
		digfillbox(24,dtab[col][row].x-bwidth,
				dtab[col][row].y-bheight,
				dtab[col][row].x+bwidth,
				dtab[col][row].y+bheight);
		diglinemode(0);
		digflush();
	}
}

/* make initial selections */
static void initselect(itemlist,count)
struct itemno_rec	*itemlist;
int	count;
{
	int	i,j,k;
	int	entry;

	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) {
			entry = dtab[i][j].entry;
			for (k=0;k<count;k++) {
				if ((itemlist[k].datatype==tree[entry].datatype) &&
				    (itemlist[k].subtype==tree[entry].subtype))
					selectbox(i,j);
			}
		}
	}
}

/* display prompt */
static void drawprompt(prompt)
char	*prompt;
{
	static char	lbuf[256];
	strncpy(lbuf,prompt,digdata.nhoriz/digdata.chwidth);
	digfillboxp(0,0,0,digdata.nhoriz-1,digdata.chheight);
	digtextp(24,0,0,lbuf);
}

static int getmousecom(col,row)
int	*row,*col;
{
	int	ch;
	int	i,j;
	int	com;

	esmenu_redisplay();

	/* get mouse command */
	do {
		ch=diggetmousech(&button,&xpos,&ypos);
		if (ch==DIG_REDRAW)
			return(COM_REDRAW);
	} while ((ch <= 0) && (button==0));

	/* see if mouse in box */
	if (button) {
		*col = -1;
		*row = -1;
		for (i=0;i<dlevel;i++) {
			for (j=0;j<dcnt[i];j++) {
				if ((xpos > dtab[i][j].x-bwidth) &&
				    (xpos < dtab[i][j].x+bwidth) &&
				    (ypos > dtab[i][j].y-bheight) &&
				    (ypos < dtab[i][j].y+bheight)) {
					*row=j;
					*col=i;
					return(COM_SELECT);
				}
			}
		}
	}

	/* see if mouse in menu */
	if (((ch > 0) || button) && ((com=esmenu_check(xpos,ypos,ch)) >= 0)) {
		return(com);
	}
	else
		return(COM_NULL);

}

/* wait for mouse button up */
static void buttonup()
{
	int	ch;

	while (button) {
		ch=diggetmousech(&button,&xpos,&ypos);
	}
}

#ifdef EMO
char *progname="estree";

main(argc,argv)
int	argc;
char	*argv[];
{
	int	i,num;
	struct itemno_rec itab[100];
	int	icnt=0;
	int	c;
	extern int optind;
	extern char *optarg;
	int	dtype;
	char	*stype;

	while ((c=getopt(argc,argv,"i:"))!=EOF) switch (c) {
	case 'i':
		if (itspec(optarg,&dtype,&stype)==0) {
			itab[icnt].datatype = dtype;
			itab[icnt].subtype = atoi(stype);
			icnt++;
		}
	}
	
	if (optind < argc) {
		digstart(DIG_DEFAULT_TERM,NULL,1);
		num=estree(argv[optind],itab,icnt,0);
		num=estree(argv[optind],itab,num,0);
		digend();
		fprintf(stderr,"%d selected items:\n",num);
		for (i=0;i<num;i++)
			fprintf(stderr,"%2d) %2d.%02d\n",
				i,itab[i].datatype,itab[i].subtype);
	}
	exit(0);
}
#endif
