/* fmload -- load FM item from formant listing produced by fmlist */

/* m.a.huckvale - january 1987 */

/* version 1.1s - November 1987
	- SFS version
*/
/* version 1.2 - March 1990
	- version that actually works
*/
/* version 1.3 - June 2002
	- option to load from ESPS format (formant + pplain)
*/

#define PROGNAME "fmload"
#define PROGVERS "1.3"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH FMLOAD SFS1 UCL
.SH NAME
fmload - load raw formant estimates item from text listing
.SH SYNOPSIS
.B fmload
(-f numformants) (-r rate) (-e) listfile sfsfile
.SH DESCRIPTION
.I fmload
creates a new FM item in the file
.I dbfile
from formant specifications listed in the text file
.I listfile.
Default input format is that written by
.I fmlist,
namely formant records listed in seconds, Hertz and dB.
Interspersed annotations (lines not beginning with space or digit) are ignored.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version.
.TP 11
.BI -f numformants
Select number of formants to load, default is 5.
.TP 11
.B -e
Text input file is in ESPS format, namely list of formant frequencies and
bandwidths per line.  The default number of formants is set to 4.  To get this format
in ESPS, run "formant file.wav", then "pplain file.fb >file.lst".
.TP 11
.BI -r rate
Frame rate of ESPS text file.  Default 100 frames/sec.
.SH OUTPUT ITEMS
.IP 12 11
Edited FM item.
.SH VERSION/AUTHOR
1.3 - Mark Huckvale
.SH SEE ALSO
fmlist
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header fmitem;	/* filing system item header for new item */
int	nformant = 5;		/* number of formants to load */
double	fdur=0.0001;		/* frame duration */

/* get formant data from text file */
int getframe(fmrec,ip)
struct fm_rec		*fmrec;	/* returned data */
FILE			*ip;		/* input file */
{
	char		line[256];
	char		*span,*strtok();
	double		p,s,g;
	int		i,f,a,b;
	static int	lin=0;
	char		lineno[10];

	/* get next data line */
	do {
		if (fgets(line,256,ip) == NULL) return(1);
		lin++;
	} while (!isdigit(line[0]) && (line[0] != ' '));
	sprintf(lineno,"%d",lin);

	/* find sections delimited by semicolons */
	for (i=strlen(line);(i>0) && (line[i] != ';');i--) line[i]='\0';
	if ((span = strtok(line,";")) == NULL)
		error("text file format error at line %s",lineno);

	/* find basic parameters of frame */
	if (sscanf(span,"%lf%lf%d%lf",&p,&s,&i,&g) != 4)
		error("text file format error at line %s",lineno);
	fmrec->posn = (int)(0.5 + p*0.001/fmitem.frameduration);
	fmrec->size = (int)(0.5 + s*0.001/fmitem.frameduration);
	fmrec->flag = i;
	fmrec->gain = (float)g;

	/* clear frame */
	for (i=0;i<nformant;i++) {
		fmrec->formant[i].freq=(float)0.0;
		fmrec->formant[i].band=(float)0.0;
		fmrec->formant[i].amp=(float)0.0;
	}
	fmrec->npeaks=0;

	/* find formant specifications */
	while (((span=strtok(NULL,";")) != NULL) && (fmrec->npeaks < nformant)) {
		if (sscanf(span,"%d%d%d",&f,&b,&a) != 3)
			error("text file format error at line %s",lineno);
		if (f) {
			fmrec->formant[fmrec->npeaks].freq=(float)f;
			fmrec->formant[fmrec->npeaks].band=(float)b;
			fmrec->formant[fmrec->npeaks].amp=(float)a;
			fmrec->npeaks++;
		}
	}

	return(0);
}

/* get ESPS formant data from text file */
int getespsframe(fmrec,ip)
struct fm_rec		*fmrec;	/* returned data */
FILE			*ip;		/* input file */
{
	char		line[256];
	char		line2[256];
	char		*p;
	int			i,n;

	/* clear frame */
	fmrec->posn = 0;
	fmrec->size = (int)(0.5 + fdur/fmitem.frameduration);
	fmrec->flag = 0;
	fmrec->gain = 0;
	for (i=0;i<nformant;i++) {
		fmrec->formant[i].freq=(float)0.0;
		fmrec->formant[i].band=(float)0.0;
		fmrec->formant[i].amp=(float)0.0;
	}
	fmrec->npeaks=0;

	/* find formant specifications */
	if (!fgets(line,256,ip)) return(1);

	/* count number of frequencies on line */
	strcpy(line2,line);
	n=0;
	p = strtok(line2," \t\r\n");
	while (p && *p) {
		n++;
		p = strtok(NULL," \t\r\n");
	}

	p = strtok(line," \t\r\n");
	i=0;
	while (p && *p && (fmrec->npeaks < nformant)) {
		if (i < n/2) {
			if (fmrec->npeaks < nformant) {
				fmrec->formant[fmrec->npeaks].freq=(float)atof(p);
				fmrec->npeaks++;
			}
		}
		else {
			if ((i-n/2) < fmrec->npeaks)
				fmrec->formant[i-n/2].band=(float)atof(p);
		}
		i++;
		p = strtok(NULL," \t\r\n");
	}

	return(0);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	char		textname[SFSMAXFILENAME];	/* text file name */
	FILE		*ip;		/* text file variable */
	int		ofid;
	/* processing variables */
	struct fm_rec	*fmrec;		/* input formant record */
	int		cnt=0;
	int		espsformat=0;
	int		ratechanged=0;
	int		posn;

	/* decode switches */
	while ( (c = getopt(argc,argv,"If:er:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Load formant estimates V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'f' :	/* number of loaded formants */
			nformant = atoi(optarg);
			break;
		case 'e':
			espsformat=1;
			if (!ratechanged) fdur=0.01;
			break;
		case 'r':
			if (atof(optarg)<=0)
				error("illegal frame rate");
			fdur = 1.0/atof(optarg);
			ratechanged=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}

	/* check for option decoding error */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-f numformants) (-e) (-r framerate) textfile dbase_file",PROGNAME);

	/* get text filename */
	if (optind < argc)
		strcpy(textname,argv[optind]);
	else
		error("no listing file specified",NULL);
	optind++;

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open text file */
	if ((ip=fopen(textname,"r")) == NULL)
		error("unable to open text file %s",textname);

	/* create FM item header */
	if (espsformat) {
		sfsheader(&fmitem,FM_TYPE,1,4,
			(sfsstruct[FM_TYPE]+nformant*sizeof(struct fm_rec_array))/4,
			fdur,0.0,1,0,0);
		sprintf(fmitem.history,"%s(file=%s,rate=%g)",PROGNAME,textname,1.0/fdur);
	}
	else {
		sfsheader(&fmitem,FM_TYPE,1,4,
			(sfsstruct[FM_TYPE]+nformant*sizeof(struct fm_rec_array))/4,
			fdur,0.0,0,0,1);
		sprintf(fmitem.history,"%s(file=%s)",PROGNAME,textname);
	}

	/* open output channel */
	if ((ofid=sfschannel(filename,&fmitem)) < 0)
		error("unable to create temporary file",NULL);

	/* get input buffer */
	if ((fmrec=(struct fm_rec *)sfsbuffer(&fmitem,1))==NULL)
		error("could not get memory buffer",NULL);

	/* load frames from text file */
	if (espsformat) {
		posn=0;
		while (getespsframe(fmrec,ip) == 0) {
			fmrec->posn = posn;
			posn += fmrec->size;
			if (sfswrite(ofid,1,fmrec) != 1)
				error("write error on temporary file",NULL);
			cnt++;
		}
	}
	else {
		while (getframe(fmrec,ip) == 0) {
			if (sfswrite(ofid,1,fmrec) != 1)
				error("write error on temporary file",NULL);
			cnt++;
		}
	}

	/* update data file */
	if (cnt == 0)
		error("no legal frames found in %s",textname);
	else if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* that's all folks */
	exit(0);
}

