/* fxlist -- list Fx data in a variety of formats */

/* M.A.Huckvale - University College London */

/* version 1.0 - October 1998 */

#define PROGNAME "fxlist"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH FXLIST 1 UCL
.SH NAME
fxlist - save speech data in a variety of formats
.SH SYNOPSIS
.B fxlist
(-i item) (-t format) (-o outfile) sfsfile
.SH DESCRIPTION
.I fxlist
is a program to list fx items.
The selected data set
is either printed on the standard output as a sequence of numbers or stored in a
foreign format datafile.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item.
.TP 11
.BI -t type
Select output file format.  Available are: RAW (binary), ESIG (Esignal format) or ASCII (default).
.TP 11
.BI -o datafile
Store the contents of the data set in the file
.I datafile
rather than listing them on the standard output.
.SH INPUT ITEMS
.IP FX 11
Any fundamental frequency item.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/
/* global structures */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <fcntl.h>
#include <string.h>
#include "sfs.h"

#define MIN(x,y) (((x)<(y))?(x):(y))

/* global data */
struct item_header	item;

/* format lookup */
void	rawproc(),esigproc(),asciiproc();
struct format_rec {
	char	*name;
	void	(*fproc)();
} formattab[]={
	{ "ASCII",	asciiproc },
	{ "RAW",	rawproc },
	{ "ESIG",	esigproc },
};
#define NUMFORMAT (sizeof(formattab)/sizeof(struct format_rec))

/* copying buffer */
#define COPYBUFSIZE	16384
short	copybuf[COPYBUFSIZE];

/* options */
int	format=0;
char	ofilename[SFSMAXFILENAME];

/* copy sample data */
void	copydata(ifid,op,numf)
int	ifid;
FILE	*op;
int	numf;
{
	int	pos;
	int	count;

	pos=0;
	while (numf > 0) {
		count = sfsread(ifid,pos,MIN(numf,COPYBUFSIZE),copybuf);
		if (count <= 0)
			error("read error on input file");
		if ((int)fwrite((char *)copybuf,2,count,op)!=count)
			error("write error on '%s'",ofilename);
		numf -= count;
		pos += count;
	}
}

/* store data in ascii file */
void asciiproc(fid,item,op)
int		fid;		/* input data file */
struct item_header *item;
FILE		*op;
{
	/* copy the samples */
	int	numf;
	int	pos;
	int	count;
	int	i;

	pos=0;
	numf=item->numframes;
	while (numf > 0) {
		count = sfsread(fid,pos,MIN(numf,COPYBUFSIZE),copybuf);
		if (count <= 0)
			error("read error on input file");
		for (i=0;i<count;i++)
			fprintf(op,"%d\n",copybuf[i]);
		numf -= count;
		pos += count;
	}
}

/* store data in binary file */
void rawproc(fid,item,op)
int		fid;		/* input data file */
struct item_header *item;
FILE		*op;
{
	/* copy the samples */
	copydata(fid,op,item->numframes);
}

/* store data in ESIGNAL file */
void esigproc(fid,item,op)
int		fid;		/* input data file */
struct item_header *item;
FILE		*op;
{
	char	esighdr[6][8];
	char	buf[128];
	int	pos,numf,count,i;

	/* write header */
	sprintf(buf,"Esignal\n");
	memcpy(esighdr[0],buf,8);
	sprintf(buf,"    1.0\n");
	memcpy(esighdr[1],buf,8);
	sprintf(buf,"  ASCII\n");
	memcpy(esighdr[2],buf,8);
	sprintf(buf,"     48\n");
	memcpy(esighdr[3],buf,8);
	sprintf(buf,"   1024\n");
	memcpy(esighdr[4],buf,8);
	sprintf(buf,"%7d\n",item->numframes);
	memcpy(esighdr[5],buf,8);
	fwrite(esighdr,8,6,op);

	sprintf(buf,"F0: DOUBLE <r>\n");
	fputs(buf,op);
	sprintf(buf,"prob_voice: DOUBLE <r>\n");
	fputs(buf,op);
	sprintf(buf,"rms: DOUBLE <r>\n");
	fputs(buf,op);
	sprintf(buf,"ac_peak: DOUBLE <r>\n");
	fputs(buf,op);
	sprintf(buf,"src_sf: DOUBLE 1 <g> %g\n",(1.0/item->frameduration));
	fputs(buf,op);
	sprintf(buf,"record_freq: DOUBLE 1 <g> %g\n",(1.0/((item->windowsize-item->overlap)*item->frameduration)));
	fputs(buf,op);
	sprintf(buf,"start_time: DOUBLE 1 <g> %g\n",item->offset);
	fputs(buf,op);

	/* save total header length */
	sprintf(buf,"%7d\n",ftell(op));
	memcpy(esighdr[4],buf,8);

	/* write data records */
	pos=0;
	numf=item->numframes;
	while (numf > 0) {
		count = sfsread(fid,pos,MIN(numf,COPYBUFSIZE),copybuf);
		if (count <= 0)
			error("read error on input file");
		for (i=0;i<count;i++) {
			fprintf(op,"%g\n",(double)copybuf[i]);
			fprintf(op,"%g\n",(copybuf[i]==0)?0.0:1.0);
			fprintf(op,"%g\n",1.0);
			fprintf(op,"%g\n",1.0);
		}
		numf -= count;
		pos += count;
	}

	/* and rewrite header with header length */
	rewind(op);
	fwrite(esighdr,8,6,op);
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* option decoding */
	extern int	optind;
	extern char	*optarg;
	int		errflg=0;
	int		c;
	int32		it;		/* selected item type */
	char		*ty;		/* item specification */
	int		initem=FX_TYPE;
	char		*intype="0";

	/* file variables */
	char		filename[SFSMAXFILENAME];	/* data file */
	int		fid;
	int		i;
	FILE		*op;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:o:t:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: FX contour export V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'o' :	/* output file */
			strcpy(ofilename,optarg);
			break;
		case 'i':	/* input item specification */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == FX_TYPE) {
					initem = it;
					intype = ty;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 't' :	/* file type */
			for (format=0;format<NUMFORMAT;format++)
				if (strcmp(optarg,formattab[format].name)==0)
					break;
			if (format >= NUMFORMAT) {
				fprintf(stderr,"known file format types are:");
				for (i=0;i<NUMFORMAT;i++)
					fprintf(stderr," %s",formattab[i].name);
				fprintf(stderr,"\n");
				error("unknown file format type '%s'",optarg);
			}
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: fxlist (-I) (-i item) (-t format) (-o outfile) sfsfile\n",NULL);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* locate start of data set */
	if ((fid=sfsopen(filename,"r",NULL)) < 0)
		error("access error on '%s'",filename);
	if (!sfsitem(fid,initem,intype,&item))
		error("cannot find input item in '%s'",filename);

	/* open output file */
	if (ofilename[0]) {
		if ((op=fopen(ofilename,"w+b"))==NULL)
			error("could not open '%s'",ofilename);
		(*formattab[format].fproc)(fid,&item,op);
		fclose(op);
	}
	else
		(*formattab[format].fproc)(fid,&item,stdout);

	/* that's all folks */
	sfsclose(fid);
	exit(0);
}


