/* stree -- DIG graphic representation of file structure */

/* m.a. huckvale - feb 1988 */

/* version 2 - May 2000 */

/*--------------------------------------------------------------------------*/
/**MAN
.TH STREE SFS1 UCL SFS
.SH NAME
stree - display item structure tree of SFS file
.SH SYNOPSIS
.B stree
(-p) file
.SH DESCRIPTION
.I stree
is a program to display the tree structure of the items in a speech data 
file.
.PP
.I Options
and their meanings are:
.TP 11
.TP 11
.B -I
Identify program and exit.
.TP 11
.B -p
Send output direct to printer.
.SH GRAPHICS
Each item is described using the item number and the first 8 characters of
the program name in the item history.  Deleted items have a cross-bar.
.SH VERSION/AUTHOR
2.0 - Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

#define PROGNAME "stree"
#define PROGVERS "2.0"
char *progname=PROGNAME;

/* global declarations */
#include <stdio.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdlib.h>
#include "sfs.h"
#include "sfstree.h"
#include "dig.h"
#include "digdata.h"

/* display structure */
#define ITXMAX	10
#define ITYMAX	10
struct {
	int	entry;
	float	x;
	float	y;
	int	select;
} dtab[ITXMAX][ITYMAX];
int	dcnt[ITXMAX];
int	xent[MAXTREELEN];
int	yent[MAXTREELEN];
int	dlevel=0;
float	bheight,bwidth,cheight,cwidth;	/* graphics constants */
int	plen;
float	xpos=0.5,ypos=0.5;			/* stored mouse x,y */

/* global data */
int	laser=0;			/* laser output */
extern char	*sfslabel[];
struct main_header head;

/* draw item box */
drawbox(x,y,it,ty,hist)
float	x,y;
int	it,ty;
char	*hist;
{
	char	iname[8];
	char	ilabel[32];
	int	llen;
	float	tlen;
	int	i;

	/* get text for box */
	sprintf(iname,"%s.%02d",sfslabel[abs(it)],ty);
	llen = strcspn(hist," (/");
	for (i=0;(i<llen) && (i<8);i++) ilabel[i]=hist[i];
	ilabel[i]='\0';
	tlen = digtextlen(ilabel);

	/* clear box */
	digflush();
	digfillbox(0,x-bwidth,y-bheight,x+bwidth,y+bheight);

	/* write outline if undeleted item */
	if (it > 0) {
		digbox(31,x-bwidth,y-bheight,x+bwidth,y+bheight);
	}
	else {
		digbox(29,x-bwidth,y-bheight,x+bwidth,y+bheight);
		digline(29,x-bwidth,y+bheight,x+bwidth,y-bheight);
	}

	/* write item number */
	digtext(24,x-2.0*cwidth,y,iname);

	/* write item label */
	digtext(24,x-tlen/2,y-1.125*cheight,ilabel);

	return(0);
}

/* display item relation tree */
display(filename)
char	*filename;
{
	int	i,j,k,nassign=0;
	int	entry,lentry;
	float	digtextlen(),gap;
	char	title[256];

	/* assign items to level in tree */
	dlevel=0;
	for (i=0;(dlevel<ITXMAX) && (nassign<treelen);i++) {
		dcnt[dlevel]=0;
		for (j=0;j<treelen;j++) {
			if (tree[j].level==i) {
				dtab[dlevel][dcnt[dlevel]].entry=j;
				dtab[dlevel][dcnt[dlevel]].select=0;
				xent[j]=dlevel;
				yent[j]=dcnt[dlevel];
				dcnt[dlevel]++;
				nassign++;
				if (dcnt[dlevel] == ITYMAX) {
					dlevel++;
					dcnt[dlevel]=0;
				}
			}
		}
		dlevel++;
	}

	/* allow space for title */
	cheight = (float)digdata.chheight/digdata.yscale;
	cwidth  = (float)digdata.chwidth/digdata.xscale;
	bheight = 1.25 * cheight;
	bwidth  = 4.5 * cwidth;
	gap = (1.0-2*dlevel*bwidth)/(dlevel+1);
	if (gap < 0) gap=0;
	plen = (1.0/cwidth);	/* max char on line */

	/* plot title */
	digclearscreen();
	sprintf(title,"file=%s speaker=%s token=%s",filename,head.speaker,head.token);
	title[plen]='\0';
	digtext(24,0.0,-1.5*cheight,title);
	digbox(20,0.0,0.0,1.0,1.0);
	digclip(0.0,0.0,1.0,1.0);

	/* calculate x,y co-ordinates */
	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) {
			dtab[i][j].x = (gap+bwidth) + i*(gap + 2*bwidth);
			dtab[i][j].y = 1.0 - ((float)(j+1))/(dcnt[i]+1);
		}
	}

	/* plot lines */
	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) {
			entry = dtab[i][j].entry;
			for (k=0;k<tree[entry].numlink;k++) {
				lentry = tree[entry].link[k];
				digline(27,dtab[xent[lentry]][yent[lentry]].x+bwidth,
					   dtab[xent[lentry]][yent[lentry]].y,
					   dtab[i][j].x-bwidth,dtab[i][j].y);
			}
		}
	}

	/* plot itemnos */
	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) {
			entry = dtab[i][j].entry;
			drawbox(dtab[i][j].x,dtab[i][j].y,
				tree[entry].datatype,
				tree[entry].subtype,
				tree[entry].history);
		}
	}

	digflush();
	return(0);
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
        /* option decoding variables */
        extern int      optind;
        extern char     *optarg;
        int             errflg=0;
        char            c;
	/* file variables */
        int             fid;
        char            filename[SFSMAXFILENAME];

        /* decode switches */
        while ( (c = getopt(argc,argv,"Ip")) != EOF) switch (c) {
                case 'I' :      /* Identify */
                        fprintf(stderr,"%s: file structure tree V%s\n",argv[0],PROGVERS);
                        exit(0);
                        break;
		case 'p' :	/* laser print */
			laser++;
			break;
                case '?' :      /* unknown */
                        errflg++;
        }
        if (errflg || (argc<2)) 
                error("usage: %s (-I) (-p) file",argv[0]);

        /* get filename */
        if (optind < argc)
                strcpy(filename,sfsfile(argv[argc-1]));
        else
                error("no database file specified",NULL);

        /* open datafile */
        if ((fid = sfsopen(filename,"r",&head)) < 0) {
                if (fid == -1)
                        error("cannot open %s",filename);
                else
                        error("access error on %s",filename);
        }
	sfsclose(fid);

        /* get tree */
        if (!sfstree(filename))
		error("cannot open %s",filename);

	/* init graphics */
	if (laser)
		digstart(DIG_DEFAULT_PRINTER,NULL,1);
	else
		digstart(DIG_DEFAULT_TERM,NULL,1);

	do {
		/* set up scaling */
		digscale(1.0,1.0+3.0*digdata.chheight/digdata.yscale,0);
		digorigin(0.0,1.5*digdata.chheight/digdata.yscale);

		/* display tree */
		display(filename);

	} while (digwait()==DIG_REDRAW);
	
	/* that's all folks */
	digend();
        exit(0);
}


