/* syout -- hardware synthesizer replay */

/* M.A. Huckvale - February 1989 */

/* version 2
	- based on DBASE program
	- but calling SFS hwsynth routines
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH SYOUT SFS1 UCL SFS
.SH NAME
syout - output synthesizer control data through hardware synthesizer
(Masscomp systems only)
.SH SYNOPSIS
.B syout
(-I) (-i item) (-n repetitions) file
.SH DESCRIPTION
.I syout
is a program to control the Quarmby hardware formant synthesizer using synthesizer
control data stored in a database file.  The synthesizer is wired to the parallel
output port /dev/dacp0/pfo0.  Frame rate is determined by the synthesizer handshake, regardless
of the frameduration in the data set.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item type and subtype.
.TP 11
.BI -n repetitions
Replay the data the given number of times.  Default 1.
.SH OUTPUT CHANNEL SELECTION
On sites where DAC and HWSYNTH replay is under software control, 
.I syout
re-directs output according to the setting of the DAC environment
variable and SFSBASE/data/dacdefs.  See playback(SFS3) and hwsynth(SFS3)
for more details.
.SH INPUT ITEMS
.IP 7.xx 11
Any synthesizer control code item
.SH VERSION/AUTHOR
2.0 - Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "syout"
#define PROGVERS "2.0"
char *progname=PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <malloc.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header syitem;	/* filing system item header for sy */
short		*sy;		/* SY frame buffer */
int		numrep=1;

/* synthesis output buffer */
unsigned short	*buff;

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item/sub-type specifiers */
	char		*ty;

	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	int		fid;
	char		*sytype="0";

	/* processing variables */
	int		i;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:n:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Formant synthesizer output V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SY_TYPE)
					sytype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'n' :	/* repeat */
			numrep = atoi(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check for option decoding error */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-n repetitions) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open datafile */
	if ((fid=sfsopen(filename,"r",NULL)) < 0) 
		error("cannot open '%s'",filename);

	/* find input item */
	if (!sfsitem(fid,SY_TYPE,sytype,&syitem))
		error("cannot find input SY item in '%s'",filename);

	/* check size */
	if (syitem.framesize != 19)
		error("wrong format SY item",NULL);

	/* get SY frame buffer */
	if ((sy=(short *)sfsbuffer(&syitem,1)) == NULL)
		error("could not get frame buffer",NULL);

	/* get output buffer */
	if ((buff=(unsigned short *)calloc(syitem.numframes+1,16*sizeof(short)))==NULL)
		error("could not get output buffer",NULL);

	/* initialise buffer - hwsynth skips first frame */
	sfsread(fid,0,1,sy);
	hwsynthcode(sy,buff);

	/* read SY data into buffer */
	for (i=0;sfsread(fid,i,1,sy);i++)
		hwsynthcode(sy,buff+16*i+16);

	/* replay */
	for (i=0;i<numrep;i++)
		hwsynth(buff,syitem.numframes+1);

	/* that's all folks */
	sfsclose(fid);
	exit(0);
}

