/* txlist -- list TX data in a variety of formats */

/* M.A.Huckvale - University College London */

/* version 1.0 - December 2007 */

#define PROGNAME "txlist"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH TXLIST 1 UCL
.SH NAME
txlist - list Tx data in a variety of formats
.SH SYNOPSIS
.B txlist
(-i item) (-S|-T) (-c) (-t format) (-o outfile) sfsfile
.SH DESCRIPTION
.I txlist
is a program to list track items.
The selected data set
is either printed on the standard output as a sequence of numbers or stored in a
foreign format datafile.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item.
.TP 11
.B -S
Output data as sample numbers.
.TP 11
.B -T
Output data as times in seconds. (default)
.TP 11
.B -c
Output data as cumulative time positions rather than period durations.
.TP 11
.BI -t type
Select output file format.  Available are: PRAAT or ASCII (default).
.TP 11
.BI -o datafile
Store the contents of the data set in the file
.I datafile
rather than listing them on the standard output.
.SH INPUT ITEMS
.IP TX 11
Any Tx item.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale.
*/
/*--------------------------------------------------------------------------*/
/* global structures */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <fcntl.h>
#include <string.h>
#include "sfs.h"

#define MIN(x,y) (((x)<(y))?(x):(y))

/* global data */
struct item_header	txitem;
int	*tx;

/* format lookup */
void	praatproc(),asciiproc();
struct format_rec {
	char	*name;
	void	(*fproc)();
} formattab[]={
	{ "ASCII",	asciiproc },
	{ "PRAAT",	praatproc },
};
#define NUMFORMAT (sizeof(formattab)/sizeof(struct format_rec))

/* copying buffer */
#define COPYBUFSIZE	16384
float	copybuf[COPYBUFSIZE];

/* options */
int	format=0;
char	ofilename[SFSMAXFILENAME];
int		dosamples=0;
int		documulative=0;

/* store data in ascii file */
void asciiproc(FILE *op)
{
	/* copy the samples */
	int	pos=0;
	int	i;

	for (i=0;i<txitem.numframes;i++) {
		pos += tx[i];
		if (dosamples) {
			if (documulative)
				fprintf(op,"%d\n",pos);
			else
				fprintf(op,"%d\n",tx[i]);
		}
		else {
			if (documulative)
				fprintf(op,"%8g\n",txitem.offset+pos*txitem.frameduration);
			else
				fprintf(op,"%8g\n",txitem.offset+tx[i]*txitem.frameduration);
		}
	}
}

/* store data in ascii file */
void praatproc(FILE *op)
{
	/* copy the samples */
	int	pos=0;
	int	i;

	for (i=0;i<txitem.numframes;i++) pos += tx[i];

	fprintf(op,"File type = \"ooTextFile\"\n");
	fprintf(op,"Object class = \"PointProcess\"\n");
	fprintf(op,"\n");
	fprintf(op,"0\n");
	fprintf(op,"%g\n",txitem.offset+pos*txitem.frameduration);
	fprintf(op,"%d\n",txitem.numframes);

	pos=0;
	for (i=0;i<txitem.numframes;i++) {
		pos += tx[i];
		fprintf(op,"%g\n",txitem.offset+tx[i]*txitem.frameduration);
	}
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* option decoding */
	extern int	optind;
	extern char	*optarg;
	int		errflg=0;
	int		c;
	int32		it;		/* selected item type */
	char		*ty;		/* item specification */
	int		initem=TX_TYPE;
	char		*intype="0";

	/* file variables */
	char		filename[SFSMAXFILENAME];	/* data file */
	int		i;
	FILE		*op;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:o:t:TSc")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Tx export V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'o' :	/* output file */
			strcpy(ofilename,optarg);
			break;
		case 'i':	/* input item specification */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == TX_TYPE) {
					initem = it;
					intype = ty;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 't' :	/* file type */
			for (format=0;format<NUMFORMAT;format++)
				if (strcmp(optarg,formattab[format].name)==0)
					break;
			if (format >= NUMFORMAT) {
				fprintf(stderr,"known file format types are:");
				for (i=0;i<NUMFORMAT;i++)
					fprintf(stderr," %s",formattab[i].name);
				fprintf(stderr,"\n");
				error("unknown file format type '%s'",optarg);
			}
			break;
		case 'S':
			dosamples=1;
			break;
		case 'T':
			dosamples=0;
			break;
		case 'c':
			documulative=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: txlist (-I) (-i item) (-S|-T) (-c) (-t format) (-o outfile) sfsfile\n",NULL);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	getitem(filename,initem,intype,&txitem,&tx);

	/* open output file */
	if (ofilename[0]) {
		if ((op=fopen(ofilename,"w"))==NULL)
			error("could not open '%s'",ofilename);
		(*formattab[format].fproc)(op);
		fclose(op);
	}
	else
		(*formattab[format].fproc)(stdout);

	/* that's all folks */
	exit(0);
}

