/*  Filename: filebuf.cpp

    Supplimentary file for providing buffered file access for the Xbase 
	project that was written by StarTech, Gary A. Kunkel.
  
    This file provides a class for providing buffered file access.

    Copyright (C) 2003 Pulse Data International (NZ) Ltd.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Contact:

		Pulse Data International (NZ) Ltd.,
		P O Box 3044,
		Christchurch,
		New Zealand.

    Email:

		enquiries@pulsedata.com

    See our website at:

        http://www.pulsedata.com/


	KeySoft 5.1 Alpha 1, 15 Aug 2003	- Windows CE Compliancy
	KeySoft 5.1 Alpha 1, 30 Jan 2004	- RAM buffering for speed
	KeySoft 5.1 Alpha 1, 13 Feb 2004	- Fixing memory leaks

*/


#include <windows.h>
#include "xbase.h"
#include "filebuf.h"

fileBuf_class::fileBuf_class () :
   _File (NULL),
   _Pos (0),
   _Heap (NULL),
   _Buffer (NULL),
   _BufferCount (0),
   _BufferStart (0),
   _HasChanged (FALSE)
{
}

BOOL fileBuf_class::Open (const char* filename, const char* mode)
{
   _File = fopen (filename, mode);

   return _File != NULL;
}

size_t fileBuf_class::Read (void* buffer, size_t size, size_t count)
{
   if (_Heap == NULL && !_LoadFileToMem ()) return 0;

   size_t bytesRequested = size * count;
   size_t bytesAvailable = _BufferCount - _Pos;

   size_t bytesToCopy = (bytesAvailable < bytesRequested)
      ? (bytesAvailable / size) * size
      : bytesRequested;

   memcpy (buffer, &_Buffer [_Pos], bytesToCopy);
   _Pos += bytesToCopy;

   return bytesToCopy / size;
}

size_t fileBuf_class::Write (const void* buffer, size_t size, size_t count)
{
   fseek (_File, _Pos, SEEK_SET);
   size_t ret = fwrite (buffer, size, count, _File);
   long writePos = ftell (_File);

   if (_Heap)
   {
      HeapDestroy (_Heap);
      _Heap = NULL;
      _Buffer = NULL;
      _BufferCount = 0;
      _BufferStart = 0;
      _Pos = 0;
   }

   if (!_LoadFileToMem ()) return 0;

   _Pos = writePos;

   return ret;
}

int fileBuf_class::Close ()
{
   int r = fclose (_File);
   _File = NULL;

   if (_Heap)
   {
      HeapDestroy (_Heap);
      _Heap = NULL;
      _Buffer = NULL;
      _BufferCount = 0;
      _BufferStart = 0;
      _Pos = 0;
   }

   return r;
}

int fileBuf_class::Seek (long offset, int origin)
{
   switch (origin)
   {
      case SEEK_CUR: _Pos += offset;               break;
      case SEEK_END: _Pos = _BufferCount + offset; break;
      case SEEK_SET: _Pos = offset;                break;
   }

   return 0;
}

long fileBuf_class::GetPos ()
{
   return _Pos;
}

int fileBuf_class::Putc (int c)
{
   fseek (_File, _Pos, SEEK_SET);
   int ret = fputc (c, _File);
   long writePos = ftell (_File);

   if (_Heap)
   {
      HeapDestroy (_Heap);
      _Buffer = NULL;
      _BufferCount = 0;
      _BufferStart = 0;
      _Pos = 0;
   }

   if (!_LoadFileToMem ()) return 0;

   _Pos = writePos;

   return ret;
}

int fileBuf_class::Getc ()
{
   if (_Heap == NULL && !_LoadFileToMem ()) return 0;
   
   BYTE c = _Buffer [_Pos];
   _Pos++;

   return c;
}

int fileBuf_class::Flush ()
{
   return fflush (_File);
}

void* fileBuf_class::FileNo ()
{
   return fileno (_File);
}

BOOL fileBuf_class::IsOpen ()
{
   return _File != NULL;
}

void fileBuf_class::operator= (fileBuf_class& rhs)
{
   long writePos = ftell (rhs._File);
   _File = rhs._File;

   if (_Heap)
   {
      HeapDestroy (_Heap);
      _Heap = NULL;
      _Buffer = NULL;
      _BufferCount = 0;
      _BufferStart = 0;
      _Pos = 0;
   }

   if (!_LoadFileToMem ()) return;

   _Pos = writePos;

}

BOOL fileBuf_class::_LoadFileToMem ()
{
   _Heap = HeapCreate (0, 0, 0);

   fseek (_File, 0, SEEK_END);
   size_t fileSize = ftell (_File);

   _Buffer = (BYTE*)HeapAlloc (_Heap, 0, fileSize);
   _BufferCount = fileSize;
   _BufferStart = 0;
   _Pos = 0;

   fseek (_File, 0, SEEK_SET);

   size_t bytesRead = fread (_Buffer, 1, fileSize, _File);

   if (bytesRead != fileSize)
   {
      HeapDestroy (_Heap);
      _Heap = NULL;
      _Buffer = NULL;
      _BufferCount = 0;
      _BufferStart = 0;
      _Pos = 0;
      return FALSE;
   }

   return TRUE;
}
