/*
 * comm.c - COM[12] interface
 *
 * V. Abell
 */

#include "touch2.h"

extern struct videoconfig Vc;

#define	BAUDLN	8
#define COL	20
#define EXITLN	18
#define	FRAMELN	12
#define PARLN	14
#define PORTLN	10
#define	STSTLN	16

short Baud[] = { 9600, 2400, 1200, 300, 0 };

struct parity Par[] = {
	{ 0, "none" },
	{ 1, "even" },
	{ 2, "odd"  },
	{ 0, NULL },
};

short Baudx = 0;
short Comopen = 0;
short Frame = 8;
short Parx = 0;
short Port = 0;
short StSt = 1;


struct menu ComPar[] = {
	{  2, 20, "Communications parameters:" },
	{  0,  0, NULL },
};


static void DisBaud();
static void DisFrame();
static void DisPar();
static void DisPort();
static void DisStst();


/*
 * CloseCom() - close COM[12] port
 */

void
CloseCom()
{
	if (Comopen) {
		(void) close_com();
		Comopen = 0;
	}
}


/*
 * DefCom() - define COM[12] parameters
 */

void
DefCom()
{
	int ch;

	DispMenu(ComPar, NULL);
	DisBaud();
	DisFrame();
	DisPar();
	DisPort();
	DisStst();
	_settextposition(EXITLN, COL);
	_outtext("X - eXit");
	for (;;) {
		if ( ! kbhit()) {
			reset_buffer();
			continue;
		}
		ch = getch();
		switch(ch) {

		case ESC:
		case 'x':
		case 'X':
			return;

		case 'b':
		case 'B':
			Baudx++;
			if (Baud[Baudx] == 0)
				Baudx = 0;
			DisBaud();
			break;

		case 'c':
		case 'C':
			Port = (Port == 0) ? 1 : 0;
			DisPort();
			break;

		case 'f':
		case 'F':
			Frame = (Frame == 8) ? 7 : 8;
			DisFrame();
			break;

		case 'p':
		case 'P':
			Parx++;
			if (Parx > 2)
				Parx = 0;
			DisPar();
			break;

		case 's':
		case 'S':
			StSt = (StSt == 1) ? 2 : 1;
			DisStst();
			break;

		case 0:
			ch = getch();
			/* fall through */
		default:
			putch(BELL);
		}
	}
}


/*
 * DisBaud() - display baud rate
 */

static void
DisBaud()
{
	char b[64];

	(void) sprintf(b, "B - baud rate: %d", Baud[Baudx]);
	ClearRow(BAUDLN, COL);
	_settextposition(BAUDLN, COL);
	_outtext(b);
}


/*
 * DisFrame() - display frame size
 */

static void
DisFrame()
{
	char b[64];

	(void) sprintf(b, "F - frame size : %d", Frame);
	ClearRow(FRAMELN, COL);
	_settextposition(FRAMELN, COL);
	_outtext(b);
}


/*
 * DisPar() - display parity
 */

static void
DisPar()
{
	char b[64];

	(void) sprintf(b, "P - parity: %s", Par[Parx].nm);
	ClearRow(PARLN, COL);
	_settextposition(PARLN, COL);
	_outtext(b);
}


/*
 * DisPort() - display COM port
 */

static void
DisPort()
{
	char b[64];

	(void) sprintf(b, "C - COM port: %d", Port + 1);
	ClearRow(PORTLN, COL);
	_settextposition(PORTLN, COL);
	_outtext(b);
}


/*
 * DisStst() - display start/stop bit count
 */

static void
DisStst()
{
	char b[64];

	(void) sprintf(b, "S - start/stop bits: %d", StSt);
	ClearRow(STSTLN, COL);
	_settextposition(STSTLN, COL);
	_outtext(b);
}


/*
 * OpenCom() - open the COM[12] port
 */

void
OpenCom()
{
	int err;
	char msg[128];

	if (Comopen)
		return;
	open_com(Port, Baud[Baudx], Par[Parx].val, StSt, Frame, &err);
	if (err) {
		(void) sprintf(msg, "Can't open COM%d port: error %d",
			Port + 1, err);
		_clearscreen(_GCLEARSCREEN);
		_settextposition(12, 25);
		_outtext(msg);
		PromptMsg("Press any key to exit.");
		(void) WaitAnyKey();
		TouchExit(1);
	}
	Comopen = 1;
}


/*
 * WaitAnyKey() - wait for any key press
 */

int
WaitAnyKey()
{
	int ch;

	for (;;) {
		if (kbhit()) {
			if ((ch = getch()) == 0)
				ch = getch();
			return(ch);
		}
		reset_buffer();
	}
}
