/*
 * conf.c - read configuration file
 *
 * V. Abell
 */

#include "touch2.h"
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

static int AsmHex(char *c);
static int AsmPrStr(char **s, char *m, char *v);
static short FindClr(char *v);
static int SetBaudRate(char *b, char *v);
static int SetBkClr(char *b, char *v);
static int SetCksum(char *b, char *v);
static int SetComPort(char *b, char *v);
static int SetDumpFmt(char *b, char *v);
static int SetPcDisp(char *b, char *v);
static int SetQDumpNm(char *b, char *v);
static int SetRDumpNm(char *b, char *v);
static int SetFrameSz(char *b, char *v);
static int SetGraphNm(char *b, char *v);
static int SetNormSp(char *b, char *v);
static int SetParity(char *b, char *v);
static int SetPromptBkClr(char *b, char *v);
static int SetPromptTxtClr(char *b, char *v);
static int SetSSBits(char *b, char *v);
static int SetTxtClr(char *b, char *v);
static int SetDispWarn(char *b, char *v);

/*
 * The Colors[] table is ordered by color index -- i.e., the color
 * index is implied from the position of an entry in the table.  So
 * don't change the order of the entries.
 */

struct clr Colors[] = {
	{ "black",		_BLACK },		/* 0 */
	{ "blue",		_BLUE },		/* 1 */
	{ "green",		_GREEN },		/* 2 */
	{ "cyan",		_CYAN },		/* 3 */
	{ "red",		_RED },			/* 4 */
	{ "magenta",		_MAGENTA },		/* 5 */
	{ "brown",		_BROWN },		/* 6 */
	{ "white",		_WHITE },		/* 7 */
	{ "gray",               _GRAY },		/* 8 */
	{ "light blue",		_LIGHTBLUE },		/* 9 */
	{ "light green",	_LIGHTGREEN },		/* 10 */
	{ "light cyan",		_LIGHTCYAN },		/* 11 */
	{ "light red",		_LIGHTRED },		/* 12 */
	{ "light magenta",	_LIGHTMAGENTA },	/* 13 */
	{ "yellow",		_YELLOW },		/* 14 */
	{ "bright white",	_BRIGHTWHITE },		/* 15 */
	{ NULL,			0L },
};

struct kcmd {
	char *nm;			/* keyword */
	int (*fun)();			/* processing function */
} Keywd [] = {
	{ "aftergraph",		SetAftGraph },		/* also -a */
	{ "ag",			SetAftGraph },		/* also -a */
	{ "bar",		SetBarChar },		/* also -b */
	{ "baudrate",		SetBaudRate },
	{ "br",			SetBaudRate },
	{ "background",		SetBkClr },
	{ "bk",			SetBkClr },
	{ "beforegraph",	SetBefGraph },		/* also -B */
	{ "b4g",		SetBefGraph },		/* also -B */
	{ "checksum",		SetCksum },		/* also -C */
	{ "cksum",		SetCksum },		/* also -C */
	{ "comport",		SetComPort },
	{ "dump",		SetDumpFmt },		/* partially -q */
	{ "gmax",		SetGmaxLvl },		/* also -G */
	{ "gmin",		SetGminLvl },		/* also -g */
	{ "graphtitle",		SetGraphTtl },
	{ "gttl",		SetGraphTtl },
	{ "intersection",	SetIntChar },		/* also -i */
	{ "ic",			SetIntChar },		/* also -i */
	{ "line",		SetLineChar },		/* also -l */
	{ "lpp",		SetLpp },		/* also -L */
	{ "pcd",		SetPcDisp },		/* also -P */
	{ "pgctdisp",		SetPcDisp },		/* also -P */
	{ "port",		SetComPort },
	{ "qdumpname",		SetQDumpNm },
	{ "rdumpname",		SetRDumpNm },
	{ "framesize",		SetFrameSz },
	{ "graphname",		SetGraphNm },
	{ "parity",		SetParity },
	{ "printer",		SetPrinter },		/* also -p */
	{ "startstop",		SetSSBits },
	{ "stst",		SetSSBits },
	{ "text",		SetTxtClr },
	{ "promptbk",		SetPromptBkClr },
	{ "pbk",		SetPromptBkClr },
	{ "prompttext",		SetPromptTxtClr },
	{ "ptxt",		SetPromptTxtClr },
	{ "warn",		SetDispWarn, },
	{ NULL,			NULL },
};

char *AftGraph = NULL;			/* after graph printer control */
char *BefGraph = NULL;			/* before graph printer control */
char Gttl[GTTLLNL+1];			/* graph title line */
short Lineval = 0;			/* line value definition status */
short PromptBkClrx = DEFPBCX;		/* prompt background color */
short PromptTxtClrx = DEFPTCX;		/* prompt text color */


/*
 * Asm1Ch() - assemble one character
 *
 * return = number of characters extracted from definition string
 *	    (-1 = error)
 */

int
Asm1Ch(s, r)
	char *s;			/* pointer to string that defines the
					 * character */
	char *r;			/* pointer to result */
{
	int b, i, n;
	register char *c;
	char tc;

	c = s;
	if (*c == '\\') {

	/*
	 * `\' is the prefix for special strings.
	 */
		c++;
		switch(*c) {

		case 'e':				/* \e = ESC */
		case 'E':
			*r = ESC;
			break;
		case 'f':				/* \f = FF */
		case 'F':
			*r =  '\f';
			break;
		case 'n':				/* \n = LF */
		case 'N':
			*r = '\n';
			break;
		case 'r':				/* \r = CR */
		case 'R':
			*r = '\r';
			break;
		case 't':				/* \t = TAB */
		case 'T':
			*r = '\t';
			break;
		case '^':				/* \^x = CTRL-x */
			tc = islower(*(++c)) ? toupper(*c) : *c;
			if (tc >= '@' && tc <= 'Z')
				*r = tc - '@';
			else
				return(0);
			break;
		case '\\':				/* \\ = \ */
			*r = '\\';
			break;
		default:
			if (*c == '0' && (*(c+1) == 't' || *(c+1) == 'o')) {

			/*
			 * \0tnnn = decimal character value
			 * \0onnn = octal character value
			 */
			    c++;
			    b = *c++ - 'o';
			    for (i = n = 0; i < 3; c++, i++) {
				if ( ! isascii(*c))
				    return(0);
				if (b) {
				    if (*c >= '0' && *c <= '9')
					n = (n * 10) + *c - '0';
				    else
					return(0);
				} else {
				    if (*c >= '0' && *c <= '7')
					n = (n << 3) + *c - '0';
				    else
					return(0);
				}
			    }
			    *r = n;
			    return(c - s);
			} else if (isxdigit(*c)) {

			/*
			 * \[0-9a-f][0-9a-f] = hexadecimal character value
			 */
			    if ((n = AsmHex(c)) == -1)
				return(0);
			    *r = n << 4;
			    if ((n = AsmHex(c+1)) == -1)
				return(0);
			    *r += n;
			    return(c + 2 - s);
			} else
			    return(0);
		}
		return(c + 1 - s);
	}
/*
 * Without a leading `\', characters represent themselves.
 */
	*r = *c;
	return(1);
}


/*
 * AsmHex() - assemble one hex character
 */

static int
AsmHex(c)
	char *c;			/* character pointer */
{
	char tc;

	if (*c >= '0' && *c <= '9')
		return((int)*c - (int)'0');
	if (isascii(*c) && isalpha(*c)) {
		tc = isupper(*c) ? tolower(*c) : *c;
		if (tc >= 'a' && tc <= 'f')
			return(10 + tc - 'a');
	}
	return(-1);
}


/*
 * AsmPrStr() - assemble printer string
 */

static int
AsmPrStr(s, m, v)
	char **s;			/* string destination pointer */
	char *m;			/* error message value */
	char *v;			/* value to assemble */
{
	char bf[PRTRCTLLN+1];
	size_t l;

	if (CvtPrtrStr(v, bf, sizeof(bf)) == -1) {
		(void) fprintf(stderr, "%s: can't convert \"%s\" to %s string.",
			Pn, v, m);
		return(0);
	}
	if (*s) {
		(void) free(*s);
		*s = NULL;
	}
	l = strlen(v) + 1;
	if ((*s = (char *)malloc(l)) == NULL) {
		(void) fprintf(stderr, "%s: no space for %s.\n", Pn, m);
		return(0);
	}
	(void) strcpy(*s, v);
	return(1);
}


/*
 * Atof() - convert digits to floating point, recognizing either `.'
 *	    or `,' as the fractional separator
 */

int
Atof(s, d, m)
	char *s;			/* string to convert */
	double *d;			/* converted value */
	char *m;			/* conversion error message */
{
	char *cp;
	double dig, rv, sc;
	short f;

	for (cp = s, f = 0, rv = 0.0; *cp; cp++) {
		if (*cp == ' ')
			continue;
		if (*cp == '.' || *cp == ',') {
			if (f) {

Atof_error:
			    if (m)
				(void) fprintf(stderr,
				    "%s: can't convert %s: %s\n", Pn, m, s);
			    return(0);
			}
			f = 1;
			sc = 0.1;
			continue;
		}
		if (*cp < '0' || *cp > '9')
			goto Atof_error;
		dig = (double)(*cp - '0');
		if (f) {
			rv += dig * sc;
			sc *= 0.1;
		} else
			rv = (rv * 10.0) + dig;
	}
	*d = rv;
	return(1);
}


/*
 * CvtPrtrStr() - convert spacing string
 */

int
CvtPrtrStr(s, r, n)
	char *s;			/* string to convert */
	char *r;			/* result buffer */	
	int n;				/* result buffer length */
{
	int i, l;

	for (i = 0; *s && i < (n - 1); i++) {
		if ((l = Asm1Ch(s, r + i)) < 1)
			return(-1);
		s += l;
	}
	if (*s)
		return(-1);
	r[i] = '\0';
	return(i);
}


/*
 * EntPrtStr() -- enter printer control strings
 */

int
EntPrtStr(a, b)
	char *a;			/* after graph control string */
	char *b;			/* before graph control string */
{
	size_t len;

	if (AftGraph) {	
		(void) free(AftGraph);
		AftGraph = NULL;
	}
	if (BefGraph) {
		(void) free(BefGraph);
		BefGraph = NULL;
	}
	len = strlen(a);
	if ((AftGraph = (char *)malloc(len + 1)) == NULL) {
		(void) fprintf(stderr,
			"%s: no space for after graph control string\n", Pn);
		return(0);
	}
	(void) strcpy(AftGraph, a);
	len = strlen(b);
	if ((BefGraph = (char *)malloc(len + 1)) == NULL) {
		(void) fprintf(stderr,
			"%s: no space for before graph control string\n", Pn);
		return(0);
	}
	(void) strcpy(BefGraph, b);
	return(1);
}


/*
 * FindClr() - find color
 */

static short
FindClr(v)
	char *v;			/* search value */
{
	short i;

	for (i = 0; Colors[i].nm; i++) {
		if (strcmpi(Colors[i].nm, v) == 0)
			return(i);
		if (strnicmp(v, "bright", 6) == 0
		&&  v[6] != ' '
		&&  strnicmp(Colors[i].nm, "bright ", 7) == 0
		&&  strcmpi(&Colors[i].nm[7], &v[6]) == 0)
			return(i);
		if (strnicmp(v, "light", 5) == 0
		&&  v[5] != ' '
		&&  strnicmp(Colors[i].nm, "light ", 6) == 0
		&&  strcmpi(&Colors[i].nm[6], &v[5]) == 0)
			return(i);
	}
	return(-1);
}


/*
 * ReadConf() - read configuration file
 */

int
ReadConf()
{
	char buf[128], *cf, *tp, *vp;
	FILE *cfs;
	short e = 1;
	short f, len;
	struct kcmd *kp, *mp;
	short rv = 1;

	if ((cf = getenv("TOUCH2")) == NULL) {
		e = 0;
		cf = "TOUCH2.CF";
	}
	if ((cfs = fopen(cf, "rt")) == NULL) {
		if ( ! e)
			return(1);
		(void) fprintf(stderr,
			"%s: can't open configuration file \"%s\"\n", Pn, cf);
		return(0);
	}
	while (fgets(buf, sizeof(buf), cfs)) {
		if (*buf == '#' || *buf == '\n')
			continue;
		if ((tp = strchr(buf, '\n')) != NULL)
			*tp = '\0';
		if ((vp = strchr(buf, '=')) == NULL) {
			(void) fprintf(stderr,
				"%s: configuration line lacks `=': %s\n",
				Pn, buf);
			rv = 0;
			continue;
		}
		for (tp = vp - 1; tp > buf; tp--) {
			if (*tp != ' ') 
				break;
		}
		len = tp + 1 - buf;
		for (vp++; *vp; vp++) {
			if (*vp != ' ');
				break;
		}
		for (f = 0, kp = Keywd; kp->nm; kp++) {
			if (strnicmp(buf, kp->nm, len) == 0) {
				f++;
				mp = kp;
			}
		}
		if (f == 1) {
			if ((*mp->fun)(buf, vp) == 0)
				rv = 0;
		} else {
			(void) fprintf(stderr,
				"%s: %s keyword in configuraton line: %s\n",
				Pn, (!f) ? "Unknown" : "Ambiguous", buf);
			rv = 0;
		}
	}
	(void) fclose(cfs);
	if ( ! TestGlev())
		rv = 0;
	return(rv);
}


/*
 * SetAftGraph() - set after graph printer control string
 */

int
SetAftGraph(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	return(AsmPrStr(&AftGraph, "after graph printer control string", v));
}


/*
 * SetBarChar() - set graph's bar character
 */

int
SetBarChar(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	int i;

	i = Asm1Ch(v, BarCh);
	if (i < 1 || v[i] != '\0') {
		(void) fprintf(stderr, "%s: bad bar character definition: %s\n",
			Pn, b);
		return(0);
	}
	return(1);
}


/*
 * SetBaudRate() - process "baudrate" line
 */

static int
SetBaudRate(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short br, bx;

	br = (short)atoi(v);
	for (bx = 0; Baud[bx]; bx++) {
		if (br == Baud[bx]) {
			Baudx = bx;
			return(1);
		}
	}
	(void) fprintf(stderr, "%s: unknown baud rate: %s\n", Pn, b);
	return(0);
}


/*
 * SetBefGraph() - set before graph printer control string
 */

int
SetBefGraph(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	return(AsmPrStr(&BefGraph, "before graph printer control string", v));
}


/*
 * SetBkClr() - set background color
 */

static int
SetBkClr(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short i;

	if ((i = FindClr(v)) < 0) {
		(void) fprintf(stderr, "%s: unknown background color: %s\n",
			Pn, b);
		return(0);
	}
	BkClrx = i;
	return(1);
}


/*
 * SetCksum() - set checksum status
 */

static int
SetCksum(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strcmpi(v, "on") == 0) {
		Cksum = 1;
		return(1);
	} else if (strcmpi(v, "off") == 0) {
		Cksum = 0;
		return(1);
	}
	(void) fprintf(stderr, "%s: unknown checksum status: %s\n", Pn, b);
	return(0);
}


/*
 * SetComPort() - set COM port number ([12])
 */

static int
SetComPort(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short p;

	p = (short) atoi(v);
	if (p == 1 || p == 2) {
		Port = p - 1;
		return(1);
	}
	(void) fprintf(stderr, "%s: illegal port number: %s\n", Pn, b);
	return(0);
}


/*
 * SetDispWarn() - set display warning status
 */

static int
SetDispWarn(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strcmpi(v, "on") == 0) {
		Warn = 1;
		return(1);
	} else if (strcmpi(v, "off") == 0) {
		Warn = 0;
		return(1);
	}
	(void) fprintf(stderr, "%s: unknown display warning status: %s\n",
		Pn, b);
	return(0);
}


/*
 * SetDumpFmt() - set dump format, raw or QuattroPro
 */

int
SetDumpFmt(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	switch (*v) {

	case 'q':
	case 'Q':
		Qp = 1;
		break;
	case 'r':
	case 'R':
		Qp = 0;
		break;
	default:
		(void) fprintf(stderr, "%s: unknown dump type: %s\n", Pn, b);
		return(0);
	}
	return(1);
}


/*
 * SetPcDisp() - set page count display status
 */

static int
SetPcDisp(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strcmpi(v, "on") == 0) {
		PcDisp = 1;
		return(1);
	} else if (strcmpi(v, "off") == 0) {
		PcDisp = 0;
		return(1);
	}
	(void) fprintf(stderr, "%s: unknown page count status: %s\n", Pn, b);
	return(0);
}


/*
 * SetQDumpNm() - set QuattroPro dump file name
 */

static int
SetQDumpNm(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strlen(v) > DUMPFNL-1) {
		(void) fprintf(stderr,
			"%s: QuattroPro dump file name too long: %s\n",
			Pn, b);
		return(0);
	}
	(void) strcpy(DumpfnQ, v);
	return(1);
}


/*
 * SetRDumpNm() - set raw dump file name
 */

static int
SetRDumpNm(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strlen(v) > DUMPFNL-1) {
		(void) fprintf(stderr,
			"%s: raw dump file name too long: %s\n",
			Pn, b);
		return(0);
	}
	(void) strcpy(DumpfnR, v);
	return(1);
}


/*
 * SetFrameSz() - set frame size ([78] bits)
 */

static int
SetFrameSz(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short f;

	f = (short) atoi(v);
	if (f == 7 || f == 8) {
		Frame = f;
		return(1);
	}
	(void) fprintf(stderr, "%s: illegal frame size: %s\n", Pn, b);
	return(0);
}


/*
 * SetGmaxLvl() - set graph's maximum level
 */

int
SetGmaxLvl(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (Atof(v, &Gmaxl, "graph's maximum level")) {
		Lineval = 1;
		return(1);
	}
	return(0);
}

/*
 * SetGminLvl() - set graph's minimum level
 */

int
SetGminLvl(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (Atof(v, &Gminl, "graph's minimum level")) {
		Lineval = 1;
		return(1);
	}
	return(0);
}


/*
 * SetGraphNm() - set graph file name
 */

static int
SetGraphNm(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strlen(v) > DUMPFNL-1) {
		(void) fprintf(stderr, "%s: graph file name too long: %s\n",
			Pn, b);
		return(0);
	}
	(void) strcpy(Graphfn, v);
	return(1);
}


/*
 * SetGraphTtl() - set graph title line
 */

int
SetGraphTtl(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strlen(v) > GTTLLNL) {
		(void) fprintf(stderr, "%s: graph title line too long: %s\n",
			Pn, b);
		return(0);
	}
	(void) strcpy(Gttl, v);
	return(1);
}


/*
 * SetIntChar() - set graph's bar nd max/min line intersection character
 */

int
SetIntChar(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	int i;

	i = Asm1Ch(v, IntCh);
	if (i < 1 || v[i] != '\0') {
		(void) fprintf(stderr,
			"%s: bad bar and max/min line intersection character definition: %s\n",
			Pn, b);
		return(0);
	}
	return(1);
}


/*
 * SetLineChar() - set graph's max/min line character
 */

int
SetLineChar(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	int i;

	i = Asm1Ch(v, LineCh);
	if (i < 1 || v[i] != '\0') {
		(void) fprintf(stderr,
			"%s: bad max/min line character definition: %s\n",
			Pn, b);
		return(0);
	}
	return(1);
}


/*
 * SetLpp() - set graph lines per page
 */

int
SetLpp(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	int t;

	if ((t = atoi(v)) <= HDRLPP) {
		(void) fprintf(stderr,
			"%s: lines per page too small (minimum = %d)\n",
			Pn, HDRLPP+1);
		return(0);
	}
	Lpp = t;
	return(1);
}


/*
 * SetPrinter() - set printer type
 */

int
SetPrinter(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	if (strcmpi(v, "epson") == 0)
		return(EntPrtStr(AFTEPSON, BEFEPSON));
	else if (strcmpi(v, "pcl4") == 0)
		return(EntPrtStr(AFTPCL4, BEFPCL4));
	else if (strcmpi(v, "ppds") == 0)
		return(EntPrtStr(AFTPPDS, BEFPPDS));
	else {
		(void) fprintf(stderr, "%s: unidentified printer: %s\n", Pn, b);
		return(0);
	}
	return(1);
}


/*
 * SetParity() - set parity ({none,even,odd})
 */

static int
SetParity(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short px;

	for (px = 0; px < 3; px++) {
		if (strcmpi(v, Par[px].nm) == 0) {
			Parx = px;
			return(1);
		}
	}
	(void) fprintf(stderr, "%s: unknown parity: %s\n", Pn, b);
	return(0);
}


/*
 * SetPromptBk() - set prompt background color
 */

static int
SetPromptBkClr(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short i;

	if ((i = FindClr(v)) < 0) {
		(void) fprintf(stderr,
			"%s: unknown prompt background color: %s\n", Pn, b);
		return(0);
	}
	if (i > 7) {
		(void) fprintf(stderr,
			"%s: illegal prompt background color: %s\n", Pn, b);
		return(0);
	}
	PromptBkClrx = i;
	return(1);
}


/*
 * SetPromptTxtClr() - set prompt text color
 */

static int
SetPromptTxtClr(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short i;

	if ((i = FindClr(v)) < 0) {
		(void) fprintf(stderr, "%s: unknown prompt text color: %s\n",
			Pn, b);
		return(0);
	}
	PromptTxtClrx = i;
	return(1);
}


/*
 * SetSSBits() - set start/stop bits ([12])
 */

static int
SetSSBits(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short s;

	s = (short) atoi(v);
	if (s == 1 || s == 2) {
		StSt = s;
		return(1);
	}
	(void) fprintf(stderr, "%s: illegal start/stop bit count: %s\n",
		Pn, b);
	return(0);
}


/*
 * SetTxtClr() - set text color
 */

static int
SetTxtClr(b, v)
	char *b;			/* configuration line buffer */
	char *v;			/* start of value string */
{
	short i;

	if ((i = FindClr(v)) < 0) {
		(void) fprintf(stderr, "%s: unknown text color: %s\n", Pn, b);
		return(0);
	}
	TxtClrx = i;
	return(1);
}


/*
 * TestGlev() - test graph level line values
 */

int
TestGlev()

{
	if (Gminl && Gmaxl && Gminl >= Gmaxl) {
		(void) fprintf(stderr,
			"%s: graph minimum level (%d) >= maximum (%d)\n",
			Pn, Gminl, Gmaxl);
		return(0);
	}
	return(1);
}
