/*
 * dump.c - dump the meter's memory
 *
 * V. Abell
 */

#include "touch2.h"
#include <stdlib.h>
#include <string.h>
#include "asyncpec.h"
#include <sys/types.h>
#include <sys/stat.h>

#define DUMPNL	256			/* dump number of lines */
#define FIELDLN	16			/* dump field length */
#define	RDYCOL	31			/* ready message column */
#define	SMROW	12			/* status message row */

char CksumErrMsg[32];			/* checksum error message */
char Date[FIELDLN];			/* dump line date */
char Dow[FIELDLN];			/* dump line day of week */
int DumpHs = 0;				/* dump header status */
int DumpLc = 0;				/* number of dump lines */
char DumpfnQ[DUMPFNL];			/* QuattroPro dump file name */
char DumpfnR[DUMPFNL];			/* raw dump file name */
FILE *Dumpfs = NULL;			/* write dump file stream */
char DumpLine[DUMPLL+1];		/* dump line */
char *DumpLp[DUMPNL];			/* dump line pointers */
char Graphfn[DUMPFNL];			/* graph file name */
FILE *Graphfs = NULL;			/* graph file stream */
FILE *Rdumpfs = NULL;			/* read dump file stream */
short Rtype;				/* dump line reading type */
char Time[FIELDLN];			/* dump line time */


struct menu DumpErr[] = {
	{ 12, 20, DumpLine },
	{  0,  0, NULL },
};

struct menu DumpRdy[] = {
	{ 12, RDYCOL, "Waiting on meter" },
	{  0,  0, NULL },
};

struct menu DumpSt[] = {
	{ 12, 27, "Waiting for meter to start dump" },
	{  0,  0, NULL },
};

#define	TYPEBUFL	64
char CheckDump[TYPEBUFL];
char DumpFile[TYPEBUFL];
char DumpScreen[TYPEBUFL];
char EraseBuf[TYPEBUFL];
char FormatBuf[TYPEBUFL];
char GraphBuf[TYPEBUFL];
char GraphTtl[GTTLLNL+1];
char ReadBuf[TYPEBUFL];

struct menu DumpType[] = {
	{  2, 10, CheckDump },
#define	CKDMPLN	0
	{  4, 10, EraseBuf },
	{  6, 10, DumpFile },
#define FILLN	2
	{  8, 10, GraphBuf },
#define	GRAPHLN	3
	{ 10, 10, FormatBuf },
#define	FMTLN	4
	{ 12, 10, ReadBuf },
#define READLN	5
	{ 14, 10, DumpScreen },
#define SCRLN	6
	{ 16, 10, "T - change graph title from:" },
	{ 17, 14,  GraphTtl},
	{ 19, 10, "X - eXit"},
	{  0,  0, NULL },
};

struct menu GraphType[] = {
	{ 12, 30, "F - write graph to file" },
	{ 14, 30, "S - draw graph on screen" },
	{  0,  0, NULL },
};

static int CheckCksum(int len);
static void DispDumpLns();
static void EraseDump();

#if	defined(UNIX)
static int GetDispLn(char *s, int n);
#else
static int GetDispLn(char *s, short n);
#endif

static int GetDumpType(char *ty);
static void RcvErr(int err);
static void ReadDump();
static int StoreDumpLn();
static void TrimSpaces(char *s);
static void WarnCantOpen(char *f);

#ifdef	UNIX
static int WarnCksum(int f);
#else
static int WarnCksum(short f);
#endif

static int WarnDumpHdr();
static void WarnDumpLen();
static int WarnExist(char *d, struct stat *s);
static void WrDumpF();


/*
 * CheckCksum() - check checksum
 */


static int
CheckCksum(len)
	int len;			/* length of line in DumpLine */
{
	char cksum[5];
	short i;
	unsigned int sum;

	if (Cksum) {
		if (len < 5) {
			(void) strcpy(CksumErrMsg, "Record is too short.");
			return(1);
		}
	/*
	 * A leading 'P' has been stripped from all dump header records,
	 * and a leading "P " has been stripped from dump file records.
	 * They need to be considered in computing the checksum.
	 */
		if (DumpLc == 0)
			sum = (unsigned int) 'P';
		else {
			if (DumpLine[0] == 'P')
				sum = 0;
			else 
				sum = (unsigned int) 'P' + (unsigned int) ' ';
		}
		for (i = 0; i < (len - 5); i++)
			sum += (unsigned int) DumpLine[i];
		(void) sprintf(cksum, "%04x", sum);
		if (strcmpi(&DumpLine[len - 4], cksum) == 0)
			return(0);
		for (i = 0; cksum[i]; i++) {
			if (isalpha(cksum[i]) && islower(cksum[i]))
				cksum[i] = toupper(cksum[i]);
		}
		(void) sprintf(CksumErrMsg, "Computed %s; read %s.",
			cksum, &DumpLine[len - 4]);
		return(1);
	}
	return(0);
}


/*
 * CheckESC() - check for ESC
 */

int
CheckESC()
{
	int ch;

	if (kbhit()) {
		if ((ch = getch()) == 0)
			ch = getch();
		if (ch == ESC)
			return(1);
		putch(BELL);
	}
	return(0);
}


/*
 * DumpMtr() - dump the meter's memory
 */

void
DumpMtr()
{
	int err, i, len;
	char dt, msg[32];

	if (GetDumpType(&dt) == 0)
		return;
	if ( ! DumpLc && dt != 'r') {
	    for (;;) {
		if (WaitRdy() == 0) {
			CloseDump();
			return;
		}
		DispMenu(DumpSt, NULL);
		if (WaitCmd("DMP", 'P') == 0) {
			(void) strcpy(DumpLine, "Dump command failed.");
			DispMenu(DumpErr,
				"Press ESC to exit; any other key to retry.");
			if ((char)WaitAnyKey() == ESC) {
				CloseDump();
				return;
			}
			continue;
		}
		_clearscreen(_GCLEARSCREEN);
		PromptMsg("Press ESC to exit.");
		len = GetDataLn(DumpLine, DUMPLL);
		if (CheckCksum(len)) {
			(void) WarnCksum(2);
			CloseDump();
			(void) EraseDump();
			return;
		}
		if (StoreDumpLn()) {
			CloseDump();
			(void) EraseDump();
			return;
		}
		if ( ! ParseHdr()) {
			if ((char)WarnDumpHdr() == ESC) {
				CloseDump();
				(void) EraseDump();
				return;
			}
		}
		_settextposition(SMROW, RDYCOL);
		_outtext("Header record read.");
		for (i = 0;;) {
			if (CheckESC()) {
				CloseDump();
				(void) EraseDump();
				return;
			}
			len = GetDataLn(DumpLine, DUMPLL);
			if (DumpLine[0] != 'P')
				break;
			if (CheckCksum(len)) {
				(void) WarnCksum(2);
				CloseDump();
				(void) EraseDump();
				return;
			}
			if (StoreDumpLn()) {
				CloseDump();
				(void) EraseDump();
				return;
			}
			if (i == 0) {
				ClearRow(SMROW, RDYCOL);
				(void) sprintf(msg,
					"Record %3d read.", DumpLc - 1);
				_settextposition(SMROW, RDYCOL);
				i = 1;
			} else {
				_settextposition(SMROW, RDYCOL+7);
				(void) sprintf(msg, "%3d", DumpLc - 1);
			}
			_outtext(msg);
		}
		if (Ndump+1 != DumpLc)
			WarnDumpLen();
		break;
	    }
	}
	switch (dt) {
	case 'f':
		WrDumpF();
		break;
	case 'g':
	case 'G':
		DrawGraph((dt == 'g') ? 0 : 1);
		break;
	case 'r':
		ReadDump();
		break;
	case 's':
		DispDumpLns();
	}
}


/*
 * DispDumpLns() - display dump lines
 */

static void
DispDumpLns()
{
	char *bf, bot[80], *cp;
	int cpl, i, j, k, l, lps;

	if (DumpLc == 0)
		return;
	cpl = Vc.numtextcols;
	lps = Vc.numtextrows;
	if ((bf = (char *)malloc((size_t)(cpl + 1))) == NULL) {
		_clearscreen(_GCLEARSCREEN);
		_settextposition(12, 20);
		_outtext("DispDump: no temporary line space");
		return;
	}

	i = 0;
	for (;;) {

	/*
	 * Display a screenload.
	 */
		_clearscreen(_GCLEARSCREEN);
		for (j = 1; j < lps && (i+j-1) < DumpLc;  j++) {
			for (cp = DumpLp[i+j-1], k = 0; k < cpl; cp++) {
				if (*cp == '"')
					continue;
				if (*cp == '\0')
					break;
				bf[k++] = *cp;
			}
			bf[k] = '\0';
			_settextposition(j, 1);
			_outtext(bf);
		}
		(void) sprintf(bot,
		  "(%d of %d) Press ESC or X to exit; Page Up/Down; Arrow Up/Down.",
		  i+1, DumpLc);
		PromptMsg(bot);
	/*
	 * Wait for a command.
	 */
		for (k = 1; k;) {
			switch((char)WaitAnyKey()) {
			case ESC:
			case 'x':
			case 'X':
				return;
			case PGDN:
				if ((i + lps - 1) < DumpLc) {
					i += lps - 1;
					k = 0;
				} else
					putch(BELL);
				break;
			case PGUP:
				if (i > 0) {
					i -= lps - 1;
					if (i < 0)
						i = 0;
					k = 0;
				} else
					putch(BELL);
				break;
			case UARW:
				if (i > 0) {
					i--;
					k = 0;
				} else
					putch(BELL);
				break;
			case DARW:
				if (i < (DumpLc - 1)) {
					i++;
					k = 0;
					break;
				}
				/* fall through */
			default:
				putch(BELL);
			}
		}
	}
} 


/*
 * EraseDump() -- erase dump
 */

static void
EraseDump()
{
	int i;

	for (i = 0; i < DumpLc; i++) {
		(void) free(DumpLp[i]);
		DumpLp[i] = NULL;
	}
	DumpLc = Ndump = 0;
}


/*
 * GetDataLn(s, n) - get meter data line
 */

int
GetDataLn(s, n)
	char *s;
	int n;
{
	char c, msg[128];
	int err, i;

	for (i = 0;;) {
		if (CheckESC()) {
			s[0] = '\0';
			return(0);
		}
		(void) check_com(&c, &err);
		if ( ! err && c) {
			if (c == CR)
				break;
			if (c == LF)
				continue;
			if (i+1 >= n)
				break;
			s[i++] = c;
		} else if (err == 6)
			continue;
		else if (err)
			RcvErr(err);
	}
	s[i] = '\0';
	return(i);
}


/*
 * GetDumpType() - get dump type
 */

static int
GetDumpType(ty)
	char *ty;			/* type response */
{
	char ch, fn[DUMPFNL];
	short i, m;
	struct stat sbuf;

	Dumpfs = Graphfs = NULL;
	for (m = 0;;) {
		if ( ! m) {
		    (void) sprintf(FormatBuf,
			"Q - change dump format to %s from %s",
			Qp ? "raw" : "QuattroPro",
			Qp ? "QuattroPro" : "raw");
		    if (Qp)
			(void) sprintf(CheckDump,
		    	    "C - %sdump check and high readings",
			    Ckdump ? "" : "don't ");
		    else
			CheckDump[0] = '\0';
		    if (DumpLc) {
			(void) sprintf(EraseBuf,
				"E - erase %d records from dump buffer",
				DumpLc);
			(void) sprintf(DumpFile,
				"F - dump %d records to file", DumpLc);
			(void) sprintf(GraphBuf,
				"G - graph %d records", DumpLc);
			ReadBuf[0] = '\0';
			(void) sprintf(DumpScreen,
				"S - dump %d records to screen", DumpLc);
		    } else {
			EraseBuf[0] = '\0';
			(void) strcpy(DumpFile, "F - dump to file");
			(void) strcpy(GraphBuf, "G - graph");
			(void) strcpy(ReadBuf, "R - read from file");
			(void) strcpy(DumpScreen, "S - dump to screen");
		    }
		    if (Gttl[0] != '\0')
			(void) sprintf(GraphTtl, "\"%s\"", Gttl);
		    else
			GraphTtl[0] = '\0';
		    DispMenu(DumpType, NULL);
		    m = 1;
		}
		switch((char)WaitAnyKey()) {

		case 'x':
		case 'X':
		case ESC:
			return(0);
		case 'c':
		case 'C':
			Ckdump = Ckdump ? 0 : 1;
			m = 0;
			break;
		case 'e':
		case 'E':
			if ( ! DumpLc)
				putch(BELL);
			else {
				(void) EraseDump();
				m = 0;
			}
			break;
		case 'f':
		case 'F':
			(void) strcpy(fn, Qp ? DumpfnQ : DumpfnR);
			if (GetInp(21, 5, "Name?", fn, fn, sizeof(fn)) == 0)
				return(0);
			if (stat(fn, &sbuf) == 0) {
				if (WarnExist(fn, &sbuf) == 0)
					return(0);
			}
			if ((Dumpfs = fopen(fn, "w+t")) != NULL) {
				(void) strcpy(Qp ? DumpfnQ : DumpfnR, fn);
				*ty = 'f';
				return(1);
			}
			WarnCantOpen(fn);
			m = 0;
			break;
		case 'g':
		case 'G':
			DispMenu(GraphType, NULL);
			if ((ch = (char)WaitAnyKey()) == ESC)
				return(0);
			switch (ch) {
			case 'f':
			case 'F':
				(void) strcpy(fn, Graphfn);
				if (GetInp(21, 5, "Name?", fn, fn, sizeof(fn))
				== 0)
					return(0);
				if (stat(fn, &sbuf) == 0) {
					if (WarnExist(fn, &sbuf) == 0)
						return(0);
				}
				if ((Graphfs = fopen(fn, "w+t")) != NULL) {
					(void) strcpy(Graphfn, fn);
					*ty = 'G';
					return(1);
				}
				WarnCantOpen(fn);
				break;
			case 's':
			case 'S':
				*ty = 'g';
				return(1);
			default:
				putch(BELL);
			}
			m = 0;
			break;
		case 'q':
		case 'Q':
			Qp = (Qp == 1) ? 0 : 1;
			m = 0;
			break;
		case 'r':
		case 'R':
			(void) strcpy(fn, DumpfnR);
			if (GetInp(21, 5, "Name?", fn, fn, sizeof(fn)) == 0)
				return(0);
			if ((Rdumpfs = fopen(fn, "r+t")) != NULL) {
				(void) strcpy(DumpfnR, fn);
				*ty = 'r';
				return(1);
			}
			WarnCantOpen(fn);
			m = 0;
			break;
		case 's':
		case 'S':
			*ty = 's';
			return(1);
		case 't':
		case 'T':
			if (GetInp(21, 7, "Title?", Gttl, GraphTtl, GTTLLNL)
			== 0)
				return(0);
			(void) strcpy(Gttl, GraphTtl);
			m = 0;
			break;
		default:
			putch(BELL);
		}
	}
}


/*
 * InitDump() - initialize dump
 */

void
InitDump()
{
	(void) strcpy(DumpfnQ, DEFQDMPF);
	(void) strcpy(DumpfnR, DEFRDMPF);
	(void) strcpy(Graphfn, DEFGRAPHF);
}



/*
 * GetDispLn(s) - get meter display line
 */

static int
GetDispLn(s, n)
	char *s;			/* destination buffer */

#if	defined(UNIX)
	int n;				/* destination buffer length */
#else
	short n;			/* destination buffer length */
#endif

{
	char c;
	int err, i;

	for (i = 0;;) {
		if (CheckESC()) {
			s[0] = '\0';
			return(0);
		}
		(void) check_com(&c, &err);
		if ( ! err && c) {
			if (c == CR) {
				s[i] = '\0';
				return(1);
			}
			if (i < n - 1)
				s[i++] = c;
			else {
				s[i] = '\0';
				return(0);
			}
		} else if (err == 6)
			continue;
		else if (err) {
			RcvErr(err);
			return(0);
		}
	}
}


/*
 * Rcverr() - handle receive error
 */

static void
RcvErr(err)
	int err;
{
	char msg[64];

	(void) sprintf(msg, "Meter receive error %d", err);
	_clearscreen(_GCLEARSCREEN);
	_settextposition(12, 20);
	_outtext(msg);
}


/*
 * ReadDump() - read dump from file
 */

static void
ReadDump()
{
	char *cp, *rv;
	int i, len;
	short err = 0;

/*
 * Read, parse and store the dump header line.
 */
	if ((rv = fgets(DumpLine, DUMPLL, Rdumpfs)) != NULL) {
		if ((cp = strrchr(DumpLine, '\n')) != NULL) {
			*cp = '\0';
			len = cp - DumpLine;
		} else {
			DumpLine[DUMPLL-1] = '\0';
			len = strlen(DumpLine);
		}
		if (CheckCksum(len)) {
			if ((char)WarnCksum(1) == ESC) {
				CloseDump();
				(void) EraseDump();
				return;
			}
		}
		if (StoreDumpLn()) {
			CloseDump();
			(void) EraseDump();
			return;
		}
	} else {
		DumpLine[0] = '\0';
		len = 0;
	}
	if (rv == NULL || ! ParseHdr()) {

	/*
	 * Handle bad dump header line.
	 */
		i = WarnDumpHdr();
		if (DumpLc)
			(void) EraseDump();
		if ((char)i == ESC) {
			CloseDump();
			return;
		}
		err = 1;
	}
/*
 * Read and store the dump lines.
 */
	while (fgets(DumpLine, DUMPLL, Rdumpfs) != NULL) {
		if ((cp = strrchr(DumpLine, '\n')) != NULL) {
			*cp = '\0';
			len = cp - DumpLine;
		} else {
			DumpLine[DUMPLL-1] = '\0';
			len = strlen(DumpLine);
		}
		if (CheckCksum(len)) {
			if ((char)WarnCksum(1) == ESC) {
				CloseDump();
				(void) EraseDump();
				return;
			}
		}
		if (StoreDumpLn()) {
			err = 1;
			break;
		}
	}
	CloseDump();
	if (err == 0 && (Ndump+1 != DumpLc))
		WarnDumpLen();
}


/*
 * StoreDumpLn() - store dump line via DumpLp[]
 */

static int
StoreDumpLn()
{
	char *cp;
	int i;
	size_t l;
	char msg[128];

	if (DumpLc >= DUMPNL) {
		(void) sprintf(msg, "Dump buffer is full (limit = %d).",
			DUMPNL);
store_warn:
		i = WarnMsg(12, (short)(((Vc.numtextcols - strlen(msg))/2)+1),
		    msg, 14, (short)(((Vc.numtextcols - strlen(DumpLine))/2)+1),
		    DumpLine, 1);
		return(((char)i == ESC) ? 1 : 0);
	}
	cp = DumpLine;
	if (*cp == 'P') {
		cp++;
		if (*cp == ' ')
			cp++;
	}
	l = (size_t)(strlen(cp) + 1);
	if ((DumpLp[DumpLc] = (char *)malloc(l)) == NULL) {
		(void) sprintf(msg, "No memory for dump line number %d.",
			DumpLc);
		goto store_warn;
	}
	(void) strcpy(DumpLp[DumpLc], cp);
	DumpLc++;
	return(0);
}


/*
 * WaitRdy() - wait for the meter to become ready
 */

int
WaitRdy()
{
	char buf[16];
	int r;

	DispMenu(DumpRdy, NULL);
	OpenCom();
	reset_buffer();
	for (;;) {
		if (CheckESC())
			return(0);
		if (buffer_length == 0)
			continue;
		if (GetDispLn(buf, sizeof(buf)) == 0)
			return(0);
		if (strcmp(buf, "INSERT")  == 0		/* ENGL */
		||  strcmp(buf, "INSERT.") == 0		/* ESPAN */
		||  strcmp(buf, "INSER. ") == 0		/* FRANC and ITALI */
		||  strcmp(buf, "IN    ")  == 0		/* NEDER */
		||  strcmp(buf, "INSIRA")  == 0		/* PORT */
		||  strcmp(buf, "SATTIN")  == 0		/* SVENS */
		||  strcmp(buf, "EINLEG")  == 0		/* DEUTS */
		||  strcmp(buf, ")))   ")  == 0)	/* SYMB */
			break;
	}
	return(GetDispLn(buf, sizeof(buf)));
}


/*
 * TrimSpaces() - trim leading and trailing spaces from string
 */

static void
TrimSpaces(s)
	char *s;			/* string address */
{
	char *s1, *s2;
 
	for (s1 = s2 = s; *s1; s1++) {
		if (s2 > s || *s1 != ' ')
			*s2++ = *s1;
	}
	while (s2 > s) {
		if (*(s2 - 1) != ' ')
			break;
		s2--;
	}
	*s2 = '\0';
}


/*
 * WarnCantOpen() - issue can't open file warning
 */

static void
WarnCantOpen(f)
	char *f;			/* file name */
{
	char msg[80];

	(void) sprintf(msg, "Can't open %s", f);
	(void) WarnMsg(12, 1, msg, 0, 0, NULL, 0);
}


/*
 * WarnCksum()
 */

static int
WarnCksum(f)

#ifdef	UNIX
	int f;			/* prompt flag for WarnMsg() */
#else
	short f;		/* prompt flag for WarnMsg() */
#endif

{
	char m1[80], m2[80];

	(void) strcpy(m1, "Checksum doesn't match for dump ");
	if (DumpLc)
		(void) sprintf(m2, "record %d.  ", DumpLc);
	else
		(void) strcpy(m2, "header record.  ");
	(void) strcat(m1, m2);
	(void) strcat(m1, CksumErrMsg);
	return(WarnMsg(12, (short)(((Vc.numtextcols - strlen(m1))/2)+1), m1,
		14, (short)(((Vc.numtextcols - strlen(DumpLine))/2)+1),
		DumpLine, f));
}


/*
 * WarnDumpHdr()
 */

static int
WarnDumpHdr()
{
	return(WarnMsg(12, 30, "Bad dump header line:", 14,
		(short)(((Vc.numtextcols - strlen(DumpLine))/2)+1),
		DumpLine, 1));
}


/*
 * WarnDumpLen() - warn about incorrect dump length
 */

static void
WarnDumpLen()
{
	char msg[80];

	(void) sprintf(msg,
		"Dump length incorrect: %d records expected, %d received.",
		Ndump, DumpLc - 1);
	(void) WarnMsg(12, 11, msg, 0, 0, NULL, 0);
}


/*
 * WarnExist() - warn that the dump file exists
 */

static int
WarnExist(d, s)
	char *d;			/* dump file name */
	struct stat *s;			/* stat buffer */
{
	char msg[80];

	_clearscreen(_GCLEARSCREEN);
	(void) sprintf(msg, "%s exists.", d);
	_settextposition(11, (short)((Vc.numtextcols - strlen(msg))/2 + 1));
	_outtext(msg);
	(void) sprintf(msg, "It is %ld bytes long.", s->st_size);
	_settextposition(13, (short)((Vc.numtextcols - strlen(msg))/2 + 1));
	_outtext(msg);
	PromptMsg("Press ESC to exit; any other key to overwrite.");
	if ((char)WaitAnyKey() == ESC)	
		return(0);
	return(1);
}


/*
 * WarnMsg() - issue warning message
 */

int
WarnMsg(r1, c1, m1, r2, c2, m2, f)

#if	defined(UNIX)
	int r1;				/* first message row */
	int c1;				/* first message column */
	char *m1;			/* first message */
	int r2;				/* second message row */
	int c2;				/* second message column */
	char *m2;			/* second message */
	int f;				/* prompt flag: 0 = any continue
					 *		1 = ESC abort
					 *		2 = any abort */
#else
	short r1;			/* first message row */
	short c1;			/* first message column */
	char *m1;			/* first message */
	short r2;			/* second message row */
	short c2;			/* second message column */
	char *m2;			/* second message */
	short f;			/* prompt flag: 0 = any continue
					 *		1 = ESC abort
					 *		2 = any abort */
#endif

{
	char *p;

	_clearscreen(_GCLEARSCREEN);
	_settextposition(r1, c1);
	_outtext(m1);
	if (m2) {
		_settextposition(r2, c2);
		_outtext(m2);
	}
	switch (f) {
	case 0:
		p = "Press any key to continue.";
		break;
	case 1:
		p = "Press ESC to exit; any other key to continue.";
		break;
	default:
		p = "Press any key to abort.";
	}
	PromptMsg(p);
	return(WaitAnyKey());
}


/*
 * WrDumpF() -- write dump lines to dump file
 */

static void
WrDumpF()
{

#if	defined(UNIX)
	int i;
#else
	short i;
#endif

	if (Qp) {
		for (i = 1; i < DumpLc; i++) {
			if (ParseDumpLn(i, 0) == 0)
				break;
			(void) TrimSpaces(Date);
			if (Ckdump == 0 && Rtype != RDREG)
				continue;
			(void) TrimSpaces(Time);
			(void) fprintf(Dumpfs, "%s,%s,%6.2f\n", Date, Time,
				Rval);
		}
	} else {
		for (i = 0; i < DumpLc; i++)
			(void) fprintf(Dumpfs, "%s\n", DumpLp[i]);
	}
	(void) fclose(Dumpfs);
	Dumpfs = NULL;
}
