/*
 * touch2.c - main program for Lifescan One Touch(R) II glucose meter
 * communication interface
 *
 * V. Abell
 */

#include "touch2.h"
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include "asyncpec.h"
#include <sys/types.h>
#include <sys/stat.h>

char Version[] = "Version 1.8, June 20, 1993";

struct menu ZeroFail[] = {
	{ 12, 14, "The command to zero the datalog may have failed." },
	{  0,  0, NULL },
};

struct menu ZeroMenu[] = {
	{  4, 29, "Zero the meter data log." },
	{  8, 26, "Are you sure you want to do this?" },
	{ 12, 23, "Please press Y (upper case) to confirm." },
	{ 13, 20, "Press any other key to exit without clearing." },
	{  0,  0, NULL },
};

struct menu MainMenu[] = {
	{  2, 13, "Lifescan One Touch(R) II Meter Communications Interface" },
	{  3, 30, Version },
#define	VERSLN	1
	{  9, 18, "C - inspect/change Communications parameters" },
	{ 10, 18, "D - Dump memory" },
	{ 11, 18, "S - inspect/display meter Status" },
	{ 12, 18, "T - read/set date and Time" },
	{ 13, 18, "X - eXit" },
	{ 14, 18, "Z - Zero the meter datalog" },
	{ 20,  8, "Copyright (C) Victor A. Abell, 1992, 1993.  All rights reserved."},
	{ 21, 16, "1715 Summit Dr., W. Lafayette, IN (317) 463-6865" },
	{  0,  0, NULL },
};

char BarCh[] = { BARCH, '\0'};		/* bar graph character */
short BkClrx =  DEFBCX;			/* background color */
short Ckdump = 0;			/* check dump status */
short Cksum = 1;			/* checksum checking status */
char IntCh[] = { INTCH, '\0' };		/* bar and min/max intersection
					 * character */
char LineCh[] = { LINECH, '\0'};	/* min/max level line character */
short PcDisp = 1;			/* page count display state */
char *Pn;				/* program name */
long PrevBkClr = BLACK;			/* previous background color */
short PrevTxtClr = WHITE;       	/* previous text color */
short Qp = 0;				/* QuattroPro output mode */
short TxtClrx = DEFTCX;			/* text color */
short Warn = 1;				/* display warning status */

static void InitClr();
static void ParseArgs(int argc, char *argv[], int err);
static void SigHandle();
static void Usage();
static void ZeroLog();


extern char *optarg;
extern int optind;


/*
 * main() - main program
 */

main(argc, argv )
	int argc;			/* argument count */
	char *argv[];			/* argument pointers */
{
	int err, i;

/*
 * Save program name.
 */
	if ((Pn = strrchr(argv[0], '/')) != NULL)
		Pn++;
	else {
#if	!defined(UNIX)
		if ((Pn = strchr(argv[0], '\\')) != NULL)
			Pn++;
		else
#endif
		Pn = argv[0];
	}

	InitDump();
	err = ! ReadConf();
	(void) ParseArgs(argc, argv, err);
	InitMenu();
	InitClr();
	InitStatus();
	(void) signal(SIGINT, SigHandle);
	MainMenu[VERSLN].col = (short)(((Vc.numtextcols-strlen(Version))/2)+1);
	DispMenu(MainMenu, NULL);
	for (;;) {
		if (kbhit()) {
			i = getch();
			switch(i) {

			case 'c':
			case 'C':
				DefCom();
				break;
			case 'd':
			case 'D':
				DumpMtr();
				break;
			case 's':
			case 'S':
				StatusMtr();
				break;
			case 't':
			case 'T':
				DateTime();
				break;
			case ESC:
			case 'x':
			case 'X':
				TouchExit(0);
			case 'z':
			case 'Z':
				ZeroLog();
				break;
			default:
				if (i == 0)
					i = getch();
				putch(BELL);
			}
			DispMenu(MainMenu, NULL);
		}
		reset_buffer();
	}
}


/*
 * CloseDump() - close dump files (abnormally)
 */

void
CloseDump()
{
	struct stat sbuf;

	if (Dumpfs != NULL) {
		(void) fclose(Dumpfs);
		Dumpfs = NULL;
		if (stat(Qp ? DumpfnQ : DumpfnR, &sbuf) == 0
		&& sbuf.st_size == 0)
			(void) unlink(Qp ? DumpfnQ : DumpfnR);
	}
	if (Graphfs != NULL) {
		(void) fclose(Graphfs);
		Graphfs = NULL;
		if (stat(Graphfn, &sbuf) == 0 && sbuf.st_size == 0)
			(void) unlink(Graphfn);
	}
	if (Rdumpfs != NULL) {
		(void) fclose(Rdumpfs);
		Rdumpfs = NULL;
	}
}


/*
 * InitClr() - initialize colors
 */

static void
InitClr()
{
	char buf[80];

	PrevTxtClr = _gettextcolor();
	_settextcolor(TxtClrx);
	PrevBkClr = _getbkcolor();
	_setbkcolor(BkClrx & 7);
	if (Vc.mode != _TEXTC80 && Warn) {
	    _clearscreen(_GCLEARSCREEN);
	    PromptMsg("Press ESC to exit; any other key to continue.");
	    _settextposition(11, 15);
	    _outtext("Warning: the display cannot be put in TEXTC80 mode.");
	    _settextposition(13, 15);
	    _outtext("This means that display operations, including color");
	    _settextposition(14, 15);
	    _outtext("selections, may be incorrect.");
	    if ((char)WaitAnyKey() == ESC)
	    	TouchExit(1);
	    _clearscreen(_GCLEARSCREEN);
	}
	if (Vc.mode == _TEXTC80 && BkClrx > 7 && Warn) {
	    _clearscreen(_GCLEARSCREEN);
	    PromptMsg("Press ESC to exit; any other key to continue.");
	    _settextposition(11, 15);
	    _outtext("Warning: for menus the display is in TEXTC80 mode.");
	    _settextposition(13, 15);
	    _outtext("Since TEXTC80 is limited to 8 background colors, the");
	    _settextposition(14, 15);
	    (void) sprintf(buf,
		"background color \"%s\" will be displayed as",
		Colors[BkClrx].nm);
	    _outtext(buf);
	    _settextposition(15, 15);
	    (void) sprintf(buf,
		"\"%s\" on menus; \"%s\" will be used only on",
		Colors[BkClrx & 7].nm, Colors[BkClrx].nm);
	    _outtext(buf);
	    _settextposition(16, 15);
	    _outtext("graph output.");
	    if ((char)WaitAnyKey() == ESC)
	    	TouchExit(1);
	    _clearscreen(_GCLEARSCREEN);
	}
}


/*
 * ParseArgs() - parse command line arguments
 */

static void
ParseArgs(argc, argv, err)
	int argc;				/* argument count */
	char *argv[];				/* argument pointers */
	int err;				/* configuration file error
						 * status */
{
	int c;
	int hlp = 0;

/*
 * Process command line arguments.
 */
	while ((c = getopt(argc, argv, "a:b:B:cCg:G:hi:l:L:p:Pqrt:w")) != EOF) {
		switch(c) {

		case 'a':	/* -a s -- set after graph printer control */
			if ( ! SetAftGraph(optarg, optarg))
				err = 1;
			break;
		case 'b':	/* -b char -- define bar graph character */
			if ( ! SetBarChar(optarg, optarg))
				err = 1;
			break;
		case 'c':	/* -c -- dump check and high readings */
			Ckdump = 1;
			break;
		case 'C':	/* -C -- don't check checksums */
			Cksum = 0;
			break;
		case 'B':	/* -B s -- set before graph printer control */
			if ( ! SetBefGraph(optarg, optarg))
				err = 1;
			break;
		case 'g':	/* -g min -- set graph minimum level value */
			if ( ! SetGmaxLvl(optarg, optarg))
				err = 1;
			break;
		case 'G':	/* -G max -- set graph maximum level value */
			if ( ! SetGminLvl(optarg, optarg))
				err = 1;
			break;
		case 'h':	/* -h -- display help */
			hlp = 1;
			break;
		case 'i':	/* -i char -- define bar and max/min
				 *	      intersection character */
			if ( ! SetIntChar(optarg, optarg))
				err = 1;
			break;
		case 'l':	/* -l char -- define max/min line character */
			if ( ! SetLineChar(optarg, optarg))
				err = 1;
			break;
		case 'L':	/* -L num -- define graph lines per page */
			if ( ! SetLpp(optarg, optarg))
				err = 1;
			break;
		case 'p':	/* -p name -- define printer */
			if ( ! SetPrinter(optarg, optarg))
				err = 1;
			break;
		case 'P':	/* -P -- turn off page count display */
			PcDisp = 0;
			break;
		case 'r':	/* -r -- turn off raw dump format */
			Qp = 1;
			break;
		case 't':	/* -t ttl -- set graph title line */
			if ( ! SetGraphTtl(optarg, optarg))
				err = 1;
			break;
		case 'w':	/* disable display warnings */
			Warn = 0;
			break;
		case '?':
			err = 1;
		}
	}
	if (AftGraph == NULL) {
		if ( ! EntPrtStr(AFTEPSON, BEFEPSON))
			err = 1;
	}
	if ( ! TestGlev())
		err = 1;
	if (err || hlp) {
		Usage();
		if (err)
			exit(1);
		exit(0);
	}
}


/*
 * SigHandle() - handle signal
 */

static void
SigHandle()
{
	TouchExit(1);
}


/*
 * TouchExit(xv) - exit
 */

void
TouchExit(xv)
	int xv;					/* exit value */
{
	CloseCom();
	CloseDump();
	_displaycursor(_GCURSORON);
	_settextcursor(Cursor);
	_setbkcolor(PrevBkClr);
	_settextcolor(PrevTxtClr);
	_settextposition(Vc.numtextrows, Vc.numtextcols);
	_outtext("\n");
	if (Mode >= 0)
		_setvideomode(Mode);
	exit(xv);
}


/*
 * Usage() -- display usage
 */

static void
Usage()
{
	(void) fprintf(stderr,
	    "%s, v%s\n", Pn, Version+1);
	(void) fprintf(stderr,
	    "usage: [-a string] [-b char] [-B string] [-cChPrw] [-g min]");
	(void) fprintf(stderr, " [-G max]\n");
	(void) fprintf(stderr,
	  "       [-i char] [-l char] [-L num] [-p name] [-t ttl]\n");
	(void) fprintf(stderr,
	    "\t-a string  set <string> as after graph printer control\n");
	(void) fprintf(stderr,
	    "\t-b char    use <char> for bar line\n");
	(void) fprintf(stderr,
	    "\t-B string  set <string> as before graph printer control\n");
	(void) fprintf(stderr,
	    "\t-c         dump check and high readings\n");
	(void) fprintf(stderr,
	    "\t-C         turn off checksum verification\n");
	(void) fprintf(stderr,
	    "\t-g min     set graph minimum level to <min>\n");
	(void) fprintf(stderr,
	    "\t           (mg/dl default = %.1f, mmol/l default = %.1f)\n",
	    DEFMGMIN, DEFMMMIN);
	(void) fprintf(stderr,
	    "\t-G max     set graph maximum level to <max>\n");
	(void) fprintf(stderr,
	    "\t           (mg/dl default = %.1f, mmol/l default = %.1f)\n",
		DEFMGMAX, DEFMMMAX);
	(void) fprintf(stderr,
	    "\t-h         display help (usage)\n");
	(void) fprintf(stderr,
	    "\t-i char    use <char> for bar and max/min intersection\n");
	(void) fprintf(stderr,
	    "\t-l char    use <char> for max/min line\n");
	(void) fprintf(stderr,
	    "\t-L num     set graph lines per page to <num>\n");
	(void) fprintf(stderr,
	    "\t           (default = %d, minimum = %d)\n", LPP, HDRLPP+1);
	(void) fprintf(stderr,
	    "\t-p name    define printer as <name> -- EPSON, PCL4 or PPDS\n");
	(void) fprintf(stderr,
	    "\t           (default = EPSON)\n");
	(void) fprintf(stderr,
	    "\t-P         turn off page count display\n");
	(void) fprintf(stderr,
	    "\t-r         change from raw to QuattroPro dump file format\n");
	(void) fprintf(stderr,
	    "\t-t ttl     set graph title line to <ttl>\n");
	(void) fprintf(stderr,
	    "\t-w         disable warnings about display mode\n");
}


/*
 * ZeroLog() - clear the meter datalog
 */

static void
ZeroLog()
{
	DispMenu(ZeroMenu, NULL);
	if ((char)WaitAnyKey() != 'Y')
		return;
	for (;;) {
		if (WaitRdy() == 0)
			return;
		if (WaitCmd("DMZ", 'Z')) {
			GetDataLn(DumpLine, DUMPLL);
			if (strcmpi(DumpLine, " 005A") == 0)
				return;
		}
		DispMenu(ZeroFail,
			"Press ESC to exit; any other key to retry.");
		if ((char)WaitAnyKey() == ESC)
			return;
	}
}
