/*			phone.cs


Rolladex system sans esql.

By William F. Foote
6/13/88


This program implements a rolodex address system.  It allows a user to
keep track of a database of addresses.  It also features address
printing, making it easy to generate address labels or physical
rolodex cards.  One can either print out all of the addresses on file,
or only those addresses marked as not yet printed.  There is also the
capability to print out a single address.

COMMAND LINE USAGE:  If called with no arguments, this program will assume
that the data file is in the file phonebook.dat.  The default directory is
"c:\rolodex", but this may be overridden by setting the environment variable
"ROLODEX_DIR" to the desired directory.  If this program is called with
a command line argument, that is assumed to be the name of a data file.
If that file name doesn't start with "\", the default directory is
prefixed onto it, thus, calling "phone blah" results in the file
"c:\rolodex\blah.dat" being read.



A word about the data file, phonebook.dat.  It consists of a header record
which contains the head of the deleted records list, followed by a number
of data records.  The data records are in internal machine format, as defined
by the card_t data structure.  The deleted records list has links with the
following meanings:

	n	where n>0	is the byte offset of the record that was
				deleted.
				deleted records list (record numbers start
				numbering at 1)
	-1			is the end of the deleted records list
	0			in a data record means that that record
				is not deleted

*/

#ifdef BSD
#define UNIX
#endif

#ifndef UNIX
#ifndef ANSI

    Compiler error--one of ANIX or UNIX must be defined

#endif
#endif

#define VERSION		"1.3"

#define LINT_ARGS	(1)	/* Enable strong type checking */
#ifdef BSD
#include <assert.h>
#include "screen.h"
#else
#include <stdassert.h>
#include <screen.h>
#include <ctype.h>
#endif
#include <time.h>
#include <stdio.h>
#include <fcntl.h>
#ifdef ANSI
#include <io.h>			/* Has lseek stuff */
#endif
#ifdef UNIX

long lseek();

#define LK_UNLCK  0	/* unlock request */
#define LK_LOCK   1	/* lock request */
#define LK_NBLCK  2	/* non-blocking lock request */
#define LK_RLCK   3	/* read permitted only lock request */
#define LK_NBRLCK 4	/* non-blocking read only lock request */

#endif

#define DEF_FILE	"phonebook"
#define	DEF_DIRECTORY	"c:\\rolodex"
#define	CARD_EXT	".dat"
#define CONFIG_EXT	".cnf"
#define UNLOAD_EXT	".unl"

#define	BACKSLASH	'\\'	/* Single character backslash */

#define	TRUE	(1)
#define	FALSE	(0)
#define	NULLP	((FILE *)0)
#define	EOLN	((char) 10)
#define FF	(12)

#define INDIVIDUAL_ASK	(2)	/* Kludge--see get_config */

#ifdef BSD
/* BSD has a funny help key, since ^Z is so cool.  */
#define HELP_KEY "^B"
#else
#define HELP_KEY "^Z"
#endif

#ifdef BSD
char *OutputDevice();
#else
FILE *popen(char *, char *);
FILE *fopen(char *, char *);
int fclose(FILE *);
int pclose(FILE *);
char *OutputDevice(char *);
#endif


char individual_device[101];	/* Device for individual labels */

char card_file[101],		/* Full path name */
     config_file[101],
     unload_file[101];

typedef struct {
	long	delete_link;	/* 0 = not deleted, -1 = end of list, # = byte offset */
	long	last_update;	/* Time/date of last update, in time() format */
	char	name[31],
		address1[31],
		address2[31],
		address3[31],
		address4[31],
		phone1[31],
		phone2[31],
		printed[2];
	}
	card_t;		/* The structure of card record entries */

typedef struct {
	long	first_deleted
	}
	header_t;	/* The structure of the header entry */

card_t 	card;		/* The "current" card */


char	match[31];

long	curr_card;	/* Byte offset of the current card */
int	card_des;	/* File descriptor of card file */


#ifdef BSD

/* Compensate for BSD's screwy tolower() definition  */

char tolower(c)

   char  c;

{
    if (c >= 'A' && c <= 'Z')
	return c - 'A' + 'a';
    else
	return c;
}

#endif


char *time_name(moment)

    /* Return the time name for the moment defined by moment.  It is
       guaranteed to be 45 characters or less long.  */

    long   moment;		/* In time() format */
    
{
     struct tm *localtime(),*ptime;
     long tptr ;
     char  temp[16];
     static char name[46];

     ptime=localtime(&moment) ;

     name[0] = '\0';
     switch (ptime->tm_wday) {
          case 0: sprintf(name,"Sunday") ; break ;
          case 1: sprintf(name,"Monday") ; break ;
          case 2: sprintf(name,"Tuesday") ; break ;
          case 3: sprintf(name,"Wednesday") ; break ;
          case 4: sprintf(name,"Thursday") ; break ;
          case 5: sprintf(name,"Friday") ; break ;
          case 6: sprintf(name,"Saturday") ; break ;
     }
     strcat(name, ", ");
     switch (ptime->tm_mon) {
          case 0: strcat(name,"January") ; break ;
          case 1: strcat(name,"February") ; break ;
          case 2: strcat(name,"March") ; break ;
          case 3: strcat(name,"April") ; break ;
          case 4: strcat(name,"May") ; break ;
          case 5: strcat(name,"June") ; break ;
          case 6: strcat(name,"July") ; break ;
          case 7: strcat(name,"August") ; break ;
          case 8: strcat(name,"September") ; break ;
          case 9: strcat(name,"October") ; break ;
          case 10:strcat(name ,"November") ; break ;
          case 11:strcat(name ,"December") ; break ;
     }
     sprintf(temp, " %d, 19%02d at ", ptime->tm_mday, ptime->tm_year);
     strcat(name, temp);
     if (ptime->tm_hour < 1)
         sprintf(temp, "%d:%02d AM", ptime->tm_hour+12, ptime->tm_min);
     else if (ptime->tm_hour < 12)
         sprintf(temp, "%d:%02d AM", ptime->tm_hour, ptime->tm_min);
     else if (ptime->tm_hour < 13)  /* i.e. it's == 12 */
         sprintf(temp, "%d:%02d PM", ptime->tm_hour, ptime->tm_min);
     else
         sprintf(temp, "%d:%02d PM", ptime->tm_hour-12, ptime->tm_min);
     strcat(name, temp);

     assert(strlen(name) <= 45);
     return name;
}

void RidTrail(s)

     /*	This routine will remove all trailing blanks from the string s.  */

    char	*s;

{
    int		i;

    for(i=strlen(s)-1; (i>=0) && (s[i] == ' '); i--);
    s[i+1] = '\0';
}


int IsBlank(s)

     /* This routine returns TRUE if s is all spaces, FALSE otherswise.  */

    char	*s;

{
    char  c;

    while (TRUE)  {
	c = *(s++);
	if (c == '\0')
	    return TRUE;
	else if (c != ' ')
	    return FALSE;
    }
}

void PadBlanks(s, l)

     /* This routine will pad the string s out so that it is l characters
	long.							*/

    char	*s;	/* The string to pad */
    int		l;	/* How long they want it to be */

{
    int		i;

    for(i=strlen(s); i<l; i++)
	s[i] = ' ';
    s[l] = '\0';
}

void SetBlank(s, l)

    char  *s;
    int   l;

    /* This routine will set s to be a string of l blanks */

{
    s[0] = '\0';
    PadBlanks(s, l);
}

void write_fail()

{
    perror("Could not write to the card file");
    exit(1);
}


void read_fail()

{
    perror("Could not read from the card file");
    exit(1);
}

void open_cards()

    /* This routine will open the current cards file at the beginning.  */

{
    header_t  header;
    int       i;
    long      l;

#ifdef ANSI
    card_des = open(card_file, O_RDWR | O_BINARY);
#endif
#ifdef UNIX
    card_des = open(card_file, O_RDWR);
#endif
    if (card_des == -1)  {
#ifdef ANSI
	card_des = open(card_file, O_RDWR | O_CREAT | O_BINARY, 00660);
#endif
#ifdef UNIX
	card_des = open(card_file, O_RDWR | O_CREAT, 00660);
#endif
	if (card_des == -1)  {
	    perror("Could not open card file");
	    exit(1);
	}
        header.first_deleted = -1;
	i=write(card_des, (char *) &header, sizeof(header_t));
	if (i != sizeof(header))
	    write_fail();
    }
    l=lseek(card_des, (long) (sizeof(header_t)), 0);	/* lseek past header */
    assert(l == sizeof(header_t));
}


void close_cards()

     /*	This routine will close the cards file */

{
    int   i;

    i=close(card_des);
    if (i != 0)  {
        perror("Could not close card file");
	exit(1);
    }
}


int read_card()

    /* This routine will get the next card from the cards file.  It will
	return TRUE if there is one, or FALSE if EOF.  */

{
    int  i;

    while (TRUE)  {
        curr_card = lseek(card_des, 0L, 1);
        assert(curr_card > 0L);		/* Shouldn't be 0, as header takes some room */
        i=read(card_des, (char *) &card, sizeof(card_t));
	if (i == 0)
	    return FALSE;		/* end of file */
        else if (i < sizeof(card_t))  {
	    printf("\n\nIn reading card, read %d bytes instead of expected %d.\n", i, sizeof(card_t));
            read_fail();
        }  else if (card.delete_link == 0L)
            return TRUE;
    }
}

void update_last_card()


     /* This routine will update the file to reflect a change in the last
        card read.  */

{
    int   i;
    long  old_pos;
    long  l;

    time(&(card.last_update));
    old_pos = lseek(card_des, 0L, 1);
    assert(old_pos > 0L);
    l=lseek(card_des, curr_card, 0);	/* Set at last card */
    assert(l == curr_card);
    i=write(card_des, (char *)&card, sizeof(card_t));
    if (i != sizeof(card_t))
        write_fail();
    l=lseek(card_des, old_pos, 0);	/* Put back where it was */
    assert(l == old_pos);
}


void delete_last_card()

     /* This routine will delete the last card read in */

{
    int   i;
    long  old_pos;
    long  l;
    header_t  head;

    old_pos = lseek(card_des, 0L, 1);
    assert(old_pos > 0L);
    l=lseek(card_des, 0L, 0);	/* move to header */
    assert(l == 0L);
#ifdef UNIX
#ifndef BSD
    i=locking(card_des, LK_RLCK, 0L);		/* Read permitted lock */
    assert(i != -1);
#endif
#endif
    i=read(card_des, (char *) &head, sizeof(header_t));
    assert(i == sizeof(header_t));
    card.delete_link = head.first_deleted;	/* Stick this record on linked list */
    head.first_deleted = curr_card;
    l=lseek(card_des, 0L, 0);	/* move to header */
    assert(l == 0L);
    i=write(card_des, (char *)&head, sizeof(header_t));
    assert(i == sizeof(header_t));
#ifdef UNIX
    l=lseek(card_des, 0L, 0);
    assert(l == 0L);
#ifndef BSD
    i=locking(card_des, LK_UNLCK, 0L);		/* Release the lock */
    assert(i != -1);
#endif
#endif
    l=lseek(card_des, old_pos, 0);	/* Put back where it was */
    assert(l == old_pos);
    update_last_card();
}

void add_card()

    /* Add a new card, in card */

{
    card_t	old_deleted_card;
    header_t  	head;
    long	new_pos;	/* pos'n of new record if is old deleted */
    int		i;
    long	l;

    card.delete_link = 0L;		/* Not deleted */
    time(&(card.last_update));
    open_cards();
    l=lseek(card_des, 0L, 0);	/* move to header */
    assert(l == 0L);
#ifdef UNIX
#ifndef BSD
    i=locking(card_des, LK_RLCK, 0L);		/* Read permitted lock */
    assert(i != -1);
#endif
#endif
    i=read(card_des, (char *) &head, sizeof(header_t));
    assert(i == sizeof(header_t));
    if (head.first_deleted == -1)  {
	l=lseek(card_des, 0L, 2);	/* move to end of file */
        assert(l > 0L);
    } else {
        new_pos = head.first_deleted;
	l=lseek(card_des, new_pos, 0);	/* move to deleted record we're about to reclaim */
        assert(l > 0L);
	l=read(card_des, (char *) &old_deleted_card, sizeof(card_t));
	assert(l == sizeof(card_t));
	head.first_deleted = old_deleted_card.delete_link;	/* Unlink it */
        l=lseek(card_des, 0L, 0);	/* move to header */
        assert(l == 0L);
        i=write(card_des, (char *)&head, sizeof(header_t));	/* write new header */
        assert(i == sizeof(header_t));
	l=lseek(card_des, new_pos, 0);	/* move to deleted record we're about to reclaim */
        assert(l > 0L);
    }
    i=write(card_des, (char *)&card, sizeof(card_t));
    assert(i == sizeof(card_t));
#ifdef UNIX
    l=lseek(card_des, 0L, 0);
    assert(l == 0L);
#ifndef BSD
    i=locking(card_des, LK_UNLCK, 0L);		/* Release the lock */
    assert(i != -1);
#endif
#endif
    close_cards();
}

int contains(s1, s2)

     char   *s1, *s2;

     /* This routine will return TRUE if string s1 contains s2
	(i.e. "bill shreds rageingly" contains "shreds")
     */

{
    int   l1, l2, i;

    l1 = strlen(s1);
    l2 = strlen(s2);
    for(i=0; i<= l1 - l2; i++)
	if (strncmp(s1+i, s2, l2) == 0)
	    return TRUE;
    return FALSE;
}


static void get_card(message, put_help)

    char   *message;
    int    put_help;	/* If TRUE, put ^Z message */

{
    char  *excl;

    excl = "!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!";
    Clear();
$SCREEN
    m=Put("s", message)
    n=Get(STR, card.name, excl);
    1=Get(STR, card.address1, excl);
    2=Get(STR, card.address2, excl);
    3=Get(STR, card.address3, excl);
    4=Get(STR, card.address4, excl);
    p=Get(STR, card.phone1, "(!!!) !!!-!!!! !!!!!!!!!!!!!!!")
    P=Get(STR, card.phone2, "(!!!) !!!-!!!! !!!!!!!!!!!!!!!")
    t=Put("s)", time_name(card.last_update))
    ?=Get(LOG, card.printed, "!")
$ENDDATA
Bill's Techno Rolodex
&m





	Name:	    &n
	Address:    &1
		    &2
		    &3
		    &4

	Phone:	    &p
		    &P


	Printed?  &?

     (Last updated on &t
$ENDSCREEN
    if (put_help)  {
	Put(22,5,"s","(Enter");
	Put(22,12,"s",HELP_KEY);
	Put(22,15,"s","for editing keys help.)");
    }
}


static void insert_cards()

{
    while (TRUE)  {
    	SetBlank(card.name, 30);
    	SetBlank(card.address1, 30);
    	SetBlank(card.address2, 30);
    	SetBlank(card.address3, 30);
	SetBlank(card.address4, 30);
	strcpy(card.phone1, "(   )    -                    ");
	strcpy(card.phone2, card.phone1);
	strcpy(card.printed, "N");
        time(&(card.last_update));
	get_card("Insert Addresses", TRUE);
	Put(20,5,PSTR,"(Leave the address blank to quit inserting.)");
	if (!ReadGS(TRUE))
	    break;
	add_card();
    }
}


static int delete_card()

{
    char  do_it[2];
    int   i;

    strcpy(do_it, "N");
    ClearGets();
    Put(21,0,"s","Do you really want to delete this card?");
    Get(21,41,LOG,do_it,"!");
    Put(22,0,"s","                                                                      ");
    Put(23,10,"s","                                                ");
    ReadGS(TRUE);
    if (do_it[0] == 'Y')  {
        delete_last_card();
	return TRUE;
    } else
	return FALSE;
}

static void edit_card()

{
    long  	edit_id;
    int		i;

    get_card("EDIT", TRUE);
    if (!ReadGS(TRUE))
	return;
    update_last_card();
}

void write_string(fp, ws)

    FILE  *fp;
    char  *ws;

    /*	This routine will write the string ws to file fp, terminated with |.
        It will preceed | or \ with \.  It won't allow a null string--it'll
	write it as one with one blank. */

{
    char   foo[81];
    char   *s;

    strcpy(foo, ws);
    assert(strlen(foo) <= 80);
    RidTrail(foo);
    s = foo;
    if (foo[0] == '\0')
        putc(' ', fp);
    while ((*s) != '\0')  {
	if ((*s) == '|' || (*s) == '\\')
	    putc('\\', fp);
	putc((*s), fp);
	s++;
    }
    putc('|', fp);
}


int get_string(fp, s, l)

    FILE  *fp;
    char  *s;
    int	  l;

    /* This routine will get a string from file fp and put it in s.  The 
       maximum length of s is l.
       It will return TRUE if successful, or FALSE if EOLN is seen.
    */

{
    int   c;
    int   i;
    int   end_seen;

    i=0;
    while ((c = getc(fp)) != EOF && c != ((int) EOLN) && c != ((int) '|'))  {
	if (i < l)  {
	    i++;
	    if (c == '\\')  {
	        c=getc(fp);
		if (c != '\\' && c != '|')  {
		    ungetc(c, fp);
		    c = '\\';
		}
	    }
	    if (c != EOF)
	        (*(s++)) = ((char)c);
	}
    }
    end_seen = (c == EOF || c == EOLN);
    while (i < l)  {
	(*(s++)) = ' ';
	i++;
    }
    (*s) = '\0';
    return !end_seen;
}


int get_eoln(fp)

    FILE  *fp;

    /*  This routine will move fp to after the next end of line.  */

{
    int   c;

    c=getc(fp);
    return c == ((int) EOLN) || c == EOF;
}

static int get_unload_card(fp, line)

    FILE  *fp;
    int line;

{
    if (!get_string(fp, card.name, 30))
        return FALSE;
    if (   get_string(fp, card.address1, 30)
	&& get_string(fp, card.address2, 30)
	&& get_string(fp, card.address3, 30)
	&& get_string(fp, card.address4, 30)
	&& get_string(fp, card.phone1, 30)
	&& get_string(fp, card.phone2, 30)
	&& get_string(fp, card.printed, 1)
	&& get_eoln(fp))  {
		return TRUE;
    } else {
        printf("\n\n\nError!  Line %d of the load file %s is not in\n", line,
		unload_file);
	printf("the correct format.\n\n");
	printf("It either has too few fields, too many fields, or extraneous characters\n");
	printf("after the last '|'.\n\n\n");
	WaitKeyA();
	return FALSE;
    }
}

static void load()

    /* Read an unload file */

{
    char  do_it[2];
    FILE  *fp;
    int   i;

    strcpy(do_it, "N");
    Clear();
$SCREEN
    n=Put("s.", unload_file)
    ?=Get(LOG, do_it, "!")
$ENDDATA
Bill's Techno Rolodex
Read Unload File





	The text file that this program expects is in \"unload\"
	format.  Fields are terminated by the vertical-bar character ('|'),
	and records are terminated by newline.  If either '\\' or '|'
	appear in a record, it should be preceeded by a backslash ('\\').

	The fields are expected to be in the following order:
            name, address1, address2, address3, address4,
	    phone1, phone2, printed

  	They will be read from the file &n
	Every address read will be inserted into the current rolodex file.


    Do you want me to read this file?  &?
$ENDSCREEN
    ReadGS(TRUE);
    if (do_it[0] == 'N')
        return;
    strcpy(do_it, "N");
    Clear();
$SCREEN
    ?=Get(LOG, do_it, "!")
    n=Put("s", unload_file)
$ENDDATA
Bill's Techno Rolodex
Confirm Read Unload File
Unload file:  &n







    WARNING:  Reading in this file could potentially add a lot of
    	      entries to your rolodex file.  Once you've read them in,
	      there is no way to get rid of them, short of individually
	      deleting each one.

	      You might want to make a backup copy of your old rolodex
	      entries before proceeding.


    Are you sure that you want me to read this file?  &?
$ENDSCREEN
    ReadGS(TRUE);
    if (do_it[0] == 'N')
        return;
    fp=fopen(unload_file, "r");
    if (fp == NULLP)  {
        perror("Couldn't open unload file for reading");
	fflush(stderr);
	printf("\n\n\n");
	WaitKeyA();
	return;
    }
    printf("Loading...");
    fflush(stdout);
    i=0;
    while(get_unload_card(fp, (++i)))  {
        if (i % 10 == 0)  {
	    printf(".");
            fflush(stdout);
	}
	if (card.phone1[0] != '(')	/* Make sure each card has (, ), and - so that these work as global */
	    card.phone1[0] = '(';	/* search conditions */
	if (card.phone1[4] != ')')
	    card.phone1[4] = ')';
	if (card.phone1[9] != '-')
	    card.phone1[9] = '-';
	if (card.phone2[0] != '(')
	    card.phone2[0] = '(';
	if (card.phone2[4] != ')')
	    card.phone2[4] = ')';
	if (card.phone2[9] != '-')
	    card.phone2[9] = '-';
	add_card();
    }
    fclose(fp);
    printf("\n\n\n%d card(s) loaded.\n\n\n", i-1);
    WaitKeyA();
}


static void unload()

    /* Generate an unload file */

{
    char  do_it[2];
    FILE  *fp;
    int   i;

    strcpy(do_it, "N");
    Clear();
$SCREEN
    n=Put("s.", unload_file)
    ?=Get(LOG, do_it, "!")
$ENDDATA
Bill's Techno Rolodex
Generate Unload File





	The text file that this program generates is in \"unload\"
	format.  Fields are terminated by the vertical-bar character ('|'),
	and records are terminated by newline.  If either '\\' or '|'
	appears in a record, it is preceeded by a backslash ('\\').

	The fields are written in the following order:
            name, address1, address2, address3, address4,
	    phone1, phone2, printed

  	They will be written into the file &n


    Do you want me to generate this file?  &?
$ENDSCREEN
    ReadGS(TRUE);
    if (do_it[0] == 'N')
        return;
    fp=fopen(unload_file, "w");
    if (fp == NULLP)  {
        perror("Couldn't open unload file for writing");
	exit(0);
    }
    printf("Unloading...");
    fflush(stdout);
    i=0;
    open_cards();
    while(read_card())  {
        if ((++i) % 10 == 0)  {
	    printf(".");
            fflush(stdout);
	}
        write_string(fp, card.name);
	write_string(fp, card.address1);
	write_string(fp, card.address2);
	write_string(fp, card.address3);
	write_string(fp, card.address4);
	write_string(fp, card.phone1);
	write_string(fp, card.phone2);
	write_string(fp, card.printed);
	fprintf(fp, "\n");
    }
    fclose(fp);
    close_cards();
    printf("\n\n\n%d card(s) unloaded.\n\n\n", i);
    WaitKeyA();
}

static void special_functs()

    /* Do the special functions menu  */

{
    char  choice[2];

    while (TRUE)  {
        strcpy(choice, " ");
        Clear();
$SCREEN
    ?=Get(STR, choice, "!")
$ENDDATA
Bill's Techno Rolodex
Special Functions









	A  Generate a text file of your addresses

	B  Read in and insert addresses from a text file


	Q  Quit and return to lookup mode


    What do you want to do?  &?
$ENDSCREEN
	ReadGS(TRUE);
	switch (choice[0])  {
	    case 'A':	unload();		break;
	    case 'B':	load();			break;
	    case 'Q':	return;
	}
    }
}

static void get_config(individual, no_across, width, height, 
			left_margin, print_phone)

    int  individual;
    int  *no_across, *width, *height, *left_margin;
    char *print_phone;

    /* 	Get the configuration information for printing.
        If individual is TRUE, get the individual configuration.
	If individual is INDIVIDUAL_ASK (=2), ask for config as well.
	NOTE:   If individual is TRUE, this routine will have the side
		effect of setting individual_device correctly.
	NOTE 2:  INDIVIDUAL_ASK implies that individual is TRUE.
    */

{
    FILE  *config;
    int   p_no_across,		/* Variables for "printing" config */
          p_width,
	  p_height,
	  p_left_margin;
    char  p_print_phone[2];
    int   i_no_across,		/* Variables for "individual" config */
          i_width,
	  i_height,
	  i_left_margin;
    char  i_print_phone[2];

    p_no_across = 2;
    i_no_across = 1;
    p_width = i_width = 40;
    p_height = 13;
    i_height = 6;
    p_left_margin = i_left_margin = 7;
    strcpy(p_print_phone, "");
    strcpy(i_print_phone, "");
    strcpy(individual_device, "");
    config = fopen(config_file, "r");
    if (config != NULLP)  {
	fscanf(config, "%d %d %d %d", &p_no_across, &p_width, &p_height, 
			&p_left_margin);
	fscanf(config, "%1s", p_print_phone);
	fscanf(config, "%d %d %d %d", &i_no_across, &i_width, &i_height, 
			&i_left_margin);
	fscanf(config, "%1s", i_print_phone);
	fscanf(config, "%100s", individual_device);
    	fclose(config);
    }
    if (strlen(p_print_phone) == 0)
        strcpy(p_print_phone, "Y");
    if (strlen(i_print_phone) == 0)
        strcpy(i_print_phone, "N");
    if (individual)  {
        (*no_across) = i_no_across;
	(*width) = i_width;
	(*height) = i_height;
	(*left_margin) = i_left_margin;
	strcpy(print_phone, i_print_phone);
    } else {
        (*no_across) = p_no_across;
	(*width) = p_width;
	(*height) = p_height;
	(*left_margin) = p_left_margin;
	strcpy(print_phone, p_print_phone);
    }
    if (individual && individual_device[0] == '\0')
        individual = INDIVIDUAL_ASK;
    if (individual == FALSE || individual == INDIVIDUAL_ASK)  {
        Clear();
$SCREEN
	a=Get(INT, no_across, "###,###");   
        b=Get(INT, width, "###,###")
        c=Get(INT, height, "###,###")
        d=Get(INT, left_margin, "###,###")
        e=Get(LOG, print_phone, "!")
$ENDDATA

			CARD PRINTING
			____ ________







     How many cards are there in a line?    &a

     What is the width of a card?	    &b

     What is the height of a card?	    &c		  (Minimum 8)
							  (0 = use form feeds)
     What left margin do you want?	    &d

     Should I print out the phone numbers?  &e
$ENDSCREEN
        ReadGS(TRUE);
	if (individual != FALSE)
	    strcpy(individual_device, OutputDevice("Rolodex Card Printer"));
        config = fopen(config_file, "w");
        if (config == NULLP)  {
	    perror("Can't open configuration file");
	    exit(1);
        }
        if (individual)  {
            i_no_across = (*no_across);
	    i_width = (*width);
	    i_height = (*height);
	    i_left_margin = (*left_margin);
	    strcpy(i_print_phone, print_phone);
        } else {
            p_no_across = (*no_across);
	    p_width = (*width);
	    p_height = (*height);
	    p_left_margin = (*left_margin);
	    strcpy(p_print_phone, print_phone);
        }
        fprintf(config, "%d\n%d\n%d\n%d\n", p_no_across, p_width, p_height, 
    			    p_left_margin);
        fprintf(config, "%s\n", p_print_phone);
        fprintf(config, "%d\n%d\n%d\n%d\n", i_no_across, i_width, i_height, 
    			    i_left_margin);
        fprintf(config, "%s\n", i_print_phone);
        fprintf(config, "%s\n", individual_device);
        fclose(config);
    }
}

static void print_card(output, flush, no_across, width, height, left_margin,
			print_phone)

    FILE  *output;
    int   flush;	/* If TRUE, don't print, but flush buffer */
    int   no_across, width, height, left_margin;
    char  *print_phone;

    /* Generate a card for the currenlty loaded address. */

{
    int   	no_lines;
    char  	*lines[5];
    int	  	line;
    static char line_buf[8][201];
    char  	temp[201];
    char  	s_left_margin[133];
    static int  across = 0;
    int   	use_ff;
    int		i;

    use_ff = height == 0;
    if (flush)  {
        if (across > 0)  {
	    for(i=0; i<height; i++)
	        if (i < 8)
		    fprintf(output, "%s\n", line_buf[i]);
	        else
		    fprintf(output, "\n");
	    if (use_ff)
	        fprintf(output, "%c", FF);
	}
	across = 0;
	return;
    }

    if (use_ff)  {
        if (print_phone)
	    height = 8;
	else
	    height = 5;
    }
    if (left_margin > 132)
	left_margin = 132;
    SetBlank(s_left_margin, left_margin);
    if (across == 0)
        for(i=0; i<8; i++)
	    strcpy(line_buf[i], "");
    across++;
    lines[0] = card.name;
    lines[1] = card.address1;
    lines[2] = card.address2;
    lines[3] = card.address3;
    lines[4] = card.address4;
    no_lines = 5;
    while (no_lines > 0 && IsBlank(lines[no_lines-1]))
        no_lines--;
    line = 0;
    for(i=0; i<no_lines; i++)  {
        sprintf(temp, "%s%s", s_left_margin, lines[i]);
	PadBlanks(temp, width);
	strcat(line_buf[line++], temp);
    }
    SetBlank(temp, width);
    strcat(line_buf[line++], temp);
    if (print_phone[0] == 'Y' && strncmp(card.phone1, "(   )    -", 10) != 0)  {
        sprintf(temp, "%s%s", s_left_margin, card.phone1);
        PadBlanks(temp, width);
        strcat(line_buf[line++], temp);
    }
    if (print_phone[0] == 'Y' && strncmp(card.phone2, "(   )    -", 10) != 0)  {
        sprintf(temp, "%s%s", s_left_margin, card.phone2);
        PadBlanks(temp, width);
        strcat(line_buf[line++], temp);
    }
    SetBlank(temp, width);
    for( ; line<8; line++)  {
	strcat(line_buf[line], temp);
    }
    if (across >= no_across)  {
	across = 0;
	for(i=0; i<height; i++)
	    if (i < 8)  {
		RidTrail(line_buf[i]);
		fprintf(output, "%s\n", line_buf[i]);
	    } else
		fprintf(output, "\n");
	if (use_ff)
	    fprintf(output, "%c", FF);
    }
}

static void print_cards()

{
    FILE  *output;
    int	  i;
    char  do_it[2];
    int   unprinted_only;
    int	  no_across, width, height, left_margin;
    char  print_phone[2];
    char  *device;
    int   no_cards;
    char  temp[201];
    char  temp2[201];

    Clear();
    strcpy(do_it, "Y");
    Put(7,0,"s","Should I only print unprinted cards?");
    Get(7,38,LOG,do_it, "!");
    ReadGS(TRUE);
    unprinted_only = do_it[0] == 'Y';
    get_config(FALSE, &no_across, &width, &height, &left_margin, print_phone);
    device=OutputDevice("Rolodex Card Printer");
    strcpy(do_it, "N");
    while (TRUE)  {
        Clear();
        Put(7,0,PSTR,"Do you want me to print samples?");
        Get(7,34,LOG,do_it,"!");
        ReadGS(TRUE);
	if (do_it[0] == 'N')
	    break;
	output = popen(device, "w");
	if (output == NULLP)  {
    	    perror("Could not write to the printer");
    	    exit(1);
	}
        SetBlank(temp, left_margin);
	strcat(temp, "******************************");
	PadBlanks(temp, width);
	strcpy(temp2, "");
	for(i=0; i<no_across; i++)
	    strcat(temp2, temp);
	RidTrail(temp2);
	for(i=0; i<height; i++)  {
	    if (i < 5 || (print_phone[0] == 'Y' && i < 8))
	        fprintf(output, "%s\n", temp2);
	    else
	        fprintf(output, "\n");
	}
	if (height == 0)		/* if using formfeeds */
	    fprintf(output, "%c", FF);
	pclose(output);
    }
    ResetTTY();
    output = popen(device, "w");
    open_cards();
    no_cards = 0;
    while (read_card())  {
	if (!unprinted_only || card.printed[0] != 'Y')  {
	    no_cards++;
	    print_card(output, FALSE, no_across, width, height,
	    		left_margin, print_phone);
        }
    }
    close_cards();
    print_card(output, TRUE, no_across, width, height,
	    	left_margin, print_phone);
    pclose(output);
    SetTTY();
    Clear();
    strcpy(do_it, "N");
    Put(7,0,"5d",no_cards);
    Put(7,6,"s","card(s) were printed.");
    Put(10,0,"s","Mark as printed?");
    Get(10,18,LOG,do_it,"!");
    ReadGS(TRUE);
    if (do_it[0] == 'Y')  {
        printf("\n\n\nMarking...");
	fflush(stdout);
	i=0;
        open_cards();
        while (read_card())  {
	    if ((++i) % 10 == 0)  {
	    	printf(".");
		fflush(stdout);
	    }
	    if (card.printed[0] != 'Y')  {
		card.printed[0] = 'Y';
		update_last_card();
	    }
	}
        close_cards();
    }
}


static void lookup_card()

{
    int   no_cards;
    char  c,u;
    char  temp[133];
    int   done;
    int   no_checked;
    static int   no_across, width, height, left_margin;
    static int   config_known=FALSE;
    static char  print_phone[2];
    FILE   *print_file;
    int    print_file_open;

    print_file_open = FALSE;
    RidTrail(match);
    if (match[0] != '\0' && match[strlen(match) - 1] == '|')
	match[strlen(match) - 1] = '\0';		/* To make trailing space match conditions possible */
    SetBlank(temp, 79);
    printf("%s%c", temp, 13);	/* Wiping out trademark notice */
    sprintf(temp, "Match condition:  %s", match);
    printf("%s  ", temp);
    fflush(stdout);
    RidTrail(temp);
    open_cards();
    no_cards = no_checked = 0;
    done = !read_card();
    while (!done)  {
        if ((++no_checked) % 10 == 0)  {
	    printf(".");
	    fflush(stdout);
	}
	if (	contains(card.name, match)
	     ||	contains(card.address1, match)
	     ||	contains(card.address2, match)
	     ||	contains(card.address3, match)
	     ||	contains(card.address4, match)
	     ||	contains(card.phone1, match)
	     ||	contains(card.phone2, match)  )  {
	        no_cards++;
	    get_card(temp, FALSE);
	    Put(22,0,"s","(space = next card, d = delete card, e = edit card, q = quit search)  ");
	    Put(23,10,"s","(p = print card, P = change print configuration)");
	    u = c = WaitKey("");
	    c = tolower(c);
	    if (c == 'd')  {
	        if (delete_card())
		    done = !read_card();
	    } else if (c == 'e')
	        edit_card();
	    else if (c == 'q')
		break;
	    else if (c == 'p')  {
	        if (u == 'P')  {
		    if (print_file_open)  {
		    	print_card(print_file, TRUE, no_across, width, height, 
				   left_margin, print_phone);
        		pclose(print_file);
			print_file_open = FALSE;
		    }
        	    get_config(INDIVIDUAL_ASK, &no_across, &width, &height, 
		               &left_margin, print_phone);
		} else if (!config_known)
        	    get_config(TRUE, &no_across, &width, &height, 
		               &left_margin, print_phone);
		if (!print_file_open)   {
		    print_file = popen(individual_device, "w");
		    if (print_file == NULLP)  {
    	    		perror("Could not write to the printer");
    	    		exit(1);
		    }
		    print_file_open = TRUE;
		}
    		print_card(print_file, FALSE, no_across, width, height, 
			left_margin, print_phone);
		fflush(print_file);
	    } else  {
		done = !read_card();
	    }
	} else
	    done = !read_card();
    }
    close_cards();
    if (no_cards == 0)  {
	printf("\n\n\nThere are no cards with that match condition.\n\n\n");
	WaitKeyA();
    }
    if (print_file_open)  {
    	print_card(print_file, TRUE, no_across, width, height, 
		   left_margin, print_phone);
        pclose(print_file);
    }
}


#ifdef BSD
static void advertize()

{
    Clear();
$SCREEN
$ENDDATA








  	Merry-Beth is rad.
$ENDSCREEN
    Put(20,0,"s","");
}

#else


static void advertize()

{
    Clear();
$SCREEN
$ENDDATA
----  Copyright (c) 1988, Charles Lei and Associates  ----

Feel free to copy or distribute this program for personal use provided
that you retain this copyright notice.  Not for resale.  Permission to
post on bulletin board systems granted.
If you use and enjoy this program, please send 20 dollars to:

    Bill Foote
    Charles Lei and Associates
    20252 Elkwood St.
    Canoga Park, CA 91306

Thank you.

Note to software developers:

    The routines that allow the fancy screen manipulation (editing keys,
    forking sub shells, etc.) are available in object code format for Microsoft
    C and Turbo C, as well as several dialects of C on Unix.   These routines
    make it easy to set up professional-looking edit screens, and facilitate 
    the writing of programs which manipulate the screen that are fully portable
    between Unix and MS-DOS.  If you are interested in purchasing these
    routines, please write to the above address, or call (818) 700-2921.
$ENDSCREEN
}

#endif

void main(argc, argv)

    int   argc;
    char  **argv;

{
    char  *getenv();
    char  *directory;

    directory = getenv("ROLODEX_DIR");
    if (directory == NULLP)
        directory = DEF_DIRECTORY;
    if (argc > 1)  {
        if (argv[1][0] == BACKSLASH)  {
            sprintf(card_file, "%s%s", argv[1], CARD_EXT);
            sprintf(config_file, "%s%s", argv[1], CONFIG_EXT);
            sprintf(unload_file, "%s%s", argv[1], UNLOAD_EXT);
	} else {
            sprintf(card_file, "%s%c%s%s", directory, BACKSLASH, argv[1], 
	    						CARD_EXT);
            sprintf(config_file, "%s%c%s%s", directory, BACKSLASH, argv[1], 
	    						CONFIG_EXT);
            sprintf(unload_file, "%s%c%s%s", directory, BACKSLASH, argv[1], 
	    						UNLOAD_EXT);
	}
    } else {
        sprintf(card_file, "%s%c%s%s", directory, BACKSLASH, 
							DEF_FILE, CARD_EXT);
        sprintf(config_file, "%s%c%s%s", directory, BACKSLASH, 
							DEF_FILE, CONFIG_EXT);
        sprintf(unload_file, "%s%c%s%s", directory, BACKSLASH, 
							DEF_FILE, UNLOAD_EXT);
    }
    assert(strlen(card_file) <= 100);
    assert(strlen(config_file) <= 100);
    assert(strlen(unload_file) <= 100);
    SetUpScrn();
    while (TRUE)  {
	Clear();
	SetBlank(match, 30);
$SCREEN
	m=Get(STR, match, "!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!")
     	f=Put("s", card_file)
	v=Put("s", VERSION)
$ENDDATA
Bill's Techno Rolodex(*) Version &v
Main Screen
File:  &f











     Whom do you want to look up?  &m

  Enter \"I\" to insert addresses, \"P\" to print out cards, or
        \"S\" for special functions.
  Enter \"Q\" to quit.

  Any card which has a field that contains what you type will be selected.


  (*Rolodex is a trademark of Rolodex Corp.)
$ENDSCREEN
	ReadGS(TRUE);
	if (strcmp(match, "I                             ") == 0)
	    insert_cards();
	else if (strcmp(match, "Q                             ") == 0)
	    break;
	else if (strcmp(match, "P                             ") == 0)
	    print_cards();
	else if (strcmp(match, "S                             ") == 0)
	    special_functs();
	else
	    lookup_card();
    }
    advertize();
    printf("\n");
    EndScrn();
}
