/*            SCREEN HANDLING PACKAGE
 *
 *               BY WILLIAM F. FOOTE
 *
 *
 *                June 6, 1988
 *


     The following procedures form a library of screen handling functions.
It also contains two decimal number formatting routines.
The calling program should have an 'include <screen.h>' statement.  It must
be compiled with ".. -ltermcap" on a unix implementation.

    This package should be compiled with "-DANSI" for the MS-DOS/ansi.sys
implementation (requires "device=ansi.sys" in the config.sys file), 
with "-DUNIX" for the (non-bsd) unix implementation, and with "-DBSD"
for the bsd implementation.


The following functions are provided:

	Clear();
			Clears the screen

	Put(y, x, type, data, {picture})
	    int   y, x, type;
	    char *data;     (Actually (dec_t *) for decimal type)
	   {char *picture;}
			Puts the string data on the screen.
			Type is a string which specifies what
			is to be put after a % in a printf line
			to output whatever data points at, for example,
			"s" for a string type.  There are also five
			numeric types, PDECIMAL, PINTEGER, PLONGINTEGER,
			PFLOAT, and PDOUBLE, in which case Put expects a pointer
			to the relevant variable, followed by a picture
			clause that will define the format in which it
			should be displayed.
			Type may not be over 20 characters long.

	Get(y, x, type, data, picture);
	    int   y, x, type;
	    char *data, *picture;
			Prepares the variable data for input
			type may only be "STRING", "LOGICAL", "DECIMAL",
			"INTEGER", "LONGINTEGER", "FLOAT" or "DOUBLE".
		        Note that "STRING" is a character constant--very
			UNLIKE the meaning of Type for Put, where it is
			a string constant.

	ReadG(AutoFinish, Function);
			Reads all the pending gets.
			Returns TRUE if data was modified, FALSE otherwise.
			Function is a function that will be called if
			  ^E is pressed.  See the remarks for ReadG
			   for further information.
			If AutoFinish is true, the user can just <CR> past
			the last field to finish editing; otherwise he'll
			have to type ^W.

    	ReadGS(AutoFinish);
			Is like ReadG, except you don't have to provide
			a special function.

        WaitKey(prompt);
	    char *prompt;
			Waits for the user to press a key, and returns
			the key pressed

	WaitKeyA();
			is equivalent to WaitKey("Press any key to continue...  ");

	RefreshGets();
			Rewrites all of the gets

	Refresh();
			Clears and repaints the screen

	SetUpScrn();
			Must be called at program startup

	SubShell();
			Will fork a subshell.

	SetTTY();	Will set the TTY mode to no echo, etc.  This is
			the default, and TTY mode must be set for ReadGS
			to work.  NOTE:  On the IBM PC version, this does
			nothing.

	ResetTTY();	Sets the TTY mode to what it was when the program
			started.  This should be done before a system(),
			or something like that.  NOTE:  ON the IBM PC version,
			this does nothing.

	ClearGets();	Clears all of the gets on the screen.  It converts
			them to equivalent puts, so that the data will still
			be visable on the screen.

	ClearPuts();	Clears all of the puts out of the internal data
			structures.  One probably won't call this function
			very often, but it might be useful if you want to
			use Put() to repeatedly write over an area of a 
			screen.  By doing a ClearPuts you lose the ability
			to refresh the screen, but you bypass the limit
			of 256 "active" puts at any one time.

	AscToDec(Asc, Result);
		char *Asc;
		dec_t *Result;
			Converts an ascii string to esql's decimal type,
			ignoring the formatting characters $, *, and ",".
			It also ignores a second decimal point, as well as
			any characters past that.

	DecToAsc(Dec, Result, Format);
		dec_t *Dec;
		char  *Result;
		char  *Format;
			Converts a decimal type variable to a formatted
			ascii string, according to the picture clause in
			Format.
			It returns TRUE if the conversion was successful
			(i.e. no overflow), FALSE otherwise.

	AscToNum(Type, Asc, Result);
			Same as AscToDec, except will work with Type =
			DECIMAL, INTEGER, or LONGINTEGER.

	NumToAsc(Type, Num, Result, Format)
			Same as DecToAsc, except wil work with Type =
			DECIMAL, INTEGER, or LONGINTEGER.

In addition, the string constant "ClrScr" is declared external, so that
it may be included in host programs.  There is also the int variable
"readg_modified", which is set TRUE if the user modifies a data field
in a ReadG, FALSE otherwise.


     At present, the get data types are "STR" ("STRING"), "LOG"
("LOGICAL") "DEC" ("DECIMAL"), "INT" ("INTEGER"), "LONGINT" ("LONGINTEGER"),
"FLOAT", and "DOUBLE".
     Type LOGICAL is the same as type STRING, except that the variable must 
be one character long, and the user is only allowed to type "Y" or "N"
as an answer to it.
     The picture clause for string data may contain any arbitrary characters
(including space), which will be forced to be a part of the string entered
by the user.  The user won't have to type these characters.  It may also
contain the following special characters:
be forced to be a part of the string entered by the user.  The user won't
have to type these characters.  It may also contain these special characters:

     	'x' or 'X'	Allow any character as input
	'!'		Allow any character, but force lower to upper case
	'#'		Allow only numbers, space, and ".".

     For example, a picture clause of "(###) ###-####" will allow the user
to enter a ten digit telephone number, and it will supply the punctuation
(and the space after the ")") automatically.
     It is assumed that data is not changed between the get and readg
calls, or any refresh calls.  If it is there will be unpredictable results.

     For decimal structures, the picture clause (= the format string) may
not contain arbitrary characters, but it may have the following characters:

	'$' Will output a digit or a sign, or a $ in place of a leading
	    blank.
	'*' Will output a digit or a sign, or a * in place of a leading
	    blank.
	'#' Specifies that a digit or a sign should be output here
	',' Will output a comma only if there are numbers to the left of
	    it, whatever character was last output otherwise.
	'.' Specifies the decimal point position (Numbers need not have
	    a decimal point).

     For example, if Dec = -257321.10,
DecToAsc(&Dec, s, " ##,###,###,###.##")    results in "       -257,321.10"
DecToAsc(&Dec, s, "$$$,$$$,$$$,$$#.##")    results in "$$$,$$$-257,321.10"
DecToAsc(&Dec, s, "***,***,***,**#.##")    results in "***,***-257,321.10"
DecToAsc(&Dec, s, "$##,###,###,###.##")    results in "      $-257,321.10"
DecToAsc(&Dec, s, ",###,###.##")	   results in "***********"

    NOTE:  In this case, DecToAsc(&Dec, &s, ",###,###.##") gives "***********".
	   Although one might expect "-257,321.10" to
	   be output, this will not be the case, because DecToAsc assumes
	   that there must be enough digit positions to fit all of the
	   digits and the sign.  DecToAsc(&Dec, &s, "####,###.##") would
	   output the desired results.

     This format string must be used for both Gets and Puts.

     Floating point and integer types use similar format strings, although
the results for exceeding the allowed with may be different.

     When readg() is executed, the terminal is put in full-screen eediting
mode.  The user may do the following things:
	<-, ->, etc.	Cursor keys:  move the cursor
        ^H		Doubles as cursor left
	home		Home key:  move to first field
        del		Delete key:  deletes character to right of cursor
        del char	Deletes character under cursor
        ins char	Inserts a blank space under cursor
	^R		Refresh the screen
	^W		Finish editing screen
	^A		Restore original values (abort)
	^J		(newline) advance to next field
	^X		Fill the rest of the string with blanks
        ^E		Do the special function for this screen
	^O		Run a program called \"other\"
	^Z		Get a help message 




    INSTALLATION NOTES FOR THE IBM PC VERSION:

This program uses ansi.sys.  The config.sys file of the user of an application
program written with this package must have "device=ansi.sys" in it.

If one desires, one can change the color of the normal and inverse text
in the program.  To do so, one shouldput these lines in his autoexec.bat file:

set S_NORM_COLOR=3f;4b
set S_INV_COLOR=3f;4b

     where  f and b are numbers representing the foreground and background
	    colors, taken from this list:

			0	Black
			1	Red
			2	Green
			3	Yellow
			4	Blue
			5	Purple
			6	Cyan
			7	White

A reasonable choice for an IBM color monitor is:
     set S_NORM_COLOR=33;40
     set S_INV_COLOR=37;44
which results in yellow on black normal text and white on blue inverse
text.


*/

X
#endif

#ifndef UNIX
#ifndef ANSI
    Compile error--one of UNIX and ANSI must be defined
#endif
#endif
#ifdef UNIX
#define cprintf printf
#define putch putchar
    /* IBM PC insists on funny cprintf in SOME cases, and on putch in place
	of putchar for console stuff */
#endif


/* ROUTINE DECLARATIONS */
#ifndef BSD
#ifdef UNIX
    char  *tgetstr(char *, char **);
    char  *tgoto(char *, int, int);
#endif
char  *getcwd(char *, int);
void  chdir(char *);
#endif
char  *calloc();


/* PROGRAM CONSTANTS */

#define MAXPUTS	256	/* Maximum number of puts */
#define MAXGETS 64	/* Maximum number of gets */
#define	TRUE	1
#define FALSE	0
#define	ESC	27
#define	NULLP	((char *) 0)

/* EDITING KEYS... */

#define BS	0	/* The other left arrow key, ^H */
#define	DEL	1	/* The delete key */
#define	REFRESH	2	/* The refresh key, ^R */
#define	FINISH	3	/* The finish editing key, ^W */
#define	RESTORE	4	/* The restore fields key, ^A */
#define	NEXTFLD	5	/* The advance to next field key, ^M */
#define	FILLBL	6	/* The fill field with blanks key, ^X */
#define	SPECFUN	7	/* The special function key, ^E */
#define OTHERPR 8	/* The run other program key, ^O */
#define	HELP	9	/* The help key, ^Z */
#define UP2     10	/* Secondary up key, ^K */
#define DOWN2   11      /* Secondary down key, ^J (BSD NOTE:  On BSD, this
			   conflicts with NEXTFLD.  Since NEXTFLD occurs in
			   the array first, it takes precidence.  */
#define RIGHT2   12	/* Secondary right key, ^L */
			/* ^H is secondary backspace, but that's already BS */
#define UP	13	/* The up arrow key */
#define	DOWN	14	/* The down arrow key */
#define LEFT	15	/* A left arrow key */
#define	RIGHT	16	/* The right arrow key */
#define	HOME	17	/* The home key */
#define DELCH   18	/* The Delete character key */
#define INSCH	19	/* The insert character key */
#ifdef UNIX
#define NOKEYS	20	/* The number of editing keys used */
#endif
#ifdef ANSI
#define	F9	20	/* The PC F9 Key--like ^E */
#define	F10	21	/* The PC F10 Key--like ^W */
#define	NOKEYS	22
#endif

/* Get data types.  These definitions are duplicated in "screen.h" */

#define STRING	's'
#define LOGICAL 'b'
#define DECIMAL	'd'
#define INTEGER 'i'
#define LONGINTEGER 'l'
#define FLOAT   'f'
#define DOUBLE  'o'

/* Put data types.  Puts may, of course, also have any data type that
   sprintf accepts (like, for example, "07x")  */

#define PDECIMAL "D"
#define PINTEGER "I"
#define PLONGINTEGER	"L"
#define PFLOAT	 "F"
#define PDOUBLE  "O"

#ifdef UNIX
#ifdef BSD

#define	STDINDES  0	/* The standard input file descriptor */

#include <sgtty.h>
#include <sys/ioctl.h>
#include <assert.h>

#else

/*  Weird things for ioctl... */

#define	STDINDES  0	/* The standard input file descriptor */
#define VMIN	  4     /* Where vmin is kept (see TTY(M) page 8) */
#include <ioctl.h>  /* My edited ioctl package. */
                            /* Allows single character input.  */

#endif

#endif

/* include files... */

#ifndef BSD
#include <stdassert.h>
#endif

#include <stdio.h>
#include <string.h>
#ifndef BSD
#include <ctype.h>
#endif
#ifndef NESQL
#ifdef UNIX
#include "/usr/informix/incl/decimal.h"	    /* esql decimal type definitions */
#endif
#ifdef ANSI
#include "\informix\incl\decimal.h"	    /* esql decimal type definitions */
#endif
#endif

/*  
              VARIABLES           */


static int	SpecFinish;		/* Flag used by ReadGF.  It indicates that the
					   special function has requested an editing finish. */
int		readg_modified;		/* Flag set by ReadG().  TRUE means that data
					   was modified.  */

typedef struct  {
			int		x,y;    /* Position on screen */
			char		*Data;  /* String put there   */
                }
		PutEntT;

static PutEntT	PutEntry[MAXPUTS];           /* The things put on the screen */

static int	NumPuts = 0;	/* The number of puts there so far. */

typedef struct 	{
		int		x, y;  /* Position on screen */
		char		Type;  /* Variable type, DECIMAL, STRING, LOGICAL,
					  INTEGER, or LONGINTEGER */
		char		*Data, /* Variable there--for Decimal, it's ascii format string */
                                *OldData, /* A copy of it for ^A--for Decimal, it's a pointer to original
					     decimal structure, which is updated upon exit of ReadG. */
                                *Picture; /* its' picture clause */
		int		PictureLen,  /* Length of picture part */
				CompPicture; /* Picture is complicated flag */
				  /* A picture is complicated if it is for a
				   * string and has any characters other than
				   * x or ! in it.  A decimal picture clause is
				   * never "Complicated".
				   */
                }
                GetEntT;

static GetEntT	GetEntry[MAXGETS];   /* The variables being read */

static int	NumGets = 0;	/* The number of gets there so far. */

static int	CRTLines;	/* The number of lines on the CRT.  */
static int	CRTCols;	/* The number of columns on the CRT.  */
static int	RevChars;	/* Number of characters RevOn takes */
       char     *ClrScr;	/* Clear screen sequence:  EXTERNAL */
static char	*Bell, *CurMove, *RevOn, *RevOff;  /* CRT sequences */
static char	*RevOnMode, *RevOffMode;
		   /* These put the CRT in and out of reverse MODE, for the
		      Wyse terminals, they are null, because mode is determined
		      by characters on the screen, not a flag setting.  */
static char	*Key[NOKEYS] =  /* The editing keys used */
		       {"\10",		/* Backspace */
			"\177",		/* Delete key */
			"\22",		/* ^R refresh key */
			"\27",		/* ^W Finish editing key */
			"\1",		/* ^A Restore fields key */
#ifdef BSD
			"\n",		/* ^J Next field key for BSD */
#else
			"\15",		/* ^M Next field key */
#endif
			"\30",		/* ^X Fill field with blanks key */
			"\5",		/* ^E special function key */
#ifdef BSD
			"\20",		/* ^P Run other program key */
			"\2",		/* ^B Help key */
#else
			"\17",		/* ^O Run other program key */
			"\32",		/* ^Z Help key */
#endif
			"\13",		/* ^K, secondary up key */
			"\12",		/* ^J, secondary down key */
			"\14"		/* ^L, secondary right key */
       	               };

#ifdef UNIX

static char	TermCapBuf[256];	/* Buffer for stuff from termcap */

#ifdef BSD
static struct sgttyb
		NewTTYMode,		/* TTY mode w/ echo off, etc... */
		OriginalTTYMode;	/* What TTY mode was at program start */
#else
static struct termio	
		NewTTYMode,		/* TTY mode w/ echo off, etc... */
		OriginalTTYMode;	/* What TTY mode was at program start */
#endif
#endif


#ifndef NDEBUG
static int	WasSetUp = FALSE;   /* To be sure that they do this */
#endif

/*  
                 FUNCTIONS         */


#ifdef BSD

	/* BSD at UCB:  ctype.h is screwed up */

static char tolower(c)

   char c;

{
    if (c >= 'A' && c <= 'Z')
	return c - 'A' + 'a';
    else
	return c;
}


static char toupper(c)

   char c;

{
    if (c >= 'a' && c <= 'z')
	return c - 'a' + 'A';
    else
	return c;
}

static int isdigit(c)

    char  c;

{
    return c >= '0' && c <= '9';
}


static int iscntrl(c)

    char  c;

{
    return c < ' ';
}


/* BSD Termcap bug:  tgetstr seems to like to return '5' at the
		     beginning of some entries.
*/

static char *my_tgetstr(a, b)

    char  *a, **b;

{
    char  *s;

    s=(char *)tgetstr(a, b);
    if (s == ((char *) 0))  {
	s="";			/* Non-existant features:  "" is more
				   reasonable than "(null)".  */
    } else {
        while ((*s) > ' ')  {
	    s++;
	}
    }
    return s;
}

#endif

#ifdef UNIX
#ifndef BSD

     /* Every Unix besides BSD still needs a slightly different tgetstr */


static char *my_tgetstr(a, b)

    char  *a, **b;

{
    char  *s;

    s=(char *)tgetstr(a, b);
    if (s == ((char *) 0))  {
	s="";			/* Non-existant features:  "" is more
				   reasonable than "(null)".  */
    }
    return s;
}



#endif
#endif


#define tgetstr(a, b) my_tgetstr(a,b)      /* Make sure that we detect things
					      like non-extant termcap
					      entries.  
					   */

static void fail(s)

    char  *s;

    /* Fail because the caller screwed up */

{
    printf("\n\nscreen.c:  %s.\n", s);
    printf("           Aborting program execution.\n");
    exit(1);
}



char *get_mem(size)

    int   size;

     /* Get memory a la calloc.  Blow up if it fails with an intelligent
	error message.  */

{
    char   *p;

    p=calloc(size, sizeof(char));
    if (p == NULLP)  {
  	fail("memory allocation error");
    }
    return p;
}

static void SetUpAssert()

     /* Make sure that they ran SetUpScrn()  */

{
    if (!WasSetUp)  {
	fail("SetUpScrn() was not called.");
    }

}

#ifdef NESQL
		/* If we're not allowed to use ESQL functions, we can't have
		   support of the decimal type  */

void dectoasc()

{
    fail("The decimal data type is not supported in this version");
}

int deccvasc()

{
     dectoasc();
}

#endif



/*VARARGS2*/

void AscToNum(Type, Asc, Result)
    char   Type;	/* The type of number */
    char   *Asc;
    long   *Result;

  /* Convert Asc to Numeric, ignoring $, *, and ",".  Also discard a second
     decimal point, and anything past it.  Result can actually be dec_t *, 
     int *, or long int * */

{
    char Buf[81];		/* A character buffer */
    int  Len;			/* Length of Buf */
    char *To;			/* Character Buffer pointer */
    char c;
    int  DPCount;		/* Count of DP's encountered so far */

    To = Buf;
    DPCount = 0;
    Len = 0;
    while ((*Asc) == ' ')
    	Asc++;		/* Trim trailing spaces */
    while ((c=(*Asc++)) != '\0' && (DPCount += c == '.') < 2)			/* Build string in Buf w/o $, *, or ","  */
	if (c != '$' && c != '*' && c != ',' && Len < 80)  {
	    (*(To++)) = c;
	    Len++;
	}
    (*To) = '\0';
    switch (Type)  {
        case DECIMAL:		deccvasc(Buf, Len, Result);    		break;
	case INTEGER:		(*((int *) Result)) = 0;
				sscanf(Buf,  "%d", Result);		break;
	case LONGINTEGER:	(*((long int *) Result)) = 0;
				sscanf(Buf, "%ld", Result);		break;
	case FLOAT:		(*((float *) Result)) = 0.0;
				sscanf(Buf, "%f", Result);		break;
	case DOUBLE:		(*((double *) Result)) = 0.0;
				sscanf(Buf, "%lf", Result);		break;
	default:		fail("Undefined type to AscToNum");
    }
}

/*VARARGS1*/

int NumToAsc(Type, Num, Result, Format)
    char  Type;
    long  *Num;
    char  *Result;
    char  *Format;


    /* Convert Num,which is of type Type, to an ascii string, left in Result,
       according to Format.
       Return TRUE if success, FALSE otherwise (i.e. overflow)  */

{
    int   DigitsLeft,		/* Excess digits to left of D.P. */
	  DigitsRight;		/* Digits to right of D.P. */
    int   TotDigits;		/* DititsLeft + Right + (1 if DPSeen)  */
    char  Buf[81],		/* Character buffer for building ascii string */
	  *fmt, 		/* Character format pointer */
	  *buf,			/* Character buffer pointer */
	  *res;			/* Character result pointer */
    int	  IsSign;		/* Flag that there is a sign  */
    int   DPSeen;		/* Flag that the D.P. has been seen */
    char  f;			/* Current format character */
    char  LastOut;		/* Character that was last output */
    int   Overflow;		/* Flag that we've got too big a number */
    int   i;
    char  c;

    if (Format[0] == '\0')  {		/* If null format string */
	if (Type == DECIMAL)  {
            Buf[80]='\0';    /* dectoasc expects null to already be there. */
	    dectoasc(Num, Buf, 80, -1);
	    for(buf=Buf+79; ((*buf)==' '); buf--)    /* Trim trailing blanks */
	   	(*buf) = '\0';
	    strcpy(Result, Buf);
	} else {
	    switch(Type)  {

		case INTEGER:	   	sprintf(Result, "%d", *((int *) Num));	  break;
		case LONGINTEGER:	sprintf(Result, "%ld", *((long int*) Num));  break;
		case FLOAT:		sprintf(Result, "%f", *((float*) Num));  break;
		case DOUBLE:		sprintf(Result, "%lf", *((float*) Num));  break;
		default:		fail("NumToAsc called with incorrect type");
	    }
	}
	return FALSE;		/* Overflow impossible */
    }
    DigitsLeft = DigitsRight = 0;	/* Count up DigitsLeft and DigitsRight */
    DPSeen = FALSE;
    for(fmt = Format; (f=(*fmt)) != '\0'; fmt++)  {
	if (f == '.')  {
	    if (DPSeen)
	        fail("NumToAsc:  two decimal points in format string");
	    DPSeen = TRUE;
  	} else if (f == '$' || f == '#' || f == '*') {
	    if (DPSeen)
		DigitsRight++;
	    else
		DigitsLeft++;
        } else if (f != ',')
	    fail("NumToAsc:  format string error");
    }
    if (DPSeen)
	TotDigits = DigitsLeft+DigitsRight+1;
    else
	TotDigits = DigitsLeft+DigitsRight;
    if (Type == DECIMAL)  {
    	Buf[80]='\0';    /* dectoasc expects null to already be there. */
    	dectoasc(Num, Buf, 80, DigitsRight);
    } else {
    	if (DPSeen && Type != FLOAT && Type != DOUBLE)
	    fail("NumToAsc:  decimal point in format for int or long");
	if (Type == INTEGER)
	    sprintf(Buf, "%d", *((int *) Num));
	else if (Type == LONGINTEGER)
	    sprintf(Buf, "%ld", *((long int *) Num));
	else if (Type == FLOAT || Type == DOUBLE)  {
	    char  temp[21];

	    sprintf(temp, "%s%d", "%.", DigitsRight);
	    assert(strlen(temp) <= 18);
	    if (Type == FLOAT)  {
		strcat(temp, "f");
		sprintf(Buf, temp, *((float *) Num));
	    } else {
		strcat(temp, "lf");
		sprintf(Buf, temp, *((double *) Num));
	    }
	} else
	    fail("NumToAsc:  invalid type");
	for(i=strlen(Buf); i<TotDigits; Buf[i++] = ' ');
	Buf[i] = '\0';			/* Pad Buf with blanks */
    }
	
    for (buf=Buf; (*buf) != '\0' && (*buf) != '.' && (*buf) != ' '; buf++)     /* Figure out how many of DigitsLeft's are excess */
	DigitsLeft--;
    if (IsSign = (Buf[0] == '-')) {
	buf = Buf+1;
	DigitsLeft++;
    }  else
	buf = Buf;
    res = Result;
    DPSeen =
    Overflow = FALSE;
    LastOut = ' ';
    for(fmt=Format; (f=(*fmt)) != '\0'; fmt++)  {
	if (f == '.')  {
	    DPSeen = TRUE;
	    (*res++) = LastOut = (*buf++);	/* Which should contain '.' */
	    Overflow = Overflow || LastOut != '.';
	} else if (f == '$' || f == '#' || f == '*')  {
	    if (DigitsLeft > 0 && !DPSeen)  {
		if (f == '#')
		    (*res++) = LastOut = ' ';
	  	else
		    (*res++) = LastOut = f;
	    } else {
	  	(*res++) = LastOut = (*buf++);
	    }
	    DigitsLeft--;
    	} else if (f == ',')  {
	    if (DigitsLeft >= 0)
		(*res++) = LastOut;
	    else
		(*res++) = LastOut = ',';
	}
    }
    (*res) = '\0';
    if ((*Result) == '$')  {		/* Move leading $ to rightmost leading blank */
	(*Result) = ' ';
	for (res=Result+1; (*res) == ' '; res++)  ;
	(*(res - 1)) = '$';
    }
    if (IsSign)  {			/* Put in sign */
	if ((c=(*Result)) != ' ' && c != '$' && c != '*')
	    Overflow = TRUE;
	else  {
	    for (res=Result+1; (*res) == ' '; res++);
	    (*(res - 1)) = '-';
	}
    }
    if (Overflow)
	for(res=Result; (*res) != '\0'; res++)
	    (*res) = '*';
    return !Overflow;
}


void AscToDec(Asc, Result)
    char   *Asc;
    long   *Result;

{
    AscToNum(DECIMAL, Asc, Result);
}


int DecToAsc(Dec, Result, Format)
    long  *Dec;
    char  *Result;
    char  *Format;

{
    return NumToAsc(DECIMAL, Dec, Result, Format);
}


static int IsNumeric(Type)
    char    Type;

     /* Return TRUE if Type is a numeric type (Decimal, Integer, or 
	LongInteger), FALSE otherwise */

{
    return (Type == INTEGER || Type == LONGINTEGER || Type == DECIMAL
	    || Type == FLOAT || Type == DOUBLE);
}

static int IsString(Type)
    char    Type;

     /* Return TRUE if Type is a string type (String or Logical),
	FALSE otherwise */

{
    return (Type == STRING || Type == LOGICAL);
}


static void SetUpTTY()
/*--------------------------------------------------------*/
/*  SetUpTTY:  Sets up the variables NewTTYMode and	  */
/*	       OriginalTTYMode.				  */
/*--------------------------------------------------------*/

{
#ifdef UNIX
#ifdef BSD
    ioctl(STDINDES, TIOCGETP, &NewTTYMode);     /* Both new & old start out as original tty mode */
    ioctl(STDINDES, TIOCGETP, &OriginalTTYMode);
    NewTTYMode.sg_flags &= (~ECHO) & (~XTABS);
    NewTTYMode.sg_flags |= CBREAK;
#else
    ioctl(STDINDES, TCGETA, &NewTTYMode);     /* Both new & old start out as original tty mode */
    ioctl(STDINDES, TCGETA, &OriginalTTYMode);
    NewTTYMode.c_lflag &= (~ICANON) & (~ECHO);
    NewTTYMode.c_iflag &= (~ICRNL);		/* Don't map ^M to ^J */
    NewTTYMode.c_cc[VMIN] = (char) 1;	/* Pass characters after 1 input */
#endif
#endif
    return;
}


void SetTTY()
/*--------------------------------------------------------*/
/*  SetTTY:  Sets TTY mode to NewTTYMode, i.e. with no    */
/*	     echo, etc...  TTY mode should normally be    */
/*   	     set like this.				  */
/*--------------------------------------------------------*/


{
#ifdef UNIX
#ifdef BSD
    ioctl(STDINDES, TIOCSETP, &NewTTYMode);
#else
    ioctl(STDINDES, TCSETA, &NewTTYMode);
#endif
#endif
    return;
}


void ResetTTY()
/*-------------------------------------------------------*/
/*  ResetTTY:  Resets the TTY mode to what it was when   */
/*	       the program was called.			 */
/*-------------------------------------------------------*/


{
#ifdef UNIX
#ifdef BSD
    ioctl(STDINDES, TIOCSETP, &OriginalTTYMode);
#else
    ioctl(STDINDES, TCSETA, &OriginalTTYMode);
#endif
#endif
    return;
}


char WaitKey(prompt)
/*---------------------------------------------*/
/* WaitKey:  Wait for a keypress (after        */
/*           displaying prompt)		       */
/*---------------------------------------------*/

char *prompt;
{
    char	     c;

    printf("%s", prompt);
#ifdef UNIX
    c = getchar();
#endif
#ifdef ANSI
    c=getch();
#endif
    printf("\n");
    return c;
}

char WaitKeyA()
/*--------------------------------------------------------*/
/* WaitAKey:  Wait for a keypress (after displaying       */
/*            "Press any key to continue..."    	  */
/*--------------------------------------------------------*/

{
    return WaitKey("Press any key to continue...  ");
}

void SubShell()
/*--------------------------------------------------------*/
/* SubShell:  Fork a subshell                             */
/*--------------------------------------------------------*/
{
    char	*Shell,
		*getenv();
    char	old_dir[82];

#ifdef UNIX
    Shell = getenv("SHELL");
#endif
#ifdef ANSI
    Shell = getenv("COMSPEC");
    getcwd(old_dir, 80);
#endif
    ResetTTY();
    system(Shell);
    SetTTY();
#ifdef ANSI
    chdir(old_dir);
#endif
}


#ifdef UNIX

void SetUpScrn()
/*----------------------------------------------------*/
/* SetUpScrn:  Initialize everything.  Call this once */
/*	       at the beginning of program execution. */
/*	       UNIX VERSION			      */
/*----------------------------------------------------*/
                    
{
    char	*termname;	/* Name of the terminal being used */
    char	**area;		/* used by termcap functions */
    char	*areapt;	/* The dummy area area points to */
    char	tcbuf[1024];	/* Buffer used by termcap functions */

    if (WasSetUp)
	fail("SetUpScrn called twice");
    WasSetUp = TRUE;

    termname = (char *) getenv("TERM");
    tgetent(tcbuf, termname);   /* Get termcap entry for this terminal */
    area = &areapt;
    (*area) = TermCapBuf;
    CRTLines = tgetnum("li");
    CRTCols  = tgetnum("co");
    Bell = "\7";
    ClrScr = (char *) tgetstr("cl", area);
    CurMove = (char *) tgetstr("cm", area);
    Key[DELCH] = (char *) tgetstr("dc",area);
    Key[INSCH] = (char *)tgetstr("ic",area);
    RevOn  = (char *) tgetstr("so", area);
    RevOff = (char *) tgetstr("se", area);
    RevChars = 0;
    RevChars = tgetnum("sg");
    if (RevChars < 0)
	RevChars = 0;
    if (RevChars == 0) {
	RevOnMode = RevOn;
	RevOffMode = RevOff;
    } else {
	RevOnMode = "";
	RevOffMode = "";
    }
    Key[UP] = (char *) tgetstr("ku", area);
    if ((Key[UP][0] == '\033') || (strlen(Key[UP]) == 1))
	/* Only accept if single character or starts with escape--
	   otherwise there's probably no termcap entry
	*/
    {
        Key[DOWN] = (char *) tgetstr("kd", area);
    	Key[LEFT] = (char *) tgetstr("kl", area);
    	Key[RIGHT] = (char *) tgetstr("kr", area);
   	Key[HOME] = (char *) tgetstr("ho", area); 
    }
    else
    {   printf("%s\n\n", ClrScr);
    	printf("Your terminal doesn't have cursor keys.  The following\n");
        printf("control keys will control the cursor:\n\n\n");
        printf("	^T	Cursor up\n");
	printf("	^F	Cursor left\n");
	printf("	^G	Cursor right\n");
	printf("	^V	Cursor down\n");
	printf("	^Y	Home cursor\n");
        printf("\n\nYou may want to write these down...  They won't be shown again.");
        printf("\n\n");
        WaitKeyA();
        Key[UP] = "\24";
        Key[DOWN] = "\26";
        Key[RIGHT] = "\7";
        Key[LEFT] = "\6";
        Key[HOME] = "\30";
    }
    SetUpTTY();
    SetTTY();		/* Set TTY up to not echo, etc.  */
}
#endif


#ifdef ANSI

void SetUpScrn()
/*----------------------------------------------------*/
/* SetUpScrn:  Initialize everything.  Call this once */
/*	       at the beginning of program execution. */
/*	       ANSI (IBM PC) VERSION		      */
/*----------------------------------------------------*/
                    
{
    char  *s,
	  *getenv();

    if (WasSetUp)
	fail("SetUpScrn called twice");
    WasSetUp = TRUE;
    s=getenv("S_INV_COLOR");
    if (s != NULLP)  {
	RevOn = get_mem(4+strlen(s));
	sprintf(RevOn, "%c[%sm", ESC, s);
    } else
	RevOn = "\33[7m";
    s=getenv("S_NORM_COLOR");
    if (s != NULLP)  {
	RevOff = get_mem(4+strlen(s));
	sprintf(RevOff, "%c[%sm", ESC, s);
    } else
	RevOff = "\33[0m";
    CRTLines = 25;
    CRTCols  = 80;
    Bell = "\7";
    ClrScr = "\33[2J";
    CurMove = "";
    Key[DELCH] = "\33S";
    Key[INSCH] = "\33R";
    RevChars = 0;
    if (RevChars == 0) {
	RevOnMode = RevOn;
	RevOffMode = RevOff;
    } else {
	RevOnMode = "";
	RevOffMode = "";
    }
    Key[UP] = "\33H";
    Key[DOWN] = "\33P";
    Key[LEFT] = "\33K";
    Key[RIGHT] = "\33M";
    Key[HOME] = "\33G";
    Key[DEL] = "\4";	/* It's ^D */
    Key[F9] = "\33C";
    Key[F10] = "\33D";
    SetUpTTY();
    SetTTY();		/* Set TTY up to not echo, etc.  */
}


char *tgoto(CurMove, x, y)

    char  *CurMove;
    int   x,y;

     /* Impersonate the termcap tgoto function.  Return a character string
	which will cause the cursor to move to the given location on the
	screen.  Note that termcap numbers columns and lines from 0, whereas
	ansi numbers them from 1--this routine compensates for that by
	adding 1 to x and y.
	NOTE:  This routine can't call sprintf(), as it is often called from
	       within printf().
     */

{
    static char  result[9];
    char	 *cp;

    x++;  y++;
    if (x > 99 || y > 99 || x < 0 || y < 0)
	return "";
    cp = result;
    (*(cp++)) = ESC;
    (*(cp++)) = '[';
    (*(cp++)) = '0' + y/10;
    (*(cp++)) = '0' + y%10;
    (*(cp++)) = ';';
    (*(cp++)) = '0' + x/10;
    (*(cp++)) = '0' + x%10;
    (*(cp++)) = 'H';
    (*(cp++)) = '\0';
    return result;
}


#endif	


void EndScrn()
/*------------------------------------------------*/
/*  EndScrn:  Clean up everything that has to be  */
/*	      cleaned up upon program termination */
/*------------------------------------------------*/

{
    SetUpAssert();
    ResetTTY();		/* Give them back normal TTY modes */
}



static void KillGets()
/*-------------------------------------------------*/
/* KillGets:  Clear all of the gets from the       */
/*             GetEntry data structure		   */
/*-------------------------------------------------*/
{
    int     i;

    SetUpAssert();
    for(i=0; i<NumGets; i++)
	if (IsString(GetEntry[i].Type))
	    free(GetEntry[i].OldData);
	else  {
	    assert(IsNumeric(GetEntry[i].Type));
	    free(GetEntry[i].Data);
	}
    NumGets = 0;
}

void ClearPuts()
/*-----------------------------------------------*/
/* ClearPuts:  Clear all of the puts from the    */
/*	       PutEntry data structure		 */
/*-----------------------------------------------*/
{
    int	    i;

    SetUpAssert();
    for(i=0; i<NumPuts; free(PutEntry[i++].Data));
    NumPuts = 0;
}



void Clear()
/*-------------------------------------------------*/
/* Clear:  Clear the screen (and all gets)         */
/*-------------------------------------------------*/
{
    KillGets();
    ClearPuts();
    cprintf("%s%s", RevOffMode, ClrScr);
#ifdef UNIX
    fflush(stdout);
#endif
}



static void OutputPut(Entry)
/*----------------------------------------------------*/
/* OutputPut:  Outputs data to the screen from a put  */
/*----------------------------------------------------*/

PutEntT	Entry;

{
    char  *s;

    s=(char *)tgoto(CurMove, Entry.x, Entry.y);
    cprintf("%s%s",s, Entry.Data);
}

/*VARARGS3*/

void Put(y, x, Type, Data)
/*----------------------------------------------------*/
/* Put:  Put an entry on the screen                   */
/*----------------------------------------------------*/

int	y, x;  /* The position to put it on the screen */
char	*Type;  /* The type of thing being put */
double	Data;   /* The data--usually a character pointer, but the biggest
		   possible data type is a double--SYSTEM DEPENDANT */
		/* Note:  For the decimal type, Data will actually be a
			  pointer to a dec_t, AND a char * which is the
			  picture clause.  In a large model program, these
			  would occupy 8 bytes, which is the size of a
			  double.  */

  {
    char	PrFormat[22];
    char	TData[133];
    PutEntT	*Entry;
    typedef struct {    			/* Format of Data for Decimal type  */
		    char     *Num;		    /* Decimal, int, or long int pointer */
		    char     *Picture;	            /* And format string */
		}	NumDataType;			/* Total length = 4 (small/medium model), or 8
						   (large model) bytes, which is <= sizeof(double) (=8 bytes )  */
    NumDataType	*NumData;

    SetUpAssert();
    if (NumPuts >= MAXPUTS)  {
	fail("Maximum number of puts exceeded");
    }
    Entry = &(PutEntry[NumPuts]);
    Entry->x = x;
    Entry->y = y;
    if (Type[0] == PDECIMAL[0])  {  	/* Special decimal type */
	NumData = (NumDataType *) &Data;
	NumToAsc(DECIMAL, (*NumData).Num, TData, (*NumData).Picture);
    } else if (Type[0] == PINTEGER[0])  {
	NumData = (NumDataType *) &Data;
	NumToAsc(INTEGER, (*NumData).Num, TData, (*NumData).Picture);
    } else if (Type[0] == PLONGINTEGER[0])  {
	NumData = (NumDataType *) &Data;
	NumToAsc(LONGINTEGER, (*NumData).Num, TData, (*NumData).Picture);
    } else if (Type[0] == PFLOAT[0])  {
	NumData = (NumDataType *) &Data;
	NumToAsc(FLOAT, (*NumData).Num, TData, (*NumData).Picture);
    } else if (Type[0] == PDOUBLE[0])  {
	NumData = (NumDataType *) &Data;
	NumToAsc(DOUBLE, (*NumData).Num, TData, (*NumData).Picture);
    } else {			/* Otherwise, send it to sprintf */
    	strcpy(PrFormat, "%");
    	strncat(PrFormat, Type, 20);
    	sprintf(TData, PrFormat, Data);
    }
    if (strlen(TData) > 132)
	fail("Put resulted in string over 132 characters");
    Entry->Data = (char *) get_mem(strlen(TData)+1);
    strcpy(Entry->Data, TData);
    OutputPut(*Entry);
    NumPuts++;
}


void ClearGets()
/*------------------------------------------------------*/
/* ClearGets:	Change all of the gets on the screen    */
/*		into puts.				*/
/*------------------------------------------------------*/
{
    int	    i;
    char    t;

    SetUpAssert();
    for(i=0; i<NumGets; i++)  {
	cprintf("%s%s", tgoto(CurMove, GetEntry[i].x-RevChars,
			     GetEntry[i].y), RevOff);
	Put(GetEntry[i].y, GetEntry[i].x, "s", GetEntry[i].Data);
    }
    KillGets();
}


static void OutputGet(y, x, Data)
/*-----------------------------------------*/
/* OutputGet:  Outputs data to the screen  */
/*             in inverse video.	   */
/*-----------------------------------------*/

int     y, x;
char    *Data;

{
    if (RevChars == 0)
    	cprintf("%s%s%s%s",tgoto(CurMove, x, y), RevOn, Data, RevOff);
    else
    {
	if (x >= RevChars)  {
	    cprintf("%s%s%s", tgoto(CurMove, x, y), Data, RevOff);
	    cprintf("%s%s", tgoto(CurMove, x-RevChars, y), RevOn);
	} else {
	    cprintf("%s%s%s",tgoto(CurMove, RevChars, y), Data, RevOff);
	    cprintf("%s%s", tgoto(CurMove, 0, y), RevOn);
	}
    }
}


void Get(y, x, Type, Data, Picture)
/*--------------------------------------------------*/
/* Get:  Prepare the variable Data to be gotten by  */
/*       ReadG().				    */
/*--------------------------------------------------*/

int	y, x;	/* The screen position */
char	Type;	/* The type of the data */
char	*Data,	/* A pointer to the string you want to get */
	*Picture; /* The picture clause */
{
    GetEntT	*Entry;

    SetUpAssert();
    if (NumGets >= MAXGETS)  {
	fail("Maximum number of gets exceeded");
    }
    Entry = &(GetEntry[NumGets]);
    Entry->y = y;
    Entry->x = x;
    Entry->Type = Type;
    Entry->PictureLen = strlen(Picture);
    Entry->Picture = Picture;
    Entry->CompPicture = FALSE;
    if (IsString(Type))  {
        char *d, *p;     /* Wherever picture has a character, force that character of data to be that. */

    	if(strlen(Data) < Entry->PictureLen)  {
	    printf("Picture:  '%s', Data:  '%s'\n", Entry->Picture, Data);
	    fail("Get() has picture clause longer than data");
  	}
    	for (d=Data, p=Picture; *p != '\0'; d++, p++)
	{
	    Entry->CompPicture = Entry->CompPicture || ( tolower(*p) != 'x' && (*p) != '!' );
	    switch (tolower(*p)) {
	        case 'x':	(*d) = (*d);				break;
	        case '#':	{ if ((!isdigit(*d)) && ((*d) != '.') && ((*d) != ' '))
				    (*d) = '0';				break;
				}
	        case '!':	
				(*d) = toupper(*d);			break;
                default:	
				(*d) = (*p);
	    }
	}
    	Entry->OldData = (char *) get_mem(strlen(Data) + 1);
    	Entry->Data = Data;
    	strcpy(Entry->OldData, Data);
	if (Type != STRING && strlen(Data) != 1)  {
	    fail("Get() error:  Logical type has length other than 1");
	}
    } else {			/* Type == DECIMAL, INTEGER, LONGINTEGER, FLOAT or DOUBLE */
	Entry->Data = (char *) get_mem(Entry->PictureLen+1);
        NumToAsc(Type, Data, Entry->Data, Picture);
	assert(strlen(Entry->Data) == Entry->PictureLen);
	Entry->OldData = (char *) Data;
    }
    OutputGet(y, x, Entry->Data);
    NumGets++;
}


void RefreshGets()
/*---------------------------------------------------*/
/* RefreshGets:  Rewrites all of the Gets.  This can */
/*               be used if something external       */
/*		 changed the data in the get.	     */
/*---------------------------------------------------*/
{
    int	    i;

    for(i=0; i<NumGets; i++)
	OutputGet(GetEntry[i].y, GetEntry[i].x, GetEntry[i].Data);
}


void Refresh()
/*---------------------------------------------------*/
/* Refresh:  Clears and repaints the screen.  It     */
/*	     leaves the cursor after the last put.   */
/*---------------------------------------------------*/
{
    int	    i;

    cprintf("%s%s", RevOffMode, ClrScr);
    for(i=0; i<NumGets; i++)
	OutputGet(GetEntry[i].y, GetEntry[i].x, GetEntry[i].Data);
    for(i=0; i<NumPuts; i++)
	OutputPut(PutEntry[i]);
}
	
/*---------------------------------------------------*/
/* ReadG:           LOCAL FUNCTIONS		     */
/*---------------------------------------------------*/



/*---------------------------------------------------*/
/* StdSpecial:    The standard special function key  */
/*                routine.  It can be used as an     */
/*                argument to ReadG().               */
/*---------------------------------------------------*/

static void StdSpecial()

{
    cprintf("%s%s", RevOffMode, ClrScr);
    printf("\n\n\n\n");
    printf("There is no special function assigned to this screen.\n\n\n");
    printf("Type \"!\" to fork a subshell.\n\n\n");
    if (WaitKeyA() == '!')
        SubShell();
}

/*--------------------------------------------------*/
/* ReadG:           Utility Functions               */
/*--------------------------------------------------*/


/*       First some utility functions
 */

static void PositionCur(Entry, Pos)
	/* Position the cursor on the screen */
    GetEntT	*Entry;
    int		*Pos;
{
    cprintf("%s", tgoto(CurMove, Entry->x+(*Pos), Entry->y));
}

static int Min(x, y)
	/* Return the minimalue */
	int	x,y;
{
    if (x>y)
	return y;
    else
	return x;
}


static int CleanNum(Entry, Pos)
    GetEntT	*Entry;
    int		*Pos;

     /* If Entry is a numeric type, clean it up (i.e. re-format it).  This
	routine will reposition the cursor, if necessary.
	Return TRUE if successful, FALSE
	otherwise (i.e. overflow).  If overflow, also echo Bell, and set
	Pos to 0. */
{
#ifdef NESQL
    long	Decimal;		/* Biggest numeric type */
#else
    dec_t	Decimal;
#endif
    int		Success;
    char	*cp;

    if (IsNumeric(Entry->Type))  {
	Success = FALSE;	/* If string is all *'s, it is overflow */
	for(cp=Entry->Data; (*cp) != '\0' && !(Success = (*cp) != '*'); cp++);
 	if (Success)  {    /* If we haven't failed so far... */
	    AscToNum(Entry->Type, Entry->Data, &Decimal);  	/* Decimal type is big enough for int or long int */
	    Success = NumToAsc(Entry->Type, &Decimal, Entry->Data, Entry->Picture);
	}
	OutputGet(Entry->y, Entry->x, Entry->Data);
	if (!Success) {
	    (*Pos) = 0;
	    PositionCur(Entry, Pos);    /* This must be BEFORE bell */
	    cprintf("%s", Bell);
	} else
	    PositionCur(Entry, Pos);
	return Success;
    } else
   	return TRUE;		/* If not decimal, always succeeds */
}


static void AlignDP(Entry, Pos)
    GetEntT	*Entry;
    int		*Pos;

     /* Align the decimal point in Entry, i.e. re-format the decimal number
	and put the cursor on top of the decimal point (or at the end of the
	string if there is none)  */

{
    char   c;

    if (Entry->Data[*Pos] != '\0')  {    /* If they're allowed to type something */
	Entry->Data[*Pos] = '.';
    	CleanNum(Entry, Pos);
    	for((*Pos)=0; ((c=Entry->Data[*Pos]) != '\0') && c != '.'; (*Pos)++);
    	PositionCur(Entry, Pos);
    }
    cprintf("%s", RevOnMode);
}


/*       These function implement the different control keys.
 */

static void DeleteCh(Entry, Pos)
    GetEntT	*Entry;
    int		*Pos;
{
    int		i;

    if (Entry->CompPicture)
	cprintf("%s", Bell);
    else if (Entry->Data[(*Pos)] != '\0')
    {
    	for(i=(*Pos); Entry->Data[i+1] != '\0'; putch(Entry->Data[i++]))
	    Entry->Data[i] = Entry->Data[i+1];
    	putch(Entry->Data[i]=' ');
        PositionCur(Entry, Pos);
    }
}

static void InsertCh(Entry, Pos)
    GetEntT	*Entry;
    int		*Pos;
{
    int		i;

    if (Entry->CompPicture)
        cprintf("%s", Bell);
    else if (Entry->Data[(*Pos)] != '\0')
    {
    	putch(' ');
    	for(i=(*Pos); Entry->Data[i+1] != '\0'; )
     	    putch(Entry->Data[i++]);
    	for(i--; i>=(*Pos); i--)
  	    Entry->Data[i+1] = Entry->Data[i];
    	Entry->Data[i+1] = ' ';
    	PositionCur(Entry, Pos);
    }
}

static void RefreshK(Entry, Pos)
    GetEntT	*Entry;
    int		*Pos;
{
    Refresh();
    PositionCur(Entry, Pos);
    cprintf("%s", RevOnMode);
}

static void Restore(Entry, Pos)
    GetEntT	*Entry;
    int		*Pos;
{
    int		i;

    for(i=0; i<NumGets; i++)
	if (IsString(GetEntry[i].Type))
	    strcpy(GetEntry[i].Data, GetEntry[i].OldData);
	else 
	    NumToAsc(GetEntry[i].Type, GetEntry[i].OldData, GetEntry[i].Data, GetEntry[i].Picture);
    RefreshK(Entry, Pos);
}

static void NextFld(Field, Pos, AutoFinish, Done)
    int		*Field, *Pos;
    int		AutoFinish, *Done;
{
    GetEntT	*Entry;

    (*Pos) = 0;
    if (CleanNum(&GetEntry[(*Field)], Pos))  {
    	if (NumGets <= ++(*Field)) {
 	    if (AutoFinish)
	    	(*Done) = TRUE;
	    (*Field) = 0;
    	}
    	Entry = &GetEntry[(*Field)];
        PositionCur(Entry, Pos);
    }
}

static void FillBl(Entry, Pos)
    GetEntT	*Entry;
    int		*Pos;
{
    int		i;
    char	Picture;

    for(i=(*Pos); Entry->Data[i] != '\0'; i++)
    {
	if (i < Entry->PictureLen)
	    Picture = tolower(Entry->Picture[i]);
	else
	    Picture = 'x';
	if (Picture == 'x' || Picture == '!' || Picture == '#'
	    || IsNumeric(Entry->Type))  {
	    putch(' ');
	    Entry->Data[i] = ' ';
	}
	else
	    putch(Entry->Data[i]);
    }
    PositionCur(Entry, Pos);
}

static void 