
/*

               Dynamic Studio Professional C++ replay routine v1.01

                             (c) 1994 Per Lindh

        Developed in cooperation with the authors of Dynamic Studio.

            Dynamic Studio is a trademark of The Loom Syndicate.
             UltraSound is a trademark of Advanced Gravis Ltd.


                                   Email:

                             pt94pli@pt.hk-r.se


                                  Postage:

                                 Per Lindh
                              c/o Gabrielsson
                                Risanaesv. 9
                              S-37273 Ronneby
                                   Sweden


                                   Phone:

                              +46-(0)457-12063

Revision:
---------

    1.00 -  First public release (94-10-17).
            Bug reports by Bjorn Nordstrom.

    1.01 -  Note decoder didn't recognize notes without sample numbers
            if there wasn't a tone portamento. Lame!!
            Effects 1ch, 1dh and 1eh were added. Set master volume,
            slide master volume up, slide master volume down.
            These effects are in the new version of Dynamic Studio, which
            is not at the time of writing this released yet....
            Bug reports by Bjorn Nordstrom. (94-10-23).


About DSMPLAY:
--------------

DSMPLAY is C/C++ replay routine for modules created with Dynamic Studio
Professional. It's main purpose is to be instructive and serve as a
guide when you write your own replayer. Therefore, all optimizations of this
code that would result in less readable code has been omitted.

This code is COPYRIGHTED!!!.
In this particular case you may make as many copies as you like and spread
it anywhere in the universe except past the neutral zone and into the
Romulan territory. You may NOT modify the code and spread it on, but, you
may modify the code for your own amusement.
This code may not be used in commercial purposes without permission from
the author.

Another purpose of this code is to make it better and better, with help from
all users. Any user with suggestions of improvement, just send it to me
(preferrably with code examples and docs) and it will be implemented...

Support of other module formats will never happen.. DSM is the ONLY one!!

As in most players, this one clicks quite a bit if loop points are very
aggressive. The best hint in removing as many clicks as possible is to load
the module into Dynamic Studio, play around with sample loop and end points
until there are very little clicks left.
Another hint is to use twice the amount of channels required and use
channel flipping. That may give some inprovements BUT, then you'll lose all
oversampling a fewer number of channels can give. The fewer channels used,
the faster the GF1 loops around the voices and can interpolate between samples.

*/



#include    <conio.h>
#include    <dos.h>
#include    <stdio.h>
#include    <stdlib.h>
#include    <malloc.h>
#include    <string.h>
#include    "types.h"       //  Standard typedefs
#include    "dsmplay.h"
#include    "dsmdefs.h"

//
//
//  Structures, macros, definitions, enumerations and constants
//
//

#define FORBID_IRQS asm pushf;  asm cli;
#define ENABLE_IRQS asm popf;

const ubyte OCR1         = 0x20;    //  8259-1 operation control register
const ubyte IMR1         = 0x21;    //  8259-1 mask register
const ubyte OCR2         = 0xA0;    //  8259-2 operation control register
const ubyte IMR2         = 0xA1;    //  8259-2 mask register
const ubyte TRUE         = 1;
const ubyte FALSE        = 0;

typedef void    interrupt (far *PVI)(...);

//  This should be commented if using a Dynamic Studio module. If you have
//  converted, say a Fasttracker module using MAKEDSM you may uncomment this.

//#define USE_ARPEGGIO


//
//
//  Global data structures and data tables
//
//


//
//  5 octave period table
//

word    PeriodTable[16*60] = {
    1712,1616,1524,1440,1356,1280,1208,1140,1076,1016, 960, 906,    //  FT00OT0
     856, 808, 762, 720, 678, 640, 604, 570, 538, 508, 480, 453,    //  FT00OT1
     428, 404, 381, 360, 339, 320, 302, 285, 269, 254, 240, 226,    //  FT00OT2
     214, 202, 190, 180, 170, 160, 151, 143, 135, 127, 120, 113,    //  FT00OT3
     107, 101,  95,  90,  85,  80,  75,  71,  67,  63,  60,  56,    //  FT00OT4

    1700,1604,1514,1430,1348,1274,1202,1134,1070,1010, 954, 900,    //  FT+1OT0
     850, 802, 757, 715, 674, 637, 601, 567, 535, 505, 477, 450,    //  FT+1OT1
     425, 401, 379, 357, 337, 318, 300, 284, 268, 253, 239, 225,    //  FT+1OT2
     213, 201, 189, 179, 169, 159, 150, 142, 134, 126, 119, 113,    //  FT+1OT3
     106, 100,  94,  89,  84,  79,  75,  71,  67,  63,  59,  56,    //  FT+1OT4

    1688,1592,1504,1418,1340,1264,1194,1126,1064,1004, 948, 894,    //  FT+2OT0
     844, 796, 752, 709, 670, 632, 597, 563, 532, 502, 474, 447,    //  FT+2OT1
     422, 398, 376, 355, 335, 316, 298, 282, 266, 251, 237, 224,    //  FT+2OT2
     211, 199, 188, 177, 167, 158, 149, 141, 133, 125, 118, 112,    //  FT+2OT3
     105,  99,  94,  88,  83,  79,  74,  70,  66,  62,  59,  56,    //  FT+2OT4

    1676,1582,1492,1408,1330,1256,1184,1118,1056, 996, 940, 888,    //  FT+3OT0
     838, 791, 746, 704, 665, 628, 592, 559, 528, 498, 470, 444,    //  FT+3OT1
     419, 395, 373, 352, 332, 314, 296, 280, 264, 249, 235, 222,    //  FT+3OT2
     209, 198, 187, 176, 166, 157, 148, 140, 132, 125, 118, 111,    //  FT+3OT3
     104,  99,  93,  88,  83,  78,  74,  70,  66,  62,  59,  55,    //  FT+3OT4

    1664,1570,1482,1398,1320,1246,1176,1110,1048, 990, 934, 882,    //  FT+4OT0
     832, 785, 741, 699, 660, 623, 588, 555, 524, 495, 467, 441,    //  FT+4OT1
     416, 392, 370, 350, 330, 312, 294, 278, 262, 247, 233, 220,    //  FT+4OT2
     208, 196, 185, 175, 165, 156, 147, 139, 131, 124, 117, 110,    //  FT+4OT3
     104,  98,  92,  87,  82,  78,  73,  69,  65,  62,  58,  55,    //  FT+4OT4

    1652,1558,1472,1388,1310,1238,1168,1102,1040, 982, 926, 874,    //  FT+5OT0
     826, 779, 736, 694, 655, 619, 584, 551, 520, 491, 463, 437,    //  FT+5OT1
     413, 390, 368, 347, 328, 309, 292, 276, 260, 245, 232, 219,    //  FT+5OT2
     206, 195, 184, 174, 164, 155, 146, 138, 130, 123, 116, 109,    //  FT+5OT3
     103,  97,  92,  87,  82,  77,  73,  69,  65,  61,  58,  54,    //  FT+5OT4

    1640,1548,1460,1378,1302,1228,1160,1094,1032, 974, 920, 868,    //  FT+6OT0
     820, 774, 730, 689, 651, 614, 580, 547, 516, 487, 460, 434,    //  FT+6OT1
     410, 387, 365, 345, 325, 307, 290, 274, 258, 244, 230, 217,    //  FT+6OT2
     205, 193, 183, 172, 163, 154, 145, 137, 129, 122, 115, 109,    //  FT+6OT3
     102,  96,  91,  86,  81,  77,  72,  68,  64,  61,  57,  54,    //  FT+6OT4

    1628,1536,1450,1368,1292,1220,1150,1086,1026, 968, 914, 862,    //  FT+7OT0
     814, 768, 725, 684, 646, 610, 575, 543, 513, 484, 457, 431,    //  FT+7OT1
     407, 384, 363, 342, 323, 305, 288, 272, 256, 242, 228, 216,    //  FT+7OT2
     204, 192, 181, 171, 161, 152, 144, 136, 128, 121, 114, 108,    //  FT+7OT3
     102,  96,  90,  85,  80,  76,  72,  68,  64,  60,  57,  54,    //  FT+7OT4

    1814,1712,1616,1524,1440,1356,1280,1208,1140,1076,1016, 960,    //  FT-8OT0
     907, 856, 808, 762, 720, 678, 640, 604, 570, 538, 508, 480,    //  FT-8OT1
     453, 428, 404, 381, 360, 339, 320, 302, 285, 269, 254, 240,    //  FT-8OT2
     226, 214, 202, 190, 180, 170, 160, 151, 143, 135, 127, 120,    //  FT-8OT3
     113, 107, 101,  95,  90,  85,  80,  75,  71,  67,  63,  60,    //  FT-8OT4

    1800,1700,1604,1514,1430,1350,1272,1202,1134,1070,1010, 954,    //  FT-7OT0
     900, 850, 802, 757, 715, 675, 636, 601, 567, 535, 505, 477,    //  FT-7OT1
     450, 425, 401, 379, 357, 337, 318, 300, 284, 268, 253, 238,    //  FT-7OT2
     225, 212, 200, 189, 179, 169, 159, 150, 142, 134, 126, 119,    //  FT-7OT3
     112, 106, 100,  94,  89,  84,  79,  75,  71,  67,  63,  59,    //  FT-7OT4

    1788,1688,1592,1504,1418,1340,1264,1194,1126,1064,1004, 948,    //  FT-6OT0
     894, 844, 796, 752, 709, 670, 632, 597, 563, 532, 502, 474,    //  FT-6OT1
     447, 422, 398, 376, 355, 335, 316, 298, 282, 266, 251, 237,    //  FT-6OT2
     223, 211, 199, 188, 177, 167, 158, 149, 141, 133, 125, 118,    //  FT-6OT3
     111, 105,  99,  94,  88,  83,  79,  74,  70,  66,  62,  59,    //  FT-6OT4

    1774,1676,1582,1492,1408,1330,1256,1184,1118,1056, 996, 940,    //  FT-5OT0
     887, 838, 791, 746, 704, 665, 628, 592, 559, 528, 498, 470,    //  FT-5OT1
     444, 419, 395, 373, 352, 332, 314, 296, 280, 264, 249, 235,    //  FT-5OT2
     222, 209, 198, 187, 176, 166, 157, 148, 140, 132, 125, 118,    //  FT-5OT3
     111, 104,  99,  93,  88,  83,  78,  74,  70,  66,  62,  59,    //  FT-5OT4

    1762,1664,1570,1482,1398,1320,1246,1176,1110,1048, 988, 934,    //  FT-4OT0
     881, 832, 785, 741, 699, 660, 623, 588, 555, 524, 494, 467,    //  FT-4OT1
     441, 416, 392, 370, 350, 330, 312, 294, 278, 262, 247, 233,    //  FT-4OT2
     220, 208, 196, 185, 175, 165, 156, 147, 139, 131, 123, 117,    //  FT-4OT3
     110, 104,  98,  92,  87,  82,  78,  73,  69,  65,  61,  58,    //  FT-4OT4

    1750,1652,1558,1472,1388,1310,1238,1168,1102,1040, 982, 926,    //  FT-3OT0
     875, 826, 779, 736, 694, 655, 619, 584, 551, 520, 491, 463,    //  FT-3OT1
     437, 413, 390, 368, 347, 328, 309, 292, 276, 260, 245, 232,    //  FT-3OT2
     219, 206, 195, 184, 174, 164, 155, 146, 138, 130, 123, 116,    //  FT-3OT3
     109, 103,  97,  92,  87,  82,  77,  73,  69,  65,  61,  58,    //  FT-3OT4

    1736,1640,1548,1460,1378,1302,1228,1160,1094,1032, 974, 920,    //  FT-2OT0
     868, 820, 774, 730, 689, 651, 614, 580, 547, 516, 487, 460,    //  FT-2OT1
     434, 410, 387, 365, 345, 325, 307, 290, 274, 258, 244, 230,    //  FT-2OT2
     217, 205, 193, 183, 172, 163, 154, 145, 137, 129, 122, 115,    //  FT-2OT3
     108, 102,  96,  91,  86,  81,  77,  72,  68,  64,  61,  57,    //  FT-2OT4

    1724,1628,1536,1450,1368,1292,1220,1150,1086,1026, 968, 914,    //  FT-1OT0
     862, 814, 768, 725, 684, 646, 610, 575, 543, 513, 484, 457,    //  FT-1OT1
     431, 407, 384, 363, 342, 323, 305, 288, 272, 256, 242, 228,    //  FT-1OT2
     216, 203, 192, 181, 171, 161, 152, 144, 136, 128, 121, 114,    //  FT-1OT3
     108, 101,  96,  90,  85,  80,  76,  72,  68,  64,  60,  57     //  FT-1OT4
};


//
//  The BPM table is precalculated to use a minimum amount of CPU time.
//  There are 255-32+1=224 different values divided in two sections:
//
//  Section #1: BPM rates 32-124        (320us Timer #2)
//
//              Table uses formula: 256-(125*(0.02/0.00032)/BPM)
//              Equals to 256-(7812.5/BPM)
//
//  Section #2: BPM rates from 125-255  (80us Timer #1)
//
//              Table uses formula: 256-(125*(0.02/0.00008)/BPM)
//              Equals to 256-(31250/BPM)
//

ubyte   BPMTable[224] = {
     12,  19,  26,  33,  39,  45,  50,  56,  61,  65,   //  TIMER2
     70,  74,  78,  82,  86,  90,  93,  97, 100, 103,   //  TIMER2
    106, 109, 111, 114, 116, 119, 121, 124, 126, 128,   //  TIMER2
    130, 132, 134, 136, 138, 139, 141, 143, 144, 146,   //  TIMER2
    147, 149, 150, 152, 153, 155, 156, 157, 158, 160,   //  TIMER2
    161, 162, 163, 164, 165, 166, 167, 168, 169, 170,   //  TIMER2
    171, 172, 173, 174, 175, 175, 176, 177, 178, 179,   //  TIMER2
    179, 180, 181, 182, 182, 183, 184, 184, 185, 186,   //  TIMER2
    186, 187, 187, 188, 189, 189, 190, 190, 191, 191,   //  TIMER2
    192, 192, 193,                                      //  TIMER2

      6,   8,  10,  12,  14,  16,  17,  19,  21,  23,   //  TIMER1
     25,  26,  28,  30,  31,  33,  34,  36,  37,  39,   //  TIMER1
     40,  42,  43,  45,  46,  48,  49,  50,  52,  53,   //  TIMER1
     54,  56,  57,  58,  59,  61,  62,  63,  64,  65,   //  TIMER1
     67,  68,  69,  70,  71,  72,  73,  74,  75,  76,   //  TIMER1
     77,  78,  79,  80,  81,  82,  83,  84,  85,  86,   //  TIMER1
     87,  88,  89,  90,  91,  92,  92,  93,  94,  95,   //  TIMER1
     96,  97,  97,  98,  99, 100, 101, 101, 102, 103,   //  TIMER1
    104, 104, 105, 106, 106, 107, 108, 109, 109, 110,   //  TIMER1
    111, 111, 112, 113, 113, 114, 115, 115, 116, 116,   //  TIMER1
    117, 118, 118, 119, 120, 120, 121, 121, 122, 122,   //  TIMER1
    123, 124, 124, 125, 125, 126, 126, 127, 127, 128,   //  TIMER1
    128, 129, 129, 130, 130, 131, 131, 132, 132, 133,   //  TIMER1
    133                                                 //  TIMER1
};


//
//  Vibrato sinewave table
//

ubyte   VibratoTable[32] = {
      0,  24,  49,  74,  97, 120, 141, 161, 180, 197, 212, 224, 235, 244,
    250, 253, 255, 253, 250, 244, 235, 224, 212, 197, 180, 161, 141, 120,
     97,  74,  49,  24
};

//
//  Logarithmic volume table indexed on linear volumes 0-64
//

uword   VolumeTable[65] = {
    20000, 45056, 47104, 48128, 48640, 49152, 50176, 51200,
    52224, 53248, 53760, 54272, 54784, 55296, 55808, 56320,
    56832, 57344, 57600, 57856, 58112, 58368, 58624, 58880,
    58936, 59192, 59448, 59704, 59960, 60216, 60472, 60728,
    60984, 61568, 61696, 61824, 61952, 62080, 62208, 62336,
    62464, 62592, 62720, 62848, 62976, 63104, 63232, 63360,
    63488, 63616, 63744, 63872, 64000, 64128, 64256, 64384,
    64512, 64640, 64768, 64896, 65024, 65152, 65280, 65408,
    65520
};


//
//  Latch values for DMA channels 1-7
//

ubyte   DMALatches[7] = {
    1, 0, 2, 0, 3, 4, 5
};


//
//  IRQ data for IRQ 0-15
//

IRQEntry    IRQTbl[16] = {
    { 0, 0xFE, 0x60, OCR1, IMR1 },  //  IRQ00
    { 0, 0xFD, 0x61, OCR1, IMR1 },  //  IRQ01
    { 1, 0xFB, 0x62, OCR1, IMR1 },  //  IRQ02
    { 3, 0xF7, 0x63, OCR1, IMR1 },  //  IRQ03
    { 0, 0xEF, 0x64, OCR1, IMR1 },  //  IRQ04
    { 2, 0xDF, 0x65, OCR1, IMR1 },  //  IRQ05
    { 0, 0xBF, 0x66, OCR1, IMR1 },  //  IRQ06
    { 4, 0x7F, 0x67, OCR1, IMR1 },  //  IRQ07
    { 0, 0xFE, 0x60, OCR2, IMR2 },  //  IRQ08
    { 0, 0xFD, 0x61, OCR2, IMR2 },  //  IRQ09
    { 0, 0xFB, 0x62, OCR2, IMR2 },  //  IRQ10
    { 5, 0xF7, 0x63, OCR2, IMR2 },  //  IRQ11
    { 6, 0xEF, 0x64, OCR2, IMR2 },  //  IRQ12
    { 0, 0xDF, 0x65, OCR2, IMR2 },  //  IRQ13
    { 0, 0xBF, 0x66, OCR2, IMR2 },  //  IRQ14
    { 7, 0x7F, 0x67, OCR2, IMR2 },  //  IRQ15
};

//
//  Frequency divisors for different voice setup's
//  They are used when calculating the GUS frequency
//

uword   FreqDivisor[19] = {
    44100,      //  14 active voices
    41160,      //  15 active voices
    38587,      //  16 active voices
    36317,      //  17 active voices
    34300,      //  18 active voices
    32494,      //  19 active voices
    30870,      //  20 active voices
    29400,      //  21 active voices
    28063,      //  22 active voices
    26843,      //  23 active voices
    25725,      //  24 active voices
    24696,      //  25 active voices
    23746,      //  26 active voices
    22866,      //  27 active voices
    22050,      //  28 active voices
    21289,      //  29 active voices
    20580,      //  30 active voices
    19916,      //  31 active voices
    19293       //  31 active voices
};


//
//  Ultrasound hardware ports, registers & setup
//

uword   basePort;           //  UltraSound base port
uword   dramDMA;            //  DMA channel used when playing
uword   gf1IRQ;             //  IRQ number used for interaction with GUS
uword   pageSelect;         //  Page select register, select new voice
uword   regSelect;          //  Register select
uword   dataLo;             //  Global data low
uword   dataHi;             //  Global data high
uword   irqStatus;          //  IRQ status register
uword   dramPort;           //  DRAM I/O
uword   irqControl;         //  IRQ control register
uword   timerControl;       //  Timer control register
uword   timerData;          //  Timer data register

ubyte   lineCounter  = 0;   //  Inbetween line counter 0 -> patSpeed
ubyte   patPos       = 0;   //  Pattern position 0-63
ubyte   patSpeed     = 6;   //  Pattern speed
ubyte   songTempo    = 125; //  Beats per minute rate
ubyte   patBreak     = 0;   //  Pattern break flag
uword   patDelayTime = 0;   //  Pattern delay time counter
ubyte   gusOk        = 0;   //  GUS init Ok ??
ubyte   effMask      = 0xff;//  Effects mask value
ubyte   curEffValue;        //  Current effect databyte (for effects loop)
ubyte   songPos;            //  Actual song position
ubyte   curChn;             //  Current channel being updated by effects
ubyte   curPat;             //  Current pattern
ubyte   numChn;             //  Number of active channels
ubyte   songLen;            //  Song length
ubyte   numSmp;             //  Samples used in module
ubyte   patLoop = 0;        //  Pattern loop control
ubyte   patLoopPos;         //  Pattern loop position
word    curPeriod;          //  Current period
uword   masterVolume;       //  Module's mastervolume

ubyte   mixImage  = 0x0b;   //  Image of mix register
ubyte   voices;             //  Number of voices used
ubyte   timerMask = 0x04;   //  Mask value of timer
ubyte   timerPort = TIMER1; //  Current timer port
ubyte   timerBPM  = 6;      //  Current speed of current timer

ubyte   SongOrders[128];    //  Orders for song
ubyte   BalanceData[16];    //  Balance data

uword   FreqTable[1815];    //  Pre-calculated frequency table

PVI      oldIntVec;         //  Old interrupt vector

DSMHeader       dHead;          //  Dynamic Module Header
ubyte far       *Notes[128];    //  Pattern pointers
Channel far     *Chans;         //  Channel information
Channel far     *actChan;       //  Active channel
Sample far      *SmpInfo;       //  Sample information

//extern  ubyte   muteChans[];

//
//
//  Prototypes
//
//

//  UltraSound lowlevel functions

void    GUS_close( void );
void    GUS_delay( void );
void    GUS_disableOutput( void );
void    GUS_enableOutput( void );
ubyte   GUS_peekByte( ulong address );
void    GUS_pokeByte( ulong address, ubyte data );
ubyte   GUS_probe( void );
void    GUS_rampLinearVolume( uword end_idx, ubyte rate, ubyte mode );
void    GUS_rampVolume( uword start, uword end, ubyte rate, ubyte mode );
void    GUS_reset( void );
void    GUS_resetIRQHandler( void );
void    GUS_setFrequency( word period );
void    GUS_setInterface( void );
void    GUS_setIRQHandler( void );
void    GUS_setPanPos( ubyte panPos );

//  Module player functions

#ifdef  USE_ARPEGGIO
void    Arpeggio( void );
#endif

ubyte   AllocPatterns( uword numPats );
void    BuildFreqTable( void );
void    DefaultVolumeSlideUp( void );
void    DumpSample( ubyte far *address, ulong SmpLoc, uword Length );
ubyte   InitLoader( void );
void    FinePortaDown( void );
void    FinePortaUp( void );
void    FineVolumeSlideDown( void );
void    FineVolumeSlideUp( void );
void    Glissando( void );
void    NoteCut( void );
void    NoteDelay( void );
void    PanControl( void );
void    ParsePatternLine( void );
void    PatternBreak( void );
void    PatternDelay( void );
void    PatternLoop( void );
void    PortamentoDown( void );
void    PortamentoDown5( void );
void    PortamentoUp( void );
void    PortamentoUp5( void );
void    PositionJump( void );
void    ReplayCallback( void );
void    RetrigChannel( void );
void    RetrigSample( void );
void    SampleOffset( void );
void    SetMasterVolume( void );
void    SetSpeedOrBPM( void );
void    SetTremoloWaveForm( void );
void    SetVibratoWaveForm( void );
void    SetVolume( void );
void    SetVolumeSmpOffset( void );
void    Simulate3D( void );
void    SlideMasterDown( void );
void    SlideMasterUp( void );
void    TonePortamento( ubyte noInit );
void    Tremolo( void );
void    UpdateLineSfx( void );
void    UpdateRealSfx( void );
void    Vibrato( ubyte noInit );
void    VolumeSlide( void );

void    interrupt far   gf1IRQHandler( void );


//
//  Set frequency of active voice
//

void    GUS_setFrequency( word period )
{
    if  ( period > 1814 )   //  Skip periods that go beyond octave 0
    {
        period = 1814;
    }

    outp( regSelect, SET_FREQUENCY );   //  Select frequency register
    outpw( dataLo, FreqTable[period] ); //  Send the new frequency
}

//
//  Set the pan position of current active voice.
//  Pan position is in the range 0-F
//

void    GUS_setPanPos( ubyte panPos )
{
    FORBID_IRQS;

    outp( regSelect, 0x0c );    //  Select balance control register
    outp( dataHi, panPos );     //  Send new pan position

    ENABLE_IRQS;
}

//
//  Try to detect an UltraSound at current base port setting
//
//  Returns TRUE if found, else returns FALSE
//

ubyte   GUS_probe( void )
{
    ubyte   save0, save1, val0, val1;

    pageSelect   = basePort + GF1_PAGE;
    regSelect    = basePort + GF1_REG_SELECT;
    dataLo       = basePort + GF1_DATA_LOW;
    dataHi       = basePort + GF1_DATA_HI;
    irqStatus    = basePort + GF1_IRQ_STAT;
    dramPort     = basePort + GF1_DRAM;
    irqControl   = basePort + GF1_IRQ_CTRL;
    timerControl = basePort + GF1_TIMER_CTRL;
    timerData    = basePort + GF1_TIMER_DATA;

//  Put card in a reset state

    outp( regSelect, MASTER_RESET );
    outp( dataHi, 0 );

    GUS_delay();
    GUS_delay();

//  Release reset state

    outp( regSelect, MASTER_RESET );
    outp( dataHi, 1 );

    GUS_delay();
    GUS_delay();

//  Save bytes at address 0 & 1 in DRAM

    save0 = GUS_peekByte( 0L );
    save1 = GUS_peekByte( 1L );

//  Poke in some new reference values

    GUS_pokeByte( 0L, 0xAA );
    GUS_pokeByte( 1L, 0x55 );

//  Now try to read the recently poke values from DRAM

    val0 = GUS_peekByte( 0L );
    val1 = GUS_peekByte( 1L );

//  Restore the old values at address 0 & 1

    GUS_pokeByte( 0L, save0 );
    GUS_pokeByte( 1L, save1 );

    if  (( val0 == 0xAA ) && ( val1 == 0x55 ))
    {
        return( TRUE );     //  YESSS!! There was an UltraSound..
    }
    else
    {
        return( FALSE );    //  NOPE!! Go out and buy one..
    }
}


//
//  This function serves as a 4.8 microsecond (or longer) delay
//  It is necessary for GUS's self-modifying bits
//

void    GUS_delay( void )
{
    inp( dramPort );
    inp( dramPort );
    inp( dramPort );
    inp( dramPort );
    inp( dramPort );
    inp( dramPort );
    inp( dramPort );
}

//
//  Read a byte from UltraSound DRAM
//

ubyte   GUS_peekByte( ulong address )
{
    ubyte   retValue;

    FORBID_IRQS;

    outp( regSelect, 0x43 );                //  Select DRAM low
    outpw( dataLo, address & 0xffff );      //  Send low 16 bits of address
    outp( regSelect, 0x44 );                //  Select DRAM high
    outp( dataHi, (ubyte)(address >> 16) ); //  Send high 8 bits of address

    retValue = inp( dramPort );             //  Read from DRAM

    ENABLE_IRQS;

    return( retValue );
}

//
//  Write a byte to UltraSound DRAM
//

void    GUS_pokeByte( ulong address, ubyte data )
{
    FORBID_IRQS;

    outp( regSelect, 0x43 );                //  Select DRAM low
    outpw( dataLo, address & 0xffff );      //  Send low 16 bits of address
    outp( regSelect, 0x44 );                //  Select DRAM high
    outp( dataHi, (ubyte)(address >> 16) ); //  Send high 8 bits of address
    outp( dramPort, data );                 //  Send data to DRAM

    ENABLE_IRQS;
}


//
//  Open and initialize the UltraSound
//

ubyte   GUS_open( ubyte numVoices )
{
    ubyte   retValue;
    char    *envPtr;
    uword   dummy;

    if  ( numVoices < 14 )  //  We can't use less than 14 voices
    {
        numVoices = 14;
    }

    if  ( numVoices <= 32 ) //  and no more than 32
    {
        voices = numVoices;

        if  (( envPtr = getenv( "ULTRASND" )) != 0 )    //  Get ULTRASND
        {
            //  Parse the environment variable into pieces
            sscanf( envPtr, "%x,%d,%d,%d", &basePort, &dramDMA, &dummy, &gf1IRQ );

            if  ( GUS_probe() )         //  Detect UltraSound
            {
                BuildFreqTable();       //  Calculate frequency table
                GUS_disableOutput();    //  Disable any output
                GUS_reset();            //  Full reset
                GUS_setInterface();     //  Program the GUS for IRQs
                GUS_setIRQHandler();    //  Program the PC for IRQs
                GUS_enableOutput();     //  Enable any output

                retValue = LOAD_OK;
            }
            else
            {
                retValue = LOAD_NOGUS;
            }
        }
        else
        {
            retValue = LOAD_NOENVVAR;
        }
    }
    else
    {
        retValue = LOAD_BADVOICES;
    }

    return( retValue );
}


//
//  Close down and deinitialize the UltraSound
//

void    GUS_close( void )
{
    GUS_disableOutput();    //  Disable any output
    GUS_reset();            //  Full reset
    GUS_resetIRQHandler();  //  Program the PC for original IRQs
}


//
//  Enable any output from the UltraSound
//

void    GUS_enableOutput( void )
{
    outp( basePort, mixImage &= ~ENABLE_OUTPUT );
}


//
//  Disable any output from the UltraSound
//

void    GUS_disableOutput( void )
{
    outp( basePort, mixImage |= ENABLE_OUTPUT );
}


//
//  Full UltraSound reset and initialization of all voices
//

void    GUS_reset( void )
{
//
//  By setting address 0 & 1 to 0 the voices that are pointing at those
//  addresses will not get summed into the output
//  2 addresses for a 16 bit voice
//

    GUS_pokeByte( 0L, 0 );
    GUS_pokeByte( 1L, 0 );

    FORBID_IRQS;

//  Turn on the reset line of the GF1

    outp( regSelect, MASTER_RESET );
    outp( dataHi, 0 );

    for ( ubyte i = 0; i < 10; i++ )
        GUS_delay();

//  Turn off the reset line of the GF1

    outp( regSelect, MASTER_RESET );
    outp( dataHi, GF1_MASTER_RESET );

    for ( i = 0; i < 10; i++ )
        GUS_delay();

//  Clear DMA, TIMER and SAMPLE interrupts

    outp( regSelect, DMA_CONTROL );
    outp( dataHi, 0 );                      //  Clear DMA Control
    outp( regSelect, TIMER_CONTROL );
    outp( dataHi, 0 );                      //  Clear TIMER Control
    outp( regSelect, SAMPLE_CONTROL );
    outp( dataHi, 0 );                      //  Clear SAMPLE Control

//  Set the number of active voices

    outp( regSelect, SET_VOICES );
    outp( dataHi, (ubyte)(voices - 1) | 0xc0 );

//  Clear interrupts on voices

    inp( irqStatus );
    outp( regSelect, DMA_CONTROL );
    inp( dataHi );                          //  Clear DMA IRQs
    outp( regSelect, SAMPLE_CONTROL );
    inp( dataHi );                          //  Clear SAMPLE IRQs
    outp( regSelect, GET_IRQV );
    inp( dataHi );                          //  Clear VOICE IRQs
	
    for ( i = 0; i < voices; i++ )
    {
        outp( pageSelect, i );  //  Select voice to adjust

        outp( regSelect, SET_CONTROL );
        outp( dataHi, VOICE_STOPPED | STOP_VOICE );     //  Stop the voice
        outp( regSelect, SET_VOLUME_CONTROL );
        outp( dataHi, VOLUME_STOPPED | STOP_VOLUME );   //  Stop the voice
		
        GUS_delay();

        //  Initialize all voice specific registers (not necessary, but cool)
        outp( regSelect, SET_FREQUENCY );
        outpw( dataLo, 0x400 );             //  Set frequency (not Hz)
        outp( regSelect, SET_START_HIGH );
        outpw( dataLo, 0 );
        outp( regSelect, SET_START_LOW );
        outpw( dataLo, 0 );
        outp( regSelect, SET_END_HIGH );
        outpw( dataLo, 0 );
        outp( regSelect, SET_END_LOW );
        outpw( dataLo, 0 );
        outp( regSelect, SET_VOLUME_RATE );
        outp( dataHi, 0x01 );
        outp( regSelect, SET_VOLUME_START );
        outp( dataHi, 0x10 );
        outp( regSelect, SET_VOLUME_END );
        outp( dataHi, 0xe0 );
        outp( regSelect, SET_VOLUME );
        outpw( dataLo, 0 );

        outp( regSelect, SET_ACC_HIGH );
        outpw( dataLo, 0 );
        outp( regSelect, SET_ACC_LOW );
        outpw( dataLo, 0 );
        outp( regSelect, SET_BALANCE );
        outp( dataHi, 0x07 );
    }

    inp( irqStatus );
    outp( regSelect, DMA_CONTROL );
    inp( dataHi );
    outp( regSelect, SAMPLE_CONTROL );
    inp( dataHi );
    outp( regSelect, GET_IRQV );
    inp( dataHi );
	
//  Set up GF1 Chip for interrupts & enable DACs

    outp( regSelect, MASTER_RESET );
    outp( dataHi, GF1_MASTER_RESET | GF1_OUTPUT_ENABLE | GF1_MASTER_IRQ );

    ENABLE_IRQS;
}

//
//  Initialize the interface between UltraSound and PC
//

void    GUS_setInterface( void )
{
    ubyte   gf1, dram = 0;
    ubyte   irq_ctrl = 0, dma_ctrl = 0;

    FORBID_IRQS;

    gf1 = IRQTbl[gf1IRQ].Latch;

    if  ( dramDMA != 0 )
    {
        dram = DMALatches[dramDMA - 1];
    }

    irq_ctrl |= gf1 | 0x40;
    dma_ctrl |= dram | 0x40;

//  Set up for Digital ASIC

    outp( basePort + 0x0f, 5 ); //  Register controls, board rev 3.4+
    outp( basePort, mixImage );
    outp( irqControl, 0 );
    outp( basePort + 0x0f, 0 );

//  First do DMA control register

    outp( basePort, mixImage );
    outp( irqControl, dma_ctrl | 0x80 );

//  IRQ control register

    outp( basePort, mixImage | 0x40 );
    outp( irqControl, irq_ctrl );
	
//  First do DMA control register (again???)

    outp( basePort, mixImage );
    outp( irqControl, dma_ctrl );

//  IRQ control register (again???)

    outp( basePort, mixImage | 0x40 );
    outp( irqControl, irq_ctrl );
	
//  just to Lock out writes to IRQ/DMA register ...

    outp( pageSelect, 0 );

//  enable output & irq, disable line & mic input

    outp( basePort, mixImage |= 0x09 );

//  just to Lock out writes to IRQ/DMA register ...

    outp( pageSelect, 0x0 );

    ENABLE_IRQS;
}


//
//  Unmask interrupt and reinstall old handler
//

void    GUS_resetIRQHandler( void )
{
    uword   temp = gf1IRQ;

    if  (( gf1IRQ != 2 ) && ( gf1IRQ != 0 ))
    {
        //  Remask GF1 interrupt
        outp( IRQTbl[gf1IRQ].IMR, inp( IRQTbl[gf1IRQ].IMR ) | ~IRQTbl[gf1IRQ].Mask );
    }

    if  ( temp != 0 )
    {
        temp += temp > 7 ? 0x68 : 0x08;

        setvect( temp, oldIntVec );
    }
}

//
//  This is the main interrupt handler
//

void    interrupt far   gf1IRQHandler(...)
{
    ubyte   irqSource;

//  clear PC's interrupt controller(s)

    outp( IRQTbl[gf1IRQ].OCR, IRQTbl[gf1IRQ].SpecEOI );

//  gotta send EOI to BOTH controllers

    if  ( gf1IRQ > 7 )
    {
        outp( OCR1, EOI );
    }

    while   ( TRUE )
    {
        if  (( irqSource = inp( irqStatus )) == 0 )
        {
            break;
        }

        if  ( irqSource & timerMask )
        {
            FORBID_IRQS;

            outp( regSelect, GET_IRQV );
            inp( dataHi );

            ReplayCallback();

            outp( regSelect, timerPort );
            outp( dataHi, timerBPM );
            outp( regSelect, TIMER_CONTROL );
            outp( dataHi, 0 );
            outp( dataHi, timerMask );

            ENABLE_IRQS;
        }
    }
}

//
//  Install a new IRQ handler and unmask it
//

void    GUS_setIRQHandler( void )
{
    uword   temp = gf1IRQ;

    if  ( temp != 0 )
    {
        temp += temp > 7 ? 0x68 : 0x08;

        oldIntVec = getvect( temp );
        setvect( temp, gf1IRQHandler );

        if  ( gf1IRQ != 0 )
        {
            //  Unmask GF1 IRQ
            outp( IRQTbl[gf1IRQ].IMR, inp( IRQTbl[gf1IRQ].IMR ) & IRQTbl[gf1IRQ].Mask );

            //  Send a specific EOI in case of pending IRQ
            outp( IRQTbl[gf1IRQ].OCR, IRQTbl[gf1IRQ].SpecEOI);
            }

        if  ( gf1IRQ > 7 )
        {
            //  Unmask IRQ 2 from 1st controller if using 2nd controller
            outp( IRQTbl[2].IMR, inp( IRQTbl[2].IMR ) & IRQTbl[2].Mask );

            //  Send a specific EOI in case of pending IRQ
            outp( IRQTbl[2].OCR, IRQTbl[2].SpecEOI );
        }
    }
}


//
//  Start timer 1 on UltraSound
//

void    GUS_startTimer( void )
{
    FORBID_IRQS;

    outp( regSelect, timerPort );       //  Select timer
    outp( dataHi, timerBPM );           //  Send IRQ speed
    outp( regSelect, TIMER_CONTROL );   //  Select control register
    outp( dataHi, timerMask );          //  Send timer mask
    outp( timerControl, 4 );
    outp( timerData, 3 );

    ENABLE_IRQS;
}


//
//  Stop one of UltraSound's two timers
//

void    GUS_stopTimer( void )
{
    FORBID_IRQS;

    outp( regSelect, TIMER_CONTROL );
    outp( dataHi, 0 );

    ENABLE_IRQS;
}


//
//  Ramp from a logarithmic volume to another at specified rate
//
//  Volume ramping is a nice way to adjust a voice's output amplitude
//  because when the DAC's output volume suddenly changes many clicks,
//  will not be heard.
//

void    GUS_rampVolume( uword start, uword end, ubyte rate, ubyte mode )
{
    uword   begin = start;
    ubyte   vmode;

    if  ( start == end )
    {
        return;
    }

    mode &= ~( VL_IRQ_PENDING | VC_ROLLOVER | STOP_VOLUME | VOLUME_STOPPED );

    if  ( start > end )
    {
        //  flip start & end if decreasing numbers ...
		start = end;
        end   = begin;
        mode |= VC_DIRECT;  //  Set decreasing volume mode bits
     }

//  Looping below 1024 or greater that 64512 can cause strange things

    if  ( start < 1024 )
    {
        start = 1024;
    }

    if  ( end > 64512 )
    {
        end = 64512;
    }

    outp( regSelect, SET_VOLUME_RATE );
    outp( dataHi, rate );
    outp( regSelect, SET_VOLUME_START );
    outp( dataHi, (ubyte)( start >> 8 ));
    outp( regSelect, SET_VOLUME_END );
    outp( dataHi, (ubyte)( end >> 8 ));

//  Set the current volume to the start volume

    outp( regSelect, SET_VOLUME );
    outpw( dataLo, begin );

//  Start ramping!!!!!!!

    outp( regSelect, SET_VOLUME_CONTROL );
    outp( dataHi, mode );

    GUS_delay();

    outp( dataHi, mode );
}

//
//  Ramp linear volumes from 0-64
//

void    GUS_rampLinearVolume( uword end_idx, ubyte rate, ubyte mode )
{
    uword   volume;
    ubyte   vmode;

    outp( regSelect, GET_VOLUME_CONTROL );

    vmode = inp( dataHi ) | ( VOLUME_STOPPED | STOP_VOLUME );

    outp( regSelect, SET_VOLUME_CONTROL );
    outp( dataHi, vmode );

    GUS_delay();

    outp( dataHi, vmode );
    outp( regSelect, GET_VOLUME );

    volume = inpw( dataLo );

    GUS_rampVolume( volume, VolumeTable[end_idx], rate, mode );
}

//
//  Stop current voice's output
//

void    GUS_stopVoice( void )
{
    ubyte   data;

    outp( regSelect, GET_CONTROL );

    data  = inp( dataHi ) | ( VOICE_STOPPED | STOP_VOICE );

    outp( regSelect, SET_CONTROL );
    outp( dataHi, data );

    GUS_delay();

    outp( dataHi, data );
}

//
//  Prime a voice with new values but don't start it
//

ubyte   GUS_primeVoice( ulong begin, ulong start, ulong end, ubyte mode )
{
    ulong   temp;
    ubyte   vmode;

//  If start is greater than end, flip 'em and turn on decrementing addresses

    if  ( start > end )
    {
        temp  = start;
		start = end;
        end   = temp;
		mode |= VC_DIRECT;
    }

//  If 16 bit data, must convert addresses
//  This is not implemented until Dynamic Studio implements it..

//    if  ( mode & VC_DATA_TYPE )
//    {
//        begin = GUS_make16Bit( begin );
//        start = GUS_make16Bit( start );
//        end   = GUS_make16Bit( end );
//    }

//  First set accumulator to beginning of data

    outp( regSelect, SET_ACC_LOW );
    outpw( dataLo, ( begin & 0x7f ) << 9 );
    outp( regSelect,SET_ACC_HIGH);
    outpw( dataLo, ( begin >> 7 ) & 0x1fff );

//  Set start loop address of buffer

    outp( regSelect, SET_START_HIGH );
    outpw( dataLo, ( start >> 7 ) & 0x1fff );
    outp( regSelect, SET_START_LOW );
    outpw( dataLo, ( start & 0x7f ) << 9 );

//  Set end address of buffer

    outp( regSelect, SET_END_HIGH );
    outpw( dataLo, ( end >> 7 ) & 0x1fff );
    outp( regSelect, SET_END_LOW );
    outpw( dataLo, ( end & 0x7f ) << 9 );

    return( mode );
}

//
//  Start a voice that has already been primed
//

void    GUS_startVoice( ubyte mode )
{
    mode &= ~( VOICE_STOPPED | STOP_VOICE );    //  Flip stop bits

    outp( regSelect, SET_CONTROL );
    outp( dataHi, mode );                       //  Send new control

    GUS_delay();

    outp( dataHi, mode );                       //  Send again
}


//
//  Main module loader
//
//  Not very documented because it will vary greatly depending on compiler,
//  memory models etc, etc. (This is surely the easiest part!!!)
//

ubyte   LoadModule( char *ModuleName )
{
    FILE        *fp;
    uword       i, numPats = 0, repeat, repLen;
    ubyte       retValue;
    ubyte far   *buffer;
    ulong       dramLoc = 0;

    for ( i = 0; i < 128; i++ )
    {
        Notes[i] = 0;           //  Clear pattern pointers
    }

    if  (( fp = fopen( ModuleName, "rb" )) != 0 )
    {
        fseek( fp, 0, SEEK_SET );
        fread( &dHead, sizeof( DSMHeader ), 1, fp );

        numChn  = dHead.NumChn;
        numSmp  = dHead.NumSmp;
        songLen = dHead.SongLen;

        masterVolume = ((uword)dHead.MasterVol * 255)/100;

        if  (( retValue = InitLoader() ) == LOAD_OK )
        {
            gusOk = TRUE;

            fread( BalanceData, numChn, 1, fp );
            fread( SongOrders, songLen, 1, fp );

            for ( i = 0; i < numChn; i++ )          //  Set initial balance
            {
                outp( pageSelect, i );              //  Select voice
                GUS_setPanPos( BalanceData[i] );    //  Set balance
            }

            for ( i = 0; i < songLen; i++ )         //  Retrieve highest patno
            {
                if  ( SongOrders[i] > numPats )
                {
                    numPats = SongOrders[i];
                }
            }

            numPats++;

            fseek( fp, numPats * numChn * 8, SEEK_CUR );    //  Skip trackinfo

            for ( i = 0; i < (uword)numSmp; i++ )
            {
                fseek( fp, 22, SEEK_CUR );

                SmpInfo[i].Bits     = fgetc( fp );

                fread( &SmpInfo[i].SmpLen, 2, 1, fp );

                SmpInfo[i].FineTune = fgetc( fp );
                SmpInfo[i].Volume   = fgetc( fp );
                SmpInfo[i].SmpLoc   = dramLoc;

                fread( &repeat, 2, 1, fp );
                fread( &repLen, 2, 1, fp );

                fseek( fp, 1, SEEK_CUR );

                if  ( repLen > 2 )
                {
                    SmpInfo[i].LoopMode = VC_LOOP_ENABLE;
                    SmpInfo[i].RepLoc   = dramLoc + repeat;
                    SmpInfo[i].SmpEnd   = SmpInfo[i].RepLoc + repLen - 1;
                }
                else
                {
                    SmpInfo[i].LoopMode = 0;
                    SmpInfo[i].RepLoc   = 0;
                    SmpInfo[i].SmpEnd   = dramLoc + SmpInfo[i].SmpLen - 1;
                }

                if  ( SmpInfo[i].SmpLen )
                {
                    dramLoc += SmpInfo[i].SmpLen + 2;
                    dramLoc &= ~16;     //  Ensure 16 byte boundary
                    dramLoc += 16;      //  for future upgrades using DMA
                }
            }

            if  ( AllocPatterns( numPats ) )
            {
                for ( i = 0; i < numPats; i++ )
                {
                    fread( Notes[i], (uword)numChn * 256, 1, fp );
                }

                if  (( buffer = (ubyte far *)farmalloc( 0x10000 )) != 0 )
                {
                    for ( i = 0; i < (uword)numSmp; i++ )
                    {
                        if  ( SmpInfo[i].SmpLen )
                        {
                            if  ( fread( buffer, SmpInfo[i].SmpLen, 1, fp ) != 1 )
                            {
                                retValue = LOAD_READERROR;
                                break;
                            }
                            else
                            {
                                retValue = LOAD_OK;
                            }

                            DumpSample( buffer, SmpInfo[i].SmpLoc, SmpInfo[i].SmpLen );
                        }
                    }

                    farfree( buffer );
                }
                else
                {
                    retValue = LOAD_MEMERROR;
                }
            }
            else
            {
                retValue = LOAD_MEMERROR;
            }
        }
        else
        {
            retValue = LOAD_MEMERROR;
        }

        fclose( fp );
    }
    else
    {
        retValue = LOAD_OPENERROR;
    }

    return( retValue );
}

//
//  Allocate memory for Sample Information & Channel Information
//

ubyte   InitLoader( void )
{
    ubyte   retValue;

    if  (( Chans = (Channel far *)farcalloc( numChn, sizeof( Channel ) )) != 0 )
    {
        if  (( SmpInfo = (Sample far *)farmalloc( (ulong)numSmp * sizeof( Sample ) )) != 0 )
        {
            if  ( strncmp( dHead.DSm, "DSm", 3 ) == 0 )
            {
                retValue = GUS_open( numChn );
            }
            else
            {
                retValue = LOAD_FMTERROR;
            }
        }
        else
        {
            retValue = LOAD_MEMERROR;
        }
    }
    else
    {
        retValue = LOAD_MEMERROR;
    }

    return( retValue );
}

//
//  Free all allocated memory
//

void    UnloadModule( void )
{
    if  ( Chans != 0 )
    {
        farfree( Chans );
    }

    if  ( SmpInfo != 0 )
    {
        farfree( SmpInfo );
    }

    for ( ubyte i = 0; i < songLen; i++ )
    {
        if  ( Notes[i] != 0 )
        {
            farfree( Notes[i] );
        }
    }

    if  ( gusOk )
    {
        GUS_close();
    }
}

//
//  Allocate memory for pattern data
//

ubyte   AllocPatterns( uword numPats )
{
    for ( uword i = 0; i < numPats; i++ )
    {
        if  (( Notes[i] = (ubyte far *)farmalloc( (uword)numChn * 256 )) == 0 )
        {
            return( FALSE );
        }
    }

    return( TRUE );
}

//
//  Dump a sample to DRAM using byte transfer
//

void    DumpSample( ubyte far *address, ulong SmpLoc, uword Length )
{
    for ( uword i = 0; i < Length; i++ )
    {
        GUS_pokeByte( SmpLoc + i, address[i] );
    }

    GUS_pokeByte( SmpLoc + i + 1, address[i] ); //  PAD
    GUS_pokeByte( SmpLoc + i + 2, address[i] ); //  PAD
}

//
//  Build a frequency table indexed by period
//

void    BuildFreqTable( void )
{
    uword   i;
    ulong   speed, temp;

    for ( i = 0; i < 54; i++ )
    {
        FreqTable[i] = 0;
    }

    for ( i = 54; i < 1815; i++ )
    {
        temp  = (ulong)FreqDivisor[voices - 14];
        speed = 3546895/i;

        //  Strange formula ??!!!??

        FreqTable[i] = (uword)((( speed << 9 ) + ( temp >> 1 )) / temp ) << 1;
    }
}

//
//  This is the main callback for the timer interrupts
//
//  tHe mODulE rEPlaYeR
//

void    ReplayCallback( void )
{
    if  ( ++lineCounter == patSpeed )   //  Are we going to get a new note??
    {
        lineCounter = 0;                //  Reset counter

        if  ( patDelayTime == 0 )       //  But not if we have a pat.delay
        {
            if  ( patPos > 63 )         //  New song position ??
            {
                songPos++;              //  Yepp!!
                patPos  = 0;
                patLoop = 0;
            }

            if  ( songPos >= songLen )  //  End of song ??
            {
                songPos = 0;            //  Yepp. Start all over..
            }

            curPat = SongOrders[songPos];   //  Get active pattern

            ParsePatternLine();         //  Parse one line in the pattern
            UpdateLineSfx();            //  Check line effects

            if  ( patBreak == 0 )
            {
                patPos++;
            }
            else
            {
                patBreak = 0;
            }
        }
        else
        {
            patDelayTime--;             //  Decrement pattern delay
        }
    }

    UpdateRealSfx();                    //  Check real-time effects

    for ( ubyte i = 0; i < numChn; i++ )
    {
        curChn  = i;
        actChan = &Chans[i];

        RetrigChannel();
    }
}

//
//  Transfer all new values to the UltraSound
//

void    RetrigChannel( void )
{
    ubyte   mode, volume;

    if  ( actChan->Volume == 0 )
    {
        actChan->PlayPer = 0;
    }

    outp( pageSelect, curChn );

    if  ( actChan->NewNote )
    {
        GUS_rampLinearVolume( 0, 0x3f, 0 );

        //  Switch to new voice

        GUS_setFrequency( (word)0 );
        GUS_stopVoice();                //  Stop the old voice

        mode = GUS_primeVoice( actChan->SmpLoc, actChan->RepLoc,
                               actChan->SmpEnd, actChan->LoopMode );

        volume = (ubyte)(( (uword)actChan->Volume * masterVolume ) / 255 );

        GUS_rampLinearVolume( volume, 0x3f, 0 );
        GUS_setFrequency( actChan->PlayPer );
        GUS_startVoice( mode );

        actChan->NewNote = 0;
    }
    else
    {
        volume = (ubyte)(( (uword)actChan->Volume * masterVolume ) / 255 );

        GUS_rampLinearVolume( volume, 0x3f, 0 );
        GUS_setFrequency( actChan->PlayPer );
    }
}

//
//  Interpret one pattern line
//

void    ParsePatternLine( void )
{
    ubyte           smpNum, noteNum;
    uword           i, j;
    NoteInfo far    *note;


    for ( i = 0; i < numChn; i++ )
    {
        note    = (NoteInfo far *)&Notes[curPat][(uword)numChn * patPos * 4 + i * 4];
        actChan = &Chans[i];

        actChan->Effect = note->Cmd;
        actChan->EffectValue = note->Dta;

        noteNum = ( note->Key >> 1 ) - 1;

        if  (( smpNum = note->SmpNum ) >= numSmp )
        {
            smpNum = 0;
        }

        if  ( smpNum != 0 )
        {
            smpNum--;

            actChan->SmpNum   = smpNum;
            actChan->Volume   = SmpInfo[smpNum].Volume;
            actChan->FineTune = SmpInfo[smpNum].FineTune;
            actChan->SmpLoc   = SmpInfo[smpNum].SmpLoc;
            actChan->RepLoc   = SmpInfo[smpNum].RepLoc;
            actChan->SmpEnd   = SmpInfo[smpNum].SmpEnd;
            actChan->LoopMode = SmpInfo[smpNum].LoopMode;

            //  Intercept any finetune changes at this point

            if  (( actChan->Effect == 0x0E ) && (( actChan->EffectValue >> 4 ) == 0x05 ))
            {
                actChan->FineTune        = actChan->EffectValue & 0x0f;
                SmpInfo[smpNum].FineTune = actChan->EffectValue & 0x0f;
            }

            if  ( note->Key != 0 )
            {
                actChan->NewNote = 1;
                actChan->OldPer  = actChan->Period;
                actChan->Period  = PeriodTable[(uword)actChan->FineTune * 60 + noteNum];
            }
            else
            {
                actChan->NewNote = 0;
            }
        }
        else
        {
            actChan->NewNote = 0;

            if  ( note->Key != 0 )
            {
                if  (( actChan->Effect == 0x03 ) || ( actChan->Effect == 0x05 ))
                {
                    actChan->WantPer  = PeriodTable[(uword)actChan->FineTune * 60 + noteNum];
                    actChan->WantNote = noteNum;
                }
                else
                {
                    actChan->Period  = PeriodTable[(uword)actChan->FineTune * 60 + noteNum];
                    actChan->NewNote = 1;
                }
            }
        }

        if  (( actChan->Effect == 0x03 ) || ( actChan->Effect == 0x05 ))
        {
            if  ( actChan->NewNote )
            {
                actChan->WantPer  = actChan->Period;
                actChan->Period   = actChan->OldPer;
                actChan->WantNote = noteNum;
            }

            if  ( actChan->WantPer == actChan->Period )
            {
                actChan->WantPer = 0;
            }
            else if ( actChan->WantPer > actChan->Period )
            {
                actChan->ToneDir = 0;
            }
            else
            {
                actChan->ToneDir = 1;
            }
        }
        else
        {
            actChan->NoteNum = noteNum;
        }

        actChan->PlayPer = actChan->Period;
    }
}

//
//  Update effects that occur once per line
//

void    UpdateLineSfx( void )
{
    for ( ubyte i = 0; i < numChn; i++ )
    {
        actChan     = &Chans[i];
        curEffValue = actChan->EffectValue;
        curChn      = i;
        curPeriod   = actChan->Period;

        if  (( actChan->Effect >> 4 ) == 2 )
        {
            SetVolumeSmpOffset();

            actChan->Effect = 0;
        }

        switch  ( actChan->Effect )
        {
            case    0x08:
                switch  (( curEffValue >> 4 ) & 0x0f )
                {
                    case    0x00:
                        PanControl();
                        break;
                    case    0x02:
                        DefaultVolumeSlideUp();
                        break;
                    case    0x03:
                        effMask = 0x0f;
                        PortamentoUp5();
                        break;
                    case    0x04:
                        effMask = 0x0f;
                        PortamentoDown5();
                        break;
                    default:
                        break;
                }
                break;
            case    0x09:
                SampleOffset();
                break;
            case    0x0B:
                PositionJump();
                break;
            case    0x0C:       //  Set volume
                SetVolume();
                break;
            case    0x0D:
                PatternBreak();
                break;
            case    0x0E:
                switch  (( curEffValue >> 4 ) & 0x0f )
                {
                    case    0x01:
                        effMask = 0x0f;
                        PortamentoUp();
                        break;
                    case    0x02:
                        effMask = 0x0f;
                        PortamentoDown();
                        break;
                    case    0x03:
                        Glissando();
                        break;
                    case    0x04:
                        SetVibratoWaveForm();
                        break;
                    case    0x06:
                        PatternLoop();
                        break;
                    case    0x07:
                        SetTremoloWaveForm();
                        break;
                    case    0x0A:
                        FineVolumeSlideUp();
                        break;
                    case    0x0B:
                        FineVolumeSlideDown();
                        break;
                    case    0x0E:
                        PatternDelay();
                        break;
                    default:
                        break;
                }
                break;
            case    0x0F:       //  Set speed or BPM
                SetSpeedOrBPM();
                break;
            case    0x13:
                Simulate3D();
                break;
            case    0x1C:
                SetMasterVolume();
                break;
            case    0x1D:
                SlideMasterUp();
                break;
            case    0x1E:
                SlideMasterDown();
                break;
            default:
                break;
        }
    }
}

//
//  Update effects that occur once per tick
//

void    UpdateRealSfx( void )
{
    for ( ubyte i = 0; i < numChn; i++ )
    {
        actChan     = &Chans[i];
        curEffValue = actChan->EffectValue;
        curChn      = i;
        curPeriod   = actChan->Period;

        //  Update Vibrato & Tremolo Positions

        if  ( actChan->WaveControl & 0x04 )
        {
            actChan->VibratoPos = 0;
        }

        if  ( actChan->WaveControl & 0x70 )
        {
            actChan->TremoloPos = 0;
        }

        switch  ( actChan->Effect )
        {
            case    0x00:
#ifdef  USE_ARPEGGIO
                Arpeggio();
#endif
                break;
            case    0x01:
                PortamentoUp();
                break;
            case    0x02:
                PortamentoDown();
                break;
            case    0x03:
                TonePortamento( 0 );
                break;
            case    0x04:
                Vibrato( 0 );
                break;
            case    0x05:
                TonePortamento( 1 );
                VolumeSlide();
                break;
            case    0x06:
                VolumeSlide();
                Vibrato( 1 );
                break;
            case    0x07:
                Tremolo();
                break;
            case    0x08:
                switch  (( curEffValue >> 4 ) & 0x0f )
                {
                    case    0x01:
                        DefaultVolumeSlideUp();
                        break;
                    default:
                        break;
                }
                break;
            case    0x0A:
                VolumeSlide();
                break;
            case    0x0E:
                switch  (( curEffValue >> 4 ) & 0x0f )
                {
                    case    0x09:
                        RetrigSample();
                        break;
                    case    0x0C:
                        NoteCut();
                        break;
                    case    0x0D:
                        NoteDelay();
                        break;
                    default:
                        break;
                }
                break;
            case    0x11:
                PortamentoUp5();
                break;
            case    0x12:
                PortamentoDown5();
                break;
            default:
                break;
        }
    }
}


//
//  Slide the frequency up by decrementing the period value
//

void    PortamentoUp( void )
{
    curPeriod -= curEffValue & effMask;
    curPeriod &= 0xfff;

    if  ( curPeriod < 0x71 )
    {
        curPeriod = 0x71;
    }

    actChan->Period  = curPeriod;
    actChan->PlayPer = curPeriod;

    effMask = 0xff;
}

//
//  Slide the frequency down by increasing the period value
//

void    PortamentoDown( void )
{
    curPeriod += curEffValue & effMask;
    curPeriod &= 0xfff;

    if  ( curPeriod >= 0x358 )
    {
        curPeriod = 0x358;
    }

    actChan->Period  = curPeriod;
    actChan->PlayPer = curPeriod;

    effMask = 0xff;
}


//
//  Slide current note into new note
//

void    TonePortamento( ubyte noInit )
{
    if  ( noInit == 0 )
    {
        if  ( curEffValue != 0 )
        {
            actChan->ToneSpeed   = curEffValue;
            actChan->EffectValue = 0;
        }
    }

    if  ( actChan->WantPer )
    {
        if  ( actChan->GlissControl )
        {
            if  ( actChan->ToneDir != 0 )
            {
                actChan->NoteNum += actChan->ToneSpeed;

                if  ( actChan->WantNote <= actChan->NoteNum )
                {
                    actChan->NoteNum = actChan->WantNote;
                    actChan->WantPer = 0;
                }
            }
            else
            {
                actChan->NoteNum -= actChan->ToneSpeed;

                if  ( actChan->WantNote >= actChan->NoteNum )
                {
                    actChan->NoteNum = actChan->WantNote;
                    actChan->WantPer = 0;
                }
            }

            actChan->Period  = PeriodTable[actChan->FineTune * 60 + actChan->NoteNum];
            actChan->PlayPer = actChan->Period;
        }
        else                                    //  Normal Tone Portamento
        {
            if  ( actChan->ToneDir == 0 )  //  Tone Porta Up
            {
                curPeriod += (uword)actChan->ToneSpeed;

                if  ( actChan->WantPer <= curPeriod )
                {
                    curPeriod        = actChan->WantPer;
                    actChan->WantPer = 0;
                }
            }
            else
            {
                curPeriod -= (uword)actChan->ToneSpeed;

                if  ( actChan->WantPer >= curPeriod )
                {
                    curPeriod        = actChan->WantPer;
                    actChan->WantPer = 0;
                }
            }
        }
    }

    actChan->Period    = curPeriod;
    actChan->PlayPer   = curPeriod;
}


//
//  Vibrate the frequency around a note
//

void    Vibrato( ubyte noInit )
{
    ubyte   vibCmd = actChan->VibratoCmd;
    ubyte   vibPos = actChan->VibratoPos;
    ubyte   wave   = actChan->WaveControl;
    ubyte   vibData;
    uword   vibOffs;

    if  ( noInit == 0 )
    {
        asm {
            xor     ah,ah
            mov     al,curEffValue
            or      al,al
            jz      Continue_Vib
            mov     cl,vibCmd
            and     al,00001111b
            jz      Mt_VibSkip
            and     cl,11110000b
            or      cl,al
        }
    Mt_VibSkip:
        asm {
            mov     al,curEffValue
            and     al,11110000b
            jz      Mt_VibSkip2
            and     cl,00001111b
            or      cl,al
        }
    Mt_VibSkip2:
        asm mov     vibCmd,cl;
    }

Continue_Vib:
    asm {
        mov     al,vibPos
        shr     al,2
        and     al,1fh
        xor     ch,ch
        mov     cl,wave
        and     cl,00000011b
        jz      Mt_Vib_Sine
        shl     al,3
        cmp     cl,1
        jz      Mt_Vib_Rampdn
        mov     cl,255
        jmp     Mt_Vib_Set
    }
Mt_Vib_Rampdn:
    asm {
        cmp     vibPos,0
        jnb     Mt_Vib_Rampdn2
        mov     cl,255
        sub     cl,al
        jmp     Mt_Vib_Set
    }
Mt_Vib_Rampdn2:
    asm {
        mov     cl,al
        jmp     Mt_Vib_Set
    }
Mt_Vib_Sine:
    asm {
        sub     bh,bh
        mov     bl,al
        and     bx,1fh
        mov     vibOffs,bx
    }

    vibData = VibratoTable[vibOffs];

    asm {
        mov     cl,vibData
    }
Mt_Vib_Set:

    asm {
        mov     al,vibCmd
        and     al,00001111b
        mul     cl
        shr     ax,7
        mov     cx,ax
        mov     ax,curPeriod
        cmp     vibPos,0
        jl      Mt_VibratoNeg
        add     ax,cx
        jmp     Mt_Vibrato3
    }
Mt_VibratoNeg:
    asm sub     ax,cx
Mt_Vibrato3:
    asm mov     vibOffs,ax

    actChan->PlayPer = vibOffs;

    asm {
        mov     al,vibCmd
        shr     al,2
        and     al,3ch
        add     vibPos,al
    }

    actChan->VibratoCmd = vibCmd;
    actChan->VibratoPos = vibPos;
}

//
//  Vibrate the sample volume
//

void    Tremolo( void )
{
    ubyte   tremCmd = actChan->TremoloCmd;
    ubyte   tremPos = actChan->TremoloPos;
    ubyte   wave    = actChan->WaveControl;
    ubyte   volume  = actChan->Volume;
    ubyte   tremData;
    uword   tremOffs;

    asm {
        xor     ah,ah
        mov     al,curEffValue
        or      al,al
        jz      Continue_Trem
        mov     cl,tremCmd
        and     al,00001111b
        jz      Mt_TreSkip
        and     cl,11110000b
        or      cl,al
    }
Mt_TreSkip:
    asm {
        mov     al,curEffValue
        and     al,11110000b
        jz      Mt_TreSkip2
        and     cl,00001111b
        or      cl,al
    }
Mt_TreSkip2:
    asm {
        mov     tremCmd,cl
    }
Continue_Trem:
    asm {
        mov     al,tremPos
        shr     al,2
        and     al,1fh
        xor     ch,ch
        mov     cl,wave
        shr     cl,4
        and     cl,3
        jz      Mt_Tre_Sine
        shl     al,3
        cmp     cl,1
        jz      Mt_Tre_Rampdn
        mov     cl,255
        jmp     Mt_Tre_Set
    }
Mt_Tre_Rampdn:
    asm {
        cmp     tremPos,0
        jnb     Mt_Tre_Rampdn2
        mov     cl,255
        sub     cl,al
        jmp     Mt_Tre_Set
    }
Mt_Tre_Rampdn2:
    asm {
        mov     cl,al
        jmp     Mt_Tre_Set
    }
Mt_Tre_Sine:
    asm {
        sub     bh,bh
        mov     bl,al
        mov     tremOffs,bx
    }

    tremData = VibratoTable[tremOffs];

    asm {
        mov     cl,tremData
    }
Mt_Tre_Set:

    asm {
        mov     al,tremCmd
        and     al,00001111b
        mul     cl
        mov     cx,ax
        shr     cx,6
        mov     al,volume
        cmp     tremPos,0
        jl      Mt_TremoloNeg
        add     al,cl
        jmp     Mt_Tremolo3
    }
Mt_TremoloNeg:
    asm sub     al,cl
Mt_Tremolo3:
    asm {
        jnb     Mt_TremoloSkip
        sub     al,al
    }
Mt_TremoloSkip:
    asm {
        cmp     al,40h
        jb      Mt_TremoloOk
        mov     al,40h
    }
Mt_TremoloOk:
    asm mov     volume,al;

    asm {
        mov     al,tremCmd
        shr     al,2
        and     al,3ch
        add     tremPos,al
    }

    actChan->TremoloCmd = tremCmd;
    actChan->TremoloPos = tremPos;
    actChan->Volume     = volume;
}


//
//  Change the sample start position
//

void    SampleOffset( void )
{
    actChan->SmpLoc += (ulong)curEffValue << 8;
    actChan->NewNote = 1;
}


//
//  Slide the volume on a channel up or down
//

void    VolumeSlide( void )
{
    if  ( curEffValue & 0xf0 )
    {
        actChan->Volume += curEffValue >> 4;

        if  ( actChan->Volume > 64 )
        {
            actChan->Volume = 64;
        }
    }
    else if ( curEffValue & 0x0f )
    {
        actChan->Volume -= curEffValue & 0x0f;

        if  ( actChan->Volume > 64 )
        {
            actChan->Volume = 0;
        }
    }
}


//
//  Jump to another position in the song
//

void    PositionJump( void )
{
    if  ( curEffValue < songLen )
    {
        patPos   = 0;
        patBreak = 1;
        songPos  = curEffValue;
    }
}



//
//  Set a channels volume to a specified value
//

void    SetVolume( void )
{
    if  ( curEffValue > 64 )
    {
        actChan->Volume = 64;
    }
    else
    {
        actChan->Volume = curEffValue;
    }
}


//
//  Break current pattern and jump to specified position in next pattern
//

void    PatternBreak( void )
{
    if  ( patBreak == 0 )
	{
        songPos++;
        patPos   = (( curEffValue >> 4 ) * 10 ) + curEffValue & 0x0f;
        patBreak = 1;
	}
}


//
//  Set speed or beats per minute rate
//

void    SetSpeedOrBPM( void )
{
    if  ( curEffValue < 32 )
    {
        patSpeed = curEffValue;
    }
    else
    {
        songTempo = curEffValue;
        timerBPM  = BPMTable[curEffValue - 32];

        if  ( curEffValue < 125 )
        {
            timerPort = TIMER2;
        }
        else
        {
            timerPort = TIMER1;
        }

        timerMask = ( timerPort - 0x45 ) << 2;
    }
}


//
//  Glissando control. Glissando on means that tone portamento will
//  slide a half note at a time.
//

void    Glissando( void )
{
    actChan->GlissControl = curEffValue & 0x0f;
}

//
//  Set the waveform type vibrato should use
//

void    SetVibratoWaveForm( void )
{
    actChan->WaveControl &= 0xf0;
    actChan->WaveControl |= curEffValue & 0x0f;
}


//
//  Loop pattern a couple of times
//

void    PatternLoop( void )
{
    ubyte   loop = curEffValue & 0x0f;

    if  ( loop == 0 )
    {
        if  ( patLoop == 0 )
        {
            patLoopPos = patPos;
        }
    }
    else
    {
        if  ( patLoop < loop )
        {
            patLoop++;
            patPos   = patLoopPos;
            patBreak = 1;
        }
        else
        {
            patLoop = 0;
        }
    }
}

//
//  Set waveform type Tremolo should use
//

void    SetTremoloWaveForm( void )
{
    actChan->WaveControl &= 0x0f;
    actChan->WaveControl |= ( curEffValue & 0x0f ) << 4;
}

//
//  Retrigger sample after a specified number of ticks
//

void    RetrigSample( void )
{
    if  ( lineCounter == curEffValue & 0x0f )
    {
        actChan->NewNote = 1;
    }
}

//
//  Fine volume slide up once per line
//

void    FineVolumeSlideUp( void )
{
    actChan->Volume += curEffValue & 0x0f;

    if  ( actChan->Volume > 64 )
    {
        actChan->Volume = 64;
    }
}

//
//  Fine volume slide down once per line
//

void    FineVolumeSlideDown( void )
{
    actChan->Volume -= curEffValue & 0x0f;

    if  ( actChan->Volume > 64 )
    {
        actChan->Volume = 0;
    }
}

//
//  Set sample volume to 0 after specified number of ticks
//

void    NoteCut( void )
{
    if  ( lineCounter == curEffValue & 0x0f )
    {
        actChan->Volume = 0;
    }
}

//
//  Delay the start of a sample until specified ticks have passed
//

void    NoteDelay( void )
{
    if  ( lineCounter == curEffValue & 0x0f )
    {
        actChan->NewNote = 1;
    }
    else
    {
        actChan->NewNote = 0;
    }
}

//
//  Delay pattern for a number of ticks
//

void    PatternDelay( void )
{
    if  ( patDelayTime == 0 )
	{
        patDelayTime = (uword)( curEffValue & 0x0f );
	}
}

//
//  Portamento up including 5 octaves
//

void    PortamentoUp5( void )
{
    curPeriod -= curEffValue & effMask;
    curPeriod &= 0xfff;

    if  ( curPeriod < 54 )
    {
        curPeriod = 54;
    }

    actChan->Period  = curPeriod;
    actChan->PlayPer = curPeriod;

    effMask = 0xff;
}

//
//  Portamento down including 5 octaves
//

void    PortamentoDown5( void )
{
    curPeriod += curEffValue & effMask;
    curPeriod &= 0xfff;

    if  ( curPeriod >= 1814 )
    {
        curPeriod = 1814;
    }

    actChan->Period  = curPeriod;
    actChan->PlayPer = curPeriod;

    effMask = 0xff;
}

//
//  Simulate a fake 3D sound
//

void    Simulate3D( void )
{
    ubyte   volume;

    asm {
        mov     al,curEffValue
        mov     bl,al
        shr     bl,6
        and     al,63
        mov     bh,al
        shr     al,3
        shr     bh,1
        or      bl,bl
        jnz     NAngle000_089
        add     al,7
        mov     ah,40h
        sub     ah,bh
        jmp     EndAngle
    }
NAngle000_089:
    asm {
        cmp     bl,1
        jnz     NAngle090_179
        mov     ah,15
        sub     ah,al
        mov     al,ah
        mov     ah,20h
        sub     ah,bh
        jmp     EndAngle
    }
NAngle090_179:
    asm {
        cmp     bl,2
        jnz     NAngle180_269
        mov     ah,8
        sub     ah,al
        mov     al,ah
        mov     ah,bh
        inc     ah
        jmp     EndAngle
    }
NAngle180_269:
    asm {
        mov     ah,20h
        add     ah,bh
    }
EndAngle:
    asm {
        mov     volume,ah
        mov     curEffValue,al
    }

    actChan->Volume = volume;

    PanControl();
}


//
//  Control balance of a channel
//

void    PanControl( void )
{
    GUS_setPanPos( curEffValue & 0x0f );
}


//
//  Volume slide up to default volume
//

void    DefaultVolumeSlideUp( void )
{
    ubyte   origVolume = SmpInfo[actChan->SmpNum].Volume;

    actChan->Volume += curEffValue & 0x0f;

    if  ( actChan->Volume > origVolume )
    {
        actChan->Volume = origVolume;
    }
}


//
//  Set volume + sample offset
//

void    SetVolumeSmpOffset( void )
{
    SampleOffset();

    actChan->Volume = (( actChan->Effect & 0x0f ) << 2 ) + 4;
}


//
//  Set master volume
//

void    SetMasterVolume( void )
{
    masterVolume = (uword)curEffValue;
}


//
//  Slide master volume up
//

void    SlideMasterUp( void )
{
    masterVolume += (uword)curEffValue;

    if  ( masterVolume > 256 )
    {
        masterVolume = 256;
    }
}

//
//  Slide master volume down
//

void    SlideMasterDown( void )
{
    if  ( curEffValue > masterVolume )
    {
        masterVolume = 0;
    }
    else
    {
        masterVolume -= (uword)curEffValue;
    }
}


#ifdef  USE_ARPEGGIO

//
//  Arpeggio
//

void    Arpeggio( void )
{
    ubyte   perOffset, arpFlag = TRUE;
    uword   tuneOffset;

    if  ( curEffValue != 0 )
    {
        switch  ( lineCounter % 3 )
        {
            case    0x00:
                arpFlag = FALSE;
                break;
            case    0x01:
                perOffset = curEffValue >> 4;
                break;
            case    0x02:
                perOffset = curEffValue & 0x0f;
                break;
            default:
                break;
        }

        if  ( arpFlag )
        {
            tuneOffset = (uword)actChan->FineTune * 60;

            for ( ubyte i = 0; i < 61; i++ )
            {
                if  ( PeriodTable[tuneOffset + i] <= curPeriod )
                {
                    actChan->PlayPer = PeriodTable[tuneOffset + i + perOffset];
                    break;
                }
            }
        }
    }
}

#endif

