
//Ŀ
//                                                                           
// SOCOM 1.0                                                                 
//                                                                           
// Gravis - Implementation                                                   
//                                                                           
// (c) 1995 P.Lindh                                                          
//                                                                           
//

#include <conio.h>
#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <socom\gravis.h>
#include <socom\socomus.h>
#include <socom\tables.h>
#include <socom\channel.h>

//Ŀ
//                                                                           
// Global variables                                                          
//                                                                           
//

extern IRQLine scIRQTable[16];
extern DMAChan scDMATable[8];
extern uword   scUsVolTable[65];
extern int     mpNumChn;
extern int     mpMasterVolume;
extern ubyte   scUsBPMTable[224];

extern AudioDevice*  socomAudio;
extern Channel       mpChannels[32];

extern void __interrupt __far SocomHandler( void );

extern void mpTickHandler( void );

//Ŀ
//                                                                           
// Constructor                                                               
//                                                                           
//

Gravis::Gravis( void )
{
   myMixImage = 0x0B;
   myResDRAM  = 32;
   myOpenFlag = false;
}

//Ŀ
//                                                                           
// Destructor                                                                
//                                                                           
//

Gravis::~Gravis( void )
{
   if ( myOpenFlag )
   {
      Close();
   }
}

//Ŀ
//                                                                           
// Allocate a block of DRAM                                                  
//                                                                           
// Returns address in 'address' variable.                                    
// Returns SOC_OK if allocation was successful or SOC_NO_DRAM if not         
// enough available DRAM.                                                    
//                                                                           
//

int   Gravis::AllocDRAM( ulong size, ulong& address )
{
   ulong ptr;
   ulong pool_size;
   ulong size_left;
   ulong prev;
   ulong next;

   // round size up to next 32 byte boundary
   size += 31;
   size &= -32L;

   ptr = myFreeDRAM;

   while ( ptr != 0 )
   {
      // check if this buffer will fit
      pool_size = PeekLong( ptr + SIZE_OFFSET );

      if ( pool_size >= size )
      {
         size_left = pool_size - size;

         if ( size_left < MEM_HEADER_SIZE )
         {
            next = PeekLong( ptr + NEXT_OFFSET );
            prev = PeekLong( ptr + PREV_OFFSET );

            if ( next != 0 )
            {
               PokeLong( next + PREV_OFFSET, prev );
            }

            if ( prev != 0 )
            {
               PokeLong( prev + NEXT_OFFSET, next );
            }
            else
            {
               myFreeDRAM = next;
            }

            address = ptr;
         }
         else
         {
            // adjust size of this pool
            PokeLong( ptr + SIZE_OFFSET, size_left );

            // calculate new address
            address = ptr + size_left;
         }

         return SOC_OK;
      }

      ptr = PeekLong( ptr + NEXT_OFFSET );   // next please
   }

   return SOC_NO_DRAM;
}

//Ŀ
//                                                                           
// Pre-calculate frequency table                                             
//                                                                           
//

void  Gravis::CalcFreqTable( void )
{
   ulong speed;
   ulong temp = 617400 / myNumVoices;

   for ( int i = 0; i < 54; i++ )
   {
      myFreqTable[i] = 0;
   }

   for ( i = 54; i < 1815; i++ )
   {
      speed = 3546895 / i;

      myFreqTable[i] = ((( speed << 9 ) + ( temp >> 1 )) / temp ) << 1;
   }
}

//Ŀ
//                                                                           
// Close down Gravis                                                         
//                                                                           
//

void  Gravis::Close( void )
{
   DisableOutput();
   Reset();
   ResetHandler();

   myOpenFlag = false;
}

//Ŀ
//                                                                           
// Convert address if DMA channel is 16 bit                                  
//                                                                           
//

ulong Gravis::ConvTo16bit( ulong address ) const
{
   ulong hold = address & 0xc0000;

   address >>= 1;
   address &= 0x1ffff;
   address |= hold;

   return address;
}

//Ŀ
//                                                                           
// Disable any output from Gravis                                            
//                                                                           
//

void  Gravis::DisableOutput( void )
{
   outp( myBasePort, myMixImage |= ENABLE_OUTPUT );
}

//Ŀ
//                                                                           
// Enable any output from Gravis                                             
//                                                                           
//

void  Gravis::EnableOutput( void )
{
   outp( myBasePort, myMixImage &= ~ENABLE_OUTPUT );
}

//Ŀ
//                                                                           
// Free a block of DRAM                                                      
//                                                                           
// Returns SOC_OK if successful or SOC_CORRUPT_DRAM if DRAM structure        
// is corrupt.                                                               
//                                                                           
//

int   Gravis::FreeDRAM( ulong size, ulong address )
{
   ulong ptr = myFreeDRAM;
   ulong prev;
   ulong next;
   ulong psize;
   ulong nnext;
   bool  flag = false;

   // round size up to next 32-byte boundary
   size += 31;
   size &= -32L;

   if ( ptr == 0 )
   {
      // no free dram. make this the first in the list
      myFreeDRAM = address;

      PokeLong( address + NEXT_OFFSET, 0 );
      PokeLong( address + PREV_OFFSET, 0 );
      PokeLong( address + SIZE_OFFSET, size );

      flag = true;
   }
   else
   {
      while ( ptr != 0 && !flag )
      {
         next = PeekLong( ptr + NEXT_OFFSET );
         prev = PeekLong( ptr + PREV_OFFSET );

         if ( address < ptr )
         {
            if ( prev == 0 )
            {
               myFreeDRAM = address;
            }
            else
            {
               PokeLong( prev + NEXT_OFFSET, address );
            }

            PokeLong( address + NEXT_OFFSET, ptr );
            PokeLong( address + PREV_OFFSET, prev );
            PokeLong( address + SIZE_OFFSET, size );
            PokeLong( ptr + PREV_OFFSET, address );

            flag = true;
         }
         else
         {
            if ( next == 0 )
            {
               PokeLong( ptr + NEXT_OFFSET, address );
               PokeLong( address + NEXT_OFFSET, 0 );
               PokeLong( address + PREV_OFFSET, ptr );
               PokeLong( address + SIZE_OFFSET, size );

               flag = true;
            }
         }

         ptr = PeekLong( ptr + NEXT_OFFSET );   // next please
      }
   }

   if ( flag )
   {
      ptr = myFreeDRAM;

      while ( ptr != 0 )
      {
         next  = PeekLong( ptr + NEXT_OFFSET );
         psize = PeekLong( ptr + SIZE_OFFSET );

         if (( ptr + psize ) == next )
         {
            nnext  = PeekLong( next + NEXT_OFFSET );
            psize += PeekLong( next + SIZE_OFFSET );

            PokeLong( ptr + SIZE_OFFSET, psize );
            PokeLong( ptr + NEXT_OFFSET, nnext );

            if ( nnext != 0 )
            {
               PokeLong( nnext + PREV_OFFSET, ptr );
            }
            else
            {
               ptr = 0;    // end of list
            }
         }
         else
         {
            ptr = PeekLong( ptr + NEXT_OFFSET );   // next please
         }
      }

      return SOC_OK;
   }
   else
   {
      return SOC_CORRUPT_DRAM;
   }
}

//Ŀ
//                                                                           
// Small delay needed when modifying certain hardware registers              
//                                                                           
//

void  Gravis::gdelay( void ) const
{
   for ( int i = 0; i < 7; i++ )
   {
      (void)inp( myDramPort );
   }
}

//Ŀ
//                                                                           
// Handle interrupts generated by Gravis                                     
//                                                                           
//

void  Gravis::Handler( void )
{
   ubyte iSource;

   // clear PC's interrupt controller
   outp( myILine->ocr, myILine->spec_eoi );

   // send EOI to both controllers
   if ( myIRQ > 7 )
   {
      outp( OCR1, EOI );
   }

   while ( true )
   {
      if (( iSource = inp( myIrqStatus )) == 0 )
      {
         break;
      }

      if ( iSource & myTimerMask )
      {
         scDisable();

         mpTickHandler();

         PrimeChannels();

         if ( Channel::myBpmChange )
         {
            myTimerBpm = scUsBPMTable[Channel::myBpm - 32];

            if ( Channel::myBpm < 125 )
            {
               myTimerPort = TIMER2;
            }
            else
            {
               myTimerPort = TIMER1;
            }

            myTimerMask = ( myTimerPort - 0x45 ) << 2;

            Channel::myBpmChange = false;
         }

         outp( myRegSelect, myTimerPort );
         outp( myDataHi, myTimerBpm );
         outp( myRegSelect, TIMER_CONTROL );
         outp( myDataHi, 0 );
         outp( myDataHi, myTimerMask );

         scEnable();
      }

      if ( iSource & ( WAVETABLE_IRQ | ENVELOPE_IRQ ))
      {
         HandleVoice();
      }
   }
}

//Ŀ
//                                                                           
// Handle voice and volume interrupts                                        
//                                                                           
//

void  Gravis::HandleVoice( void )
{
   ulong waveIgnore;
   ulong volumeIgnore;
   ulong voiceBit;
   ubyte voice;
   ubyte iSource;
   ubyte temp1;
   ubyte temp2;

   volumeIgnore = 0;

   while ( true )
   {
      scDisable();

      outp( myRegSelect, GET_IRQV );

      iSource = inp( myDataHi );

      scEnable();

      voice = iSource & 0x1f;

      iSource &= ( VOICE_VOLUME_IRQ | VOICE_WAVE_IRQ );

      if ( iSource == ( VOICE_VOLUME_IRQ | VOICE_WAVE_IRQ ))
      {
         break;
      }

      voiceBit = 1L << voice;

      if (!( iSource & VOICE_VOLUME_IRQ ))
      {
         if (!( volumeIgnore & voiceBit ))
         {
            volumeIgnore |= voiceBit;

            outp( myVocSelect, voice );

            VolumeHandler( voice );
         }
      }
   }
}

//Ŀ
//                                                                           
// Initialize memory allocation routines                                     
//                                                                           
// Return the size of the DRAM in bytes                                      
//                                                                           
//

int   Gravis::InitDRAM( void )
{
   ulong pool_len;
   ulong address;
   ulong size;

   if ( myResDRAM > 256 * 1024 )
   {
      myResDRAM = 32;
   }

   size     = SizeDRAM();
   pool_len = size * 1024 - 32 - myResDRAM;

   myFreeDRAM = myResDRAM;

   PokeLong( myResDRAM + NEXT_OFFSET, 0 );
   PokeLong( myResDRAM + PREV_OFFSET, 0 );
   PokeLong( myResDRAM + SIZE_OFFSET, pool_len );

   // this section is necessary to split the DRAM into pools that do not
   // cross 256K boundaries. The GF1 cannot play 16 bit data across
   // 256K boundaries

   if ( pool_len > 256 * 1024 )
   {
      AllocDRAM( pool_len, address );

      if ( pool_len > 256 * 3 * 1024 )
      {
         FreeDRAM( POOL_SIZE, 256 * 3 * 1024 + 32 );
      }

      if ( pool_len > 256 * 2 * 1024 )
      {
         FreeDRAM( POOL_SIZE, 256 * 2 * 1024 + 32 );
      }

      FreeDRAM( POOL_SIZE, 256 * 1 * 1024 + 32 );
      FreeDRAM( 256 * 1024 - myResDRAM, myResDRAM );
   }

   return size;
}

//Ŀ
//                                                                           
// Load a module into the AudioDevice                                        
//                                                                           
//

int   Gravis::LoadModule( Module* aMod, int digVoices )
{
   int   retValue = SOC_OK;

   if ( myOpenFlag )
   {
      Close();
   }

   if (( retValue = Open( aMod->myNumChn, digVoices )) == SOC_OK )
   {
      for ( int i = 0; i < aMod->myNumSmp; i++ )
      {
         if ( aMod->mySamples[i].myLength != 0 )
         {
            if (( retValue = LoadSample( &aMod->mySamples[i] )) != SOC_OK )
            {
               break;
            }
         }
      }
   }

   return retValue;
}

//Ŀ
//                                                                           
// Download a sample into DRAM                                               
//                                                                           
//

int   Gravis::LoadSample( Sample* smp )
{
   ulong location;
   ulong reppos;
   ulong repend;
   ulong length = smp->myLength;
   int   retValue;
   int   i;

   if (( retValue = AllocDRAM( length + 8, location )) == SOC_OK )
   {
      smp->myGravisLoc = location;

      for ( i = 0; i < length; i++ )
      {
         PokeByte( location + i, smp->myAddress[i] );
      }

      if ( smp->myRepPos > length )
      {
         smp->myRepPos      = 0;
         smp->myRepEnd      = 2;
         smp->mySampleMode &= ~VC_LOOP_ENABLE;
      }

      if ( smp->myRepEnd > length )
      {
         smp->myRepEnd = length;
      }

      if ( smp->mySampleMode & VC_LOOP_ENABLE )
      {
         reppos = location + smp->myRepPos;
         repend = reppos + smp->myRepEnd;

         for ( i = 0; i < 8; i++ )
         {
            PokeByte( repend + i, PeekByte( reppos + i ));
         }
      }
      else
      {
         reppos = location;
         repend = location + length;

         for ( i = 0; i < 8; i++ )
         {
            PokeByte( repend + i, 0 );
         }
      }

      smp->myRepPos = reppos;
      smp->myRepEnd = repend;
   }

   return retValue;
}

//Ŀ
//                                                                           
// Open and initialize Gravis                                                
//                                                                           
//

int   Gravis::Open( int musVoices, int digVoices )
{
   int   retValue;
   int   temp;
   int   voices = musVoices + digVoices;
   char* envPtr;

   if ( voices < 32 )
   {
      if ( voices < 14 )
      {
         voices = 14;
      }

      myNumVoices = musVoices;

      InitVoiceAlloc();

      myNumVoices = voices;

      if (( envPtr = getenv( "ULTRASND" )) != 0 )
      {
         sscanf( envPtr, "%x,%d,%d,%d,%d", &myBasePort, &myDMA, &temp, &myIRQ, &temp );

         if ( Probe() )
         {
            myTimerPort = TIMER1;
            myTimerMask = 0x04;
            myTimerBpm  = 6;

            socomAudio = this;

            myILine = &scIRQTable[myIRQ];

            CalcFreqTable();
            DisableOutput();
            Reset();
            SetInterface();
            SetHandler();
            InitDRAM();
            EnableOutput();

            myOpenFlag = true;

            retValue = SOC_OK;
         }
         else
         {
            retValue = SOC_NO_CARD;
         }
      }
      else
      {
         retValue = SOC_NO_ENVVAR;
      }
   }
   else
   {
      retValue = SOC_TOO_MANY_VOICES;
   }

   return retValue;
}

//Ŀ
//                                                                           
// Read a byte from DRAM                                                     
//                                                                           
//

ubyte Gravis::PeekByte( ulong address ) const
{
   ubyte value;

   scDisable();

   outp( myRegSelect, SET_DRAM_LOW );
   outpw( myDataLo, address );
   outp( myRegSelect, SET_DRAM_HIGH );
   outp( myDataHi, address >> 16 );

   value = inp( myDramPort );

   scEnable();

   return value;
}

//Ŀ
//                                                                           
// Read a 32-bit value from DRAM                                             
//                                                                           
//

ulong Gravis::PeekLong( ulong address ) const
{
   ulong    value;
   ubyte*   ptr = (ubyte*)&value;

   for ( int i = 0; i < 4; i++ )
   {
      *ptr++ = PeekByte( address++ );
   }

   return value;
}

//Ŀ
//                                                                           
// Play a module                                                             
//                                                                           
//

void  Gravis::Play( Module* aMod )
{
   aMod->StartPlay();

   StartPlayer();
}

//Ŀ
//                                                                           
// Write a byte to DRAM                                                      
//                                                                           
//

void  Gravis::PokeByte( ulong address, ubyte data ) const
{
   scDisable();

   outp( myRegSelect, SET_DRAM_LOW );
   outpw( myDataLo, address );
   outp( myRegSelect, SET_DRAM_HIGH );
   outp( myDataHi, address >> 16 );
   outp( myDramPort, data );

   scEnable();
}

//Ŀ
//                                                                           
// Write a 32-bit value to DRAM                                              
//                                                                           
//

void  Gravis::PokeLong( ulong address, ulong value ) const
{
   ubyte*   ptr = (ubyte*)&value;

   for ( int i = 0; i < 4; i++ )
   {
      PokeByte( address++, *ptr++ );
   }
}

//Ŀ
//                                                                           
// Prime channels with new values                                            
//                                                                           
//

void  Gravis::PrimeChannels( void )
{
   Channel* chn;
   ubyte    volume;

   for ( int i = 0; i < mpNumChn; i++ )
   {
      chn = &mpChannels[i];

      outp( myVocSelect, i );

      if ( chn->myKick )
      {
         outp( myRegSelect, GET_VOLUME );

         if ( inpw( myDataLo ) < 1600 )
         {
            VolumeHandler( i );
         }
         else
         {
            RampVolume( 0, VL_WAVE_IRQ );
         }
      }
      else
      {
         if ( chn->myPlayVolume != chn->myOldVolume )
         {
            volume = ( ubyte(chn->myPlayVolume) * mpMasterVolume ) / 255;

            RampVolume( volume, 0 );

            chn->myOldVolume = chn->myPlayVolume;
         }

         SetPeriod( chn->myPlayPeriod );
         SetBalance( chn->myPanPos );
      }
   }
}

//Ŀ
//                                                                           
// Prime a voice with new values but don't start it                          
//                                                                           
//

ubyte Gravis::PrimeVoice( ulong begin, ulong start, ulong end, ubyte mode ) const
{
   ulong temp;

   if ( start > end )
   {
      temp  = start;
      start = end;
      end   = temp;
      mode |= VC_DIRECT;
   }

   if ( mode & VC_DATA_TYPE )
   {
      begin = ConvTo16bit( begin );
      start = ConvTo16bit( start );
      end   = ConvTo16bit( end );
   }

   // first set accumulator to beginning of data

   outp( myRegSelect, SET_ACC_LOW );
   outpw( myDataLo, begin << 9 );
   outp( myRegSelect,SET_ACC_HIGH);
   outpw( myDataLo, begin >> 7 );

   // set start loop address of buffer

   outp( myRegSelect, SET_START_HIGH );
   outpw( myDataLo, start >> 7 );
   outp( myRegSelect, SET_START_LOW );
   outpw( myDataLo, start << 9 );

   // set end address of buffer

   outp( myRegSelect, SET_END_HIGH );
   outpw( myDataLo, end >> 7 );
   outp( myRegSelect, SET_END_LOW );
   outpw( myDataLo, end << 9 );

   return mode;
}

//Ŀ
//                                                                           
// Try to detect a Gravis UltraSound                                         
//                                                                           
//

bool  Gravis::Probe( void )
{
   ubyte save0;
   ubyte save1;
   ubyte value0;
   ubyte value1;

   myIrqStatus    = myBasePort + GF1_IRQ_STAT;
   myTimerControl = myBasePort + GF1_TIMER_CTRL;
   myTimerData    = myBasePort + GF1_TIMER_DATA;
   myIrqControl   = myBasePort + GF1_IRQ_CTRL;
   myVocSelect    = myBasePort + GF1_PAGE;
   myRegSelect    = myBasePort + GF1_REG_SELECT;
   myDataLo       = myBasePort + GF1_DATA_LOW;
   myDataHi       = myBasePort + GF1_DATA_HIGH;
   myDramPort     = myBasePort + GF1_DRAM;

   // pull a reset on the GF1
   outp( myRegSelect, MASTER_RESET );
   outp( myDataHi, 0x00 );

   // delay a little while
   gdelay();
   gdelay();

   // release reset
   outp( myRegSelect, MASTER_RESET );
   outp( myDataHi, GF1_MASTER_RESET );

   // delay a little while
   gdelay();
   gdelay();

   // save bytes at address 0 and 1
   save0 = PeekByte( 0L );
   save1 = PeekByte( 1L );

   // poke in some new reference values
   PokeByte( 0L, 0xAA );
   PokeByte( 1L, 0x55 );

   // read the reference values
   value0 = PeekByte( 0L );
   value1 = PeekByte( 1L );

   // restore old values
   PokeByte( 0L, save0 );
   PokeByte( 1L, save1 );

   return value0 == 0xAA && value1 == 0x55;
}

//Ŀ
//                                                                           
// Slide current voice's volume to destination volume                        
//                                                                           
//

void  Gravis::RampVolume( ubyte endidx, ubyte mode ) const
{
   ubyte vMode;
   uword curVol;
   uword endVol;
   uword begVol;

   StopVolume();

   outp( myRegSelect, GET_VOLUME );

   curVol = inpw( myDataLo );
   endVol = scUsVolTable[endidx];

   if ( curVol != endVol )
   {
      begVol = curVol;

      if ( curVol > endVol )
      {
         curVol = endVol;
         endVol = begVol;
         mode  |= VC_DIRECT;
      }

      if ( endVol > 64512 )
      {
         endVol = 64512;
      }

      outp( myRegSelect, SET_VOLUME_START );
      outp( myDataHi, curVol >> 8 );
      outp( myRegSelect, SET_VOLUME_END );
      outp( myDataHi, endVol >> 8 );

      outp( myRegSelect, SET_VOLUME );
      outpw( myDataLo, begVol );

      outp( myRegSelect, SET_VOLUME_CONTROL );
      outp( myDataHi, mode );

      gdelay();

      outp( myDataHi, mode );
   }
}

//Ŀ
//                                                                           
// Full reset and initialization of all voices                               
//                                                                           
//

void  Gravis::Reset( void ) const
{
   PokeByte( 0L, 0 );
   PokeByte( 1L, 0 );

   scDisable();

   // pull a reset on the GF1
   outp( myRegSelect, MASTER_RESET );
   outp( myDataHi, 0 );

   // delay a while
   for ( int i = 0; i < 10; i++ )
   {
      gdelay();
   }

   // release reset
   outp( myRegSelect, MASTER_RESET );
   outp( myDataHi, GF1_MASTER_RESET );

   for ( i = 0; i < 10; i++ )
   {
      gdelay();
   }

   // clear interrupts
   outp( myRegSelect, DMA_CONTROL );
   outp( myDataHi, 0 );
   outp( myRegSelect, TIMER_CONTROL );
   outp( myDataHi, 0 );
   outp( myRegSelect, SAMPLE_CONTROL );
   outp( myDataHi, 0 );

   // set number of active voices
   outp( myRegSelect, SET_VOICES );
   outp( myDataHi, ( myNumVoices - 1 ) | 0xC0 );

   // clear interrupts on voices
   (void)inp( myIrqStatus );
   outp( myRegSelect, DMA_CONTROL );
   (void)inp( myDataHi );
   outp( myRegSelect, SAMPLE_CONTROL );
   (void)inp( myDataHi );
   outp( myRegSelect, GET_IRQV );
   (void)inp( myDataHi );

   for ( i = 0; i < myNumVoices; i++ )
   {
      outp( myVocSelect, i );

      outp( myRegSelect, SET_CONTROL );
      outp( myDataHi, VOICE_STOPPED | STOP_VOICE );
      outp( myRegSelect, SET_VOLUME_CONTROL );
      outp( myDataHi, VOLUME_STOPPED | STOP_VOLUME );

      gdelay();

      outp( myRegSelect, SET_FREQUENCY );
      outpw( myDataLo, 0x400 );             //  Set frequency (not Hz)
      outp( myRegSelect, SET_START_HIGH );
      outpw( myDataLo, 0 );
      outp( myRegSelect, SET_START_LOW );
      outpw( myDataLo, 0 );
      outp( myRegSelect, SET_END_HIGH );
      outpw( myDataLo, 0 );
      outp( myRegSelect, SET_END_LOW );
      outpw( myDataLo, 0 );
      outp( myRegSelect, SET_VOLUME_RATE );
      outp( myDataHi, 0x3f );
      outp( myRegSelect, SET_VOLUME_START );
      outp( myDataHi, 0x10 );
      outp( myRegSelect, SET_VOLUME_END );
      outp( myDataHi, 0xe0 );
      outp( myRegSelect, SET_VOLUME );
      outpw( myDataLo, 0 );
      outp( myRegSelect, SET_ACC_HIGH );
      outpw( myDataLo, 0 );
      outp( myRegSelect, SET_ACC_LOW );
      outpw( myDataLo, 0 );
      outp( myRegSelect, SET_BALANCE );
      outp( myDataHi, 0x07 );

   }

   (void)inp( myIrqStatus );
   outp( myRegSelect, DMA_CONTROL );
   (void)inp( myDataHi );
   outp( myRegSelect, SAMPLE_CONTROL );
   (void)inp( myDataHi );
   outp( myRegSelect, GET_IRQV );
   (void)inp( myDataHi );

   // set up GF1 chip for interrupts and enable DACs
   outp( myRegSelect, MASTER_RESET );
   outp( myDataHi, GF1_MASTER_RESET | GF1_OUTPUT_ENABLE | GF1_MASTER_IRQ );

   scEnable();
}

//Ŀ
//                                                                           
// Restore interrupts and handler                                            
//                                                                           
//

void  Gravis::ResetHandler( void ) const
{
   int   temp = myIRQ;

   if ( temp != 0 )
   {
      if ( temp != 2 )
      {
         outp( myILine->imr, inp( myILine->imr ) | ~myILine->mask );
      }

      temp += temp > 7 ? 0x68 : 0x08;

      _dos_setvect( temp, myOldIntVec );
   }
}

//Ŀ
//                                                                           
// Set balance of current voice                                              
//                                                                           
//

void  Gravis::SetBalance( ubyte panpos ) const
{
   outp( myRegSelect, SET_BALANCE );
   outp( myDataHi, panpos );
}

//Ŀ
//                                                                           
// Install a new interrupt handler                                           
//                                                                           
//

void  Gravis::SetHandler( void )
{
   int   temp = myIRQ;

   if ( temp != 0 )
   {
      temp += temp > 7 ? 0x68 : 0x08;

      myOldIntVec = _dos_getvect( temp );

      _dos_setvect( temp, SocomHandler );

      outp( myILine->imr, inp( myILine->imr ) & myILine->mask );
      outp( myILine->ocr, myILine->spec_eoi );

      if ( myIRQ > 7 )
      {
         IRQLine* iLine = &scIRQTable[2];

         outp( iLine->imr, inp( iLine->imr ) & iLine->mask );
         outp( iLine->ocr, iLine->spec_eoi );
      }
   }
}

//Ŀ
//                                                                           
// Program Gravis for IRQs and DMA                                           
//                                                                           
//

void  Gravis::SetInterface( void )
{
   ubyte iControl, dControl;

   dControl = scDMATable[myDMA].latch | 0x40;
   iControl = myILine->latch | 0x40;

   scDisable();

   // set up for digital ASIC
   outp( myBasePort + 0x0f, 0x05 );
   outp( myBasePort, myMixImage );
   outp( myIrqControl, 0 );
   outp( myBasePort + 0x0f, 0 );

   // DMA control register
   outp( myBasePort, myMixImage );
   outp( myIrqControl, dControl | 0x80 );

   // IRQ control register
   outp( myBasePort, myMixImage | 0x40 );
   outp( myIrqControl, iControl );

   // DMA control register
   outp( myBasePort, myMixImage );
   outp( myIrqControl, dControl );

   // IRQ control register
   outp( myBasePort, myMixImage | 0x40 );
   outp( myIrqControl, iControl );

   // just to lock out writes to IRQ/DMA register
   outp( myVocSelect, 0 );

   // enable output and IRQ, disable line & mic input
   outp( myBasePort, myMixImage |= 0x09 );

   // just to lock out writes to IRQ/DMA register
   outp( myVocSelect, 0 );

   scEnable();
}

//Ŀ
//                                                                           
// Set current voice's period                                                
//                                                                           
//

void  Gravis::SetPeriod( int period ) const
{
   if ( period > 1814 )
   {
      period = 1814;
   }

   outp( myRegSelect, SET_FREQUENCY );
   outpw( myDataLo, myFreqTable[period] );
}

//Ŀ
//                                                                           
// Determine the amount of DRAM on-board                                     
//                                                                           
// Returns amount in bytes                                                   
//                                                                           
//

ulong Gravis::SizeDRAM( void )
{
   ubyte save0;
   ubyte save1;
   ulong address;

   save0 = PeekByte( 0 );           // save 1st location

   PokeByte( 0, 0xAA );             // poke in a new byte

   if ( PeekByte( 0 ) != 0xAA )     // check if there is a 1st block of DRAM
   {
      return 0;
   }

   PokeByte( 0, 0x00 );             // zero it out so we can check for mirroring

   for ( ulong i = 1; i < 1024; i++ )
   {
      if ( PeekByte( 0 ) != 0 )     // check for mirroring
      {
         break;
      }

      address = i << 10;

      save1 = PeekByte( address );  // save location so we don't destroy it

      PokeByte( address, 0xAA );    // write a new byte at location

      if ( PeekByte( address ) != 0xAA )
      {
         break;
      }

      PokeByte( address, save1 );   // restore location
   }

   PokeByte( 0, save0 );            // restore 1st location

   return i << 10;
}

//Ŀ
//                                                                           
// Start module replayer                                                     
//                                                                           
//

void  Gravis::StartPlayer( void )
{
   scDisable();

   outp( myRegSelect, myTimerPort );
   outp( myDataHi, myTimerBpm );
   outp( myRegSelect, TIMER_CONTROL );
   outp( myDataHi, myTimerMask );
   outp( myTimerControl, 0x04 );
   outp( myTimerData, 0x03 );

   scEnable();
}

//Ŀ
//                                                                           
// Start up current voice                                                    
//                                                                           
//

void  Gravis::StartVoice( ubyte mode ) const
{
   mode &= ~( VOICE_STOPPED | STOP_VOICE );

   outp( myRegSelect, SET_CONTROL );
   outp( myDataHi, mode );

   gdelay();

   outp( myDataHi, mode );
}

//Ŀ
//                                                                           
// Stop current voice                                                        
//                                                                           
//

void  Gravis::StopVoice( void ) const
{
   ubyte vMode;

   outp( myRegSelect, GET_CONTROL );

   vMode = inp( myDataHi ) | ( VOICE_STOPPED | STOP_VOICE );

   outp( myRegSelect, SET_CONTROL );
   outp( myDataHi, vMode );

   gdelay();

   outp( myDataHi, vMode );
}

//Ŀ
//                                                                           
// Stop current voice's volume                                               
//                                                                           
//

void  Gravis::StopVolume( void ) const
{
   ubyte vMode;

   outp( myRegSelect, GET_VOLUME_CONTROL );

   vMode = inp( myDataHi ) | ( VOLUME_STOPPED | STOP_VOLUME );

   outp( myRegSelect, SET_VOLUME_CONTROL );
   outp( myDataHi, vMode );

   gdelay();

   outp( myDataHi, vMode );
}

//Ŀ
//                                                                           
// Stop playing module                                                       
//                                                                           
//

void  Gravis::Stop( Module* aMod )
{
   scDisable();

   outp( myRegSelect, TIMER_CONTROL );
   outp( myDataHi, 0x00 );

   scEnable();

   aMod->StopPlay();
}

//Ŀ
//                                                                           
// Unload a module from device driver                                        
//                                                                           
//

int   Gravis::UnloadModule( Module* aMod )
{
   int      retValue = SOC_OK;
   Sample*  smp;

   for ( int i = 0; i < aMod->myNumSmp; i++ )
   {
      smp = &aMod->mySamples[i];

      if ( smp->myGravisLoc != 0 && smp->myLength != 0 )
      {
         if (( retValue = FreeDRAM( smp->myLength, smp->myGravisLoc )) != SOC_OK )
         {
            break;
         }

         smp->myGravisLoc = 0;
      }
   }

   return retValue;
}

//Ŀ
//                                                                           
// Volume interrupt handler                                                  
//                                                                           
//

void  Gravis::VolumeHandler( int voice )
{
   Channel* chn = &mpChannels[voice];
   Sample*  smp = &Channel::mySamples[chn->mySampleNum];
   ubyte    volume;
   ubyte    mode;

   outp( myVocSelect, voice );

   if ( chn->myKick )
   {
      chn->myKick = 0;

      StopVoice();

      SetPeriod( chn->myPlayPeriod );

      SetBalance( chn->myPanPos );

      chn->myOldVolume = chn->myPlayVolume;

      volume = ( ubyte(chn->myPlayVolume) * mpMasterVolume ) / 255;

      RampVolume( volume, 0 );

      mode = PrimeVoice( smp->myGravisLoc + chn->myStartPos, smp->myRepPos,
         smp->myRepEnd, smp->mySampleMode );

      StartVoice( mode );
   }
}

//Ŀ
//                                                                           
// Play a standalone digital sample                                          
//                                                                           
//

void  Gravis::PlaySample( ubyte voice, ubyte panPos, Sample* smp )
{
   ubyte mode;
   ubyte pan = panPos & 0x0f;

   scDisable();

   outp( myVocSelect, voice );

   RampVolume( 0, 0 );
   StopVoice();
   SetBalance( pan );
   RampVolume( smp->myVolume, 0 );
   SetPeriod( smp->myPeriod );

   mode = PrimeVoice( smp->myGravisLoc, smp->myRepPos, smp->myRepEnd, smp->mySampleMode );
   StartVoice( mode );

   scEnable();
}

//Ŀ
//                                                                           
// Stop a running sample                                                     
//                                                                           
//

void  Gravis::StopSample( int voice ) const
{
   scDisable();

   outp( myVocSelect, voice );   // select correct voice

   RampVolume( 0, 0 );           // slide volume to 0
   StopVoice();                  // stop the voice

   scEnable();
}

