/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev and Christian Lupien.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#define GATOS_ATI_C 1

#include "gatos.h"
#include "ati.h"
#include "atiregs.h"

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

/* Private global vars */
static u32 scale_cntl=DISABLE_VIDEO ;
static int stride=0 ;	/* FrameBuffer pixel stride (bytes per pixel) */
static u32 save_off_pitch=0, save_h_total_disp=0, save_v_total_disp=0 ;
static u8 ecp_div=0 ;
static int firstpoll=1 ;

/* ------------------------------------------------------------------------ */
/* Initialization routines */

int ati_init(void) {

  u32 id=CONFIG_CHIP_ID ; int d, v, f, m, regs=0 ;

  /* Both chip register apertures enabled (chip in use by X server) ? */
  if (id == 0xFFFFFFFF || VIDEO_FORMAT == 0xFFFFFFFF) RETURN(ENXIO) ;

  /* Chip Class must be 0x00 (VGA device) */
  if (id & 0x00FF0000) RETURN(ENODEV) ;

  /* Check the FrameBuffer address */
  if ( ((CONFIG_CNTL<<18) & 0xFFC00000) != gatos.fbaddr ) RETURN(EINVAL) ;

  /* ATI chip device id and chip revision */
  d = gatos.ati.deviceid = id & 0x0000FFFF ;		/* Device ID */
  v = gatos.ati.revision = (id & 0x07000000) >> 24 ;	/* Chip version */
  f = (id & 0x38000000) >> 27 ;		/* Chip Foundry */
  m = (id & 0xC0000000) >> 30 ;		/* Chip Version Minor */

  /* Set detailed ATI chip ID flags (struct ati) */
  ati_chipid(d,v,f,m) ;

  /* Older chips w/o video capture and overlay scaler */
  if (ati.gx || ati.cx || ati.ct || ati.et) RETURN(ENOSYS) ;

  /* Amount of Video RAM */
  gatos.atiram = MEM_CNTL & 15 ;
  if (gatos.atiram < 8) gatos.atiram = 512*(gatos.atiram+1) ;
  else if (gatos.atiram < 12) gatos.atiram = 1024*(gatos.atiram-3) ;
  else gatos.atiram = 2048*(gatos.atiram-7) ;

  /* Place video capture buffers at top of video memory, reserve *
   * 4kb for registers in framebuffer (only if 8MB videomem)     */
  if (gatos.regaddr == 0xFFFFFFFF && gatos.videoram == 8192) regs = 4 ;
  gatos.captbufsize = (gatos.buffermem - regs) >> 1 ;
  gatos.buffer0 = 1024*(gatos.videoram - regs) - 1024*gatos.captbufsize ;
  gatos.buffer1 = gatos.buffer0                - 1024*gatos.captbufsize ;

  /* Initial ATI register programming */

  scale_cntl         = ENABLE_VIDEO ;
  TRIG_CNTL          = 0x00000000 ;
  OVERLAY_SCALE_CNTL = DISABLE_VIDEO ;

  CAPTURE_BUF0_OFFSET = gatos.buffer0 ;
  CAPTURE_BUF1_OFFSET = gatos.buffer1 ;

  OVERLAY_VIDEO_CLR = 0x00000000 ;
  OVERLAY_VIDEO_MSK = 0x00000000 ;
  OVERLAY_KEY_CNTL  = 0x00000050 ;

  SCALER_COLOR_CNTL = 0x00101000 ;
  SCALER_H_COEFF0   = 0x00002000 ;
  SCALER_H_COEFF1   = 0x0D06200D ;
  SCALER_H_COEFF2   = 0x0D0A1C0D ;
  SCALER_H_COEFF3   = 0x0C0E1A0C ;
  SCALER_H_COEFF4   = 0x0C14140C ;

  VIDEO_FORMAT      = 0x000B000B ;

  ati_setcolorkey() ; ati_pollscreen(0) ; RETURN0 ; }

/* ------------------------------------------------------------------------ */
/* Set detailed ATI chip ID flags */

static int ati_chipid(int d, int v, int f, int m) {

  int match=0, vfm=(v<<8)|(f<<4)|m, vm=(v<<4)|m ;

  memset(&ati,0,sizeof(ati)) ; /* Clear all flags first */

  switch (d) {	/* Chip type (ID) */

    case 0x00D7: ati.gx = 1 ;					/* (GX) */
      switch (v) {
        case 0:  ati.gx_c = match = 1 ; break ;
        case 1:  ati.gx_d = match = 1 ; break ;
        case 2:  ati.gx_e = match = 1 ; break ;
        case 3:
        default: ati.gx_f = match = 1 ; break ; }
      break ;
    case 0x0057: ati.cx = match = 1 ; break ;			/* (CX) */
    case 0x4354: ati.ct = match = 1 ; break ;			/* CT */
    case 0x4554: ati.et = match = 1 ; break ;			/* ET */
    case 0x4C54: ati.lt = match = 1 ; break ;			/* LT */

    /* VT */
    case 0x5654:						/* VT */
    case 0x5655:						/* VU */
      ati.vt = 1 ;
      switch (vfm) {
        case 0x010: ati.vt_a3n1 = match = 1 ; break ;
        case 0x000: ati.vt_a3s1 = match = 1 ; break ;
        case 0x011: ati.vt_a4n1 = match = 1 ; break ;
        case 0x001: ati.vt_a4s1 = match = 1 ; break ;
        case 0x100: ati.vt_b1s1 = match = 1 ; break ;
        case 0x230: ati.vt_b2u1 = match = 1 ; break ;
        case 0x231: ati.vt_b2u2 = match = 1 ; break ;
        case 0x232: ati.vt_b2u3 = match = 1 ; break ; }
      if (match && v>0) ati.doesvtb = 1 ; break ;

    /* GT */
    case 0x4754:						/* GT */
    case 0x4755:						/* GU */
      ati.gt = 1 ;
      switch (vfm) {
        case 0x000: if (d==0x4754) ati.gt_a2s1 = match = 1 ; break ;
        case 0x100: if (d==0x4754) ati.gt_b1s1 = match = 1 ; break ;
        case 0x101: if (d==0x4754) ati.gt_b1s2 = match = 1 ; break ;
        case 0x230: ati.gt_b2u1 = match = 1 ; break ;
        case 0x231: ati.gt_b2u2 = match = 1 ; break ;
        case 0x232: ati.gt_b2u3 = match = 1 ; break ;
        default: break ; }
      if (match && v>0) ati.doesvtb = 1 ; break ;

    /* Rage IIC */
    case 0x4756:						/* GV */
    case 0x475A:						/* GZ */
    case 0x4757:						/* GW */
    case 0x4759:						/* GY */
    case 0x5656:						/* VV */
    case 0x5657:						/* VW */
      ati.gt = 1 ;
      switch (vm) {
        case 0x20: switch (d) {
            case 0x4756: ati.gv_b3u1 = match = 1 ; break ;
            case 0x475A: ati.gz_b3u1 = match = 1 ; break ;
            case 0x4757: ati.gw_b3u1 = match = 1 ; break ;
            case 0x4759: ati.gy_b3u1 = match = 1 ; break ;
            case 0x5656: ati.vv_b3u1 = match = ati.vt = 1 ; ati.gt = 0 ; break ;
            case 0x5657: ati.vw_b3u1 = match = ati.vt = 1 ; ati.gt = 0 ; break ; }
          ati.gt2c = ati.gt2c_b3u1 = ati.doesvtb = 1 ; break ;
        case 0x21: switch (d) {
            case 0x4756: ati.gv_b3u2 = match = 1 ; break ;
            case 0x475A: ati.gz_b3u2 = match = 1 ; break ;
            case 0x4757: ati.gw_b3u2 = match = 1 ; break ;
            case 0x4759: ati.gy_b3u2 = match = 1 ; break ;
            case 0x5656: ati.vv_b3u2 = match = ati.vt = 1 ; ati.gt = 0 ; break ;
            case 0x5657: ati.vw_b3u2 = match = ati.vt = 1 ; ati.gt = 0 ; break ; }
          ati.gt2c = ati.gt2c_b3u2 = ati.doesvtb = 1 ; break ;
        default: break ; }
      break ;

    /* Rage LT */
    case 0x4C47:						/* LG */
      ati.lg = match = ati.lg_a1s1 = ati.doesvtb = 1 ; break ;

    /* Rage LT PRO (LT-C) */
    case 0x4C44:						/* LD */
    case 0x4C49:						/* LI */
    case 0x4C50:						/* LP */
    case 0x4C51:						/* LQ */
    case 0x4C42:						/* LB */
      ati.lt_c = ati.doesltc = ati.doesgtc = 1 ;
      switch (vm) {
        case 0x00: switch (d) {
            case 0x4C44: ati.ld_c1u1 = match = 1 ; break ;
            case 0x4C49: ati.li_c1u1 = match = 1 ; break ;
            case 0x4C50: ati.lp_c1u1 = match = 1 ; break ;
            case 0x4C51: ati.lq_c1u1 = match = 1 ; break ;
            case 0x4C42: ati.lb_c1u1 = match = 1 ; break ; }
          ati.lt_c1u1 = 1 ; break ;
        case 0x41: switch (d) {
            case 0x4C44: ati.ld_c2u1 = match = 1 ; break ;
            case 0x4C49: ati.li_c2u1 = match = 1 ; break ;
            case 0x4C50: ati.lp_c2u1 = match = 1 ; break ;
            case 0x4C51: ati.lq_c2u1 = match = 1 ; break ;
            case 0x4C42: ati.lb_c2u1 = match = 1 ; break ; }
          ati.lt_c2u1 = 1 ; break ;
        case 0x42: switch (d) {
            case 0x4C44: ati.ld_c2u2 = match = 1 ; break ;
            case 0x4C49: ati.li_c2u2 = match = 1 ; break ;
            case 0x4C50: ati.lp_c2u2 = match = 1 ; break ;
            case 0x4C51: ati.lq_c2u2 = match = 1 ; break ;
            case 0x4C42: ati.lb_c2u2 = match = 1 ; break ; }
          ati.lt_c2u2 = 1 ; break ;
        case 0x43: switch (d) {
            case 0x4C44: ati.ld_c2u3 = match = 1 ; break ;
            case 0x4C49: ati.li_c2u3 = match = 1 ; break ;
            case 0x4C50: ati.lp_c2u3 = match = 1 ; break ;
            case 0x4C51: ati.lq_c2u3 = match = 1 ; break ;
            case 0x4C42: ati.lb_c2u3 = match = 1 ; break ; }
          ati.lt_c2u3 = 1 ; break ;
        default: ati.lt_c2u3 = ati.lb_c2u3 = 1 ; break ; }
      break ;

    /* Rage Mobility */
    case 0x4C4D:						/* LM */
    case 0x4C52:						/* LR */
      ati.lt_c = ati.doesltc = ati.doesgtc = 1 ;
      switch (vm) {
        case 0x40: switch (d) {
            case 0x4C4D: ati.lm_a1t1 = match = 1 ; break ;
            case 0x4C52: ati.lr_a1t1 = match = 1 ; break ; }
          ati.ltm = 1 ; break ;
        default: break ; }
      break ;

    /* Rage XL/XC */
    case 0x474D:						/* GM */
    case 0x474E:						/* GN */
    case 0x474F:						/* GO */
  /*case 0x4750: The Mach64 GP is also a GT_C ?!?		 * GP */
    case 0x4752:						/* GR */
    case 0x4753:						/* GS */
      ati.lt_c = ati.ltm = ati.doesltc = ati.doesgtc = 1 ;
      switch (vm) {
        case 0x40: switch (d) {
            case 0x474D: ati.gm_a1t1 = match = 1 ; break ;
            case 0x474E: ati.gn_a1t1 = match = 1 ; break ;
            case 0x474F: ati.go_a1t1 = match = 1 ; break ;
          /*case 0x4750: ati.gp_a1t1 = match = 1 ; break ; => GT_C below */
            case 0x4752: ati.gr_a1t1 = match = 1 ; break ;
            case 0x4753: ati.gs_a1t1 = match = 1 ; break ;
          ati.xlxc = 1 ; break ; }
        default: break ; }
      break ;

    /* GT_C */
    case 0x4750: /* The Mach64 GP is also a Rage XL/XC ?!?	 * GP */
      ati.lt_c = ati.ltm = ati.doesltc = ati.doesgtc = 1 ;
      if (vm==0x40) ati.gp_a1t1 = ati.xlxc = match = 1 ;
    case 0x4744:						/* GD */
    case 0x4749:						/* GI */
    case 0x4751:						/* GQ */
    case 0x4742:						/* GB */
      ati.gt = ati.gt_c = ati.doesgtc = 1 ;
      switch (vfm) {
        case 0x330: switch (d) {
            case 0x4744: ati.gd_c1u1 = match = 1 ; break ;
            case 0x4750: ati.gp_c1u1 = match = 1 ; break ;
            case 0x4749: ati.gi_c1u1 = match = 1 ; break ;
            case 0x4742: ati.gb_c1u1 = match = 1 ; break ; }
          ati.gt_c1u1 = 1 ; break ;
        case 0x331: switch (d) {
            case 0x4744: ati.gd_c1u2 = match = 1 ; break ;
            case 0x4750: ati.gp_c1u2 = match = 1 ; break ;
            case 0x4749: ati.gi_c1u2 = match = 1 ; break ;
            case 0x4742: ati.gb_c1u2 = match = 1 ; break ; }
          ati.gt_c1u2 = 1 ; break ;
        case 0x430: switch (d) {
            case 0x4744: ati.gd_c2u1 = match = 1 ; break ;
            case 0x4750: ati.gp_c2u1 = match = 1 ; break ;
            case 0x4749: ati.gi_c2u1 = match = 1 ; break ;
            case 0x4742: ati.gb_c2u1 = match = 1 ; break ;
            case 0x4751: ati.gq_c2u1 = match = 1 ; break ; }
          ati.gt_c2u1 = 1 ; break ;
        default:
          if (vm!=0x41) break ;
          switch (d) {
            case 0x4744: ati.gd_c2u2 = match = 1 ; break ;
            case 0x4750: ati.gp_c2u2 = match = 1 ; break ;
            case 0x4749: ati.gi_c2u2 = match = 1 ; break ;
            case 0x4742: ati.gb_c2u2 = match = 1 ; break ;
            case 0x4751: ati.gq_c2u2 = match = 1 ; break ; }
          ati.gt_c2u2 = 1 ; break ; }
      break ;

    default: break ; }

  /* Select latest if no exact match */
  if (!match) {
    ati.gt = ati.gt_c = ati.gb_c2u2 = ati.gt_c2u2 = ati.doesgtc = 1 ; }

  return match ; }

/* ------------------------------------------------------------------------ */
/* Public routines */

/* Overlay Window Colorkeying */
int ati_setcolorkey(void) {
  if ((CRTC_GEN_CNTL & 0x01000000))		/* Extended mode */
    switch ((CRTC_GEN_CNTL & 0x00000700) >> 8) {
      case 1: OVERLAY_GRAPHICS_MSK = 0x0000000F ; stride = 1 ; break ;
      case 2: OVERLAY_GRAPHICS_MSK = 0x000000FF ; stride = 1 ; break ;
      case 3: OVERLAY_GRAPHICS_MSK = 0x00007FFF ; stride = 2 ; break ;
      case 4: OVERLAY_GRAPHICS_MSK = 0x0000FFFF ; stride = 2 ; break ;
      case 5: OVERLAY_GRAPHICS_MSK = 0x00FFFFFF ; stride = 4 ; break ;
      case 6: OVERLAY_GRAPHICS_MSK = 0x00FFFFFF ; stride = 4 ; break ;
      default: RETURN(ENODEV) ; break ; }
  else {					/* VGA mode: assume 8bpp */
    OVERLAY_GRAPHICS_MSK = 0x000000FF ; stride = 1 ; }
  OVERLAY_GRAPHICS_CLR = gatos.colorkey ; RETURN0 ; }

/* Enable/display video capture to capture buffers */
void ati_enable_capture(int enable) {
  CAPTURE_CONFIG = (enable) ? 0x10000015 : 0x00000000 ; }

/* Enable/disable video display in Overlay Window */
void ati_enable_video(int enable) {
  OVERLAY_SCALE_CNTL = (enable) ? scale_cntl : DISABLE_VIDEO ; }

/* Capture Buffer Mode */
void ati_buffer_mode(void) {
  /* TODO: Single/double buffer, field/frame capture */
}

/* ------------------------------------------------------------------------ */
/* Size in pixels of incoming video stream (full frame) */
int ati_setcaptsize(void) {

  int v=0 ;

  if (gatos.xcapt > hactive[gatos.hcaptmax]) RETURN(EOVERFLOW) ;
  if (gatos.ycapt > vactive[gatos.vcaptmax]) RETURN(EOVERFLOW) ;

  /* NTSC, PAL or SECAM ? */
  switch (gatos.format) {
    case 1: case 2: case 4:			/* NTSC */
      v = 23 ;
      if (gatos.xcapt > 640) RETURN(EINVAL) ;
      if (gatos.ycapt > 480) RETURN(EINVAL) ;
      if (gatos.ycapt <= 360) v = 22 ;
      if (gatos.ycapt <= 240) v = 21 ;
      if (gatos.ycapt <= 120) v = 20 ;
      if (gatos.ycapt <=  60) v = 19 ; break ;
    case 3: case 5: case 7:			/* PAL */
      v = 24 ;
      if (gatos.xcapt > 768) RETURN(EINVAL) ;
      if (gatos.ycapt > 576) RETURN(EINVAL) ;
      if (gatos.ycapt <= 432) v = 23 ;
      if (gatos.ycapt <= 288) v = 21 ;
      if (gatos.ycapt <= 144) v = 19 ;
      if (gatos.ycapt <=  72) v = 18 ; break ;
    case 6:					/* SECAM */
      v = 25 ;
      if (gatos.xcapt > 768) RETURN(EINVAL) ;
      if (gatos.ycapt > 576) RETURN(EINVAL) ;
      if (gatos.ycapt <= 432) v = 24 ;
      if (gatos.ycapt <= 288) v = 22 ;
      if (gatos.ycapt <= 144) v = 20 ;
      if (gatos.ycapt <=  72) v = 19 ; break ;
    default: RETURN(EINVAL) ; }

#if defined(GATOSBUTTONS) && GATOSBUTTONS >= 1
  v += gatosbuttons[0] ;
#endif

  OVERLAY_SCALE_CNTL = DISABLE_VIDEO ;
  SCALER_BUF_PITCH   = gatos.xcapt ;
  CAPTURE_X_WIDTH    = XY(2*gatos.xcapt,0) ;
  CAPTURE_START_END  = XY(v-1+(gatos.ycapt>>1),v) ;
  TRIG_CNTL          = 0x80000000 ;
  if (gatos.video) OVERLAY_SCALE_CNTL = scale_cntl ;

  RETURN0 ; }

/* ------------------------------------------------------------------------ */

int ati_pollscreen(int sig) {

  u8 pll_vclk_cntl, pll_ref_div, vclk_post_div, vclk_fb_div, save ;
  u32 disp, pitch ; int i ;

  /* Return if called from signal handler and nothing changed */
  if ( sig && (CRTC_OFF_PITCH == save_off_pitch)
    && (CRTC_H_TOTAL_DISP == save_h_total_disp)
    && (CRTC_V_TOTAL_DISP == save_v_total_disp) ) return 0 ;

  /* Remember new values for next time */
  save_off_pitch = CRTC_OFF_PITCH ;
  save_h_total_disp = CRTC_H_TOTAL_DISP ;
  save_v_total_disp = CRTC_V_TOTAL_DISP ;

  /* Get monitor dotclock, check for Overlay Scaler clock limit */
  save = CLOCK_CNTL1 ; i = CLOCK_CNTL0 & 3 ;
  CLOCK_CNTL1 = 2<<2 ; pll_ref_div = CLOCK_CNTL2 ;
  CLOCK_CNTL1 = 5<<2 ; pll_vclk_cntl = CLOCK_CNTL2 ;
  CLOCK_CNTL1 = 6<<2 ; vclk_post_div = (CLOCK_CNTL2>>(i+i)) & 3 ;
  CLOCK_CNTL1 = (7+i)<<2 ; vclk_fb_div = CLOCK_CNTL2 ;
  gatos.dotclock = 2.0*gatos.refclock*vclk_fb_div
                   / (pll_ref_div*(1<<vclk_post_div)) ;

  /* ecp_div: 0=dotclock, 1=dotclock/2, 2=dotclock/4 */
  ecp_div = gatos.dotclock /
    (gatos.aticard[gatos.cardidx].dotclock + gatos.overclock) ;
  if (ecp_div>2) ecp_div = 2 ;
  if ((pll_vclk_cntl&0x30) != ecp_div<<4) {
    CLOCK_CNTL1 = (5<<2)|2 ;
    CLOCK_CNTL2 = (pll_vclk_cntl&0xCF) | (ecp_div<<4) ; }
  CLOCK_CNTL1 = save ;

  /* Screen (monitor) dimensions */
  if ((CRTC_GEN_CNTL & 0x01000000)) {	/* Extended mode */
    gatos.xdim = (CRTC_H_TOTAL_DISP>>16) + 1 ; gatos.xdim *= 8 ;
    gatos.ydim = (CRTC_V_TOTAL_DISP>>16) + 1 ; }
  else {				/* VGA mode */
    /* TODO: decode VGA registers */
    gatos.xdim = 1280 ; gatos.ydim = 1024 ; }

  /* Desktop panning */
  disp = 8 * (CRTC_OFF_PITCH & 0x000FFFFF) / stride ;
  pitch = 8 * ((CRTC_OFF_PITCH & 0xFFC00000) >> 22) ;
  gatos.ypan = disp/pitch ; gatos.xpan = disp - pitch*gatos.ypan ;

  /* Tell user about the new values */
  if (GATOSASYNC || (VERBOSE && firstpoll)) fprintf(stderr,
    "%s Screen %dx%d+%d+%d, Dotclock %f MHz, Scaler Clock Factor %d\n",
      (firstpoll)?"==>":"***", gatos.xdim, gatos.ydim,
      gatos.xpan, gatos.ypan, gatos.dotclock, 1<<ecp_div) ;
  firstpoll = 0 ;

  /* Caller should call ati_setgeometry() */
  return 1 ; }

/* ------------------------------------------------------------------------ */
/* Size and position of Overlay Scaler window */
void ati_setgeometry(void) {

  int xs, ys, xc, yc, x0, y0, x1, x2, y1, y2, xscale, yscale, offset=0 ;

  /* Window size and position */
  xs = gatos.xsize ; x0 = gatos.xpos - gatos.xpan ;
  ys = gatos.ysize ; y0 = gatos.ypos - gatos.ypan ;

  /* Disable video before reprogramming registers */
  OVERLAY_SCALE_CNTL = DISABLE_VIDEO ;

  /* Return (with video disabled) if window totally outside screen */
  if (x0+gatos.xsize <= 0 || y0+gatos.ysize <= 0 ||
      x0 >= gatos.xdim || y0 >= gatos.ydim) return ;

  /* Number of pixels outside screen:
   * x1=left, x2=right, y1=top, y2=bottom */
  x1 = -x0 ; x2 = x0+xs-gatos.xdim ; if (x1<0) x1 = 0 ; if (x2<0) x2 = 0 ;
  y1 = -y0 ; y2 = y0+ys-gatos.ydim ; if (y1<0) y1 = 0 ; if (y2<0) y2 = 0 ;

  /* Size and position of on-screen part of window */
  xs -= x1+x2 ; if (x0<0) x0 = 0 ;
  ys -= y1+y2 ; if (y0<0) y0 = 0 ;

  /* Video scaling factors */
  xc = gatos.xcapt*xs/gatos.xsize ; xscale = xc*4096*(1<<ecp_div)/xs ;
  yc = gatos.ycapt*ys/gatos.ysize ; yscale = yc*2048/ys ;

  /* Offset Scaler buffers from Capture buffers *
   * if upper left window corner outside screen */
  if (x1+y1) offset = 2*x1*gatos.xcapt/gatos.xsize +
    ((y1*gatos.ycapt/gatos.ysize)&~1) * gatos.xcapt ;

  /* Program the chip */
  SCALER_BUF0_OFFSET = gatos.buffer0+offset ;
  SCALER_BUF1_OFFSET = gatos.buffer1+offset ;
  SCALER_HEIGHT_WIDTH = XY(xc,yc>>1) ;
  OVERLAY_SCALE_INC = XY(xscale,yscale) ;
  OVERLAY_Y_X_START = XY(x0,y0) | 0x80000000 ;
  OVERLAY_Y_X_END = XY(x0+xs-1,y0+ys-1) ;
  if (gatos.video) OVERLAY_SCALE_CNTL = scale_cntl ; }

/* ------------------------------------------------------------------------ */

/* Note:
 * Gamma and Red Temp correction only affects
 * video displayed in the Overlay Scaler window.
 * Data in the capture buffers is not affected.
 */

/* Overlay Scaler window gamma correction */
int ati_setgamma(void) {
  scale_cntl = ENABLE_VIDEO ;
  if (gatos.video) OVERLAY_SCALE_CNTL = scale_cntl ;
  RETURN0 ; }

/* Overlay Scaler window red color temp (cold=9800K or warm=6500K) */
int ati_setcold(void) {
  scale_cntl = ENABLE_VIDEO ;
  if (gatos.video) OVERLAY_SCALE_CNTL = scale_cntl ;
  RETURN0 ; }

/* ------------------------------------------------------------------------ */
/* Capture and playback routines */

void ati_capture(FILE *file, int wait) {

  volatile void *buf0=ATIFB+CAPTURE_BUF0_OFFSET,
                *buf1=ATIFB+CAPTURE_BUF1_OFFSET ;
  int i ;

  /* Capture (no sync to data arrival; requires kernel support :-( */
  while (1) {
    fprintf(file,"%3d %3d\n",gatos.xcapt,gatos.ycapt) ;
    for ( i=0 ; i<gatos.ycapt ; i+=2 ) {
      fwrite((void*)(buf0+i*gatos.xcapt),2,gatos.xcapt,file) ;
      fwrite((void*)(buf1+i*gatos.xcapt),2,gatos.xcapt,file) ; }
    fflush(file) ; if (gatos.stop) break ; if (wait) usleep(wait) ; } }

void ati_playback(FILE *file, int wait) {

  int i, h, v, xsave=gatos.xcapt, ysave=gatos.ycapt, res ; char buf[9] ;
  volatile void *buf0=ATIFB+CAPTURE_BUF0_OFFSET,
                *buf1=ATIFB+CAPTURE_BUF1_OFFSET ;

  CAPTURE_CONFIG = 0x00000000 ;

  /* Playback (no sync; requires kernel support :-( */
  while ( fread(buf,1,8,file) == 8 ) {
    res = sscanf(buf,"%3d %3d\n",&h,&v) ;
    if (h!=gatos.xcapt || v!=gatos.ycapt) {
      gatos_setcapturesize(gatos.xcapt,gatos.ycapt) ;
      gatos_setgeometry(gatos.xsize,gatos.ysize,gatos.xpos,gatos.ypos) ; }
    for ( i=0 ; i<gatos.ycapt ; i+=2 ) {
      fread((void*)(buf0+i*gatos.xcapt),2,gatos.xcapt,file) ;
      fread((void*)(buf1+i*gatos.xcapt),2,gatos.xcapt,file) ; }
    if (gatos.stop) break ; if (wait) usleep(wait) ; }

  if (gatos.xcapt!=xsave || gatos.ycapt!=ysave) {
    gatos_setcapturesize(xsave,ysave) ;
    gatos_setgeometry(gatos.xsize,gatos.ysize,gatos.xpos,gatos.ypos) ; }

  /* Done */
  CAPTURE_CONFIG = 0x10000015 ; }

/* ------------------------------------------------------------------------ */
/* Debug and report routines */

#define DUMPREG(MEMADDR,NAME)	\
	fprintf(stderr,"%s: %-22s (%s) = 0x%08X\n", \
	gatos.ati.ident,#NAME,MEMADDR,NAME)

void ati_dumpregs(void) {
  int i ; u8 tmp ;
  DUMPREG("MEM_0_00",CRTC_H_TOTAL_DISP) ;
  DUMPREG("MEM_0_02",CRTC_V_TOTAL_DISP) ;
  DUMPREG("MEM_0_05",CRTC_OFF_PITCH) ;
  DUMPREG("MEM_0_06",CRTC_INT_CNTL) ;
  DUMPREG("MEM_0_07",CRTC_GEN_CNTL) ;
  DUMPREG("MEM_0_0F",I2C_CNTL_0) ;
  DUMPREG("MEM_0_1E",GP_IO) ;
  DUMPREG("MEM_0_24",CLOCK_CNTL) ;
  DUMPREG("MEM_0_2C",MEM_CNTL) ;
  DUMPREG("MEM_0_2F",I2C_CNTL_1) ;
  DUMPREG("MEM_0_31",DAC_CNTL) ;
  DUMPREG("MEM_0_32",EXT_DAC_REGS) ;
  DUMPREG("MEM_0_34",GEN_TEST_CNTL) ;
  DUMPREG("MEM_0_37",CONFIG_CNTL) ;
  DUMPREG("MEM_0_38",CONFIG_CHIP_ID) ;
  DUMPREG("MEM_1_00",OVERLAY_Y_X_START) ;
  DUMPREG("MEM_1_01",OVERLAY_Y_X_END) ;
  DUMPREG("MEM_1_02",OVERLAY_VIDEO_CLR) ;
  DUMPREG("MEM_1_03",OVERLAY_VIDEO_MSK) ;
  DUMPREG("MEM_1_04",OVERLAY_GRAPHICS_CLR) ;
  DUMPREG("MEM_1_05",OVERLAY_GRAPHICS_MSK) ;
  DUMPREG("MEM_1_06",OVERLAY_KEY_CNTL) ;
  DUMPREG("MEM_1_08",OVERLAY_SCALE_INC) ;
  DUMPREG("MEM_1_09",OVERLAY_SCALE_CNTL) ;
  DUMPREG("MEM_1_0A",SCALER_HEIGHT_WIDTH) ;
  DUMPREG("MEM_1_0D",SCALER_BUF0_OFFSET) ;
  DUMPREG("MEM_1_0E",SCALER_BUF1_OFFSET) ;
  DUMPREG("MEM_1_0F",SCALER_BUF_PITCH) ;
  DUMPREG("MEM_1_10",CAPTURE_START_END) ;
  DUMPREG("MEM_1_11",CAPTURE_X_WIDTH) ;
  DUMPREG("MEM_1_12",VIDEO_FORMAT) ;
  DUMPREG("MEM_1_13",VBI_START_END) ;
  DUMPREG("MEM_1_14",CAPTURE_CONFIG) ;
  DUMPREG("MEM_1_15",TRIG_CNTL) ;
  DUMPREG("MEM_1_16",OVERLAY_EXCLUSIVE_HORZ) ;
  DUMPREG("MEM_1_17",OVERLAY_EXCLUSIVE_VERT) ;
  DUMPREG("MEM_1_18",VBI_WIDTH) ;
  DUMPREG("MEM_1_20",CAPTURE_BUF0_OFFSET) ;
  DUMPREG("MEM_1_21",CAPTURE_BUF1_OFFSET) ;
  DUMPREG("MEM_1_22",ONESHOT_BUF_OFFSET) ;
  DUMPREG("MEM_1_30",MPP_CONFIG) ;
  DUMPREG("MEM_1_31",MPP_STROBE_SEQ) ;
  DUMPREG("MEM_1_32",MPP_ADDR) ;
  DUMPREG("MEM_1_33",MPP_DATA) ;
  DUMPREG("MEM_1_40",TVO_CNTL) ;
  DUMPREG("MEM_1_54",SCALER_COLOR_CNTL) ;
  DUMPREG("MEM_1_55",SCALER_H_COEFF0) ;
  DUMPREG("MEM_1_56",SCALER_H_COEFF1) ;
  DUMPREG("MEM_1_57",SCALER_H_COEFF2) ;
  DUMPREG("MEM_1_58",SCALER_H_COEFF3) ;
  DUMPREG("MEM_1_59",SCALER_H_COEFF4) ;
  DUMPREG("MEM_1_75",SCALER_BUF0_OFFSET_U) ;
  DUMPREG("MEM_1_76",SCALER_BUF0_OFFSET_V) ;
  DUMPREG("MEM_1_77",SCALER_BUF1_OFFSET_U) ;
  DUMPREG("MEM_1_78",SCALER_BUF1_OFFSET_V) ;
  tmp = CLOCK_CNTL1 ;
  for ( i=0 ; i<32 ; i++ ) {
    if (i%16==0) fprintf(stderr,"%s: PLL[%2d..%2d] =",gatos.ati.ident,i,i+15) ;
    CLOCK_CNTL1 = i<<2 ; fprintf(stderr," %02X",CLOCK_CNTL2) ;
    if ((i+1)%16==0) fprintf(stderr,"\n") ; }
  CLOCK_CNTL1 = tmp ;
}
