/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev and Christian Lupien.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#define GATOS_BOARD_C 1

#include "gatos.h"
#include "board.h"
#include "atiregs.h"
#include "i2c.h"
#include "bt829.h"
#include "fi12xx.h"

#include <stdio.h>
#include <errno.h>
#include <string.h>

#define AUDIOCHIP(ADDR,CHIP,IDENT,NAME)	\
	gatos.##CHIP##.addr = 0xFF ; if (i2c_device(ADDR)) { \
	  gatos.##CHIP##.addr = ADDR ; i2c_register(ADDR,IDENT,NAME) ; } \
	has_##CHIP = (gatos.##CHIP##.addr != 0xFF)

/* Private global vars */
static int has_tda8425=0, has_tda9850=0, has_tda9851=0, has_msp3410=0 ;
static int tda8425reg[9] ;
static char *idents[CARDTYPES+1] = {
  NULL, "ST.ALONE", "INTEGRATED", "EXTPOD",
  "AIW", "AIWPRO", "AIW128", "NEC", "INTEL" } ;
static char *names[CARDTYPES+1] = {
  NULL, "Stand-Alone", "Integrated", "External POD",
  "All-In-Wonder", "All-In-Wonder Pro", "All-In-Wonder 128",
  "NEC", "Intel" } ;

/* ------------------------------------------------------------------------ */
/* Initialization routine */

int board_init(void) {

  u8 sum=0, data[16] ; int i, j ; char name[64] ;
  u8 *ptr=gatos.aticard[gatos.cardidx].biosinfo ;

  /* Identify Capture Board */
  gatos.cardtype = board_identify(&gatos.board.addr,&gatos.boardinfo) ;
  snprintf(name,sizeof(name),"%s Capture Board",names[gatos.cardtype]) ;

  /* Register Capture Board */
  if (gatos.board.addr == 0xFF) {
    if (VERBOSE) { printf("%s\n",name) ; fflush(stdout) ; }
    if (*ptr&0x0F) gatos.tunertype = (*ptr<=MAXTUNERTYPE) ? *ptr : 0 ; }
  else {
    i2c_register(gatos.board.addr,idents[gatos.cardtype],name) ;
    board_setbyte() ; /* Enable Capture Board */
    gatos.tunertype = gatos.boardinfo & 0x0F ;
    if (gatos.tunertype==15 && *ptr<=MAXTUNERTYPE) gatos.tunertype = *ptr ;
    if (gatos.tunertype>MAXTUNERTYPE) gatos.tunertype = 0 ; }

  /* I2C EEPROM */
  gatos.eeprom.addr = 0xFF ;
  if (i2c_device(0xA8)) {
    data[0] = 0x00 ; i2c_write(0xA8,data,1) ;
    for ( i=0 ; i<8 ; i++ ) {
      i2c_readn(0xA8,data,16) ; for ( j=0 ; j<16 ; j++ ) sum ^= data[j] ; }
    if (sum) i2c_register(0xA8,"EEPROM","EEPROM (checksum error)") ;
    else { i2c_register(0xA8,"EEPROM",NULL) ; gatos.eeprom.addr = 0xA8 ; } }

  /* Audio chips */

  AUDIOCHIP(0x82,tda8425,"TDA8425","TDA8425 Stereo Audio Processor") ;
  AUDIOCHIP(0xB4,tda9850,"TDA9850","TDA9850 BTSC Stereo+SAP Audio Processor") ;
  AUDIOCHIP(0xB6,tda9851,"TDA9851","TDA9851 BTSC Economic Stereo Audio Processor") ;
  AUDIOCHIP(0x80,msp3410,"MSP3410","MSP3410 Multi-Sound Audio Processor") ;

  for ( i=0 ; i<9 ; i++ ) { tda8425reg[i] = -1 ; }

  /* FIXME /AA */
  if (has_tda8425 || has_tda9850 || has_msp3410) gatos.stereo = 1 ;

  board_setbyte() ; board_setaudio() ; RETURN0 ; }

/* ------------------------------------------------------------------------ */
/* Public routines */

int board_setmux(void) {
  board_setaudio() ; board_setbyte() ; RETURN0 ; }

int board_setvol(int v) {
  board_setaudio() ; RETURN0 ; }

int board_setmute(void) {
  board_setaudio() ; board_setbyte() ; RETURN0 ; }

int board_setstereo(void) {
  board_setbyte() ; RETURN0 ; }

int board_setsap(void) {
  board_setbyte() ; RETURN0 ; }

int board_hasvol(void) {
  return has_tda8425 || has_tda9850 || has_msp3410 ; }

int board_hasstereo(void) {
  return has_tda8425 || has_tda9850 || has_msp3410 ; }

int board_hassap(void) {
  return has_tda9850 || has_msp3410 ; }

int board_isstereo(void){
  if (gatos.board.addr == 0xFF) return 0 ;
  return !(i2c_read(gatos.board.addr)&0x20) ; }

/* ------------------------------------------------------------------------ */
/* Private routines */

static u8 board_identify(u8 *addr, u8 *info) {

  u8 *ptr=gatos.aticard[gatos.cardidx].biosinfo ;
  
  /* Stand-alone cards */

  *addr = 0x70 ; *info = 0xFF ; i2c_write(*addr,info,1) ;
  if ((*info=i2c_read(*addr)) != 0xFF && !gatos.nosa) return CARD_STAND_ALONE ;

  *addr = 0x40 ; *info = 0xFF ; i2c_write(*addr,info,1) ;
  if ((*info=i2c_read(*addr)) != 0xFF && !gatos.nosa) return CARD_STAND_ALONE ;

  *addr = 0x78 ; *info = 0xFF ; i2c_write(*addr,info,1) ;
  if ((*info=i2c_read(*addr)) != 0xFF && !gatos.nosa) return CARD_NEC ;

  /* All-In-Wonder, All-In-Wonder Pro or All-In-Wonder 128 */
  *addr = 0x76 ; *info = 0x7F ; i2c_write(*addr,info,1) ;
  *info = i2c_read(*addr) ;
  if ( (*ptr != 0xFF && *ptr&0x0F) || *info != 0xFF || gatos.i2c_mode==5 ) {
    if (*info == 0xFF) *addr = 0xFF ;
    if (*ptr != 0xFF && *ptr&0x0F && *(ptr+4) == 1) return CARD_INTEL_BOARD ;
    switch (gatos.ati.deviceid) {
      case 0x5245: case 0x5246: case 0x524B: case 0x524C:
        return CARD_ALL_IN_WONDER_128 ;
      case 0x4742: case 0x4744: case 0x4749: case 0x4750: case 0x4751:
        return CARD_ALL_IN_WONDER_PRO ;
      default:
        if (*addr == 0xFF) {
          if (!gatos.debug&GATOSQUIET) fprintf(stderr,
            "GATOS: Warning: All-In-Wonder Board Control not detected\n") ;
          *addr = 0x76 ; *info = i2c_read(*addr) ; }
        return CARD_ALL_IN_WONDER ; } }

  /* External POD cards */
  *addr = 0x72 ; if ( (*info=i2c_read(*addr)) != 0xFF ) return CARD_EXTERNAL_POD ;
  *addr = 0x42 ; if ( (*info=i2c_read(*addr)) != 0xFF ) return CARD_EXTERNAL_POD ;

  /* Integrated card ? */
  *addr = 0xFF ; *info = 0xFF ; return CARD_INTEGRATED ; }

/* Board info/control byte:
 * 0x80: bt829: 0:enable 1:disable
 * 0x40: sound: 0:tuner 1:video
 * 0x20: 0:stereo 1:mono (R/O?)
 * 0x10: 0:mute 1:unmute
 * 0x0F: Tuner type (R/O)
 */

static int board_setbyte(void) {

  u8 info ; u32 tmp ;

  /*if (gatos.board.addr == 0xFF) {*/
    tmp = EXT_DAC_REGS & 0xFFFFFFFC ;
    if (!gatos.audio) tmp |= 0x2 ;
    else if (gatos.mux==2) tmp |= 0x1 ;
    EXT_DAC_REGS = tmp ; /*}*/

  if (gatos.board.addr != 0xFF) {
    info = 0x2F | ((gatos.mux!=2)<<6) | (gatos.audio<<4) ;
    i2c_write(gatos.board.addr,&info,1) ; }

  RETURN0 ; }

static int board_setaudio(void) {

  u8 tmp, data[5] ;

  if (has_tda8425) {
    tmp = (gatos.mute) ? 0xC0 : 0xFC ;
    TDA8425(0x00,tmp) ; TDA8425(0x01,tmp) ;
    TDA8425(0x02,0xF6) ; TDA8425(0x03,0xF6) ;
    tmp = 0xC0 | ((!gatos.audio)<<5) | (gatos.mux==2) |
          ((gatos.stereo) ? 0x1E : (gatos.sap) ? 0x02 : 0x04) ;
    TDA8425(0x08,tmp) ; }

  if (has_tda9850 && gatos.eeprom.addr != 0xFF) {
    data[0] = 0x00 ; i2c_write(gatos.eeprom.addr,data,1) ;
    i2c_readn(gatos.eeprom.addr,data,5) ;
    if (!memcmp(data,"ATI",3)) {
      TDA9850(0x08,data[3]&0x1F) ; TDA9850(0x08,(data[4]&0x1F)|0x80) ;
      TDA9850(0x04,0x07) ; TDA9850(0x05,0x07) ; TDA9850(0x06,0x40) ;
      TDA9850(0x07,0x07) ; TDA9850(0x08,0x10) ; TDA9850(0x09,0x10) ;
      TDA9850(0x0A,0x03) ; }
    if (gatos.mux == 2) {
      TDA9850(0x04,0x0F) ; TDA9850(0x05,0x0F) ;
      TDA9850(0x07,0x07) ; TDA9850(0x08,0x00) ;
      TDA9850(0x09,0x00) ; TDA9850(0x0A,0x03) ; }
    TDA9850(0x06,(gatos.stereo<<6)|(gatos.sap<<7)) ; }

  if (has_tda9851) {
    data[0] = (BTREAD(P_IO)&0xFC) | (gatos.mux==2) ;
    BTWRITE(P_IO,data[0]) ;
    data[0] = (i2c_read(gatos.tda9851.addr) & 0x01) | 0x04 ;
    i2c_write(gatos.tda9851.addr,data,1) ; }

  if (has_msp3410) { /* ??? */ }

  RETURN0 ; }

#if 0 /* NIY */
static int MSP3410(u8 sub, u16 reg, u16 val) {
  u8 data[5] = {sub,reg>>8,reg&0xFF,val>>8,val&0xFF} ;
  i2c_write(gatos.msp3410.addr,data,5) ; RETURN0 ; }
#endif

static u16 MSP3410get(u8 sub, u16 reg) {
  int val=0 ;
  /* TODO: I2C 16bit read routine (routines below not exported)
  i2c_start() ; i2c_sendbyte(gatos.msp3410.addr) ;
  i2c_sendbyte(sub) ; i2c_sendbyte(reg>>8) ; i2c_sendbyte(reg&0xFF) ;
  i2c_start() ; i2c_sendbyte(gatos.msp3410.addr) ;
  val = i2c_readbyte(0) << 16 ; val += i2c_readbyte(1) ; i2c_stop() ;
  */
  return val ; }

/* ------------------------------------------------------------------------ */
/* Debug and report routines */

#define DUMPREG \
  if (gatos.board.addr!=0xFF) fprintf(stderr,"%s: %-4s = 0x%02X\n", \
    i2c_ident(gatos.board.addr),"INFO",i2c_read(gatos.board.addr))
#define DUMP8425(REG) \
  fprintf(stderr,"%s: 0x%02X = 0x%02X\n", \
    i2c_ident(gatos.tda8425.addr),REG,tda8425reg[REG])
#define DUMP9850(REG) \
  fprintf(stderr,"%s: 0x%02X = 0x%02X\n", \
    i2c_ident(gatos.tda9850.addr),REG,i2c_readreg8(gatos.tda9850.addr,REG))
#define DUMP9851 \
  fprintf(stderr,"%s: %-4s = 0x%02X\n", \
    i2c_ident(gatos.tda9851.addr),"STAT",i2c_read(gatos.tda9851.addr))
#define DUMP3410(REG) \
  fprintf(stderr,"%s: 0x%04X = 0x%04X\n", \
    i2c_ident(gatos.tda9850.addr),REG,MSP3410get(0x13,REG))

#include <sys/types.h>
#include <signal.h>
#include <unistd.h>

void board_dumpregs(void) {
  DUMPREG ;
  if (has_tda8425) {	/* TDA8425 registers cannot be   */
    DUMP8425(0x00) ;	/* read, so we have to cheat :-( */
    DUMP8425(0x01) ;
    DUMP8425(0x02) ;
    DUMP8425(0x03) ;
    DUMP8425(0x08) ; }
  if (has_tda9850) {
    DUMP9850(0x04) ;
    DUMP9850(0x05) ;
    DUMP9850(0x06) ;
    DUMP9850(0x07) ;
    DUMP9850(0x08) ;
    DUMP9850(0x09) ;
    DUMP9850(0x0A) ; }
  if (has_tda9851) DUMP9851 ;
  if (has_msp3410) {
    DUMP3410(0x0000) ;
    DUMP3410(0x0006) ;
    DUMP3410(0x0007) ; } }
