/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev and Christian Lupien.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#define GATOS_I2C_C 1

#include "gatos.h"
#include "i2c.h"
#include "ati.h"
#include "i2csw.h"
#include "i2cmpp.h"

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <sys/time.h>

/* Private global vars */
static int active=0 ;
static char *i2cident[128], *i2cname[128] ;
static char* driver[MAXI2CDRIVER+1] = {
  NULL, "Type A I2C Driver", "Type B I2C Driver (GP_IO)",
  "Type C I2C Driver (Rage PRO)", "Type LG I2C Driver (GP_IO)",
  "Type TB I2C Driver (MPP)" } ;

/* ------------------------------------------------------------------------ */
/* Initialization routine */

int i2c_init(void) {

  int i, found=0, trymodes=(1<<(MAXI2CDRIVER+1))-2 ;

#ifndef USE_DELAY_S
  if (gatos.bogomips) i2c_ddelay = gatos.bogomips/8 ; else RETURN(EINVAL) ;
#endif

  /* If i2c_mode not set in gatos.conf, choose
   * default based on ATI chip ID and BIOS data. */
  if (!gatos.i2c_mode) {
    if ( ati.vt_a3n1 || ati.vt_a3s1 || ati.gt_a2s1
      || ati.vt_a4n1 || ati.vt_a4s1 )		gatos.i2c_mode = 1 ;
    else if ( ati.lg_a1s1 || ati.lt )		gatos.i2c_mode = 4 ;
    else if ( ati.doesvtb || ati.gt_c1u1 ||
              ati.gt_c1u2 || ati.gt2c )		gatos.i2c_mode = 2 ;
    else if (ati.lt_c)				gatos.i2c_mode = 3 ;
    else if ( ati.gt_c2u1 || ati.gt_c2u2 )
            if ( ati.gp_c2u1 || ati.gq_c2u1 ||
                 ati.gp_c2u2 || ati.gq_c2u2 ) {	gatos.i2c_mode = 2 ;
						gatos.bugfix_mppread = 0x81 ; }
            else				gatos.i2c_mode = 3 ;
         else if ( ati.gp_c2u1 || ati.gq_c2u1 ||
                   ati.gp_c2u2 || ati.gq_c2u2 )	gatos.i2c_mode = 2 ;
              else				gatos.i2c_mode = 3 ; }
  if (gatos.i2c_mode == 2 &&
      gatos.aticard[gatos.cardidx].biosdata48 != 0xFFFF)
    gatos.i2c_mode = 5 ;

  /* Try all known interfaces */
  while (trymodes && !found ) {
    while ((gatos.i2c_mode<MAXI2CDRIVER) && !(trymodes&(1<<gatos.i2c_mode)))
      gatos.i2c_mode++ ;
    trymodes &= ~(1<<gatos.i2c_mode) ;
    found = (gatos.i2c_mode == 5) ? !i2c_init_mpp() : !i2c_init_sw() ;
    if (!found) {
      if (VERBOSE) fprintf(stderr,"I2C mode %d failed\n",gatos.i2c_mode) ;
      gatos.i2c_mode = 0 ; } }

  /* All known I2C interfaces failed initialization :-( */
  if (!found) RETURN(ENODEV) ;

  /* Found an I2C driver that works :-) */
  if (VERBOSE) printf("Using %s\n",driver[gatos.i2c_mode]) ;
  for ( i=0 ; i<128 ; i++ ) { i2cident[i] = NULL ; i2cname[i] = NULL ; }
  RETURN0 ; }

static char str[12] ;

char* i2c_ident(u8 addr) {
  if (i2cident[addr>>1]) return i2cident[addr>>1] ;
  snprintf(str,sizeof(str),"UNKN @ 0x%02X",addr) ; return str ; }

char* i2c_name(u8 addr) {
  if (i2cname[addr>>1]) return i2cname[addr>>1] ;
  snprintf(str,sizeof(str),"UNKN @ 0x%02X",addr) ; return str ; }

/* ------------------------------------------------------------------------ */
/* High-level (API) routines */

u8 i2c_read(u8 addr) {
  u8 data ; i2c_start() ; i2c_sendbyte(addr|1) ;
  data = i2c_readbyte(1) ; i2c_stop() ; return data ; }

int i2c_readn(u8 addr, u8 *data, int count) {
  i2c_start() ; i2c_sendbyte(addr|1) ;
  while (count > 0) { --count ; *data = i2c_readbyte((count!=0)) ; ++data ; }
  i2c_stop() ; return 0 ; }

int i2c_write(u8 addr, u8 *data, int count) {
  int nack=-1 ;
  i2c_start() ; i2c_sendbyte(addr) ;
  while (count > 0) { nack = i2c_sendbyte(*data) ; ++data ; --count ; }
  i2c_stop() ; return nack ? -1 : 0 ; }

u8 i2c_readreg8(u8 addr, u8 reg) {
  u8 data ;
  i2c_start() ; i2c_sendbyte(addr) ; i2c_sendbyte(reg) ;
  i2c_start() ; i2c_sendbyte(addr|1) ; data = i2c_readbyte(1) ;
  i2c_stop() ; return data ; }

int i2c_writereg8(u8 addr, u8 reg, u8 value) {
  i2c_start() ; i2c_sendbyte(addr) ; i2c_sendbyte(reg) ;
  i2c_sendbyte(value) ; i2c_stop() ; return 0 ; }

int i2c_device(u8 addr) {
  int nack ; i2c_start() ; nack = i2c_sendbyte(addr) ;
  i2c_stop() ; return !nack ; }

/* ------------------------------------------------------------------------ */
/* Low-level (driver) routines */

#ifndef USE_DELAY_S
/* delay n microseconds */
void i2c_delay(u32 n) {
  while (n > 0) { volatile int x = i2c_ddelay ; while (x > 0) --x ; --n ; } }
#endif

static void i2c_start(void) {
  if (I2CTRAFFIC) fprintf(stderr,(active) ? " <" : "I2C: <") ; active = 1 ;
  if (gatos.i2c_mode == 5) i2c_start_mpp() ; else i2c_start_sw() ; }

static void i2c_stop(void) {
  if (gatos.i2c_mode == 5) i2c_stop_mpp() ; else i2c_stop_sw() ;
  if (I2CTRAFFIC) fprintf(stderr," >\n") ; active = 0 ; }

static u8 i2c_readbyte(int last) {
  u8 data=0xFF ;
  data = (gatos.i2c_mode==5) ? i2c_readbyte_mpp(last) : i2c_readbyte_sw(last) ;
  if (I2CTRAFFIC) fprintf(stderr," =%02X%c", data, last?'-':'+') ;
  return data ; }

static int i2c_sendbyte(u8 data) {
  int nack=-1 ;
  nack = (gatos.i2c_mode==5) ? i2c_sendbyte_mpp(data) : i2c_sendbyte_sw(data) ;
  if (I2CTRAFFIC) fprintf(stderr," %02X%c", data, nack?'-':'+') ;
  return nack ; }

/* ------------------------------------------------------------------------ */

static volatile int N=0, N1=0, N2=0, loop=0 ;

#define SECS	2

static void i2c_alarm(int sig) {
  if (loop) ((N1) ? N2 : N1) = N ; if (N2) loop = 0 ; }

void i2c_info(void) {
  int i, d=gatos.debug ; struct sigaction act, oldact ; struct itimerval it ;
  gatos.debug &= ~4 ;
  for ( i=0 ; i<256 ; i+=2 )
    if (i2c_device(i))
      if (!i2cident[i>>1]) i2c_register(i,"UNKN","Unclaimed Device") ;
      else fprintf(stderr,"I2C unit @ 0x%02X: %s\n",i,i2cname[i>>1]) ;
    else if (i2cident[i>>1]) fprintf(stderr,
      "I2C unit @ 0x%02X: %s DISAPPEARED\n",i,i2cname[i>>1]) ;
  fprintf(stderr,"I2C clock frequency is ... ") ;
  it.it_value.tv_sec  = 0 ;     it.it_interval.tv_sec  = SECS ;
  it.it_value.tv_usec = 10000 ; it.it_interval.tv_usec = 0 ;
  act.sa_flags = 0 ; act.sa_restorer = NULL ;
  sigemptyset(&act.sa_mask) ; act.sa_handler = &i2c_alarm ;
  sigaction(SIGVTALRM,&act,&oldact) ; setitimer(ITIMER_VIRTUAL,&it,NULL) ;
  N1 = N2 = 0 ; loop = 1 ; i2c_start() ;
  for ( N=0 ; loop ; N++ ) i2c_sendbyte(0xFF) ;
  it.it_value.tv_sec  = it.it_interval.tv_sec  = 0 ;
  it.it_value.tv_usec = it.it_interval.tv_usec = 0 ;
  setitimer(ITIMER_VIRTUAL,&it,NULL) ; sigaction(SIGVTALRM,&oldact,NULL) ;
  i2c_stop() ; N = N2-N1 ;
  fprintf(stderr,"%.1f kHz\n", 9.0*N/(1000.0*SECS)) ; gatos.debug = d ; }

void i2c_register(u8 addr, char *ident, char *name) {
  if (!name) name = ident ;
  i2cident[addr>>1] = (char*) malloc(8+strlen(ident)) ;
  i2cname[addr>>1] =  (char*) malloc(1+strlen(name)) ;
  if (i2cident[addr>>1])
    snprintf(i2cident[addr>>1],8+strlen(ident),"%s @ 0x%02X",ident,addr) ;
  if (i2cname[addr>>1]) strcpy(i2cname[addr>>1],name) ;
  if (VERBOSE) printf("I2C unit @ 0x%02X: %s\n",addr,name) ; }
