//------------------------------------------------------------------------------
// medialibrary.js - Content Manager user interface helper functions.
//
// INTEL CONFIDENTIAL
//	Copyright 2004-2006 Intel Corporation All Rights Reserved.  The source code
//	contained or described herein and all documents related to the source code
//	(Material) are owned by Intel Corportion or its suppliers or licensors.
//	Title to the Material remains with Intel Corporation or its suppliers and
//	licensors. The Material may contain trade secrets and 	proprietary and
//	confidential information of Intel Corporation and its 	suppliers and
//	licensors, and is protected by worldwide copyright and trade secret laws
//	and treaty provisions. No part of the Material may be used, copied,
//	reproduced, modified, published, uploaded, posted, transmitted, distributed,
//	or disclosed in any way without Intels prior express written permission. 
//	
//	No license under any patent, copyright, trade secret or other intellectual
//	property right is granted to or conferred upon you by disclosure or delivery
//	of the Materials, either expressly, by implication, inducement, estoppel or
//	otherwise. Any license under such intellectual property rights must be express
//	and approved by Intel in writing.
//
//	Unless otherwise agreed by Intel in writing, you may not remove or alter this
//	notice or any other notice embedded in Materials by Intel or Intels suppliers
//	or licensors in any way.
//
//------------------------------------------------------------------------------



//------------------------------------------------------------------------------
// refreshInterval - The interval at which network/file system UI is refreshed.
//------------------------------------------------------------------------------

var refreshInterval = 3000;
var refreshIntervalShort = 1500;


//------------------------------------------------------------------------------
// Filesystem access constants
//------------------------------------------------------------------------------
var FS_UNINITIALIZED			= 0x00000000;
var FS_FAILED					= 0xFFFFFFFF;
var FS_READY					= 0x00000001;
var FS_INITIALIZING				= 0x00000002;
var FS_ACCESSDENIED				= 0xF0000001;



//------------------------------------------------------------------------------
// str2html - Remove < > & from strings to make appropriate as HTML text.
//------------------------------------------------------------------------------

function str2html(str)
{
	str = str.replace(/&/g, "&amp;");
	str = str.replace(/</g, "&lt;");
	str = str.replace(/>/g, "&gt;");
	return str;
}



//------------------------------------------------------------------------------
// pp2html - Make paragraph appropriate to HTML, including wrap+truncate.
//------------------------------------------------------------------------------

function pp2html(text, length, wrap)
{
	if (text.length > length)
		text = text.substring(0, length-1) + "...";

	if (wrap)
	{
		var temp = text;
		text = "";
		while (temp.length)
		{
			text = text + str2html(temp.substring(0, wrap));
			temp = temp.substring(wrap, temp.length);
			if (temp.length)
				text += "<br>";
		}
	}
	else
		text = str2html(text);

	return text;
}



//============================= Base List Objects ==============================



//------------------------------------------------------------------------------
// listLookup - Keep a lookup array of lists to allow these to be indexed by
// number, for use in events and timeout handlers.
//------------------------------------------------------------------------------

var listLookup = new Array();



//------------------------------------------------------------------------------
// BaseItem - Base class for a series of items in lists or grids/scrolls.
//------------------------------------------------------------------------------

function BaseItem()
{
	var item = new Object();
	item.list = null;
	item.waiting = false;
	item.update = 0;

	// Redraw this item
	item.Redraw = function()
			{ this.list.UpdateStart(); this.list.UpdateFinish(); }

	// UpdateStart/Finish used to manage nested updates	
	item.UpdateStart = function()
			{ this.update++; }
	item.UpdateFinish = function()
			{ if (!--this.update) this.Redraw(); }

	return item;
}


//------------------------------------------------------------------------------
// TextItem - Plain text item for lists or grids/scrolls.
//------------------------------------------------------------------------------

TextItem.prototype = new BaseItem();
TextItem.prototype.constructor = TextItem;
TextItem.prototype.Redraw = TextItem__Redraw;

function TextItem(str) {
	this.str = str;
}

function TextItem__Redraw() {
	return this.str;
}



//------------------------------------------------------------------------------
// BaseList - Base class for a series of lists or grids/scrolls.
//------------------------------------------------------------------------------

function BaseList()
{
	var list = new BaseItem();
	list.lookupIndex = listLookup.length;
	listLookup[list.lookupIndex] = list;

	list.items = new Array();
	list.itemsWaiting = new Array();
	list.pending = false;
	

	// Set the list display grid/scroll and optional info widget
	list.SetDisplay = function(grid, info)
			{ BaseListSetDisplay(this, grid, info); };
		
	// Redraw this list		
	list.Redraw = function()
	{
		if (this.grid)
			GridSetViewport(this.grid, this.grid.viewFirst, this.items.length + (this.pending?1:0),
					this.grid.viewCollapse, this.grid.viewEscape);
	}

	// Redraw the "pending" item in this list
	list.RedrawPending = function()
	{
		return "";
	}

	// Add an item to the list
	list.AddItem = function(item)
	{
		this.UpdateStart();
		this.items[this.items.length] = item;
		item.list = this;
		this.UpdateFinish();
	}

	// Manage the display, focus and actions on an item
	list.ItemFocus = function(index, focus)
			{ return focus ? ((index+1)+" "+languageGridOf+" "+this.items.length) : ""; }
	list.ItemAction = function(index)
			{	}
	
	// Waiting area methods
	list.AddItemWait = BaseListAddItemWait;
	list.FinishItemWait = BaseListFinishItemWait;
	list.RemoveItemWait = BaseListRemoveItemWait;
	
	// List event handlers
	list.Action = BaseListAction;
	list.Focus = BaseListFocus;
	list.Update = BaseListUpdate;

	return list;
}



//------------------------------------------------------------------------------
// BaseList::Focus - Grid/scroll item change of focus.
//------------------------------------------------------------------------------

function BaseListFocus(cell, focus, list)
{
	var stats = list.grid.viewPane.stats;
	var index = list.grid.viewFirst+(cell.navRow*list.grid.navCols)+cell.navCol;
	if (stats)
		stats.label.innerHTML = list.ItemFocus(index, focus);
}



//------------------------------------------------------------------------------
// BaseList::Action - Grid/scroll item action (click or press).
//------------------------------------------------------------------------------

function BaseListAction(cell, list)
{
	var index = list.grid.viewFirst+(cell.navRow*list.grid.navCols)+cell.navCol;
	list.ItemAction(index);	
}



//------------------------------------------------------------------------------
// BaseList::Update - Grid/scroll contents changed or scrolled, so redraw.
//------------------------------------------------------------------------------

function BaseListUpdate(grid, start, count)
{
	var list = grid.list;

	var row = 0, col = 0;
	for (var i = start; i < start+count; i++)
	{
		var index = grid.viewFirst + i;
		var inner = (index>=list.items.length)?list.RedrawPending():list.items[index].Redraw();
		GridSetCell(grid, row, col, inner, BaseListAction, null, list);
		
		if (++col >= grid.navCols)
		{
			col = 0;
			row++;
		}
	}
}



//------------------------------------------------------------------------------
// BaseList::SetDisplay - Set the grid/scroll display and optional info field.
// Assign focus/action to each cell, and insure updates happen correctly.
//------------------------------------------------------------------------------

function BaseListSetDisplay(list, grid, info)
{
	list.UpdateStart();
	list.grid = grid;
	list.info = info;
	grid.list = list;
	grid.cbUpdate = list.Update;
	
	for (var row = 0; row < grid.navRows; row++)
	{
		for (var col = 0; col < grid.navCols; col++)
		{
			grid.navGrid[row][col].cbAction = list.Action;
			grid.navGrid[row][col].cbFocus = list.Focus;
			grid.navGrid[row][col].cbData = list;
		}
	}
	list.UpdateFinish();
}



//------------------------------------------------------------------------------
// BaseList::AddItemWait - Add the given item to this list's waiting area.
//------------------------------------------------------------------------------

function BaseListAddItemWait(item)
{
	this.UpdateStart();
	if (item)
	{
		this.itemsWaiting[this.itemsWaiting.length] = item;
		item.list = this;
		item.waiting = true;
	}
	this.UpdateFinish();
} // BaseListAddItemWait()



//------------------------------------------------------------------------------
// BaseList::FinishItemWait - Given item is finished waiting; move it to the
// active list area.
//------------------------------------------------------------------------------

function BaseListFinishItemWait(item)
{
	this.UpdateStart();
	if (item)
		if (item.waiting)
		{
			this.RemoveItemWait(item);
			item.waiting = false;
			this.AddItem(item);
		}
	this.UpdateFinish();
} // BaseListFinishItemWait()



//------------------------------------------------------------------------------
// BaseList::RemoveItemWait - Remove the given item from this list's waiting 
// area.
//------------------------------------------------------------------------------

function BaseListRemoveItemWait(item)
{
	this.UpdateStart();
	if (item)
		if (item.waiting)
		{
			// Find item
			for (i = 0; i < this.itemsWaiting.length; i++)
			{
				if (this.itemsWaiting[i] == item)
				{
					item.waiting = false;
					this.itemsWaiting.splice(i, 1);
					break;
				}
			}
			
		}
	this.UpdateFinish();
} // BaseListRemoveItemWait()




//=============================== Monitor List =================================

var MONITOR_DEFAULT = 0;
var MONITOR_FOLDER = 1;
var MONITOR_DRIVE = 2;
var MONITOR_CDS = 3;
var MONITOR_SPECIAL = -1;

//------------------------------------------------------------------------------
// MonitorItem - Represents a folder or the CDS aggregation state for a
// specific servers, or for all servers.
//------------------------------------------------------------------------------

function MonitorItem(path, label, computer, checked, type)
{
	var monitor = new BaseItem();
	monitor.label = label;
	monitor.path = path;
	monitor.breadcrumb = path;
	monitor.key = path.toUpperCase();
	monitor.checked = checked
		? CHECKED_ON
		: CHECKED_OFF;
	monitor.type = type;
	monitor.computer = computer;
	
	// Special initialization for each monitor type.
	if (type == MONITOR_FOLDER) {
		if (path.length == 2 && path.indexOf(':') == 1)
			monitor.type = MONITOR_DRIVE;
	}
	else if (type == MONITOR_CDS) {
		if (updateInfo.addAllServers.post)
			monitor.checked = CHECKED_ANCESTOR;
		monitor.breadcrumb = label;
	}
	if (!monitor.computer)
		monitor.computer = '';
	if (monitor.computer == 'localhost') {
		monitor.computer = '';
	}
	

	// Redraw the item with its appropriate icon
	monitor.Redraw = function()
	{

		var zoom = document.body.style.zoom > 0 ? 
			document.body.style.zoom : 1;
		if (zoom != truncateLastZoom) {
			// zoom change, re-init truncation
			truncateLastZoom = zoom;
			TruncateInitialize();
		}

		// Generate truncated label text.
		var rect = this.list.grid.elem.getBoundingClientRect();
		var rowWidth = (rect.right - rect.left);
		rowWidth -= 120 * zoom;
		var labelWidth = rowWidth;
		if (this.computer && this.computer != "")
			labelWidth = rowWidth * .70;
		var content = Truncate(this.label, labelWidth);
		
		// Generate truncated computer name text.
		computerWidth = rowWidth * .30;
		var computer;
		if (this.computer && this.computer != "")
			computer = Truncate(this.computer, computerWidth);
		else
			computer = "";
		
		// Generate checkbox class.
		var cbox = 'class="TreeCheckOffSelect"';
		switch (this.checked) {
			case CHECKED_ON:
				cbox = 'class="TreeCheckOnSelect"';
				break;
			case CHECKED_ANCESTOR:
				cbox = 'class="TreeCheckOnInactive"';
				break;
			case CHECKED_OFF:
			default:
				break;
		}

		// Generate icon tag.
		var icon = '<span style="margin-left:10px"></span>';
		switch (this.type) {
			case MONITOR_FOLDER:
				icon = '<span class="folderIconClosed" style="margin-left:10px"></span>';
				break;
			case MONITOR_DRIVE:
				icon = '<span class="driveIcon" style="margin-left:10px"></span>';
				break;
			case MONITOR_CDS:
				// Extra indentation to show child relationship to select-all checkbox.
				icon = '<span style="margin-left:50px"></span>';
				break;
			case MONITOR_SPECIAL:
			default:
				break;
		}

		var inner = '<table border="0" width="100%" cellpadding="0" cellspacing="0">';
		inner += '<tr valign="center"><td>';
		inner += '<span ' + cbox + '></span>';
		inner += icon;
		inner += '<span style="padding-left:10px;vertical-align:middle">';
		inner += content + '</span>';
		inner += '</td>';
		inner += '<td align="right">';
		inner += '<span class="TreeCheckOnInactive" style="visibility:hidden"></span>';
		inner += '<span style="vertical-align:middle">';
		inner += computer + '</span>';
		inner += '</td></tr></table>';
		return inner;
	}
	
	return monitor;
}



//------------------------------------------------------------------------------
// MonitorList - LTD0
//------------------------------------------------------------------------------

function MonitorList(grid, info, actionButton)
{
	var monitors = new BaseList();
	monitors.actionButton = actionButton;

	// Focus updates the "breadcrumb" trail as well as item count
	monitors.ItemFocus = function(index, focus)
	{
		var rect = this.grid.elem.getBoundingClientRect();
		var width =  rect.right - rect.left - 200;
		if (width != truncateLastWidth) {
			// width change, re-init truncation
			truncateLastWidth = width;
			TruncateInitialize();
		}
		this.info.elem.innerHTML = focus
			? TruncatePath(this.items[index].breadcrumb, width)
			: "";
		return focus ? ((index+1)+" "+languageGridOf+" "+this.items.length) : "";
	}

	monitors.ItemAction = function(index)
	{
		var item = this.items[index];
		if (item.checked == CHECKED_ANCESTOR)
			// Take no action because this checkbox is disabled.
			return;
		
		this.UpdateStart();

		item.checked = (item.checked == CHECKED_OFF)
			? CHECKED_ON
			: CHECKED_OFF;

		var monitorState = null;
		if (item.type == MONITOR_FOLDER || item.type == MONITOR_DRIVE)
			monitorState = updateInfo.folderState[item.key];
		else if (item.type == MONITOR_CDS)
			monitorState = updateInfo.serverState[item.key];

		if (monitorState) {
			// Should always be true for an actual item
			monitorState.post = item.checked;
		}

		else if (item.type == MONITOR_SPECIAL) {
			// This is our special checkbox to check on/off all CDS.
			updateInfo.addAllServers.post = item.checked;
			for (var i = index + 1; i < this.items.length; i++) {
				var curItem = this.items[i];
				curItem.checked = item.checked
					? CHECKED_ANCESTOR
					: (updateInfo.serverState[curItem.key].post
						? CHECKED_ON
						: CHECKED_OFF);
			}
		}

		this.CheckSelection();
		this.UpdateFinish();
	}
	
	monitors.RedrawPending = function()
	{
		return "FOO"; // Needed?
	}
	
	monitors.CheckSelection = function() {
		var changed = false;
		// Find at least one change in folder selection for
		// monitoring.
		for (var key in updateInfo.folderState) {
			var monitorState = updateInfo.folderState[key];
			if (monitorState.pre != monitorState.post) {
				changed = true;
				break;
			}
		}
		if (!changed) {
			// Check servers now
			for (var key in updateInfo.serverState) {
				var monitorState = updateInfo.serverState[key];
				if (monitorState.pre != monitorState.post) {
					changed = true;
					break;
				}
			}
			if (updateInfo.addAllServers.pre != updateInfo.addAllServers.post) {
				changed = true;
			}
		}			
		if (changed) {
			// If we have a change, enable the action button.
			WidgetSetNavigation(this.actionButton, 'leaf');
		}
		else {
			// No change, disable action button.
			WidgetSetNavigation(this.actionButton, 'inactive');
		}
	}

	// Put preexisting monitor information into our media update structures
	// if we haven't yet.
	if (!updateInfo) {
		updateInfo = new UpdateInfo();
	}
	
	try {
		if (!updateInfo.initialized) {
			updateInfo.Reinitialize();
			updateInfo.FetchMonitors();
			updateInfo.EstablishDefaultFolders(mediaFlow.toLowerCase() == "setup");
			updateInfo.EstablishExcludedFolders();
			updateInfo.initialized = true;
		}
	}
	catch (ex) {
		throw "EXCEPTION_LOSTCOMMS";
	}

	monitors.UpdateStart();
		
	// Display default paths first.	
	var folderDefault = updateInfo.folderDefault;
	for (var i = 0; i < folderDefault.length; i++) {
		// Put the path in displayed MonitorList.
		monitors.AddItem(new MonitorItem(folderDefault[i].path,		// breadcrumb/path
										 folderDefault[i].label,	// label
										 "",						// computer name
										 folderDefault[i].post,		// checked
										 MONITOR_FOLDER));			// type
	}
	
	// Iterate through folder select table to separate local and remote folders.
	var folderLocal = new Array;
	var folderRemote = new Array;
	for (var key in updateInfo.folderState) {
		var monitorState = updateInfo.folderState[key];
		if (updateInfo.IsDefaultFolder(monitorState.path))
			// Don't show folder if we've already displayed it.
			continue;
		if (monitorState.path.indexOf(":") == 1) {
			// Starts with drive: reference, so assume local folder
			folderLocal.push(monitorState);
		}
		else {
			// Not a local folder
			folderRemote.push(monitorState);
		}
	}	
	
	// Display local folders.
	folderLocal.sort();
	for (var i = 0; i < folderLocal.length; i++) {
		var monitorState = folderLocal[i];
		if (!monitorState.pre && !monitorState.post)
			// Don't show the folder if it wasn't being monitored before
			// and also isn't selected now.
			continue;
		monitors.AddItem(new MonitorItem(monitorState.path,		// breadcrumb/path
										 monitorState.label,	// label
										 "",					// computer name
										 monitorState.post,		// checked
										 MONITOR_FOLDER));		// type		
	}
	
	// Display remote folders.
	folderRemote.sort();
	for (var i = 0; i < folderRemote.length; i++) {
		var monitorState = folderRemote[i];
		if (!monitorState.pre && !monitorState.post)
			// Don't show the folder if it wasn't being monitored before
			// and also isn't selected now.
			continue;
		var devName = Path2DeviceName(monitorState.path);
		monitors.AddItem(new MonitorItem(monitorState.path,		// breadcrumb/path
										 monitorState.label,	// label
										 devName,				// computer name
										 monitorState.post,		// checked
										 MONITOR_FOLDER));		// type		
	}

	// Select all servers checkbox item.
	monitors.AddItem(new MonitorItem("",							// breadcrumb/path
		LayoutTranslate(languageXML, "setup.custom.servers.all"),	// label
		"",															// computer name
		updateInfo.addAllServers.post,								// checked
		MONITOR_SPECIAL));											// type
	
	// Display CDS in sorted order.
	var servers = new Array;
	// Move server hash data to sortable array.
	for (var key in updateInfo.serverState) {
		servers.push(updateInfo.serverState[key]);
	}	
	servers.sort();
	for (var i = 0; i < servers.length; i++) {
		var monitorState = servers[i];
		monitors.AddItem(new MonitorItem(monitorState.path,		// path
										 monitorState.label,	// label
										 "",					// computer name
										 monitorState.post,		// checked
										 MONITOR_CDS));			// type
	}

	if (grid)
		monitors.SetDisplay(grid, info);

	monitors.UpdateFinish();

	return monitors;
	
}



//================================== Device List ===============================



//------------------------------------------------------------------------------
// DeviceItem - Create a new device item based on the XML node description.
//------------------------------------------------------------------------------

function DeviceItem(node)
{
	var device = new BaseItem();

	// Properties
	device.name = "";
	device.uid = "";
	device.friendly = "";
	device.description = "";
	device.type = "desktop";	// Must correlate to an icon type in stylesheets.dll
	device.localhost = (node.tagName == "localhost");
	device.status = device.localhost ? "available" : "";
	device.shares = -1;
	device.fs = null;
	device.root = null;
	device.searching = true;

	// Save localhost device as a global for special use.
	if (device.localhost)
		deviceLocalhost = device;	

	// Methods
	device.Redraw = DeviceItemRedraw;
	device.Refresh = DeviceItemRefresh;
	device.RefreshShares = DeviceItemRefreshShares;
	device.BrowseFiles = DeviceItemBrowseFiles;
	
	// Initiate first refresh with given XML node
	device.Refresh(node);	
	
	return device;
}


//------------------------------------------------------------------------------
// DeviceItemRedraw - Redraw the device name, icon and availability
//------------------------------------------------------------------------------

function DeviceItemRedraw()
{
	var overlay = "";
	var type = this.type;

	if (this.status == "unavailable") {
		type += "Absent";
	}
	else if (this.localhost) {
		overlay = "folderOverlay_closed";
	}
	// If it's available, pick an overlay based on what shares are available
	else if (this.shares > 0) {
		overlay = "folderOverlay_closed"; 
	}
	else if (this.searching) {
		overlay = "searchOverlay"; 
	}
	else if (this.status == "connectedNotVerified") {
		overlay = "searchOverlay";
	}
	else if (this.shares == 0) {
		overlay = "folderOverlay_unshared";
	}

		
	// Create the label for this device; force it to wrap and truncate as needed
	var label = pp2html(this.friendly ? this.friendly : this.name, 14, 8);
	return label + '<div class="'+ type + '"><div class="'+overlay+'"></div></div><br>';
}


//------------------------------------------------------------------------------
// DeviceItemRefresh - Refresh the device data based on the given XML node
//------------------------------------------------------------------------------

function DeviceItemRefresh(node)
{

	try
	{	
		this.name = Unescape((node.getElementsByTagName("name")).item(0).text);
	}
	catch(e) { /* Ignore */ }
	
	try
	{	
		this.uid = (node.getElementsByTagName("uid")).item(0).text;
	}
	catch(e) { /* Ignore */ }
	
	try
	{	
		this.friendly = Unescape((node.getElementsByTagName("friendly")).item(0).text);
	}
	catch(e) { /* Ignore */ }
	
	try
	{	
		this.description = (node.getElementsByTagName("description")).item(0).text;
	}
	catch(e) { /* Ignore */ }
	
	try
	{
		// Only refresh status if currently held status is not connected.
		if (this.status != "connected")
			this.status = node.attributes.getNamedItem("status").value;
	}
	catch(e) { /* Ignore */ }
	
	try
	{	
		var typeElem = node.getElementsByTagName("type");
		if (typeElem.length > 0)
		{
			var typeNode = typeElem[0];
			var type = typeNode.attributes.getNamedItem("class").value;
			if (type.toLowerCase() == "computer" || type.toLowerCase() == "other_pc")
				this.type = "desktop";
			else if (type.toLowerCase().indexOf("intel upnp server") != -1)
				this.type = "desktop";
			else if (type.toLowerCase() == "mediadevices")
				this.type = "dma";
			else if (type.toLowerCase().indexOf("router") != -1)
				this.type = "router";
			else
				this.type = "unknownDevice";
		}
	}
	catch(e) { /* Ignore */ }
	
	try
	{	
		this.ip = (node.getElementsByTagName("ipv4")).item(0).text;
	}
	catch(e) { /* Ignore */ }

	// If this item is on waiting list, and it is a connected PC, finish
	// waiting and move it to the active list.
	if (this.waiting && this.list) {
		if (this.status == "connected" && this.type == "desktop")
			this.list.FinishItemWait(this);
	}

}


//------------------------------------------------------------------------------
// DeviceItemRefresh - Refresh the device data based on the given XML node
//------------------------------------------------------------------------------

function DeviceItemRefreshShares()
{

	if (!this.list)
		return;

	this.list.UpdateStart();

	var thisDevice = this;
	var refreshAgain = false;

	try {

		if (this.status == "unavailable") {
			this.searching = false;
			this.shares = 0;
		}

		if (this.searching) {
			if (!this.fs) {
				// Filesystem hasn't been instantiated.
				if (this.BrowseFiles()) {
					// Initializing in background, try checking shares later.
					refreshAgain = true;
				}
				else {
					throw "Device.Refresh: Failed to initialize filesystem for device " + device.name;
				}
			} // if (!this.fs)
			else {
				// Filesystem has been instantiated, now check status of initialization
				if (this.fs.status == FS_READY) {
					// We know for sure that this device is actually connected.
					this.status = "connected";
					if (this.shares == -1) {
						// root folder enumeration hasn't started yet
						this.root = this.fs.rootFolder;
						if (this.root) {
							if (!this.root.enumerateBackground())
								throw "Device.RefreshShares: Failed to initialize filesystem for device " + device.name;
							this.shares = this.root.childNodes.length;
							refreshAgain = true;
						}
						else {
							throw "Device.RefreshShares: Failed to initialize filesystem for device " + device.name;
						}
					}
					else {
						if (this.root.complete) {
							this.searching = false;
							this.shares = this.root.childNodes.length;
						}
						else if (this.root.error) {
							this.searching = false;
							this.shares = 0;
						}
						else {
							this.shares = this.root.childNodes.length;
							refreshAgain = true;
						}
					}
				}
				else if (this.fs.status == FS_INITIALIZING) {
					// Initializing in background, try checking shares later.
					refreshAgain = true;
				}
				else {
					// Filesystem status is failed or unknown
					throw "Device.RefreshShares: Failed to initialize filesystem for device " + device.name;
				}
			} // else
		} // if (this.searching)

	}
	catch (e) {
		// Failed to initialize filesystem.
		this.searching = false;
		this.shares = 0;
		if (this.fs.status == FS_ACCESSDENIED) {
			// If access was denied, the device is connected but 
			// no accessible shares.
			this.status = "connected";
		}
	}
	
	// If item is still on waiting list, and we found shares, finish
	// waiting and move it to the active list.  Also move to active list
	// if we know this device is connected and is a PC.
	if (this.waiting) {
		var isConnectedPc = this.status == "connected" && this.type == "desktop";
		if (this.shares > 0 || isConnectedPc)
			this.list.FinishItemWait(this);
	}
		
	// If item is still on waiting list, and network map is complete, then 
	// no chance of getting on the active list.  So remove this item.
	if (!refreshAgain && this.waiting && this.list.networkMapComplete)
		this.list.RemoveItemWait(this);

	// Update device list pending flag now that device item waiting status
	// is refreshed
	this.list.AdjustPending();

	this.list.UpdateFinish();

	if (refreshAgain)
		setTimeout(function () {thisDevice.RefreshShares();}, refreshInterval);

}


//------------------------------------------------------------------------------
// DeviceItemRefresh - Initialize filesystem interface on device.  Return
// true on success.
//------------------------------------------------------------------------------

function DeviceItemBrowseFiles()
{
	this.fs = new ActiveXObject("ShellUtilities.Filesystem");
	if (!this.fs)
		return false;
	this.rootPath = this.localhost ? "" : "\\\\" + this.name;
	if (!this.fs.initializeBackground(this.rootPath)) {
		return false;
	}
	return true;
}



//------------------------------------------------------------------------------
// DeviceList - Create the list of network map devices, and initiate the
// recurring refresh until the network is finished building.
//------------------------------------------------------------------------------

function DeviceList(grid, info)
{

	var devices = new BaseList();

	// Properties
	devices.networkMap = null;
	devices.networkMapComplete = false;
	
	// Methods
	
	devices.SetDisplay = DeviceListSetDisplay;
	devices.AdjustPending = DeviceListAdjustPending;
	
	devices.GetDeviceByName = function(name) 
	{
		if (name.toUpperCase() == "LOCALHOST")
			return deviceLocalhost;
		for (var i = 0; i < this.items.length; i++) {
			var curName = this.items[i].name;
			if (!curName)
				continue;
			if (curName.toUpperCase() == name.toUpperCase())
				return this.items[i];
		}
		return null;
	}
	
	devices.GetDeviceByIp = function(ip)
	{
		for (var i = 0; i < this.items.length; i++) {
			var curIp = this.items[i].ip;
			if (!curIp)
				continue;
			if (curIp == ip)
				return this.items[i];
		}
		return null;
	}

	devices.ItemAction = function(index)
	{
		var device = this.items[index];
		var proceed = false;
		if (!device)
			return;
		if (device.name && device.status != "unavailable")
		{
			this.current = device;
			PageLoad("#FOLDER");
		}
	}
	
	devices.ItemFocus = function(index, focus)
	{
		var device = this.items[index];
		var desc = '';

		if (!device)
			// Hovering over the 'connection' icon.
			return "";

		if (focus)
		{
			// Determine the proper description based on device status & shares
			if (device.status == "unavailable")
				desc = languageDeviceUnavailable;
			else {
				if (device.localhost)
					desc = languageDeviceAll;
				else if (device.shares > 0)
					desc = languageDeviceShared + " " + device.shares;
				else if (device.searching)
					desc = languageDeviceSearching;
				else if (device.shares == 0)
					desc = languageDeviceNone;
				else
					desc = languageDeviceSearching;
			}

			// Determine the location of the focus line on the screen
			var ofs = this.grid.child[index-this.grid.viewFirst];
			var lineWidth = Math.floor(ofs.offsetWidth/2) - this.lineTop.elem.offsetLeft;
			while (ofs)
			{
				lineWidth += ofs.offsetLeft;
				ofs = ofs.offsetParent;
			}
			
			this.lineTop.elem.style.width = lineWidth+'px';
		}

		// Update the description area text
		this.info.elem.innerHTML = focus ? desc : '';

		// Adjust the position of the focus line
		this.lineTop.elem.style.display = focus ? 'block' : 'none';
		this.lineBot.elem.style.display = focus ? 'block' : 'none';

		return "";
	}
	
	devices.SetDisplay = function(grid, info)
			{ DeviceListSetDisplay(this, grid, info); };

	devices.SetLine = function(lineTop, lineBot, bar)
	{
		this.UpdateStart();
		this.lineTop = lineTop;
		this.lineTop.elem.className = 'NetmapLineTop';
		this.lineTop.elem.style.display = 'none';
		this.lineBot = lineBot;
		this.lineBot.elem.className = 'NetmapLineBot';
		this.lineBot.elem.style.display = 'none';
		this.bar = bar;
//		this.bar.elem.className = 'NetmapBar';
		this.UpdateFinish();	
	}

	devices.RedrawPending = function()
	{
		return '<div class="searchNetworkLarge"></div><br>';
	}


	// Attempt to build the network map
	try
	{
		devices.networkMap = new ActiveXObject("IMSShellUtilities.NetworkMap");
	}
	catch(e)
	{
		devices.networkMap = null;
	}



	devices.UpdateStart();

	DeviceListRefresh(devices);

	if (grid)
		devices.SetDisplay(grid, info);

	devices.UpdateFinish();

	return devices;
	
}


//------------------------------------------------------------------------------
// DeviceList::KeyStep - Switch grid navigation from vertical to horizontal.
//------------------------------------------------------------------------------

function DeviceListKeyStep(grid, step, vert)
{
	// Vertical navigation is allowed to leave the grid
	if (vert)
		return false;

	// Are we stepping above the top of the grid?
	var viewFirst = grid.viewFirst + step;
	if (viewFirst < 0)
	{
		viewFirst = 0;
		if (!grid.viewFirst)
			WidgetSetFocus(grid.navGrid[0][0]);
	}


	// Are we stepping below the bottom of the grid?
	if ((step>0) && (viewFirst + grid.navRows*grid.navCols > grid.viewTotal))
	{
		viewFirst = grid.viewTotal-grid.navRows*grid.navCols;
		if (viewFirst < 0)
			viewFirst = 0;
	}
	
	
	// Update the current viewport with the new parameters
	if (viewFirst != grid.viewFirst)
		GridSetViewport(grid, viewFirst, grid.viewTotal, grid.viewCollapse);
	return true;
}



//------------------------------------------------------------------------------
// DeviceList::KeyPage - Switch from page to single, vertical to horizontal.
//------------------------------------------------------------------------------

function DeviceListKeyPage(grid, step)
{
	return grid.cbKeyStep(grid, step*4, false);
}



//------------------------------------------------------------------------------
// DeviceList::SetDisplay - Set the grid+info widgets in which the device list
// will be displayed, and insure this is rendered for the first time.
//------------------------------------------------------------------------------

function DeviceListSetDisplay(devices, grid, info)
{
	devices.UpdateStart();

	try {
		BaseListSetDisplay(devices, grid, info);

		grid.elem.id = "netMapScroll";
		grid.cbKeyStep = DeviceListKeyStep;
		grid.cbKeyPage = DeviceListKeyPage;
		if (grid.viewPane.stats)
			grid.viewPane.stats.id = "netMapStats";

		for (var i = 0; i < grid.child.length; i++)
		{
			grid.child[i].align = "center";
			grid.child[i].valign = "top";
		}
		
		info.elem.className = "NetmapDescription";
	}
	catch (e) {
	}
	
	devices.UpdateFinish();
}



//------------------------------------------------------------------------------
// DeviceListAdjustPending - Check if this list is still pending and revise
// pending flag as appriate.  This is called after any actions that may
// change the pending status of this list.
//------------------------------------------------------------------------------

function DeviceListAdjustPending()
{
	// Device list is still pending if network map is not complete or if items
	// are still on the waiting list
	this.UpdateStart();
	this.pending = !this.networkMapComplete || (this.itemsWaiting.length > 0);
	this.UpdateFinish();
}



//------------------------------------------------------------------------------
// DeviceListRefresh - Read the most recent network map and refresh the
// device list based on the items found there.  If the pending flag is set,
// schedule this function to be called again after some interval.
//------------------------------------------------------------------------------

function DeviceListRefresh(devices)
{
	devices.UpdateStart();

	try
	{
		networkXML = new ActiveXObject("Msxml2.DOMDocument"); // .4.0");
		networkXML.async = false;
	 	networkXML.resolveExternals = false; 
		networkXML.validateOnParse = true; 
		networkXML.setProperty("SelectionLanguage", "XPath");

		var str = devices.networkMap.GetXmlString();
		networkXML.loadXML(str);

		// Create or refresh the localhost
		var localhostNodeList = networkXML.selectNodes("/network/localhost");
		if (!devices.items.length) {
			var deviceItem = new DeviceItem(localhostNodeList[0]);
			devices.AddItem(deviceItem);
			deviceItem.RefreshShares();
		}


		// Get the other devices on the network
		var deviceNodeList = networkXML.selectNodes("/network/device");
		for (var i = 0; i < deviceNodeList.length; i++)
		{
			var deviceNode = deviceNodeList[i];
			var deviceItem = null;
			
			// If its already in the list, use existing item and refresh it.
			var name = '';
			try
			{
				name = (deviceNode.getElementsByTagName("name")).item(0).text;
			}
			catch (ex) 
			{
			}
			if (name) {
				name = Unescape(name);
				// Search active device list
				for (var j = 0; j < devices.items.length; j++) {
					if (devices.items[j].name.toUpperCase() == name.toUpperCase()) {
						deviceItem = devices.items[j];
						break;
					}
				}
				// Search waiting list if we didn't find it in active list
				if (!deviceItem) {
					for (var j = 0; j < devices.itemsWaiting.length; j++) {
						if (devices.itemsWaiting[j].name.toUpperCase() == name.toUpperCase()) {
							deviceItem = devices.itemsWaiting[j];
							break;
						}
					}
				}
			}
			else {
				continue;
			}
			
			// Otherwise, create a new item and add it to the waiting list.
			// A device is moved from the waiting list to the active device list
			// if RefreshShares finds shares on the device, or if the device
			// is verified to be connected and is a computer/PC.
			if (!deviceItem) {
				deviceItem = new DeviceItem(deviceNode);
				if (deviceItem.type != "router" && deviceItem.status != "unavailable") {
					if (deviceItem.status == "connected" && deviceItem.type == "desktop") {
						// Connected PCs automatically go on the active list.
						devices.AddItem(deviceItem);
					}
					else {
						// All other devices go to the waiting list until we find shares
						// on them.
						devices.AddItemWait(deviceItem);
					}
					deviceItem.RefreshShares();
				}
			}
			else {
				deviceItem.Refresh(deviceNode);
			}
		} // for (var i...)
			

		// Check for a pending flag, indicating the network map is still pending completion
		var pendingNodes = networkXML.selectNodes("/network/pending");
		var status = null;
		if (pendingNodes) {
			if (pendingNodes[0]) {
				status = pendingNodes[0].text;
			}
		}
		var networkMapPending = (status && (status != "0") && (status != "false"));
		
		if (networkMapPending) {
			// Network map still pending, set up device list refresh after specific time interval
			setTimeout("DeviceListRefresh(listLookup["+devices.lookupIndex+"])", refreshInterval);
		}
		else {
			// Network map complete, so connection status for all devices has been updated.
			// The only devices left on the waiting list are non-PC devices
			// with searching complete and no shares found, or with searching still underway.
			// Remove the devices for which searching is complete.
			devices.networkMapComplete = true;
			for (var i = 0; i < devices.itemsWaiting.length; i++) {
				if (!devices.itemsWaiting[i].searching)
					devices.RemoveItemWait(devices.itemsWaiting[i]);
			}
			devices.AdjustPending();	// adjust pending flag now that we are complete
		}


	}
	catch(e)
	{
	}

	devices.UpdateFinish();
}



//================================== Truncation ===============================

// TruncateInitialize()
// Initialize truncation system.
function TruncateInitialize() {
	try {
		var fontSize = parent.document.body.style.zoom > 0 ?
			Math.ceil(26 * parent.document.body.style.zoom) : 26;
		Toolbox.SetFont(fontSize, 1, "Arial");  
	}
	catch (ex) {
	}
}

// Truncate()
// Truncate generic text, such as a label or name.
function Truncate(text, width) {
	var newText = '';
	try {
		newText = Toolbox.Truncate(text, width);
	}
	catch (ex) {
		newText = text;
	}
	return newText;
}

// TruncatePath()
// Truncate a pathname.
function TruncatePath(path, max) {
	var newPath = '';
	try {
		newPath = Toolbox.TruncatePath(path, max);
	}
	catch (ex) {
		newPath = path;
	}
	return newPath;
}



//================================== Utility ===============================


// Unescape()
// Replace character codes in an escaped string with the characters themselves.
function Unescape(str) {
	var result = "";
	var p;
	var convert;
	while ((p = str.indexOf("%")) != -1) {
		if (p + 2 < str.length) {
			result = result + str.substring(0, p);
			convert = str.substring(p+1, p+3);
			result = result + String.fromCharCode(parseInt(convert, 16));
			str = str.substring(p+3, str.length);
		}
		else {
			break;
		}
	}
	result = result + str;
	return result;
}
