// mediaone.js - Content Manager M1 media server interaction classes
//
// INTEL CONFIDENTIAL
//	Copyright 2004-2006 Intel Corporation All Rights Reserved.  The source code
//	contained or described herein and all documents related to the source code
//	(Material) are owned by Intel Corportion or its suppliers or licensors.
//	Title to the Material remains with Intel Corporation or its suppliers and
//	licensors. The Material may contain trade secrets and 	proprietary and
//	confidential information of Intel Corporation and its 	suppliers and
//	licensors, and is protected by worldwide copyright and trade secret laws
//	and treaty provisions. No part of the Material may be used, copied,
//	reproduced, modified, published, uploaded, posted, transmitted, distributed,
//	or disclosed in any way without Intels prior express written permission. 
//	
//	No license under any patent, copyright, trade secret or other intellectual
//	property right is granted to or conferred upon you by disclosure or delivery
//	of the Materials, either expressly, by implication, inducement, estoppel or
//	otherwise. Any license under such intellectual property rights must be express
//	and approved by Intel in writing.
//
//	Unless otherwise agreed by Intel in writing, you may not remove or alter this
//	notice or any other notice embedded in Materials by Intel or Intels suppliers
//	or licensors in any way.
//

// Dependencies:
// MxMediaOne object, embedded in container HTML file
// Toolbox object, embedded in container HTML file


//var DEBUGSTR = "";
//var DEBUGTIME = null;

// Global connection manager
var connectionManager;
var updateManager = null;
var updateInfo = null;		// Data structures used for media updates.


// Local M1 Server interfaces
var localServer;		// Server access object [MXServer]
var aggregator;			// Aggregation interface to use/block other CDS [MXAggregator]
var synchronizer;		// Synchronization interface to use/block folders [MXSynchronizer]
var fileServices;		// Local M1 Server file services interface [MXFileServices] 

var checkServerTimeout;	// Handle from setTimeout in event connection status checking isn't working.
var checkServerRetry;
var checkServerRepeat = null;

// bitwise flags that define the file type.
var M1_DIR_ATTR_FOLDER		= 0x0002;
var M1_DIR_ATTR_HIDDEN		= 0x0020;
var M1_DIR_ATTR_NETWORK		= 0x0100;
var M1_DIR_ATTR_OPTICAL		= 0x0200;
var M1_DIR_ATTR_REMOVABLE	= 0x0400;

// always create M1 filesystem object, even if it doesn't exist
var M1_FILE_FLAG_ALWAYS		= 0x0002;

var Tools = null;

// Default folders
var CSIDL_PERSONAL = 0x0005;
var CSIDL_COMMON_DOCUMENTS = 0x002e;

// Exclude folders
var CSIDL_PROGRAM_FILES = 0x0026;
var CSIDL_WINDOWS = 0x0024;
var CSIDL_SYSTEM = 0x0025;
var SYSTEMDRIVE = "%SYSTEMDRIVE%";
var SYSTEMDRIVEDEFAULT = "C:";


function StartMediaOne()
{
//DEBUGTIME = new Date();
//DEBUGSTR = "StartMediaOne ";
	try
	{
		MXMediaOne.start();
		MXMediaOne.addFeatures(2);
	 
		connectionManager = new ConnectionManager();
		connectionManager.connect();
	}
	catch(e)
	{
	}
}

function StopMediaOne()
{
  try
  {
      connectionManager.disconnect();
      if (localServer != null){
         localServer.dispose();
         localServer = null;
         fileServices = null;
         aggregator = null;
         synchronizer = null;
      }
      MXMediaOne.stop();
  }
  catch(e)
  {
  }

}

// CheckMediaOneExecute()
// Execute the given callback function only once a connection to the M1 server
// has been verified.  If a connection isn't verified after a certain
// time, execute the given failure callback.
function CheckMediaOneExecute(cbExec, cbFail, timeLimit) {
    checkServerTimeout = setTimeout(function () {CheckFail(cbFail)}, timeLimit);
    checkServerRetry = setTimeout(function () {CheckServer(cbExec)}, 500);
}

function CheckFail(cbFail) {
	ClearTimeoutsMediaOne();
	if (cbFail)
		cbFail();
}

function CheckServer(cbExec) { 
	if (connectionManager == null) {
		checkServerRetry = setTimeout(function () {CheckServer(cbExec)}, 500);
	}
	else if (connectionManager.getStatus() == 'connected') {
		clearTimeout(checkServerTimeout);
		if (cbExec)
			cbExec();
	}
	else if (connectionManager.getStatus() == 'error') {
		PageLoad('#SERVERLOSTCOMMS');
	}
	else if (connectionManager.getStatus() == 'stopped') {
		PageLoad('#SERVERSTOPPED');
	}
	else {
		checkServerRetry = setTimeout(function () {CheckServer(cbExec)}, 500);
	}
}

function ClearTimeoutsMediaOne() {
	if (checkServerTimeout)
		clearTimeout(checkServerTimeout);
	if (checkServerRetry)
		clearTimeout(checkServerRetry);
}


function CheckMediaOneRepeat(cbFail, cbParam) {
	try {
		var checkStatus = 0;
		checkStatus = synchronizer.status;
		mxThrowIfFailed(synchronizer);   
		var connectionStatus = connectionManager.getStatus();
		if (connectionStatus == 'stopped' || connectionStatus == 'error')
			throw "CheckMediaOneRepeat: not connected";
		checkServerRepeat = setTimeout(function () {CheckMediaOneRepeat(cbFail, cbParam)}, 3000);
	}
	catch (ex) {
		cbFail(cbParam);
	}
}

function CheckMediaOneStop() {
	if (checkServerRepeat) {
		clearTimeout(checkServerRepeat);
		checkServerRepeat = null;
	}
}



//------------------------------------------------------------------------------
// class ConnectionManager
// Connection manager manages service stopping, starting, and connecting
// of MediaOne service.
//------------------------------------------------------------------------------

// Possible states (returned via getStatus()):
//  'stopped'      : service is stopped, we can't connect
//  'disconnected' : service is running, but we are not connected
//  'connecting'   : service is running, we are connecting
//  'connected'    : service is running and we are connected
//  'stopping'     : service is stopping, we are not connected
//  'starting'     : service is starting, we are not connected
//  'error'        : an error occurred

var SERVICE_STOPPED              = 0x00000001;
var SERVICE_START_PENDING        = 0x00000002;
var SERVICE_STOP_PENDING         = 0x00000003;
var SERVICE_RUNNING              = 0x00000004;
var SERVICE_CONTINUE_PENDING     = 0x00000005;
var SERVICE_PAUSE_PENDING        = 0x00000006;
var SERVICE_PAUSED               = 0x00000007;


function ConnectionManager()
{
	this.disconnect(); // ensure disconnected state
}

ConnectionManager.prototype.connect = function()
{
//DEBUGTIME = new Date();
//DEBUGSTR = "connect(0) ";
	try {
		if (Toolbox.QueryServiceStatus() != SERVICE_STOPPED)
		{
			MXMediaOne.start(); // flush discovery cache
			this.disconnect();
			// Call the working function, separated because it relies on setTimeout.
			ConnectionManager__connecting();
		}
	}
	catch (ex) {
		this.status = 'error';
	}
}

ConnectionManager.prototype.restart = function()
{
//DEBUGTIME = new Date();
//DEBUGSTR = "restart(0) ";
	this.disconnect();
	ConnectionManager__stop();
}

ConnectionManager.prototype.disconnect = function()
{
//var DEBUGTIMEINT = new Date;
//DEBUGSTR += "disconnect(" + (DEBUGTIMEINT.getTime() - DEBUGTIME.getTime()) + ") ";
	try {
		if (this.timeoutConnecting) {
			clearTimeout(this.timeoutConnecting)
		}
		if (localServer != null){
			localServer.dispose();
		}
	}
	catch (ex) {
		// Absorb exception, this is a cleanup function
	}

	this.timeoutConnecting = null;
	localServer = null;
	fileServices = null;
	aggregator = null;
	synchronizer = null;
	  
	this.status = 'disconnected'; 
}

ConnectionManager.prototype.getStatus = function()
{
	try {
		if (!this.isActive() && 
			Toolbox.QueryServiceStatus() == SERVICE_STOPPED)
		{
			this.disconnect();
			this.status = 'stopped';
		}
	}
	catch (ex) {
		this.status = 'error';
	}
	return this.status;
}

ConnectionManager.prototype.isActive = function()
{
	return this.status == 'stopping' ||
		this.status == 'starting' ||
		this.status == 'connecting';
}


function ConnectionManager__connecting() {
//var DEBUGTIMEINT = new Date;
//DEBUGSTR += "connecting(" + (DEBUGTIMEINT.getTime() - DEBUGTIME.getTime()) + ") ";
	try {
		this.timeoutConnecting = null;
		connectionManager.status = 'connecting';
		localServer = MXMediaOne.localServer;
		if (localServer) {
			fileServices = localServer.osServices.fileServices;
			aggregator = localServer.aggregator;
			synchronizer = localServer.synchronizer;
			if (fileServices && aggregator && synchronizer) {
				synchronizer.connectSync();
				aggregator.connectAgg();
				if (synchronizer.connected && aggregator.connected) {
//var DEBUGTIMEINTINT = new Date;
//DEBUGSTR += "connected(" + (DEBUGTIMEINTINT.getTime() - DEBUGTIME.getTime()) + ") ";
//alert(DEBUGSTR);
					connectionManager.status = 'connected';
					return;
				}
			}
		}
		// if our connection attempt fails, try again
		this.timeoutConnecting = setTimeout("ConnectionManager__connecting()", 1000);
	}
	catch (ex) {
		connectionManager.status = 'error';
//DEBUGSTR += "error ";
//alert(DEBUGSTR);
	} 
}

function ConnectionManager__stop()
{
//var DEBUGTIMEINT = new Date;
//DEBUGSTR += "stop(" + (DEBUGTIMEINT.getTime() - DEBUGTIME.getTime()) + ") ";
	try {
		connectionManager.status = 'stopping';
		if (Toolbox.StopService())
		{
			setTimeout("ConnectionManager__stopping()", 1000);
		}
		else
		{
			connectionManager.status = 'error';
		} 
	}
	catch (ex) {
		connectionManager.status = 'error';
	}		
}

function ConnectionManager__starting()
{
//var DEBUGTIMEINT = new Date;
//DEBUGSTR += "starting(" + (DEBUGTIMEINT.getTime() - DEBUGTIME.getTime()) + ") ";
	try
	{
		connectionManager.status = 'starting';
		if (Toolbox.QueryServiceStatus() == SERVICE_RUNNING)
		{
			this.timeoutConnecting = setTimeout("ConnectionManager__connecting()", 1000);
		}
		else
		{
			setTimeout("ConnectionManager__starting()", 1000);
		}
	}
	catch (e)
	{
		connectionManager.status = 'error';
//DEBUGSTR += "error ";
//alert(DEBUGSTR);
	}
}

function ConnectionManager__stopping()
{
//var DEBUGTIMEINT = new Date;
//DEBUGSTR += "stopping(" + (DEBUGTIMEINT.getTime() - DEBUGTIME.getTime()) + ") ";
	try
	{
		var status = Toolbox.QueryServiceStatus(); 
		if (status == SERVICE_STOPPED)
		{
			// service is stopped, re-start
			var success = false;
		 
			// We'd also like to make sure the rest of the EF stack is up...
			try
			{
				var monitor = new ActiveXObject("IntelMediaServer.ServiceMonitor");
				monitor.StartServices();
				success = true;
//DEBUGSTR += "monitor.StartServices() ";
			}
			catch (ex)
			{
				// An error occurred, starting the whole stack,
				// let's at least get the media server going
				success = Toolbox.StartService();
//DEBUGSTR += "Toolbox.StartService() ";
			}

			if (success)
			{
				connectionManager.status = 'starting';
				setTimeout("ConnectionManager__starting()", 1000);
			}
			else
			{
				connectionManager.status = 'error';
//DEBUGSTR += "error ";
//alert(DEBUGSTR);
			} 
		}
		else
		{
			setTimeout("ConnectionManager__stopping()", 1000);
		}
	}
	catch (ex)
	{
		connectionManager.status = 'error';  
//DEBUGSTR += "error ";
//alert(DEBUGSTR);
	}
}



//------------------------------------------------------------------------------
// class UpdateManager
// Performs the actual media update with the media server.
//------------------------------------------------------------------------------
// public
UpdateManager.prototype.Update = UpdateManager__Update;
UpdateManager.prototype.Stop = UpdateManager__Stop;
// protected
UpdateManager.prototype.Progress = UpdateManager__Progress;
UpdateManager.prototype.Fail = UpdateManager__Fail;
// private
UpdateManager.prototype.UpdateStart = UpdateManager__UpdateStart;
UpdateManager.prototype.RemoveFolders = UpdateManager__RemoveFolders;
UpdateManager.prototype.AddFolders = UpdateManager__AddFolders;
UpdateManager.prototype.UpdateServers = UpdateManager__UpdateServers;


// UpdateManager() constructor
// updateInformation	The UpdateInfo structure containing the information to do the update
// cbInitialized		Callback to notify that process has initialized & is self-sufficient
// cbProgress			Callback for progress notificaion
// cbComplete			Callback for notification that update is complete
// cbFail				Callback for failure notification
function UpdateManager(updateInformation, cbInitialized, cbProgress, cbComplete, cbFail) {

	this.updateInfo = updateInformation;
	this.cbInitialized = cbInitialized;
	this.cbProgress = cbProgress;
	this.cbComplete = cbComplete;
	this.cbFail = cbFail;

	this.aggregating = false;
	this.synchronizing = false;

	this.pending = null;
	this.complete = null;
	
	this.totalScanned = 0;
	this.totalMusic = 0;
	this.totalPictures = 0;
	this.totalVideos = 0;
	this.totalBlocked = 0;
}


// static
function UpdateManager__Fail(updateMgr) {
	if (updateMgr.cbFail)
		updateMgr.cbFail();
}


function UpdateManager__Update() {

	try {
		
		CheckMediaOneRepeat(UpdateManager__Fail, updateManager);

		// Run through our folder and server monitoring state structures
		// and store changes in the appropriate arrays.
		this.updateInfo.StoreChanges(UpdateManager__UpdateStartHelper);

	}
	catch (ex) {
		this.Stop();
		if (this.cbFail)
			this.cbFail();
	}

}


// private - only called through Update()
// static
function UpdateManager__UpdateStartHelper() {
	if (updateManager)
		updateManager.UpdateStart();
}


// private - only called through UpdateStartHelper()
function UpdateManager__UpdateStart() {

	try {
	
		// Shortcut locals.
		var folderAdd = this.updateInfo.folderAdd;
		var folderRemove = this.updateInfo.folderRemove;
		var serverAdd = this.updateInfo.serverAdd;
		var serverRemove = this.updateInfo.serverRemove;

		this.complete = new Array;
		
		// Array of pending update actions: revoke read access all folders,
		// grant read access all folders, synchronize individual folders,
		// and aggregate individual servers.
		this.pending = new Array;
		
		this.totalScanned = 0;
		this.totalMusic = 0;
		this.totalPictures = 0;
		this.totalVideos = 0;
		this.totalBlocked = 0;

		localServer.resetCount(-1);
		mxThrowIfFailed(localServer);

		if ((folderAdd.length > 0) || (folderRemove.length > 0)) {
			synchronizer.notifyChanges();
			mxThrowIfFailed(synchronizer);
			this.synchronizing = true;
		}
  
		if ((serverAdd.length > 0) || (serverRemove.length > 0)) {
			aggregator.notifyChanges();
			mxThrowIfFailed(aggregator);
			this.aggregating = true;
		}

		if (folderRemove.length > 0) {
			// If we have folders to remove, start folder remove procedure.
			this.RemoveFolders();
		}

		else {
			// No folders to remove, so start folder add procedure.
			if (folderAdd.length > 0)
				this.AddFolders();
			else
				this.UpdateServers();
		}
	}

	catch (ex) {
		this.Stop();
		if (this.cbFail)
			this.cbFail();
	}

}

// try/catch covered by UpdateManager.Update().
function UpdateManager__RemoveFolders() {

	// Shortcut locals.
	var folderRemove = this.updateInfo.folderRemove;

	// If a folder has these attributes, we can't change its permissions.
	var permMask = M1_DIR_ATTR_NETWORK | M1_DIR_ATTR_OPTICAL |
		M1_DIR_ATTR_REMOVABLE;
	
	var pendingRevoke = new Array;

	for (i = 0; i < folderRemove.length; i++) {
		pendingRevoke.push(i);
	}

	// prevent completion until we have tried to sync everything!
// LTD: timeout???
	this.pending[0] = 'revoke';
	this.pendingRevoke = pendingRevoke;

	for (i = 0; i < folderRemove.length; i++) {

		var monitor = folderRemove[i].monitor;
		mxThrowIfFailed(folderRemove[i]);

		// Only remove the folder from monitoring if we are actually monitoring it.
		if (monitor)
		{
			var syncParams = new Object;
			syncParams.monitor = monitor;
			syncParams.updateMgr = this;
			
			// Determine if network path to avoid attribute check (blocks if bad path)
			var path = Url2Path(folderRemove[i].url);
			var isNetworkPath = false;	
			if (path)
				isNetworkPath = (path.indexOf('\\\\') == 0);
			
			if (!isNetworkPath && ((folderRemove[i].attributes & permMask) == 0)) {
				// we can change permissions
				Toolbox.RevokeReadAsync(path, 
										UpdateManager__UnsyncFolder, 
										syncParams);
			}
			else {
				// don't change permissions, because we can't
				UpdateManager__CallUnsyncFolder(syncParams);
			}
		} // if monitor
		
		else {
			// Not monitoring this folder anymore.
			// We won't really unsync it because of false parameter,
			// but we do need to remove folder from pending list.
			UpdateManager__UnsyncFolder(0, false, syncParams);
		} // else
		
	} // for
}


// Helper function needed to do setTimeout and maintain a unique
// scope for syncParams.
function UpdateManager__CallUnsyncFolder(syncParams) {
	setTimeout(function () {UpdateManager__UnsyncFolder(0, true, syncParams)}, 1);
}

// static
function UpdateManager__UnsyncFolder(errorCode, result, syncParams)
{

	try {

		var monitor = syncParams.monitor;
		var updateMgr = syncParams.updateMgr;

		if (errorCode == 0 && result == true) {
			if (monitor)
				synchronizer.unshare(monitor);
			updateMgr.totalBlocked++;
		}

		updateMgr.pendingRevoke.pop();

		// If no more unsyncs pending, move on to next stage, which
		// is syncing folders if we have folders, or else updating 
		// aggregated servers.
		if (updateMgr.pendingRevoke.length <= 0) {
			// Take 'revoke' pending entry out.
			updateMgr.pending.splice(0, 1);
			if (updateMgr.updateInfo.folderAdd.length > 0)
				updateMgr.AddFolders();
			else
				updateMgr.UpdateServers();
		}
	}
	
	catch (ex) {
		var exUpdateMgr = null;
		if (syncParams) {
			if (syncParams.updateMgr) {
				exUpdateMgr = syncParams.updateMgr;
			}
			else
				exUpdateMgr = updateManager;
		}
		else
			exUpdateMgr = updateManager;
		if (exUpdateMgr) {
			exUpdateMgr.Stop();
			if (exUpdateMgr.cbFail)
				exUpdateMgr.cbFail();
		}
	}

}

// try/catch established by UpdateManager.Update() and UpdateManager.RemoveFolders()
function UpdateManager__AddFolders()
{

	var folderAdd = this.updateInfo.folderAdd;

	// If a folder has these attributes, we can't change its permissions.
	var permMask = M1_DIR_ATTR_NETWORK | M1_DIR_ATTR_OPTICAL |
		M1_DIR_ATTR_REMOVABLE;
	
	var pendingGrant = new Array;

	for (i = 0; i < folderAdd.length; i++) {
		pendingGrant.push(i);
	}

	// prevent completion until we have tried to sync everything!
// LTD: timeout???
	this.pending[0] = 'grant';
	this.pendingGrant = pendingGrant;

	for (i = 0; i < folderAdd.length; i++) {

		var syncParams = new Object;
		syncParams.fileObj = folderAdd[i];
		syncParams.updateMgr = this;

		// Determine if network path to avoid attribute check (blocks if bad path)
		var path = Url2Path(folderAdd[i].url);
		var isNetworkPath = false;	
		if (path)
			isNetworkPath = (path.indexOf('\\\\') == 0);
		
		if (!isNetworkPath && ((folderAdd[i].attributes & permMask) == 0)) {
			// we can change permissions
			Toolbox.GrantReadAsync(path, 
									UpdateManager__SyncFolder, 
									syncParams);
		}
		else {
			// don't change permissions, because we can't
			UpdateManager__CallSyncFolder(syncParams);
		}
		
	} // for

}


// Helper function needed to do setTimeout and maintain a unique
// scope for syncParams.
function UpdateManager__CallSyncFolder(syncParams) {
	setTimeout(function () {UpdateManager__SyncFolder(0, true, syncParams)}, 1);
}


// static
function UpdateManager__SyncFolder(errorCode, result, syncParams)
{

	try {
	
		var fileObj = syncParams.fileObj;
		var updateMgr = syncParams.updateMgr;

		if (synchronizer.share(fileObj)) {
			updateMgr.pending.push(fileObj.url);
		}

		updateMgr.pendingGrant.pop();

		// If no more syncs pending, move on to next stage, updating 
		// aggregated servers.
		if (updateMgr.pendingGrant.length <= 0) {
			// Take 'grant' pending entry out.
			updateMgr.pending.splice(0, 1);
			updateMgr.UpdateServers();
		}
	}
	
	catch (ex) {
		var exUpdateMgr = null;
		if (syncParams) {
			if (syncParams.updateMgr) {
				exUpdateMgr = syncParams.updateMgr;
			}
			else
				exUpdateMgr = updateManager;
		}
		else
			exUpdateMgr = updateManager;
		if (exUpdateMgr) {
			exUpdateMgr.Stop();
			if (exUpdateMgr.cbFail)
				exUpdateMgr.cbFail();
		}
	}

}


// try/catch handled by UpdateManager.Update() and UpdateManager.Sync/UnsyncFolder()
function UpdateManager__UpdateServers()
{
	var serverAdd = this.updateInfo.serverAdd;
	var serverRemove = this.updateInfo.serverRemove;

	if (this.updateInfo.addAllServers.post != this.updateInfo.addAllServers.pre) {
		if (this.updateInfo.addAllServers.post) {
			aggregator.excludeByDefault = false;
		}
		else {
			aggregator.excludeByDefault = true;
		}
	}

	if (serverRemove) {
		if (serverRemove.length > 0) {
			for (i = 0; i < serverRemove.length; i++) {
				aggregator.unaggregate(serverRemove[i]);
				this.totalBlocked++;
				mxThrowIfFailed(aggregator);
			}
		}
	}

	if (serverAdd) {
		if (serverAdd.length > 0) {
			for (i = 0; i < serverAdd.length; i++) {
				aggregator.aggregate(serverAdd[i]);
				mxThrowIfFailed(aggregator);
// LTD: For now we will just assume aggregation is
// complete after calling aggregate(), since for some reason
// .aggregate doesn't call the progress handler.
	//			this.pending.push(serverAdd[i].identifier);
				this.complete.push(ServerName2DeviceUrl(serverAdd[i].name));
			}
		}
	}
	
	// Notify client that the update process has initialized and is
	// self-sufficient.
	if (this.cbInitialized)
		this.cbInitialized();

	if (this.pending.length == 0) {
		if (this.cbComplete != null) {
			this.Stop();
			this.updateInfo.StoreMediaPcs();
			this.cbComplete(0, 0, 0, 0, this.totalBlocked);
		}
	}
		
}


function UpdateManager__Progress(source, url, status, scanned, music, pictures, videos) {

//alert('source=' + source + ';url=' + url + ';scanned=' + scanned + ';music=' + music +
//  ';pictures=' + pictures + ';videos=' + videos);

	try {
	
		// Ensure we don't trigger anything after update is finished.
		if (!this.synchronizing && !this.aggregating) {
			this.Stop();
			return;
		}
	   
		var isServer = (source == "IMXAggregator");
		url = isServer ? url : NormalizePath(url);
		
		// Check if the given progress URL is in our pending list.  If it is,
		// take it off the pending list and mark it complete.
		for (var i = 0; i < this.pending.length; i++) {
			var pending_url = isServer ? this.pending[i] : 
				NormalizePath(this.pending[i]);
			if (pending_url == url)	{
				if (status >= 0) 
					// add to complete list, success
					this.complete.push(url);
				this.pending.splice(i, 1);
				break;
			}
		}

		// Update statistics.
		this.totalScanned += scanned;
		this.totalMusic  += music;
		this.totalPictures += pictures;
		this.totalVideos += videos;

		if (this.pending.length > 0) {
			// We still have pending update actions.
			if (this.cbProgress != null) {
				this.cbProgress(this.totalScanned, this.totalMusic, 
								this.totalPictures, this.totalVideos);
			}
		}
		else {
			// We are done with our update.
			if (this.cbComplete != null) {
				this.Stop();
				// Populate a list of media PCs we are now monitoring.
				this.updateInfo.StoreMediaPcs();
				if (this.cbComplete)
					this.cbComplete(this.totalScanned, this.totalMusic, 
						this.totalPictures, this.totalVideos,
						this.totalBlocked);
			}
		}
	}
	
	catch (ex) {
		this.Stop();
		if (this.cbFail)
			this.cbFail();
	}
	
}


function UpdateManager__Stop() {

	// Stop continuously checking for server connectivity.
	CheckMediaOneStop();

	try {
		if (this.synchronizing)
			synchronizer.unnotifyChanges();
		if (this.aggregating)
			aggregator.unnotifyChanges();
	}
	catch (ex) {
		// absorb exceptions, this is a cleanup function
	}

	this.synchronizing = false;
	this.aggregating = false;
	
}


// Progress handler attached to MXMediaOne object.  Calls UpdateManager__Progress
// for the global updateManager.
function MXMediaOne::importProgress(source, url, status, scanned, music, pictures, videos)
{
	if (!updateManager)
		return;

	updateManager.Progress(source, url, status, scanned, music, pictures, videos);

}



//------------------------------------------------------------------------------
// class UpdateInfo
// Information structures used for media updating.
//------------------------------------------------------------------------------

// public
UpdateInfo.prototype.Reinitialize = UpdateInfo__Reinitialize;
UpdateInfo.prototype.FetchMonitors = UpdateInfo__FetchMonitors;
UpdateInfo.prototype.StoreChanges = UpdateInfo__StoreChanges;
UpdateInfo.prototype.StoreMediaPcs = UpdateInfo__StoreMediaPcs;
UpdateInfo.prototype.EstablishDefaultFolders = UpdateInfo__EstablishDefaultFolders;
UpdateInfo.prototype.EstablishExcludedFolders = UpdateInfo__EstablishExcludedFolders;
UpdateInfo.prototype.IsDefaultFolder = UpdateInfo__IsDefaultFolder;


function UpdateInfo() {
	this.serverState = new Array;
	this.folderState = new Array;
	this.serverAdd = new Array;
	this.serverRemove = new Array;
	this.folderAdd = new Array;
	this.folderRemove = new Array;
	this.folderDefault = new Array;
	this.folderDefaultLookup = new Array;
	this.folderExclude = new Array;
	this.mediaPcs = new Array;			// Info table of PCs from which media is added.
	this.addAllServers = null;			// Special MonitorState object to track opt-in aggregation
	this.hasCustomSelections = false;	// Flag if user made additional Custom selections (not
										// just default folders).  This value is only set
										// when update is complete.
	this.initialized = false;

	if (!Tools)
		Tools = new ActiveXObject("ShellUtilities.Tools");

}


function UpdateInfo__Reinitialize() {
	this.serverState = new Array;
	this.folderState = new Array;
	this.serverAdd = new Array;
	this.serverRemove = new Array;
	this.folderAdd = new Array;
	this.folderRemove = new Array;
	this.folderDefault = new Array;
	this.folderDefaultLookup = new Array;
	this.folderExclude = new Array;
	this.mediaPcs = new Array;			// Info table of PCs from which media is added.
	this.addAllServers = null;			// Special MonitorState object to track opt-in aggregation
}


function UpdateInfo__FetchMonitors() {

	// Fetch all currently monitored folders.
	// Set up array of monitors.
	var monitorIter = synchronizer.monitors;
	if (monitorIter == null)
		mxThrow(synchronizer);
	var monitors = VBArray(monitorIter.next(1000000)).toArray();
		mxThrowIfFailed(monitorIter);
	monitorIter = null;

	if (monitors)
		for (var i = 0; i < monitors.length; i++) {
			Tools.SuppressNoDisk(true);
			var fileObj = monitors[i].fileSystemObject; 
			// file system object may be null if no longer accessible
			if (fileObj) {
				var url = fileObj.url;
				if (url) {
					var path = StripTrailSlash(Url2Path(fileObj.url));
					var key = path.toUpperCase();
					var monitorState = null;
					// Generate & insert a new monitorState object if one doesn't exist
					// for this path.
					if (this.folderState[key] == null) {          
						monitorState = new MonitorState(path);
						this.folderState[key] = monitorState;       
					}
					else {
						monitorState = this.folderState[key];
					}
					monitorState.label = StripTrailSlash(fileObj.name);
					monitorState.pre = true;
					monitorState.mid = true;
					monitorState.post = true;
				}
			}
		}
	
	// Fetch all UPnP CDS on the network.

	// Set up array of servers.
	var serverIter = MXMediaOne.servers;
	mxThrowIfFailed(MXMediaOne);
	var servers = VBArray(serverIter.next(1000000)).toArray();
	mxThrowIfFailed(serverIter);
	serverIter.dispose();
	serverIter = null;

	if (servers)
		for (var i = 0; i < servers.length; i++) {
			var id = servers[i].identifier;
			if (id) {
				var key = id.toUpperCase();
				var monitorState = null;
				// Check that the server is not the localhost Intel Media Server.
				var deviceName = ServerName2DeviceName(servers[i].name);
				if (deviceName && deviceLocalhost && deviceLocalhost.name) {
					if (deviceName.toUpperCase() == deviceLocalhost.name.toUpperCase()) {
						var cdsName = ServerName2CdsName(servers[i].name);
						if (cdsName.toUpperCase().indexOf("INTEL") >= 0)
							continue;
					}
				}
				
				// Generate & insert a new monitorState object if one doesn't exist
				// for this path.
				if (this.serverState[key] == null) {          
					monitorState = new MonitorState(id);
					this.serverState[key] = monitorState;       
				}
				else {
					monitorState = this.serverState[key];
				}
				monitorState.label = servers[i].name;
				// Save server object for later, because there is no API to retrieve
				// a server object by identifier if it isn't aggregated.
				monitorState.server = servers[i];
				if (aggregator.isAggregated(servers[i])) {
					monitorState.pre = true;
					monitorState.mid = true;
					monitorState.post = true;
				}
				else {
					monitorState.pre = false;
					monitorState.mid = false;
					monitorState.post = false;
				}
			} // if (id)
		} // for
	
	// Retrieve opt-in aggregation entry.
	this.addAllServers = new MonitorState("<AddAllServers>");
// LTD: put actual opt-in setting into pre and post
	this.addAllServers.pre = aggregator.excludeByDefault ? false : true;
	this.addAllServers.post = this.addAllServers.pre;	
	
}


function UpdateInfo__EstablishDefaultFolders(forceSelect) {

	var folderDef = null;
	var path = "";
	var label = "";
	var key = "";
	
	var folderDefault = this.folderDefault = new Array;
	var folderDefaultLookup = this.folderDefaultLookup = new Array;
	
	var folderState = this.folderState;

	// Establish default folders in listed order.
	
	// My Documents
	try {
		path = Toolbox.ShellFolderPath(CSIDL_PERSONAL);
		if (path) {
			key = path.toUpperCase();
    		if (folderState[key])
				folderDef = folderState[key];
			else {
				folderDef = new MonitorState(path);
			}
			label = "";
			label = Tools ? Tools.ShellFolderNameCustom(CSIDL_PERSONAL) : null;
			if (label && label != "") {
				folderDef.label = label;
			}
			else {
				label = Tools ? Tools.ShellFolderNameLocal(CSIDL_PERSONAL) : null;
				if (label && label != "") {
					folderDef.label = label;
				}
				else {
					folderDef.label = Path2FileName(path);
				}
			}
			folderDefault.push(folderDef);
		}
	}
	catch (ex) {
		// absorb
	}
 
	// Shared Documents
	try {
		path = Toolbox.ShellFolderPath(CSIDL_COMMON_DOCUMENTS);
		if (path) {
			key = path.toUpperCase();
			if (folderState[key])
				folderDef = folderState[key];
			else {
				folderDef = new MonitorState(path);
			}
			label = "";
			label = Tools ? Tools.ShellFolderNameCustom(CSIDL_COMMON_DOCUMENTS) : null;
			if (label && label != "") {
				folderDef.label = label;
			}
			else {
				label = Tools ? Tools.ShellFolderNameLocal(CSIDL_COMMON_DOCUMENTS) : null;
				if (label && label != "") {
					folderDef.label = label;
				}
				else {
					folderDef.label = Path2FileName(path);
				}
			}
			folderDefault.push(folderDef);
		}
	}
	catch (ex) {
		// absorb
	}
    
    // Establish lookup hashtable for default folders for optimization.
    // Also put them in the folderState hashtable.
    for (var i = 0; i < folderDefault.length; i++) {
		if (forceSelect)
			folderDefault[i].post = true;
		path = folderDefault[i].path;
		if (path) {
			key = path.toUpperCase();
			folderDefaultLookup[key] = folderDefault[i];
			folderState[key] = folderDefault[i];
		}
    }
    
}


function UpdateInfo__IsDefaultFolder(path) {
	var key = path.toUpperCase();
	return this.folderDefaultLookup[key] ? true : false;
}


function UpdateInfo__EstablishExcludedFolders() {
	var folderExclude = this.folderExclude = new Array;
	var path = "";

	// Program Files
	try {
		path = Toolbox.ShellFolderPath(CSIDL_PROGRAM_FILES);
		if (path) {
			key = path.toUpperCase();
			folderExclude[key] = new MonitorState(path);
		}
	}
	catch (ex) {
		// absorb
	}

	// System Root (Windows)
	try {
		path = Toolbox.ShellFolderPath(CSIDL_WINDOWS);
		if (path) {
			key = path.toUpperCase();
			folderExclude[key] = new MonitorState(path);
		}
	}
	catch (ex) {
		// absorb
	}

	// System Drive (usually C:)
	try {
		path = Toolbox.ExpandEnvironmentStrings(SYSTEMDRIVE);
		if (!path)
			path = SYSTEMDRIVEDEFAULT;
		if (path) {
			key = path.toUpperCase();
			folderExclude[key] = new MonitorState(path);
		}
	}
	catch (ex) {
		// absorb
	}

}


function UpdateInfo__StoreChanges(cbStore) {

	// Clear out arrays so we can rebuild them.
	this.serverAdd = new Array;
	this.serverRemove = new Array;
	this.folderAdd = new Array;
	this.folderRemove = new Array;
	this.cbStore = cbStore;				// Callback once changes are stored.

	// Store server monitoring changes to the serverAdd/Remove arrays
	// as appropriate.
	if (this.addAllServers.post) {
		// Don't store any server monitoring changes if user chose
		// automatic aggregation.
	} // if this.addAllServers.post
	else {
		// User chose to manually select servers, so comply with selections
		for (var key in this.serverState) {
			var monitorState = this.serverState[key];
			if (monitorState != null) {
				var serverObj = monitorState.server;
				if (serverObj == null)
					continue;

				if (this.addAllServers.pre) {
					if (aggregator.isAggregated(serverObj)) {
						if (monitorState.post == false) {
							this.serverRemove.push(serverObj);
						}
					}
					if (monitorState.post == true) {
						this.serverAdd.push(serverObj);
					}										
				}
				else {
					if (aggregator.isAggregated(serverObj)) {
						if (monitorState.post == false) {
							this.serverRemove.push(serverObj);
						}
					}
					else {
						if (monitorState.post == true) {
							this.serverAdd.push(serverObj);
						}
					} // else
				}
			} // if monitorState
		} // for
	}
	
	// Store folder monitoring changes to the folderAdd/Remove arrays
	// as appropriate.
	for (var key in this.folderState) {
		var monitorState = this.folderState[key];
		if (monitorState != null) {
			var fileObj = fileServices.byURL2(
				Path2Url(monitorState.path), 
				M1_FILE_FLAG_ALWAYS
				);
			if (!fileObj)
				continue;

			if (monitorState.pre == true) {
				if (monitorState.post != true) {
					this.folderRemove.push(fileObj);
				}
			}
			else {
				if (monitorState.post == true) {
					this.folderAdd.push(fileObj);
				}
			} // else
		} // if monitorState
	} // for
	
	// Call the client's post-StoreChanges callback.
	if (this.cbStore)
		this.cbStore();
	
} // UpdateInfo__StoreFolders()


// UpdateInfo__StoreMediaPcs()
// Stores a list of all PCs on which media is being tracked by the CDS,
// using the monitors information.
function UpdateInfo__StoreMediaPcs() {

	this.mediaPcs = new Array;		// String[] (device names)
	this.serverState = new Array;	// MonitorState[]
	this.folderState = new Array;	// MonitorState[]
	this.updatedCustom = false;		// User made custom selections

	// Refresh our monitors server/folderState to see what media PCs we have now.
	this.FetchMonitors();

	// Populate media PCs based on server monitor list.
	for (var serverKey in this.serverState) {
		try {
			if (!this.serverState[serverKey].pre)
				// This server isn't aggregated; continue.
				continue;
			if (!this.serverState[serverKey].label)
				continue;
			var devName = this.serverState[serverKey].label;
			var devKey = devName.toUpperCase();
			if (!this.mediaPcs[devKey]) {
				this.mediaPcs[devKey] = devName;
			}
			// If at this point, user made additional custom selection to
			// aggregate a media server.
			this.hasCustomSelections = true;
		}
		catch (e) {
		}
	}
	
	// Populate media PCs based on folder monitor list.
	for (var folderKey in this.folderState) {
		try {
			var path = this.folderState[folderKey].path;
			if (!path)
				continue;
			if (!this.IsDefaultFolder(path))
				// User made additional custom selections to
				// share a non-default folder.
				this.hasCustomSelections = true;
			var devName = Path2DeviceName(path);
			var devKey = devName.toUpperCase();
			// Try to retrieve a friendly name, if we have created a device list.
			if (deviceList) {
				var device = deviceList.GetDeviceByName(devName);
				if (device) {
					devName = device.name;
					devKey = devName.toUpperCase();
					if (device.friendly) {
						if (device.friendly != "") {
							this.mediaPcs[devKey] = device.friendly;
							continue;
						}
					}
				}
			}
			// If we didn't get a friendly device name, use what we have.
			if (!this.mediaPcs[devKey]) {
				this.mediaPcs[devKey] = devName;
			}
		}
		catch (e) {
		}
	}


}


// UpdateInfo__StoreMediaPcsNew()
// Stores a list of all PCs media was newly searched on in the mediaPcs hash, given
// an array of Urls that were synchronized.
function UpdateInfo__StoreMediaPcsNew(addUrls) {

	for (var i = 0; i < addUrls.length; i++) {
		var devName = Url2DeviceName(addUrls[i]);
		var devKey = devName.toUpperCase();
		// Try to retrieve a friendly name, if we have created a device list.
		if (deviceList) {
			var device = deviceList.GetDeviceByName(devName);
			if (device) {
				devName = device.name;
				devKey = devName.toUpperCase();
				if (device.friendly) {
					if (device.friendly != "") {
						this.mediaPcs[devKey] = device.friendly;
						continue;
					}
				}
			}
		}
		// If we didn't get a friendly device name, use what we have.
		if (!this.mediaPcs[devKey]) {
			this.mediaPcs[devKey] = devName;
		}		
	}
}

MonitorState.prototype.toString = MonitorState__toString;

// MonitorState constructor
// Initialize a MonitorState object, used by some of the hash tables in
// UpdateInfo.
function MonitorState(path) {
	this.pre = false;
	this.post = false;
	this.mid = false;
	this.postPending = false;
	this.path = path;
	this.label = "";
}

function MonitorState__toString() {
	if (!this.path)
		return "";
	if (!this.path.indexOf('://'))
		// just a pathname
		return this.path.toUpperCase();
	// Server name, requires special sort string
	return this.label.toUpperCase();
}



//------------------------------------------------------------------------------
// class MXException
// MediaOne exceptions.
//------------------------------------------------------------------------------

function MXException(errorSource)
{
  this.errorCode = errorSource.error.result;
  this.errorSource = errorSource;
  this.errorString = errorSource.error.action;
}

function mxThrow(obj)
{
  throw new MXException(obj);
}

function mxThrowIfFailed(obj)
{
  if (obj.error.result != MERROR_OK)
    throw new MXException(obj);
}



//------------------------------------------------------------------------------
// Utility functions
//------------------------------------------------------------------------------

function Path2Url(path) {
	if (!path)
		return "";
	return 'file://' + path;
}

function Url2Path(url) {
	if (!url)
		return "";
	var urlPieces = url.split('://');
	var path = (urlPieces.length > 1) ? urlPieces[1] : url;
	return path;
}

function StripTrailSlash(path) {
	if (!path)
		return "";
	if (path.lastIndexOf('\\') == (path.length-1))
		return path.slice(0, -1);
	else
		return path;
}

function NormalizePath(path)
{
  path = path.replace(/\\/g, '/');
  if (path.charAt(path.length - 1) != '/')
    path += '/';
  return path;
}

function Url2DeviceName(url) {
	if (!url)
		return '';
	var path = Url2Path(url);
	return Path2DeviceName(path);
}

// Cheap kludge to avoid making CDS server names a special case
// apart from regular file share devices
function ServerName2DeviceUrl(serverName) {
	if (!serverName)
		return '';
	return 'file:////' + ServerName2DeviceName(serverName);
}

// Convert UPnP friendly server name to device name
// Example:
// Intel Media Server (DEEZ)
// converts to
// DEEZ
function ServerName2DeviceName(serverName) {
	if (!serverName)
		return '';
	var begin = serverName.lastIndexOf('(') + 1;
	var end = serverName.lastIndexOf(')');
	if ((begin <= 0) || (end < 0) || (begin >= end))
		return serverName;
	return serverName.substring(begin, end);
}

function ServerName2CdsName(serverName) {
	if (!serverName)
		return '';
	var end = serverName.lastIndexOf('(');
	if (end <= 0)
		return serverName;
	return serverName.substring(0, end);
}

function Path2DeviceName(path) {
	if (!path)
		return '';
	if (path.indexOf(':') == 1)
		// local file, starts with drive reference
		return 'localhost';
	var pathPieces = path.split('\\');
	// 2 slashes before device name
	if (pathPieces.length > 2)
		return pathPieces[2];
	else
		return '';
}

function Path2FileName(path) {
	var fileName = '';
	if (!path)
		return '';
	var lastIndexSlash = path.lastIndexOf('\\');
	var len = path.length;
	// If no slashes, this is already a filename.
	if (lastIndexSlash == -1)
		return path;
	// if this is a trailing slash, strip it.
	if (lastIndexSlash == len - 1) {
		fileName = path.substring(0, lastIndexSlash);
		lastIndexSlash = path.lastIndexOf('\\');
		// If no slashes, this is now a filename.
		if (lastIndexSlash == -1)
			return fileName;
	}
	else
		fileName = path;
	return fileName.substring(lastIndexSlash+1, path.length);
}

function ServerUrl2Ip(url) {
	var ip = '';
	if (!url)
		return '';
	var indexIpStart = url.indexOf('://');
	if (indexIpStart == -1)
		return '';
	ip = url.substring(indexIpStart+3, url.length);
	var indexIpColon = ip.indexOf(':');
	var indexIpSlash = ip.indexOf('/');
	if (indexIpColon == -1 && indexIpSlash == -1)
		return ip;
	else if (indexIpSlash == -1)
		return ip.substring(0, indexIpColon);
	else if (indexIpColon == -1)
		return ip.substring(0, indexIpSlash);
	else if (indexIpColon < indexIpSlash)
		return ip.substring(0, indexIpColon);
	else	
		return ip.substring(0, indexIpSlash);
}