// treelist.js - Content Manager tree list support
//
// INTEL CONFIDENTIAL
//	Copyright 2004-2006 Intel Corporation All Rights Reserved.  The source code
//	contained or described herein and all documents related to the source code
//	(Material) are owned by Intel Corportion or its suppliers or licensors.
//	Title to the Material remains with Intel Corporation or its suppliers and
//	licensors. The Material may contain trade secrets and proprietary and
//	confidential information of Intel Corporation and its suppliers and
//	licensors, and is protected by worldwide copyright and trade secret laws
//	and treaty provisions. No part of the Material may be used, copied,
//	reproduced, modified, published, uploaded, posted, transmitted, distributed,
//	or disclosed in any way without Intels prior express written permission. 
//	
//	No license under any patent, copyright, trade secret or other intellectual
//	property right is granted to or conferred upon you by disclosure or delivery
//	of the Materials, either expressly, by implication, inducement, estoppel or
//	otherwise. Any license under such intellectual property rights must be express
//	and approved by Intel in writing.
//
//	Unless otherwise agreed by Intel in writing, you may not remove or alter this
//	notice or any other notice embedded in Materials by Intel or Intels suppliers
//	or licensors in any way.
//



// Dependencies on globals ("extern")
// extern var updateInfo;


// TreeList control constants
// Checkbox status for each TreeItem
var CHECKED_OFF = 0;
var CHECKED_ON = 1;
var CHECKED_ANCESTOR = 2;
// Subcontrol focus for TreeList
var TREEFOCUS_CHECKBOX = 0;
var TREEFOCUS_PLUSMINUS = 1;

// FILESYSTEM specific constants
var FSTYPE_DIRECTORY =			0x000000F0;		// Group of FSTYPEs, all can be enumerated as a folder.
var FSTYPE_FOLDER =				0x00000010;		// Generic folder on local machine
var FSTYPE_DRIVE =				0x00000020;		// Drive types
var FSTYPE_DRIVE_HARD =			0x00000021;
var FSTYPE_DRIVE_CD	=			0x00000022;
var FSTYPE_DRIVE_DVD =			0x00000023;
var FSTYPE_DRIVE_REMOVABLE =	0x00000024;
var FSTYPE_SHARE =				0x00000040;		// Network share.
var FSTYPE_FOLDER_SHARE =		0x00000041;		// Folder subordinate to a share.



//------------------------------------------------------------------------------
// class TreeItem
//------------------------------------------------------------------------------
TreeItem.prototype = new BaseItem();
TreeItem.prototype.constuctor = TreeItem.prototype.TreeItem = TreeItem;
TreeItem.prototype.AppendChild = TreeItem__AppendChild;
TreeItem.prototype.Check = TreeItem__Check;
TreeItem.prototype.Expand = TreeItem__Expand;
TreeItem.prototype.GetNextDisplayItem = TreeItem__GetNextDisplayItem;
TreeItem.prototype.HasChildren = TreeItem__HasChildren;
TreeItem.prototype.IsSelectable = TreeItem__IsSelectable;
TreeItem.prototype.IsCheckboxShown = TreeItem__IsCheckboxShown;
TreeItem.prototype.Update = TreeItem__Update;


// TreeItem constructor
// Initialize an independent TreeItem, based on a BaseItem.
function TreeItem(label, info) {

	this.depth = 0;
	this.expanded = false;
	this.checked = CHECKED_OFF;

	this.nextItem = null;
	this.previousItem = null;

	this.parent = null;
	this.nextSibling = null;
	this.previousSibling = null;
	this.firstChild = null;
	this.lastChild = null;

	this.label = label;
	this.info = info;

}


// TreeItem__AppendChild()
// Append the given TreeItem as a child of this TreeItem, adjusting all necessary
// pointers.  Two sets of pointers are adjusted--tree relationship pointers
// and flat list pointers.
function TreeItem__AppendChild(newItem) {

	var newNextItem = null;	// save the value of applicable next item before we overwrite it

	// Obvious item settings.
	newItem.parent = this;
	newItem.list = this.list;
	newItem.depth = this.depth + 1;
	newItem.checked = this.checked ? CHECKED_ANCESTOR : CHECKED_OFF;
	
	if (!this.firstChild) {
		// The newItem is the current item's first child.
		
		this.firstChild = newItem;
		
		// Set flat list pointers.
		newNextItem = this.nextItem;	// The newItem's nextItem will be set
										// to the current item's nextItem in the flat list,
										// since the current item has no children.
		this.nextItem = newItem;		// Now that the current item's nextItem
										// has been saved, we can set the current
										// item's nextItem in the flat list
										// to be the newItem (first child).
		newItem.previousItem = this;	// Since the current item had no children
										// before, we know that the newItem's
										// previousItem in the flat list should
										// be the current item (newItem's parent).
	}
	else {
		// Children already exist in this tree.
		// Flat list modifications now become much more complicated.
		
		// Find the closest ancestor with a nextSibling, as
		// that sibling will be newItem's nextItem in the flat list.
		var curAncestor = this;
		while (!curAncestor.nextSibling) {
			curAncestor = curAncestor.parent;
			if (!curAncestor) {
				break;
			}
		}
		newNextItem = curAncestor ? curAncestor.nextSibling : null;
		
		// Find the farthest, rightmost descendant of
		// the current item (before newItem is appended)
		// as that will be newItem's previousItem in the flat
		// list, and we will need to set that item's nextItem
		// to newItem.
		var curDescendant = this.lastChild;
		while (curDescendant.lastChild) {
			curDescendant = curDescendant.lastChild;
		}
		curDescendant.nextItem = newItem;
		this.lastChild.nextSibling = newItem;
		newItem.previousItem = curDescendant;
		
		// newItem's previousSibling will always be 
		// the old lastChild.
		newItem.previousSibling = this.lastChild;
	}


	newItem.nextItem = newNextItem;
	if (newItem.nextItem) {
		newItem.nextItem.previousItem = newItem;
	}
	
	// Since we are appending newItem, we know that this item's lastChild
	// should be newItem.
	this.lastChild = newItem;

}


// TreeItem__Check()
// Check/uncheck this TreeItem's checkbox, and all descendants, based
// on the last checkbox value.  Update the display.
function TreeItem__Check(noAction) {

	if (!this.IsSelectable())
		return;
	
	this.UpdateStart();

	var lastChecked = this.checked;
	
	this.checked = this.checked ? CHECKED_OFF : CHECKED_ON;
	if (this.CheckAction && !noAction)
		this.CheckAction(lastChecked);
	
	var curItem = this.nextItem;
	if (curItem) {
		var childCheck = this.checked ? CHECKED_ANCESTOR : CHECKED_OFF;
		while (curItem.depth > this.depth) {
			lastChecked = curItem.checked;
			curItem.checked = childCheck;
			if (curItem.CheckAction && !noAction)
				curItem.CheckAction(lastChecked);
			curItem = curItem.nextItem;
			if (!curItem)
				break;
		}
	}


	if (this.list.myDocsFs) {
		if (this.info.indexOf(this.list.myDocsPath) != -1) {
			// Selection has changed for a mirrored node, so update the list.
			this.list.RefreshSelections();
		}	
	}

	// Selection has changed; update status of action button if applicable.
	if (this.list.CheckSelection)
		this.list.CheckSelection();

	this.UpdateFinish();
}


// TreeItem__Expand()
// Expand/collapse this TreeItem's children, based on the last value
// of the expanded flag.
function TreeItem__Expand() {

	if (!this.HasChildren())
		return;
		
	this.UpdateStart();
	this.expanded = !this.expanded;

	if (this.expanded) {
		if (this.PopulateChildren && !this.fetchStarted) {
			// Execute only if a dynamic PopulateChildren function exists and the
			// population hasn't started yet.
			this.PopulateChildren();
		}
	}

	this.list.lengthVisible = -1;	// Force recalculation of lengthVisible.
	this.UpdateFinish();
}


// TreeItem__GetNextDisplayItem()
// Return the next displayable item after this item.  In other words,
// return the next item in the TreeList that isn't hidden by
// a collapsed ancestor.
function TreeItem__GetNextDisplayItem() {

	if (this.expanded)
		return this.nextItem;
	if (this.nextSibling)
		return this.nextSibling;
			
	// Find the closest ancestor with a nextSibling, because
	// that nextSibling will be the next displayable item in the flat list.
	var curAncestor = this.parent;
	if (!curAncestor)
		return null;
	while (!curAncestor.nextSibling) {
		curAncestor = curAncestor.parent;
		if (!curAncestor) {
			break;
		}
	}
	return (curAncestor ? curAncestor.nextSibling : null);
}


// TreeItem__HasChildren()
// Returns true if this TreeItem has children, false otherwise.
function TreeItem__HasChildren() {
	return this.firstChild ? true : false;
}


// virtual
// TreeItem__IsCheckboxShown()
// Return true if this TreeItem is to have a checkbox shown.
// Always returns true.
function TreeItem__IsCheckboxShown() {
	return true;
}


// TreeItem__IsSelectable()
// Return true if this TreeItem is selectable, false otherwise.
// The condition for selectability of the basic TreeItem is that
// an ancestor of this item isn't selected.
function TreeItem__IsSelectable() {
	if (this.checked == CHECKED_ANCESTOR)
		return false;
	return true;
}


// TreeItem__Update()
// Generate the HTML for displaying this TreeItem.
function TreeItem__Update() {

	// Set up appropriate checkbox class & mouse handler.
	var cbox = '';
	var cboxMouse = '';
	if (this.IsCheckboxShown()) {
		var cbox = '';
		var cboxSelect = (this.list.subFocus == TREEFOCUS_CHECKBOX) ? 'Select' : '';
		switch (this.checked) {
			case CHECKED_ON:
				cbox = 'class="TreeCheckOn' + cboxSelect + '"';
				break;
			case CHECKED_ANCESTOR:
				cbox = 'class="TreeCheckOnInactive"';
				break;
			case CHECKED_OFF:
			default:
				cbox = 'class="TreeCheckOff' + cboxSelect + '"';
				break;
		}
		// Mouse event handler to assign focus to checkbox if mouse passes over.
		// Pass the handler the cell this checkbox is contained in
		// and the fact that a checkbox was moused into.
		var cboxMouse = (this.checked == CHECKED_ON || this.checked == CHECKED_OFF) 
			? 'onMouseEnter="TreeList__MouseEnter(this.parentNode.widgetIndex, TREEFOCUS_CHECKBOX);"'
				+ 'onMouseOut="TreeList__MouseOut(this.parentNode.widgetIndex, TREEFOCUS_CHECKBOX);"'
			: '';
	}
	else {
		cbox = 'style="width:40px; height:40px"';
	}
			
	// Set up appropriate plus/minus class & mouse handler.
	var plus = '';
	var plusSelect = this.list.subFocus == TREEFOCUS_PLUSMINUS ? 'Select' : '';
	if (this.expanded)
		plus = 'class="TreeMinus' + plusSelect + '"';
	else
		plus = this.HasChildren() ? 'class="TreePlus' + plusSelect + '"' : 
			'style="width:40px; height:40px"';
	// Mouse event handler to assign focus to plus/minus if mouse passes over.
	// Pass the handler the cell this plus/minus is contained in
	// and the fact that a plus/minus was moused into.
	var plusMouse = (this.expanded || this.HasChildren())
		? 'onMouseEnter="TreeList__MouseEnter(this.parentNode.widgetIndex, TREEFOCUS_PLUSMINUS);"'
			+ 'onMouseOut="TreeList__MouseOut(this.parentNode.widgetIndex, TREEFOCUS_PLUSMINUS);"'
		: '';

	// If this TreeItem has an IconClass() function defined, get the class name
	// for the icon associated with this TreeItem.
	var iconClass = this.IconClass ? this.IconClass() : '';

	// Truncate the label as necessary
	var rect = this.list.grid.elem.getBoundingClientRect();
	var zoom = document.body.style.zoom > 0 ? 
		document.body.style.zoom : 1;
	if (zoom != truncateLastZoom) {
		// zoom change, re-init truncation
		truncateLastZoom = zoom;
		TruncateInitialize();
	}
	var width = rect.right - rect.left;
	width -= 180 * zoom;
	// Truncation is dependent on depth/indentation of the item
	width -= (this.depth-1) * 36 * zoom;
	var content = Truncate(this.label, width);

	// Actual HTML we will provide for this TreeItem.
	var inner = '<span ' + cboxMouse + ' ' + cbox + ' subcontrol></span>';
	inner += '<span style="padding-left:' + ((this.depth-1) * 40 + 10) + '"></span>';
	inner += '<span ' + plusMouse + ' ' + plus 
		+ ' subcontrol></span>';
	inner += '<span class="' + iconClass + '" style="margin-left:10px"></span>';
 	inner += '<span style="padding-left:10px;vertical-align:middle">' + content + '</span>';

	return inner;
}



//------------------------------------------------------------------------------
// class TreeList
//------------------------------------------------------------------------------
TreeList.prototype = new BaseList();
TreeList.prototype.constuctor = TreeList.prototype.TreeList = TreeList;
TreeList.prototype.Clean = TreeList__Clean;
TreeList.prototype.Redraw = TreeList__Redraw;
TreeList.prototype.SetDisplay = TreeList__SetDisplay;
TreeList.prototype.GetLengthVisible = TreeList__GetLengthVisible;
TreeList.prototype.UpdateRow = TreeList__UpdateRow;
// Callbacks for TreeList's associated grid.
TreeList.prototype.Action = TreeList__Action;
TreeList.prototype.Focus = TreeList__Focus;
TreeList.prototype.KeyStep = TreeList__KeyStep;
TreeList.prototype.Update = TreeList__Update;


// TreeList constructor
// Allocate and initialize a TreeList, based on a BaseList.
function TreeList(grid, info) {

	this.subFocus = TREEFOCUS_CHECKBOX;
	this.subFocusAction = false;
	this.subFocusClicked = false;
	this.lengthVisible = 0;

	this.root = this.startItem = new TreeItem("", "");
	this.root.list = this;
	this.root.expanded = true;

	if (grid)
		this.SetDisplay(grid, info);

}


// TreeList__Action()
// Handler that executes when a button is pressed and a cell in the TreeList's
// grid is in focus, so carry out the appropriate action on the associated item.
//
// cell		Cell widget that received focus
// list		Data associated with widget
function TreeList__Action(cell, list)
{
	if (!list.subAction && list.subActionClicked) {
		list.subActionClicked = false;
		return;
	}
	var item = cell.item;
	if (list.subFocus == TREEFOCUS_CHECKBOX) {
		item.Check();
	}
	else if (list.subFocus == TREEFOCUS_PLUSMINUS) {
		item.Expand();
	}
}


// TreeList__Clean()
// Clean up all circular references created by the TreeList.
function TreeList__Clean() {

	// Visually clean TreeList.
	GridSetViewport(this.grid, 0, 0,
		this.grid.viewCollapse, this.grid.viewEscape);
	this.grid.viewPane.stats.label.innerHTML = "";
	this.info.elem.innerHTML = "";
				
	// Remove cell backreferences to TreeItems.
	for (var row = 0; row < this.grid.navRows; row++) {
		this.grid.navGrid[row][0].elem.innerHTML = "";
		this.grid.navGrid[row][0].item = null;
		this.grid.navGrid[row][0].elem.onclick = null;
		this.grid.navGrid[row][0].elem.onmouseover = null;
	}
	
	// Remove grid backreference to this TreeList.
	this.grid.list = null;

	// Clear out all pointers in the entire TreeList.
	var item = this.startItem;
	var nextItem = null;
	while (item) {
		nextItem = item.nextItem;
		
		item.nextItem = null;
		item.previousItem = null;

		item.parent = null;
		item.nextSibling = null;
		item.previousSibling = null;
		item.firstChild = null;
		item.lastChild = null;

		item = nextItem;
	}
	
}


// TreeList__Focus()
// Handler that executes when a cell in the TreeList's grid receives focus.
//
// cell		Cell widget that received focus
// focus	Whether cell received focus
// list		Data associated with widget
function TreeList__Focus(cell, focus, list)
{
	
	if (!list)
		// This focus trigger wasn't caused by an item in our list.
		return;

	var stats = list.grid.viewPane.stats;
	var index = list.grid.viewFirst+(cell.navRow*list.grid.navCols)+cell.navCol;
	var grid = cell.navTable;	// Retrieve cell's parent grid.
	var item = cell.item;		// Retrieve TreeItem associated with cell.

	if (focus && item) {

		if (stats) {
			var rect = list.grid.elem.getBoundingClientRect();
			var width =  rect.right - rect.left - 200;
			if (width != truncateLastWidth) {
				// width change, re-init truncation
				truncateLastWidth = width;
				TruncateInitialize();
			}
			list.info.elem.innerHTML = TruncatePath(cell.item.info, width);
			statsInner = (index+1)+" "+languageGridOf+" "+list.GetLengthVisible();
			stats.label.innerHTML = statsInner;
		}
	
		// prevent nav to non-existent check
		if (list.subFocus == TREEFOCUS_CHECKBOX && !item.IsSelectable()) {
			list.subFocus = TREEFOCUS_PLUSMINUS;
		}
		else if (list.subFocus == TREEFOCUS_PLUSMINUS && 
				!item.HasChildren() && !item.expanded) {
			list.subFocus = TREEFOCUS_CHECKBOX;
		}

		list.UpdateRow(cell.navRow, cell.item);
	}
	else {
		// Lost focus, kill breadcrumb / stats.
		list.info.elem.innerHTML = "";
		stats.label.innerHTML = "";
	}

}


// TreeList__GetLengthVisible()
// Get length of this TreeList, counting only visible items (some may be invisible due to collapsed items).
function TreeList__GetLengthVisible() {

	// If we have a valid cached value for length of visible item list, use it.
	if (this.lengthVisible != -1)
		return this.lengthVisible;

	var curItem = this.startItem.nextItem;
	var count = 0;
	var list = this;
	
	while (curItem) {
		// Jump to next item/index.
		curItem = curItem.GetNextDisplayItem();
		count++;	
	}
	
	this.lengthVisible = count;
	return count;
	
}

// TreeList__KeyStep()
// Handler that executes when a key is pressed and a cell in the TreeList grid
// has focus.  Only captures left/right keyboard/remote events.
//
// cell		Cell in which key was pressed
// step		Direction of key (<0 if up/left, >0 if down/right)
// vert		True if key was for vertical nav (up/down)
function TreeList__KeyStep(cell, step, vert)
{

	// Only handle left-right (non-vertical) navigation.
	if (!vert) {
		var grid = cell.navTable;	// Retrieve cell's parent grid.
		var list = grid.list;		// Retrieve TreeList associated with grid.
		var item = cell.item;		// Retrieve TreeItem associated with cell.

		// If left was pressed, we are currently focused on the plus/minus sign, 
		// and the checkbox for this item is selectable, put focus on checkbox.
		if (step < 0 && list.subFocus == TREEFOCUS_PLUSMINUS && item.IsSelectable()) {
			list.subFocus = TREEFOCUS_CHECKBOX;
			list.UpdateRow(cell.navRow, cell.item);
			return true;
		}

		// If right was pressed, we are currently focused on the checkbox,
		// and this item has a plus/minus sign (has children), put focus on plus/minus.
		else if (step > 0 && list.subFocus == TREEFOCUS_CHECKBOX && item.HasChildren()) {
			list.subFocus = TREEFOCUS_PLUSMINUS;
			list.UpdateRow(cell.navRow, cell.item);
			return true;
		}
	}
	
	// Pass vertical navigation up the chain.
	return false;
}


// TreeList__Click()
// onClick handler for tree list items.  Sets a flag that
// a mouse was clicked to execute an action, then performs
// normal widget Click functionality.
function TreeList__Click() {
	try {
		var cell = widgetNavIndex[this.widgetIndex];
		if (cell) {
			var list = cell.navTable.list;
			if (list)
				list.subActionClicked = true;
			WidgetKeyAction(cell);
		}
	}
	catch (e) {
	}
}


// TreeList__MouseOver()
// onMouseOver handler for tree list items.  Determines whether
// mouse is currently over a subcontrol, then performs normal
// widget MouseOver functionality.
function TreeList__MouseOver() {
	try {
		var cell = widgetNavIndex[this.widgetIndex];
		if (cell) {
			var list = cell.navTable.list;
			if (list) {
				list.subActionClicked = false;
				if (window.event.toElement.subcontrol)
					list.subAction = true;
				else
					list.subAction = false;
			}
			if (cell != widgetNavFocus) 
				WidgetSetFocus(cell); 
		}
	}
	catch (e) {
	}
}


// TreeList__MouseEnter()
// onMouseEnter handler for the checkbox and plus/minus subcontrols.
// If the cursor enters one of those subcontrols and the subcontrol is
// selectable, highlight that subcontrol.  Also enable it for action.
function TreeList__MouseEnter(widgetIndex, subFocus) {
	try {
		var cell = widgetNavIndex[widgetIndex];
		if (cell) {
			var list = cell.navTable.list;
			if (list) {
				list.subAction = true;
				if (subFocus == list.subFocus)
					return;
				list.subFocus = subFocus;
				list.UpdateRow(cell.navRow, cell.item);
			}
		}
	}
	catch (e) {
	}
}


// TreeList__MouseOut()
// onMouseOut handler for the checkbox and plus/minus subcontrols.
// If the cursor exits a subcontrol, disable it for action.
function TreeList__MouseOut(widgetIndex, subFocus) {
	try {
		var cell = widgetNavIndex[widgetIndex];
		if (cell) {
			var list = cell.navTable.list;
			if (list)
				list.subAction = false;
		}
	}
	catch (e) {
	}
}


// TreeList__Redraw()
// Redraw this TreeList.
//
// Need a different redraw function because our data structure is different from BaseList.
function TreeList__Redraw() {
	if (this.grid)
		GridSetViewport(this.grid, this.grid.viewFirst, this.GetLengthVisible(),
				this.grid.viewCollapse, this.grid.viewEscape);
}


// TreeList__SetDisplay()
// Initialize grid/info display associated with this TreeList.  Set up references
// to the grid and the info display in the TreeList object, and assign callback
// functions for each cell.
//
// Override BaseList's SetDisplay because TreeList requires assigning a KeyStep
// callback, which BaseList does not do.
function TreeList__SetDisplay(grid, info)
{
	this.UpdateStart();
	this.grid = grid;
	this.info = info;

	grid.list = this;
	grid.cbUpdate = this.Update;
	
	for (var row = 0; row < grid.navRows; row++)
	{
		for (var col = 0; col < grid.navCols; col++)
		{
			grid.navGrid[row][col].cbAction = this.Action;
			grid.navGrid[row][col].cbFocus = this.Focus;
			grid.navGrid[row][col].cbKeyStep = this.KeyStep;	// Changed from BaseList
			grid.navGrid[row][col].cbData = this;
		}
	}
	this.UpdateFinish();
}


// TreeList__Update(grid, start, count)
// static (no concept of "this")
// Update the display for this TreeList.
//
// grid		Grid on which the TreeList is displayed
// start	Start row in grid to update (NOT start index in data structure)
// count	Number of rows to update
function TreeList__Update(grid, start, count) {

	var index = 0;
	var startIndex = start + grid.viewFirst;	// Start index in data structure
	var endIndex = startIndex + count;
	var list = grid.list;
	var curItem = list.startItem.nextItem;
	var row = start;
	
	while (curItem && index < endIndex) {

		// If item falls within viewport, actually put its display
		// in the scroll.
		if ((index >= startIndex) && (index < endIndex)) {
			list.UpdateRow(row, curItem);
			row++;
		}
		
		curItem = curItem.GetNextDisplayItem();
		index++;
		
	}

}


// TreeList__UpdateRow()
// Update the contents of an individual row.  Used by TreeList__Focus(),
// TreeList__KeyStep(), TreeList__MouseEnter(), and TreeList__Update().
function TreeList__UpdateRow(row, item) {
	var inner = item.Update();				// Get refreshed cell contents.
	ScrollSetRow(this.grid, row, inner,		// Set cell contents.
		this.Action, null, this);
	this.grid.navGrid[row][0].item = item;	// Attach the item to the cell.
	// Custom mouse click handler to intercept clicks then pass them to normal widget handler.
	this.grid.navGrid[row][0].elem.onclick = TreeList__Click;
	this.grid.navGrid[row][0].elem.onmouseover = TreeList__MouseOver;
}



//------------------------------------------------------------------------------
// class FsTreeItem
// A TreeItem representing an FsFolder object.
//------------------------------------------------------------------------------
FsTreeItem.prototype = new TreeItem("", "");
FsTreeItem.prototype.constructor = FsTreeItem.prototype.FsTreeItem = FsTreeItem;
FsTreeItem.prototype.CheckAction = FsTreeItem__CheckAction;
FsTreeItem.prototype.HasChildren = FsTreeItem__HasChildren;
FsTreeItem.prototype.IconClass = FsTreeItem__IconClass;
FsTreeItem.prototype.IsCheckboxShown = FsTreeItem__IsCheckboxShown;
FsTreeItem.prototype.IsSelectable = FsTreeItem__IsSelectable;
FsTreeItem.prototype.PopulateChildren = FsTreeItem__PopulateChildren;



// FsTreeItem constructor
// Allocate and initialize an independent filesystem TreeItem.
function FsTreeItem(label, info, folder) {
	this.TreeItem(label, info);

	// Filesystem-specific properties.
	this.folder = folder;
	this.key = folder.path.toUpperCase();
	this.fetchIndex = 0;
	this.fetchStarted = false;
	this.fetchFinished = false;	
}


// FsTreeItem__CheckAction()
// Perform updates to media update structures when this item is checked/unchecked.
function FsTreeItem__CheckAction(lastCheckedState) {

	var monitorState = null;
	if (this.checked == CHECKED_ANCESTOR) {
		return;
	}

	if (this.checked == CHECKED_ON) {
		if (!(monitorState = updateInfo.folderState[this.key])) {
			monitorState = new MonitorState(this.folder.path);
			updateInfo.folderState[this.key] = monitorState;
			monitorState.pre = false;
			monitorState.mid = false;
			monitorState.label = this.folder.name;
		}
		monitorState.postPending = true;
	}
	else { // CHECKED_OFF

		if (monitorState = updateInfo.folderState[this.key]) {
			if (lastCheckedState == CHECKED_ANCESTOR) {
				if (monitorState.postPending) {
					// For the case where this was previously selected before
					// an ancestor was selected, keep it selected.
					this.checked = CHECKED_ON;
				}
			}
			else {
				// If this was directly unchecked, or wasn't selected prior
				// to the ancestor being selected, unselect it.
				monitorState.postPending = false;
			}
		}
		
	}

}


// FsTreeItem__HasChildren()
// Returns true if this filesystem TreeItem has children, false otherwise.
// Differs from TreeItem__HasChildren() in that if children are not populated
// yet, we check for children in underlying FsFolder object.
function FsTreeItem__HasChildren() {
	var result = false;
	if (this.depth > 12) {
		// don't allow display deeper than 12
		return false;
	}
	if (this.firstChild) {
		result = true;
	}
	else {
		if (this.folder)
			result = this.folder.hasChildNodes();
	}
	return result;
}


// FsTreeItem__IconClass()
// Return item icon class.
function FsTreeItem__IconClass() {
	return this.folder
		? ((this.folder.type & FSTYPE_DRIVE)
		   ? 'driveIcon' 
		   : (this.expanded 
		      ? 'folderIconOpen'
		      : 'folderIconClosed'))
		: 'cdsIcon';
}


// FsTreeItem__IsCheckboxShown()
// Return true if this item is to have a checkbox shown.
// False if the item is one of the excluded folders.
function FsTreeItem__IsCheckboxShown() {
	if (updateInfo.folderExclude[this.key])
		return false;
	else
		return true;
}

// FsTreeItem__IsSelectable()
// Return true if this TreeItem is selectable, false otherwise.
// The condition for selectability of the filesystem TreeItem is that
// an ancestor of this item isn't selected, or the item isn't
// in the exclude list.
function FsTreeItem__IsSelectable() {
	if (this.checked == CHECKED_ANCESTOR || updateInfo.folderExclude[this.key])
		return false;
	return true;
}

// FsTreeItem__PopulateChildren()
// Dynamically populate the children of this item in the background.
function FsTreeItem__PopulateChildren() {
	if (!this.fetchStarted) {
		if (!this.folder.enumerateBackground()) {
			alert("Background enumeration of folder failed");
			return;
		}
		this.fetchStarted = true;
	}
	if (this.folder.error) {
		this.fetchFinished = true;
		if (this.folder.childNodes.length <= 0)
			this.expanded = false;
		this.Redraw();
		return;
	}
	if (!this.folder.complete) {
		if (this.fetchIndex >= this.folder.childNodes.length) {
			// we're not ready for this index yet, try later.
			FsTreeItem__PopulateChildrenHelper(this);
			return;
		}
	}
	
	var fetchFolder = this.folder.childNodes.item(this.fetchIndex);
	if (!fetchFolder) {
		this.Redraw();
		this.fetchFinished = true;
		return;
	}
	
	var newItem = new FsTreeItem(fetchFolder.name, fetchFolder.path, fetchFolder);
	this.AppendChild(newItem);
	
	// Override checkbox state of item based on whether it was preselected.
	// Skip items that are already checked by ancestry.
	var monitorState = null;
	if (!newItem.checked) {
		if (monitorState = updateInfo.folderState[newItem.key]) {
			// Turn checkbox value to on, if selection info exists for this path
			// and its "post" selection value is true.  Post means we will
			// be including the folder as a synchronized monitor when we are done
			// with our selections.
			if (monitorState.postPending) {
				newItem.checked = CHECKED_ON;
			}
		}
	}

	this.fetchIndex++;
	this.list.lengthVisible++;
	
	// Only do a screen refresh every 10 items, not every item.
	if ((this.fetchIndex % 10) == 0) {
		this.Redraw();
		FsTreeItem__PopulateChildrenHelper(this);
	}
	else {
		this.PopulateChildren();
	}
}

function FsTreeItem__PopulateChildrenHelper(item) {
	setTimeout(function () {item.PopulateChildren()});
}



//------------------------------------------------------------------------------
// class FsTreeList
// A TreeList of FsTreeItems, representing a filesystem.
//------------------------------------------------------------------------------
FsTreeList.prototype = new TreeList(null, null);
FsTreeList.prototype.constructor = FsTreeList.prototype.FsTreeList = FsTreeList;
FsTreeList.prototype.CheckSelection = FsTreeList__CheckSelection;
FsTreeList.prototype.CommitSelections = FsTreeList__CommitSelections;
FsTreeList.prototype.RefreshSelections = FsTreeList__RefreshSelections;


// FsTreeList constructor
// Initialize a FsTreeList, which is a slightly modified
// version of a TreeList that handles FsTreeItems and FsFolders
// rather than basic TreeItems.
//
// grid				Grid widget on which this FsTreeList will be displayed.
// info				Info label on which FsTreeItem extra info will be displayed.
// fs				Filesystem object from which to fetch data for this FsTreeList.
// actionButton		Action button widget to enable/disable when selection has changed
function FsTreeList(grid, info, actionButton, fs, pathMyDocuments) {

	this.TreeList(grid, info);

	this.actionButton = actionButton;
	
	// Update intermediate selection tracking values for monitors
	// so we can tell if a selection change has been made since this particular 
	// FsTreeList has been initialized.
	for (var key in updateInfo.folderState) {
		var monitorState = updateInfo.folderState[key];
		monitorState.mid = monitorState.post;
		monitorState.postPending = monitorState.post;
	}

	// Override the default startItem.
	this.root = this.startItem = new FsTreeItem("", "", fs.rootFolder);
	this.root.list = this;
	this.root.expanded = true;

	// Override the default startItem.
	if (pathMyDocuments) {
		var myDocsFs = new ActiveXObject("ShellUtilities.Filesystem");
		if (myDocsFs.initialize(pathMyDocuments)) {
			var myDocsFolder = myDocsFs.rootFolder;
			var myDocsLabel = myDocsFolder.path.substr(myDocsFolder.path.lastIndexOf("\\")+1);
			var myDocsItem = new FsTreeItem(myDocsLabel, myDocsFolder.path, myDocsFolder);
			this.root.AppendChild(myDocsItem);
			this.lengthVisible++;
			this.myDocsFs = myDocsFs; // Need reference to the filesystem object or else it will die.
			this.myDocsPath = pathMyDocuments;

			// Override checkbox state of item based on whether it was preselected.
			var monitorState = null;
			if (monitorState = updateInfo.folderState[myDocsItem.key]) {
				if (monitorState.postPending) {
					myDocsItem.checked = CHECKED_ON;
				}
			}
		}
	}

	this.root.PopulateChildren();

}


function FsTreeList__CheckSelection() {

	var changed = false;
	// Find at least one change in folder selection for
	// monitoring.
	for (var key in updateInfo.folderState) {
		var monitorState = updateInfo.folderState[key];
		if (monitorState.mid != monitorState.postPending) {
			changed = true;
			break;
		}
	}
	if (changed) {
		// If we have a change, enable the action button.
		WidgetSetNavigation(this.actionButton, 'leaf');
	}
	else {
		// No change, disable action button.
		WidgetSetNavigation(this.actionButton, 'inactive');
	}
}


function FsTreeList__CommitSelections() {
	// Commit all pending selections to actual selections.
	for (var key in updateInfo.folderState) {
		var monitorState = updateInfo.folderState[key];
		monitorState.post = monitorState.postPending;
	} // for (var key in updateInfo.folderState)
}


function FsTreeList__RefreshSelections() {

	var curItem = this.startItem.nextItem;
	while (curItem) {
		var monitorState = null;
		if (monitorState = updateInfo.folderState[curItem.key]) {
			if (curItem.checked == CHECKED_OFF && monitorState.postPending)
				curItem.Check(true);
			if (curItem.checked == CHECKED_ON && !monitorState.postPending)
				curItem.Check(true);
		}
		curItem = curItem.nextItem;
	}	
	
	this.Redraw();
}
