// DemoView.cpp : implementation of the CDemoView class
//

#include "stdafx.h"
#include <math.h>
#include <mmsystem.h>
#include "Demo.h"
#include "Scene.h"
#include "DemoView.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

inline D3DVALUE RND(D3DVALUE v) {
	return D3DVAL(rand())*v/RAND_MAX;
}

/////////////////////////////////////////////////////////////////////
// Global data and objects
CDemoView* g_AppView = NULL;
CScene Scene;

/////////////////////////////////////////////////////////////////////
// External function-prototypes
enum APPMSGTYPE {MSG_NONE, MSGERR_APPMUSTEXIT, MSGWARN_SWITCHTOSOFTWARE,
	MSGWARN_CANTDOFULLSCREEN};
void DisplayFrameworkError(HRESULT hr, APPMSGTYPE errType);

/////////////////////////////////////////////////////////////////////
// Name: FullScreenWndProc()
// Desc: The WndProc funtion used when the app is in fullscreen mode. This is
//		 needed simply to trap the ESC key.
LRESULT CALLBACK FullScreenWndProc(HWND hWnd, UINT uMsg, WPARAM wParam,
	LPARAM lParam)
{
	MSG Msg = {hWnd, uMsg, wParam, lParam};
	if ( TranslateAccelerator(hWnd, g_AppView->m_hAcc, &Msg) )
	{
		uMsg = Msg.message;
		wParam = Msg.wParam;
		lParam = Msg.lParam;
	}

	switch ( uMsg ) {
	case WM_KEYDOWN:
	case WM_COMMAND:
		return g_AppView->SendMessage(uMsg, wParam, lParam);
	default:
		return DefWindowProc(hWnd, uMsg, wParam, lParam);
	}
}

/////////////////////////////////////////////////////////////////////////////
// CDemoView

CDemoView::CDemoView()
{
	m_pFramework			= NULL;
    m_pDeviceInfo           = NULL;
    m_pFullScreenDeviceInfo = NULL;
	m_bReady				= FALSE;
	m_bPaused				= FALSE;
	m_bSingleStep			= FALSE;
	m_bWindowed 			= TRUE;
	g_AppView				= this;
	m_dwBaseTime			= timeGetTime();
	m_bTexture				= TRUE;
	m_bWireframe			= FALSE;
	m_hAcc = LoadAccelerators(AfxGetApp()->m_hInstance, MAKEINTRESOURCE(IDR_MAINFRAME));
}

CDemoView::~CDemoView()
{
	Scene.Release();
	Cleanup3DEnvironment();
}

BEGIN_MESSAGE_MAP(CDemoView, CWnd)
	//{{AFX_MSG_MAP(CDemoView)
	ON_WM_KEYDOWN()
	ON_COMMAND(MENU_FULLSCREEN, OnToggleFullscreen)
	ON_COMMAND(MENU_CHANGEMODE, OnChangeDevice)
	ON_COMMAND(MENU_PAUSE, OnPause)
	ON_UPDATE_COMMAND_UI(MENU_PAUSE, OnUpdatePause)
	ON_UPDATE_COMMAND_UI(MENU_FULLSCREEN, OnUpdateFullscreen)
	ON_COMMAND(MENU_EXIT, OnExit)
	ON_COMMAND(MENU_DITHERING, OnDithering)
	ON_UPDATE_COMMAND_UI(MENU_DITHERING, OnUpdateDithering)
	ON_COMMAND(MENU_LINEAR_FILTER, OnLinearFilter)
	ON_UPDATE_COMMAND_UI(MENU_LINEAR_FILTER, OnUpdateLinearFilter)
	ON_COMMAND(MENU_POINT_FILTER, OnPointFilter)
	ON_UPDATE_COMMAND_UI(MENU_POINT_FILTER, OnUpdatePointFilter)
	ON_COMMAND(MENU_MIPPOINT_FILTER, OnMippointFilter)
	ON_UPDATE_COMMAND_UI(MENU_MIPPOINT_FILTER, OnUpdateMippointFilter)
	ON_COMMAND(MENU_MIPLINEAR_FILTER, OnMiplinearFilter)
	ON_UPDATE_COMMAND_UI(MENU_MIPLINEAR_FILTER, OnUpdateMiplinearFilter)
	ON_COMMAND(MENU_LINEARMIPPOINT_FILTER, OnLinearmippointFilter)
	ON_UPDATE_COMMAND_UI(MENU_LINEARMIPPOINT_FILTER, OnUpdateLinearmippointFilter)
	ON_COMMAND(MENU_LINEARMIPLINEAR_FILTER, OnLinearmiplinearFilter)
	ON_UPDATE_COMMAND_UI(MENU_LINEARMIPLINEAR_FILTER, OnUpdateLinearmiplinearFilter)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CDemoView message handlers
BOOL CDemoView::PreCreateWindow(CREATESTRUCT& cs) 
{
	if (!CWnd::PreCreateWindow(cs))
		return FALSE;

	cs.dwExStyle |= WS_EX_CLIENTEDGE;
	cs.style &= ~WS_BORDER;
	cs.lpszClass = AfxRegisterWndClass(CS_HREDRAW|CS_VREDRAW|CS_DBLCLKS, 
		::LoadCursor(NULL, IDC_ARROW), HBRUSH(COLOR_WINDOW+1), NULL);

	return TRUE;
}

void CDemoView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags)
{
	switch ( nChar ) {
	// Movement
	case VK_RIGHT:
		Scene.m_CameraPos.x += 1.0f;
		break;
	case VK_LEFT:
		Scene.m_CameraPos.x -= 1.0f;
		break;
	case VK_UP:
		Scene.m_CameraPos.z += 1.0f;
		break;
	case VK_F5:
		Scene.m_b3DNow = !Scene.m_b3DNow;
		break;
	case VK_F2:
		m_bTexture = !m_bTexture;
		if ( m_bTexture )
		{
			Scene.m_Mesh->SetGroupTexture(0, (LPDIRECT3DRMTEXTURE)Scene.m_Texture);
			Scene.m_Mesh->SetGroupMapping(0, D3DRMMAP_WRAPU|D3DRMMAP_WRAPV|D3DRMMAP_PERSPCORRECT);
		}
		else
			Scene.m_Mesh->SetGroupTexture(0, NULL);
		break;
	case VK_F3:
		m_bWireframe = !m_bWireframe;
		if ( m_bWireframe )
			Scene.m_Mesh->SetGroupQuality(0, D3DRMFILL_WIREFRAME|D3DRMLIGHT_ON|D3DRMSHADE_GOURAUD);
		else
			Scene.m_Mesh->SetGroupQuality(0, D3DRMFILL_SOLID|D3DRMLIGHT_ON|D3DRMSHADE_GOURAUD);
		break;
	case VK_F4:
		{
			EFFECTOR Effector;
			Effector.Init(RND(500), 1 + RND(2));
			Effector.Center.x = RND(COLS/2) - COLS/4;
			Effector.Center.x = RND(ROWS/2) - ROWS/4;
			Effector.Center.y = RND(1);
			// Collide surface with effectors
			Scene.m_Surface.Deform(Effector);
		}
		break;
	case VK_DOWN:
		Scene.m_CameraPos.z -= 1.0f;
		break;
	// Rotation
	case 'L':
		Scene.m_Alpha += PI/64;
		Scene.m_ZAxis.z = D3DVAL(sin(Scene.m_Alpha));
		Scene.m_ZAxis.x = D3DVAL(cos(Scene.m_Alpha));
		Scene.m_Camera->SetOrientation(Scene.m_Scene, Scene.m_ZAxis.x, Scene.m_ZAxis.y, Scene.m_ZAxis.z,
			0.0f, 1.0f, 0.0f);
		break;
	case 'R':
		Scene.m_Alpha -= PI/64;
		Scene.m_ZAxis.z = D3DVAL(sin(Scene.m_Alpha));
		Scene.m_ZAxis.x = D3DVAL(cos(Scene.m_Alpha));
		Scene.m_Camera->SetOrientation(Scene.m_Scene, Scene.m_ZAxis.x, Scene.m_ZAxis.y, Scene.m_ZAxis.z,
			0.0f, 1.0f, 0.0f);
		break;
	default:;
		CWnd::OnKeyDown(nChar, nRepCnt, nFlags);
	}
	Scene.SetCamera(Scene.m_CameraPos.x, Scene.m_CameraPos.z);
	Render3DEnvironment();
}

/////////////////////////////////////////////////////////////////////
// Name: OnInitialUpdate()
// Desc: When the AppForm object is created, this function is called to
//		 initialize it. Here we getting access ptrs to some of the controls,
//		 and setting the initial state of some of them as well.
void CDemoView::InitialUpdate() 
{
	m_bReady = FALSE;
	
	// Enumerate available D3D devices, passing a callback that allows devices
	// to be accepted/rejected based on what capabilities the app requires.
	HRESULT hr;
	if ( FAILED(hr = D3DEnum_EnumerateDevices(NULL)) )
	{
		DisplayFrameworkError(hr, MSGERR_APPMUSTEXIT);
		return;
	}

    // Select a default device to render with
    D3DEnum_SelectDefaultDevice(&m_pDeviceInfo);
    UpdateDeviceInfo();

    // Check if we could not get a device that renders into a window, which
    // means the display must be 16- or 256-color mode. If so, let's bail.
    if( FALSE == m_pDeviceInfo->bWindowed )
    {
        Cleanup3DEnvironment();
        DisplayFrameworkError( D3DFWERR_INVALIDMODE, MSGERR_APPMUSTEXIT );
        return;
    }

	// Initialize the 3D environment for the app
	if ( FAILED(hr = Initialize3DEnvironment()) )
	{
		Cleanup3DEnvironment();
		DisplayFrameworkError(hr, MSGERR_APPMUSTEXIT);
		return;
	}
	Scene.SetCamera(Scene.m_CameraPos.x, Scene.m_CameraPos.z);

	// Register a class for a fullscreen window
	WNDCLASS wndClass = {CS_HREDRAW | CS_VREDRAW, FullScreenWndProc, 0, 0, NULL,
		NULL, NULL, (HBRUSH)GetStockObject(WHITE_BRUSH), NULL,
		TEXT("Fullscreen Window")};
	RegisterClass(&wndClass);
	m_bReady = TRUE;
}

/////////////////////////////////////////////////////////////////////
// Name: UpdateDeviceInfo()
// Desc: Saves information about the currently selected device. 
void CDemoView::UpdateDeviceInfo()
{
    // Save the selected device information. However, if the device can
    // only do fullscreen operation, then use software emulation for
    // windowed mode.
    m_pFullScreenDeviceInfo = m_pDeviceInfo;
    
    if ( m_pDeviceInfo->ddDriverCaps.dwCaps2 & DDCAPS2_CANRENDERWINDOWED )
        m_bUsingHELForWindowedMode = FALSE;
    else
    {
        D3DEnum_SelectDefaultDevice( &m_pDeviceInfo, D3DENUM_SOFTWAREONLY );
        m_bUsingHELForWindowedMode = TRUE;
    }
}

/////////////////////////////////////////////////////////////////////
// Name: AppInitialize()
// Desc: Initializes the sample framework, then calls the app-specific function
//		 to initialize device specific objects. This code is structured to
//		 handled any errors that may occur duing initialization
HRESULT CDemoView::AppInitialize(HWND hWnd)
{
    GUID*           pDriverGUID;
    GUID*           pDeviceGUID;
    DDSURFACEDESC2* pMode;
    DWORD           dwFrameworkFlags;
    HRESULT         hr;

    if ( m_bWindowed )
    {
        pDriverGUID      = m_pDeviceInfo->pDriverGUID;
        pDeviceGUID      = m_pDeviceInfo->pDeviceGUID;
        pMode            = NULL;
        dwFrameworkFlags = D3DFW_ZBUFFER;
    }
    else
    {
        pDriverGUID      = m_pFullScreenDeviceInfo->pDriverGUID;
        pDeviceGUID      = m_pFullScreenDeviceInfo->pDeviceGUID;
        pMode            = &m_pFullScreenDeviceInfo->ddsdFullscreenMode;
        dwFrameworkFlags = D3DFW_ZBUFFER|D3DFW_FULLSCREEN;
    }

    // Initialize the D3D framework
    if ( SUCCEEDED(hr = m_pFramework->Initialize(hWnd, pDriverGUID,
		pDeviceGUID, pMode, dwFrameworkFlags)) )
    {
	    D3DTextr_RestoreAllTextures(m_pFramework->GetD3DDevice());
		LPDIRECT3D2 pD3D = NULL;
		LPDIRECT3DDEVICE2 pDevice = NULL;
		m_pFramework->GetDirect3D()->QueryInterface(IID_IDirect3D2, (void**)&pD3D);
		m_pFramework->GetD3DDevice()->QueryInterface(IID_IDirect3DDevice2, (void**)&pDevice);

		// Let the app run its startup code which creates the 3d scene.
		if ( FAILED(hr = Scene.CreateDevAndView(pD3D, pDevice)) )
		/*if ( SUCCEEDED(Scene.CreateDevAndView(pDriverGUID, hWnd,
			m_pFramework->GetDirectDraw(),
			m_pFramework->GetBackBuffer())) )*/
		{
			D3DTextr_InvalidateAllTextures();
			m_pFramework->DestroyObjects();
            AfxGetMainWnd()->PostMessage(WM_CLOSE, 0, 0);
        }
		SAFE_RELEASE(pDevice);
		SAFE_RELEASE(pD3D);
    }
    return hr;
}

/////////////////////////////////////////////////////////////////////
// Name: Initialize3DEnvironment()
// Desc: One-time initialization for the 3D portion of the app. Any changes to
//		 the device will be handled with the Change3DEnvironment() function.
HRESULT CDemoView::Initialize3DEnvironment()
{
	HRESULT hr;

	// Initialize the app
	if ( !Scene.Create() || !Scene.Build() || !Scene.SetCamera(0, 0) )
		return E_FAIL;

    // Create a new CD3DFramework class. This class does all of our D3D
    // initialization and manages the common D3D objects.
    if ( NULL == (m_pFramework = new CD3DFramework()) )
        return E_OUTOFMEMORY;

    // Finally, initialize the framework and scene.
    if ( SUCCEEDED(hr = AppInitialize(m_hWnd)) )
        return S_OK;

    // If we get here, the first initialization passed failed. If that was with a
    // hardware device, try again using a software rasterizer instead.
    if ( m_pDeviceInfo->bHardware )
    {
        // Try again with a software rasterizer
        DisplayFrameworkError(hr, MSGWARN_SWITCHTOSOFTWARE);
        D3DEnum_SelectDefaultDevice(&m_pDeviceInfo, D3DENUM_SOFTWAREONLY);
        UpdateDeviceInfo();
        hr = AppInitialize(m_hWnd);
    }
    return hr;
}

/////////////////////////////////////////////////////////////////////
// Name: Change3DEnvironment()
// Desc: Handles driver, device, and/or mode changes for the app.
HRESULT CDemoView::Change3DEnvironment()
{
	if ( NULL == m_pFramework )
		return E_FAIL;

    // In case we're coming from a fullscreen mode, restore the window size
    HWND hWnd = m_hWnd;
    if ( m_bWindowed == FALSE )
        hWnd = m_hwndRenderFullScreen;

	// Release all objects that need to be re-created for the new device
	D3DTextr_InvalidateAllTextures();
	SAFE_RELEASE(Scene.m_Device);
	SAFE_RELEASE(Scene.m_Viewport);
	m_pFramework->DestroyObjects();

    // Inform the framework class of the device change. It will internally
    // re-create valid surfaces, a d3ddevice, etc.
    return AppInitialize(hWnd);
}

/////////////////////////////////////////////////////////////////////
// Name: Render3DEnvironment()
// Desc: Draws the scene
HRESULT CDemoView::Render3DEnvironment()
{
	if ( !m_bReady )
		return S_OK;

	// If fullscreen, check the cooperative level before rendering
	if ( !m_bWindowed )
		if ( FAILED(m_pFramework->GetDirectDraw()->TestCooperativeLevel()) )
		{
			// If we lost fullscreen mode, let's bail out of it for real
			GoWindowed();
			return S_OK;
		}
	
	// FrameMove (animate) the scene
	if ( !m_bPaused || m_bSingleStep )
		m_bSingleStep = FALSE;

	// Render the scene
	Scene.Render(m_bPaused);

	// Keep track of the frame rate, and output it every 500 milliseconds to
	// the appropiate UI control.
	static DWORD dwLastTime = 0;
	static DWORD dwNumFrames = 0;
	dwNumFrames++;
	
	static CString S, S1;
	if ( timeGetTime() - dwLastTime > 500 )
	{
		FLOAT fFPS = dwNumFrames*1000.0f/(timeGetTime() - dwLastTime);
		dwLastTime = timeGetTime();
		dwNumFrames=0;
		S.Format("%4.1f fps", fFPS);
		if ( Scene.m_b3DNow )
			S += " (3DNow!)";
	}
	OutputText(0, 0, S);
	S1.Format("%iK cycles", (int)Scene.m_Cycles/1000);
	OutputText(0, 18, S1);

	// Output any special messages
	if ( m_bWindowed && m_bUsingHELForWindowedMode )
	{
		OutputText(0, 0, "Note: Must go fullscreen");
		OutputText(0, 18, "for the selected device");
	}

	// Show the frame on the primary surface.
	if ( m_pFramework->ShowFrame() == DDERR_SURFACELOST )
		m_pFramework->RestoreSurfaces();
	return S_OK;
}

/////////////////////////////////////////////////////////////////////
// Name: OutputText()
// Desc: Draws text on the window.
HRESULT CDemoView::OutputText(DWORD x, DWORD y, LPCSTR str)
{
	// Get a DC for the render surface. Then, write out the buffer
	HDC hDC;
	if ( SUCCEEDED(m_pFramework->GetRenderSurface()->GetDC(&hDC)) )
	{
		SetTextColor(hDC, RGB(255, 255, 0));
		SetBkMode(hDC, TRANSPARENT);
		ExtTextOut(hDC, x, y, 0, NULL, str, strlen(str), NULL);
		m_pFramework->GetRenderSurface()->ReleaseDC(hDC);
	}
	return S_OK;
}

/////////////////////////////////////////////////////////////////////
// Name: Move()
// Desc: Called when the render window is moved
void CDemoView::Move(int x, int y)
{
	// Inform the framework of the new window position
	if ( m_pFramework && m_bReady )
	{
		RECT rc;
		GetClientRect(&rc);
		ClientToScreen((POINT*)&rc.left);
		ClientToScreen((POINT*)&rc.right);
		m_pFramework->Move((SHORT)rc.left, (SHORT)rc.top);
	}
}

/////////////////////////////////////////////////////////////////////
// Name: Cleanup3DEnvironment()
// Desc: Cleanup scene objects
void CDemoView::Cleanup3DEnvironment()
{
	if ( m_pFramework )
	{
		//App_DeleteDeviceObjects();
		D3DTextr_InvalidateAllTextures();
		SAFE_DELETE(m_pFramework);
	}
	D3DEnum_FreeResources();
}

/////////////////////////////////////////////////////////////////////
// Name: GoFullScreen()
// Desc: Makes the 3D rendering go to fullscreen mode
void CDemoView::GoFullScreen()
{
	HRESULT hr;
	if ( FALSE == m_bWindowed )
		return;

	m_bWindowed = FALSE;

	// Create a new fullscreen window
	RECT rc = {0, 0, 100, 100};
	m_hwndRenderFullScreen = CreateWindow(TEXT("Fullscreen Window"), NULL,
		WS_OVERLAPPED, CW_USEDEFAULT, CW_USEDEFAULT, 100, 100,
		m_hWnd, 0L, NULL, 0L);
	
	if ( FAILED(hr = Change3DEnvironment()) )
	{
		Scene.m_Viewport = NULL;
		Scene.m_Device = NULL;
        GoWindowed();
        DisplayFrameworkError(hr, MSGWARN_CANTDOFULLSCREEN);
	}
}

/////////////////////////////////////////////////////////////////////
// Name: GoWindowed()
// Desc: Makes the 3D rendering go back to windowed mode
void CDemoView::GoWindowed()
{
	if ( m_bWindowed )
		return;
	m_bWindowed = TRUE;

	if ( FAILED(Change3DEnvironment()) )
	{
		Scene.m_Viewport = NULL;
		Scene.m_Device = NULL;
        AfxGetMainWnd()->PostMessage(WM_CLOSE, 0, 0);
		return;
	}

	if ( m_hwndRenderFullScreen )
		::DestroyWindow(m_hwndRenderFullScreen);
	m_hwndRenderFullScreen = NULL;
}

/////////////////////////////////////////////////////////////////////
// Name: DisplayFrameworkError()
// Desc: Displays error messages in a message box
void DisplayFrameworkError(HRESULT hr, APPMSGTYPE errType)
{
	CHAR strMsg[512];

	switch ( hr )
	{
		case D3DENUMERR_NOCOMPATIBLEDEVICES:
			strcpy(strMsg, TEXT("Could not find any compatible Direct3D\n"
				"devices."));
			break;
		case D3DENUMERR_SUGGESTREFRAST:
			strcpy(strMsg, TEXT("Could not find any compatible devices.\n\n"
				"Try enabling the reference rasterizer using\n"
				"EnableRefRast.reg."));
			break;
		case D3DENUMERR_ENUMERATIONFAILED:
			strcpy(strMsg, TEXT("Enumeration failed. Your system may be in an\n"
				"unstable state and need to be rebooted"));
			break;
		case D3DFWERR_INITIALIZATIONFAILED:
			strcpy(strMsg, TEXT("Generic initialization error.\n\nEnable "
				"debug output for detailed information."));
			break;
		case D3DFWERR_NODIRECTDRAW:
			strcpy(strMsg, TEXT("No DirectDraw"));
			break;
		case D3DFWERR_NODIRECT3D:
			strcpy(strMsg, TEXT("No Direct3D"));
			break;
		case D3DFWERR_INVALIDMODE:
			strcpy(strMsg, TEXT("This sample requires a 16-bit (or higher) "
				"display mode\nto run in a window.\n\nPlease switch "
				"your desktop settings accordingly."));
			break;
		case D3DFWERR_COULDNTSETCOOPLEVEL:
			strcpy(strMsg, TEXT("Could not set Cooperative Level"));
			break;
		case D3DFWERR_NO3DDEVICE:
			strcpy(strMsg, TEXT("No 3D Device"));
			break;
		case D3DFWERR_NOZBUFFER:
			strcpy(strMsg, TEXT("No ZBuffer"));
			break;
		case D3DFWERR_NOVIEWPORT:
			strcpy(strMsg, TEXT("No Viewport"));
			break;
		case D3DFWERR_NOPRIMARY:
			strcpy(strMsg, TEXT("No primary"));
			break;
		case D3DFWERR_NOCLIPPER:
			strcpy(strMsg, TEXT("No Clipper"));
			break;
		case D3DFWERR_BADDISPLAYMODE:
			strcpy(strMsg, TEXT("Bad display mode"));
			break;
		case D3DFWERR_NOBACKBUFFER:
			strcpy(strMsg, TEXT("No backbuffer"));
			break;
		case D3DFWERR_NONZEROREFCOUNT:
			strcpy(strMsg, TEXT("Non-zero ref count"));
			break;
		case D3DFWERR_NORENDERTARGET:
			strcpy(strMsg, TEXT("No render target"));
			break;
		case E_OUTOFMEMORY:
			strcpy(strMsg, TEXT("Not enough memory!"));
			break;
		case DDERR_OUTOFVIDEOMEMORY:
			strcpy(strMsg, TEXT("There was insufficient video memory "
				"to use the\nhardware device."));
			break;
		default:
			strcpy(strMsg, TEXT("Generic application error.\n\nEnable "
				"debug output for detailed information."));
	}

	if ( MSGERR_APPMUSTEXIT == errType )
	{
		strcat( strMsg, TEXT("\n\nThis sample will now exit."));
		MessageBox( NULL, strMsg, "MFC Fog", MB_ICONERROR|MB_OK );
	}
	else
	{
		if ( MSGWARN_SWITCHTOSOFTWARE == errType )
			strcat( strMsg, TEXT("\n\nSwitching to software rasterizer."));
        if( MSGWARN_CANTDOFULLSCREEN == errType )
            strcat( strMsg, TEXT("\n\nCan't do fullscreen mode.") );
		MessageBox( NULL, strMsg, "MFC Fog", MB_ICONWARNING|MB_OK );
	}
}

/////////////////////////////////////////////////////////////////////
// The remaining code handles the UI for the MFC-based app.

/////////////////////////////////////////////////////////////////////
// Name: OnChangeDevice()
// Desc: Use hit the "Change Driver.." button. Display the dialog for the user
//		 to select a new driver/device/mode, and call Change3DEnvironment to
//		 use the new driver/device/mode.
void CDemoView::OnChangeDevice()
{
    HRESULT hr;

    if ( m_bReady )
    {
        m_bReady = FALSE;
        D3DEnum_UserChangeDevice(&m_pDeviceInfo);
        UpdateDeviceInfo();
        if ( FAILED(hr = Change3DEnvironment()) )
        {
            // Handle the error case 
            DisplayFrameworkError(hr, MSGERR_APPMUSTEXIT);
            AfxGetMainWnd()->PostMessage(WM_CLOSE, 0, 0);
        }
		else
			m_bReady = TRUE;
    }
}

/////////////////////////////////////////////////////////////////////
// Name: OnToggleFullScreen()
// Desc: Called when user toggles the fullscreen mode
void CDemoView::OnToggleFullscreen()
{
	if ( m_bWindowed )
		GoFullScreen();
	else
		GoWindowed();
}

void CDemoView::OnUpdateFullscreen(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(!m_bWindowed);
}

/////////////////////////////////////////////////////////////////////
// Name: OnCameraOptions()
// Desc: Handles UI for when user pauses or singlesteps the camera
void CDemoView::OnPause() 
{
	m_bPaused = !m_bPaused;
	if ( m_bPaused )
		m_dwPauseTime = timeGetTime();
	else
		m_dwBaseTime += timeGetTime() - m_dwPauseTime;
}

void CDemoView::OnUpdatePause(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(m_bPaused);	
}

void CDemoView::OnExit() 
{
	PostQuitMessage(0);
}

void CDemoView::OnDithering() 
{
	Scene.m_bDithering = !Scene.m_bDithering;
	Scene.SetRenderState();	
}

void CDemoView::OnUpdateDithering(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(Scene.m_bDithering);
}

/////////////////////////////////////////////////////////////////////
// Texture filtering
void CDemoView::OnPointFilter() 
{
	Scene.m_TextureQuality = D3DRMTEXTURE_NEAREST;
	Scene.SetRenderState();	
}

void CDemoView::OnUpdatePointFilter(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(Scene.m_TextureQuality == D3DRMTEXTURE_NEAREST);
}

void CDemoView::OnLinearFilter() 
{
	Scene.m_TextureQuality = D3DRMTEXTURE_LINEAR;
	Scene.SetRenderState();	
}

void CDemoView::OnUpdateLinearFilter(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(Scene.m_TextureQuality == D3DRMTEXTURE_LINEAR);
}

void CDemoView::OnMippointFilter() 
{
	Scene.m_TextureQuality = D3DRMTEXTURE_MIPNEAREST;
	Scene.SetRenderState();	
}

void CDemoView::OnUpdateMippointFilter(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(Scene.m_TextureQuality == D3DRMTEXTURE_MIPNEAREST);
}

void CDemoView::OnMiplinearFilter() 
{
	Scene.m_TextureQuality = D3DRMTEXTURE_MIPLINEAR;
	Scene.SetRenderState();	
}

void CDemoView::OnUpdateMiplinearFilter(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(Scene.m_TextureQuality == D3DRMTEXTURE_MIPLINEAR);
}

void CDemoView::OnLinearmippointFilter() 
{
	Scene.m_TextureQuality = D3DRMTEXTURE_LINEARMIPNEAREST;
	Scene.SetRenderState();	
}

void CDemoView::OnUpdateLinearmippointFilter(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(Scene.m_TextureQuality == D3DRMTEXTURE_LINEARMIPNEAREST);
}

void CDemoView::OnLinearmiplinearFilter() 
{
	Scene.m_TextureQuality = D3DRMTEXTURE_LINEARMIPLINEAR;
	Scene.SetRenderState();	
}

void CDemoView::OnUpdateLinearmiplinearFilter(CCmdUI* pCmdUI) 
{
	pCmdUI->SetCheck(Scene.m_TextureQuality == D3DRMTEXTURE_LINEARMIPLINEAR);
}
