/////////////////////////////////////////////////////////////////////////////
//

#include "stdafx.h"
#include <math.h>
#include <stdlib.h>
#include <time.h>
#include "Scene.h"

/////////////////////////////////////////////////////////////////////////////
// Move callback
void __cdecl MoveCallback(LPDIRECT3DRMFRAME3 pFrame, VOID* pArg,
	D3DVALUE dT)
{
	CScene* pScene = (CScene*)pArg;
	pScene->DoDeformations(dT);
}

/////////////////////////////////////////////////////////////////////////////
// CScene
CScene::CScene()
{
    m_D3DRM = NULL;
	m_Device = NULL;
	m_Viewport = NULL;
    m_Scene = NULL;
	m_Camera = NULL;
	m_Mesh = NULL;
	m_Lights = NULL;
	m_Texture = NULL;
	m_Ambient = NULL;
	m_Parallel = NULL;
	m_Background = NULL;
	m_Alpha = PI/2;
	m_CameraPos.x = D3DVAL(0);
	m_CameraPos.z = -D3DVAL(15);
	m_bDithering = TRUE;
	m_b3DNow = FALSE;
	m_RenderQuality = D3DRMLIGHT_ON|D3DRMFILL_SOLID|D3DRMSHADE_GOURAUD;
	m_TextureQuality = D3DRMTEXTURE_LINEARMIPLINEAR;
	m_Cycles = 0;
}

CScene::~CScene()
{
}

void CScene::Release()
{
	SAFE_RELEASE(m_Mesh);
	SAFE_RELEASE(m_Lights);
	SAFE_RELEASE(m_Texture);
	SAFE_RELEASE(m_Ambient);
	SAFE_RELEASE(m_Parallel);
	SAFE_RELEASE(m_Background);

	SAFE_RELEASE(m_Camera);
	SAFE_RELEASE(m_Frame);
	SAFE_RELEASE(m_Scene);
	SAFE_RELEASE(m_Viewport);
	SAFE_RELEASE(m_Device);
	SAFE_RELEASE(m_D3DRM);
}

BOOL CScene::Create()
{
	// Create the D3DRM object and the D3DRM window object
	LPDIRECT3DRM pD3DRM;
	if ( FAILED(Direct3DRMCreate(&pD3DRM)) )
	{
		MSG("Failed to create Direct3DRM.\n");
		return FALSE;
	}

	// Now query for the D3DRM3 object
	if ( FAILED(pD3DRM->QueryInterface(IID_IDirect3DRM3, (void**)&m_D3DRM)) )
	{
		pD3DRM->Release();
		MSG("Failed query for  Direct3DRM3.\n");
		return FALSE;
	}
	SAFE_RELEASE(pD3DRM);

	// Create the master scene frame and camera frame
	if ( FAILED(m_D3DRM->CreateFrame(NULL, &m_Scene)) )
	{
		MSG("Failed to create the master scene frame.\n");
		return FALSE;
	}
	m_Scene->SetZbufferMode(D3DRMZBUFFER_ENABLE);
	if ( FAILED(m_D3DRM->CreateFrame(m_Scene, &m_Camera) ) )
	{
		MSG("Failed to create the camera's frame.\n");
		return FALSE;
	}
	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
// Create the D3DRM device and viewport with the given D3D Driver and of
// the specified size
HRESULT CScene::CreateDevAndView(/*GUID* pDriverGIUD, HWND hWnd,
	LPDIRECTDRAW4 lpDD, LPDIRECTDRAWSURFACE4 lpBack*/
	LPDIRECT3D2 lpD3D, LPDIRECT3DDEVICE2 lpD3DDev)
{
	HRESULT hr;
	// Create the D3DRM device from this window and using the specified D3D
	// Driver.
	if ( FAILED(hr = m_D3DRM->CreateDeviceFromD3D(lpD3D, lpD3DDev, &m_Device)) )
		return hr;

	/*LPDIRECTDRAW pDD = NULL;
	LPDIRECTDRAWSURFACE pBack = NULL;
	lpDD->QueryInterface(IID_IDirectDraw, (void**)&pDD);
	lpBack->QueryInterface(IID_IDirectDrawSurface, (void**)&pBack);

    /*LPDIRECTDRAWCLIPPER pClipper;
    if ( FAILED(hr = pDD->CreateClipper(0, &pClipper, NULL)) )
        return hr;
    // Associate the clipper with the window
    pClipper->SetHWnd(0, hWnd);
	if ( FAILED(hr = m_D3DRM->CreateDeviceFromClipper(pClipper, NULL,
		640, 480, &m_Device)) )
		return hr;*/
	// Create the D3DRM device from this window and using the specified D3D
	// Driver.
	/*if ( FAILED(hr = m_D3DRM->CreateDeviceFromSurface(pDriverGIUD, pDD,
		pBack, 0, &m_Device)) )
		return hr;*/

	// Create the D3DRM viewport using the camera frame.  Set the background
	// depth to a large number.  The width and height may be slightly
	// adjusted, so get them from the device to be sure.
	int Width  = m_Device->GetWidth();
	int Height = m_Device->GetHeight();
	
	if ( FAILED(hr = m_D3DRM->CreateViewport(m_Device, m_Camera, 0, 0,
		Width, Height, &m_Viewport)) )
	{
		SAFE_RELEASE(m_Device);
		return hr;
	}

	m_Viewport->SetBack(D3DVAL(1000));
	D3DVALUE left, right, bottom, top;
	m_Viewport->GetPlane(&left, &right, &bottom, &top);
	m_Viewport->SetPlane(left/10, right/10, bottom/10, top/10);
	m_Viewport->SetField(D3DVAL(0.8));
	//m_Device->SetDither(TRUE);

	// Set the texture quality (point or linear filtering)
	if ( m_Device->GetTextureQuality() != m_TextureQuality )
		if ( FAILED(m_Device->SetTextureQuality(m_TextureQuality)) )
		{
			m_TextureQuality = D3DRMTEXTURE_LINEAR;
			if ( FAILED(m_Device->SetTextureQuality(D3DRMTEXTURE_LINEAR)) )
			{
				m_TextureQuality = D3DRMTEXTURE_NEAREST;
				m_Device->SetTextureQuality(m_TextureQuality);
			}
		}

	// Set the render quality, fill mode, lighting state and color shade info
	SetRenderState();
	return D3DRM_OK;
}

BOOL CScene::SetCamera(D3DVALUE x, D3DVALUE z)
{
	return !FAILED(m_Camera->SetPosition(m_Scene, x, D3DVAL(4), z));
}

/////////////////////////////////////////////////////////////////////////////
// Clear the viewport, render the next frame and update the window
BOOL CScene::Render(BOOL bNoMove)
{
	if ( m_Viewport && SUCCEEDED(m_Viewport->ForceUpdate(0, 0, m_Viewport->GetWidth(),
		m_Viewport->GetHeight())) )
		// Tick the scene
		if ( bNoMove || SUCCEEDED(m_Scene->Move(D3DVAL(0.05))) )
			// Clear the viewport
			if ( SUCCEEDED(m_Viewport->Clear(D3DRMCLEAR_ALL)) )
				// Render the scene to the viewport
				if ( SUCCEEDED(m_Viewport->Render(m_Scene)) )
					if ( SUCCEEDED(m_Device->Update()) )
						return TRUE;
	//MSG("Could not render the scene.\n");
	return FALSE;
}

/////////////////////////////////////////////////////////////////////////////
// Set the render quality, dither toggle and shade info if any of them
// has changed
void CScene::SetRenderState()
{
	// Set the render quality (light toggle, fill mode, shade mode)
	if ( m_Device->GetQuality() != m_RenderQuality )
		m_Device->SetQuality(m_RenderQuality);

	// Set dithering toggle
	if ( m_Device->GetDither() != m_bDithering )
		m_Device->SetDither(m_bDithering);

	// Set the texture quality (point or linear filtering)
	if ( m_Device->GetTextureQuality() != m_TextureQuality )
		m_Device->SetTextureQuality(m_TextureQuality);

	HDC hdc = GetDC(NULL);
	int BPP = GetDeviceCaps(hdc, BITSPIXEL);
	ReleaseDC(NULL, hdc);

	// Set shade info based on current bits per pixel
	switch ( BPP )
	{
		case 1:
			m_Device->SetShades(4);
			m_D3DRM->SetDefaultTextureShades(4);
			break;
		case 16:
			m_Device->SetShades(32);
			m_D3DRM->SetDefaultTextureColors(64);
			m_D3DRM->SetDefaultTextureShades(32);
			break;
		case 24:
		case 32:
			m_Device->SetShades(256);
			m_D3DRM->SetDefaultTextureColors(64);
			m_D3DRM->SetDefaultTextureShades(256);
			break;
	}
}

/////////////////////////////////////////////////////////////////////////////
BOOL CScene::Build()
{
	int i, k;
	D3DRMGROUPINDEX id;
	BOOL bStatus = FALSE;
	LPDIRECT3DRMFRAME3 pFrame = NULL, pDupFrame = NULL;

	// Initialize the fog
	if ( FAILED(m_Scene->SetSceneFogEnable(TRUE)) )
		goto generic_error;
	if ( FAILED(m_Scene->SetSceneFogParams(D3DVAL(0), ROWS, D3DVAL(0.005))) )
		goto generic_error;
	
	// Initialize the lights in the scene
	if ( FAILED(m_D3DRM->CreateFrame(m_Scene, &m_Lights)) )
		goto generic_error;
	m_Lights->SetOrientation(m_Scene, D3DVAL(-1), D3DVAL(-1), D3DVAL(0),
		D3DVAL(0), D3DVAL(1), D3DVAL(0));
	
	if ( FAILED(m_D3DRM->CreateLightRGB(D3DRMLIGHT_DIRECTIONAL,
		0.95f, 0.95f, 0.95f, &m_Parallel)) )
		goto generic_error;
	m_Parallel->SetConstantAttenuation(D3DVAL(0));
	
	if ( FAILED(m_Lights->AddLight(m_Parallel)) )
		goto generic_error;
	if ( FAILED(m_D3DRM->CreateLightRGB(D3DRMLIGHT_AMBIENT,
		0.8f, 0.8f, 0.8f, &m_Ambient)) )
		goto generic_error;
	if ( FAILED(m_Scene->AddLight(m_Ambient)) )
		goto generic_error;
	
	// Load texture
    if ( FAILED(m_D3DRM->LoadTexture("abalone.bmp", &m_Texture)) )
	{
		MSG("Failed to load abalone.bmp.\n");
		goto generic_error;
    }
	m_Texture->GenerateMIPMap(0);

	// Create landscape m_Frame
	if ( FAILED(m_D3DRM->CreateFrame(m_Scene, &m_Frame)) )
		goto generic_error;
	if ( FAILED(m_D3DRM->CreateMesh(&m_Mesh)) )
		goto generic_error;

	// Init deformation objects
	m_Surface.Init();
	EFFECTOR Effector;
	Effector.Init(200.0f, 1.0f);
	Effector.Center.y = 0.5f;
	// Performance measurement block
	{
		__int64 Ticks = 0x7FFFFFFFFFFFFFFF;
		for ( i = 0; i < 20; i++ )
		{
			__int64 StartTicks, EndTicks;
			// Flush the data cache if necessary
			// memset(data, 0, sizeof_data_cache);
			_asm {
				cpuid
				rdtsc
				mov		dword ptr StartTicks,eax
				mov		dword ptr StartTicks[4],edx
			}
			// Code which performance is measured
			m_Surface.Update3DNow(0.1f);
			__asm {
				cpuid
				rdtsc
				mov		dword ptr EndTicks,eax
				mov		dword ptr EndTicks[4],edx
			}
			if ( EndTicks - StartTicks < Ticks )
				Ticks = EndTicks - StartTicks;
		}
		CString S;
		S.Format("Surface Deformation Demo\nWritten by Max I. Fomitchev (maxf@webzone.net)\n3DNow! Routine processed 10,000 vertices in %iK cycles", Ticks/1000);
		AfxMessageBox(S);
	}
	m_Surface.Deform(Effector);
	m_Surface.GetData(m_Vertex);

	// Set texture coordinates
	for ( i = 0, k = 0; i < m_Surface.NumRows; i++ )
		for ( int j = 0; j < m_Surface.NumCols; j++, k++ )
		{
			m_Vertex[k].tu = D3DVAL(j%8)/8;
			m_Vertex[k].tv = D3DVAL(i%8)/8;
			//m_Vertex[k].color = RGBA_MAKE(255, 255, 255, 0);
		}

	// Init faces data with triangles
	for ( i = 0, k = 0; i < m_Surface.NumRows - 1; i++ )
		for ( int j = 0; j < m_Surface.NumCols - 1; j++ )
	{
		m_FaceData[k++] = i*m_Surface.NumCols + j;
		m_FaceData[k++] = (i + 1)*m_Surface.NumCols + j;
		m_FaceData[k++] = (i + 1)*m_Surface.NumCols + j + 1;
		m_FaceData[k++] = i*m_Surface.NumCols + j;
		m_FaceData[k++] = (i + 1)*m_Surface.NumCols + j + 1;
		m_FaceData[k++] = i*m_Surface.NumCols + j + 1;
	}

    if ( FAILED(m_Mesh->AddGroup(m_Surface.NumVertex,
		(m_Surface.NumRows - 1)*(m_Surface.NumCols - 1)*2,
		3, m_FaceData, &id)) )
		return FALSE;
	m_Mesh->SetVertices(id, 0, m_Surface.NumVertex, m_Vertex);
	m_Mesh->SetGroupTexture(0, (LPDIRECT3DRMTEXTURE)m_Texture);
	m_Mesh->SetGroupMapping(0, D3DRMMAP_WRAPU|D3DRMMAP_WRAPV|D3DRMMAP_PERSPCORRECT);

	// Add land mesh visual
	if ( FAILED(m_Frame->AddVisual((LPDIRECT3DRMVISUAL)m_Mesh)) )
		return FALSE;
	// Add motion callback
	if ( FAILED(m_Scene->AddMoveCallback(MoveCallback, this, D3DRMCALLBACK_PREORDER)) )
		return FALSE;

	// Sphere
	//if ( !(pFrame = LoadMesh("sphere.x", D3DVAL(0), D3DVAL(3))) )
	//	goto generic_error;
	bStatus = TRUE;
	
generic_error:
    SAFE_RELEASE(m_Background);
	SAFE_RELEASE(m_Ambient);
	SAFE_RELEASE(m_Parallel);
	return bStatus;
}

LPDIRECT3DRMFRAME3 CScene::LoadMesh(LPCSTR FileName, D3DVALUE x, D3DVALUE z)
{
	LPDIRECT3DRMMESHBUILDER3 builder = NULL;
	LPDIRECT3DRMMESH mesh = NULL;
    LPDIRECT3DRMFRAME3 frame = NULL;
	D3DRMBOX box;
	int count;

	if ( FAILED(m_D3DRM->CreateMeshBuilder(&builder)) )
		goto generic_error;
	
	if ( FAILED(builder->Load((void*)FileName, NULL, D3DRMLOAD_FROMFILE,
		NULL, NULL)) )
		goto generic_error;

	count = builder->GetFaceCount();
	if ( FAILED(builder->Scale(0.1f, 0.1f, 0.1f)) )
		goto generic_error;
	// Adjust coordinates
	if ( FAILED(builder->GetBox(&box)) )
		return FALSE;
	if ( FAILED(builder->Translate(-box.min.x, -box.min.y, -box.min.z)) )
		return FALSE;
	if ( FAILED(builder->CreateMesh(&mesh)) )
		goto generic_error;
	if ( FAILED(m_D3DRM->CreateFrame(m_Scene, &frame)) )
		goto generic_error;
	if ( FAILED(frame->AddVisual((LPDIRECT3DRMVISUAL)mesh)) )
		goto generic_error;

generic_error:
	SAFE_RELEASE(mesh);
	SAFE_RELEASE(builder);
	return frame;
}

void CScene::DoDeformations(float dT)
{
	__int64 StartTicks, EndTicks;
	_asm {
		cpuid
		rdtsc
		mov		dword ptr StartTicks,eax
		mov		dword ptr StartTicks[4],edx
	}
	// Calculate surface deformations
	if ( m_b3DNow )
		m_Surface.Update3DNow(dT);
	else
		m_Surface.Update(dT);
	__asm {
		cpuid
		rdtsc
		mov		dword ptr EndTicks,eax
		mov		dword ptr EndTicks[4],edx
	}
	m_Cycles = EndTicks - StartTicks;
	// Set new vertices
	m_Surface.GetData(m_Vertex);
	m_Mesh->SetVertices(0, 0, m_Surface.NumVertex, m_Vertex);
}
