//==========================================================================
// mixdev.c
//
// This source file manages the "Choose Mixer" (IDD_MIXERS) dialog. This
// dialog contains a dropdown listbox which will be filled in with the
// names of all installed Mixer devices in the computer. The user can
// select one of these Mixer devices, and it then becomes the "currently
// open Mixer device" which other dialogs in the program can display
// information about. For example, the main window will display info about
// the lines in this Mixer device.
//
// This dialog also contains a readonly, multi-line Edit box that
// displays some generic information about the selected Mixer device, such
// as the version number of its mixer driver, Manufacturer and Product
// code of the audio card for this Mixer, etc.
//
// The IDD_MIXERS dialog is presented when the user selects the
// "File -> Mixer driver..." menu item in the main window.
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
// KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR
// PURPOSE.
//
// Copyright (C) 1993 - 1997  Microsoft Corporation.  All Rights Reserved.
// Modified 2001 by Jeff Glatt
//==========================================================================

#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>
#include <mmsystem.h>

#include "resource.h"
#include "mixapp.h"





// Handle of the currently open Mixer device
HMIXER			MixerHandle = 0;

// Strings
const TCHAR		gszMixerErr[] = TEXT("Error #%lu getting info upon Mixer #%lu!");
const TCHAR		gszMixerErr2[] = TEXT("Error #%lu opening Mixer #%lu!");
const TCHAR		gszMixerErr3[] = TEXT("Error #%lu closing previously open Mixer!");





// ***************************** closeMixer() **************************
// Closes any currently open Mixer.

void closeMixer(void)
{
	// Close any open mixer device
	if (MixerHandle)
	{
		mixerClose(MixerHandle);
		MixerHandle = 0;
	}
}





// **************************** openMixer() *************************
// Opens the requested mixer device, and returns its handle. Closes
// any previously open mixer device. Sets the main window's titlebar
// to display the new mixer's name.
//
// ARGS:
//		HWND hwnd:		Window that receives messages from the mixer
//						device.
//
//		UINT mixerID:	ID # of new mixer device to open.
//
// RETURN:
//		Handle of new, currently open mixer, or 0 if an error. Note
//		that if this function fails, it will attempt to return the
//		handle of any previously open mixer without closing it.

HMIXER openMixer(HWND hwnd, UINT mixerID)
{
	MMRESULT		err;
	HMIXER			newmixer;
	MIXERCAPS		mixercaps;
	LPTSTR			ptr;
	TCHAR			buffer[MAXPNAMELEN + APP_MAX_APP_NAME_CHARS];

	// Get info about requested mixer device
	if ((err = mixerGetDevCaps(mixerID, &mixercaps, sizeof(MIXERCAPS))))
	{
		// Can't query the new mixer device
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, gszMixerErr, err, mixerID);

		// Return handle to previously opened mixer without closing it
		return(MixerHandle);
	}

    // Open the new (requested) mixer. Note that we specify our main window as a callback
	// so it can receive messages from the Mixer if someone else changes its settings while
	// we have it open
	if ((err = mixerOpen(&newmixer, mixerID, (DWORD)hwnd, 0L, CALLBACK_WINDOW)))
	{
		// Can't open the new mixer device
		doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, gszMixerErr2, err, mixerID);
		return(MixerHandle);
	}

	// Was there a previously opened mixer?
	if (MixerHandle)
	{
		// Close the previously opened mixer device
		if ((err = mixerClose(MixerHandle)))
		{
			mixerClose(newmixer);
			doMsgBox(hwnd, MB_OK|MB_ICONEXCLAMATION, gszMixerErr3, err);
			return(MixerHandle);
		}
	}

	// Save new handle as currently open mixer
	MixerHandle = newmixer;

	// Load the app's name
	ptr = &buffer[0];
	if (!(mixercaps.fdwSupport = LoadString(MyInstance, IDS_APP_NAME, &buffer[0], SIZEOF(buffer))))
	{
		// Get the error message associated with this error # from Windows
//		FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, 0, GetLastError(), MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), &buffer[0], SIZEOF(buffer), 0);
//		MessageBox(MainWindow, &buffer[0], "ERROR", MB_OK|MB_ICONEXCLAMATION);
		buffer[0] = 0;
	}
	else
	{
		ptr += mixercaps.fdwSupport;
		*(ptr)++ = ':';
		*(ptr)++ = ' ';
	}

	// Append the Mixer device name
	lstrcpy(ptr, &mixercaps.szPname[0]);

	// Set this string as the main window title
	SetWindowText(hwnd, (LPCTSTR)&buffer[0]);

	// Return handle to the newly opened mixer
	return(newmixer);
}





// ************************** showMixerCaps() **************************
// Displays info about a mixer device using the passed MIXERCAPS struct
// (which was already filled in via a call to mixerGetDevCaps()). Called
// by mixerDevDlgProc()'s WM_INITDIALOG handling.
//
// ARGS:
//		HWND hedit:				Handle to readonly edit control.
//
//		UINT mixerID:			ID # of desired mixer device.
//
//		LPMIXERCAPS pmxcaps:	Filled in MIXERCAPS.
//
// RETURNS:

BOOL showMixerCaps(HWND hedit, UINT mixerID, LPMIXERCAPS pmxcaps)
{
	SetWindowRedraw(hedit, FALSE);

	SetWindowFont(hedit, GetStockFont(SYSTEM_FIXED_FONT), FALSE);

	MEditPrintF(hedit, NULL);

	MEditPrintF(hedit, "%18s: %u", (LPCTSTR)"Device Id", mixerID);
	MEditPrintF(hedit, "%18s: %u", (LPCTSTR)"Manufacturer Id", pmxcaps->wMid);
	MEditPrintF(hedit, "%18s: %u", (LPCTSTR)"Product Id", pmxcaps->wPid);
	MEditPrintF(hedit, "%18s: %u.%.02u", (LPCTSTR)"Driver Version",
				(pmxcaps->vDriverVersion >> 8),
				(pmxcaps->vDriverVersion & 0x00FF));
	MEditPrintF(hedit, "%18s: %.08lXh", (LPCTSTR)"Support Flags", pmxcaps->fdwSupport);
	MEditPrintF(hedit, "%18s: %lu", (LPCTSTR)"Destination lines", pmxcaps->cDestinations);

	SetWindowRedraw(hedit, TRUE);

	return(TRUE);
}





// *********************** mixerDevDlgProc() ************************
// Dialog Procedure for the Mixers (IDD_MIXERS) dialog.
//
// The IDD_MIXERS dialog is presented when the user selects the
// "File -> Mixer driver..." menu item in the main window.
//
// ARGS:
//		Standard args for a dialog procedure.
//
// RETURNS:
//		Standard return for a dialog procedure.

BOOL APIENTRY mixerDevDlgProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
		case WM_INITDIALOG:
		{
			HWND		hcb;
			UINT		cMixerDevs;
			MIXERCAPS	mixercaps;
			
			// Save the ID # of the currently open mixer in the main window's DWL_USER
			// field. This is passed to us from the main window procedure, when it creates
			// the IDD_MIXERS dialog. We need to save this just in case the user
			// changes the currently selected Mixer while the IDD_MIXERS dialog is
			// open, but finally clicks on the "Cancel" button. In that case, we need to
			// restore the currently open Mixer to what it was when the IDD_MIXERS
			// dialog was first created
			SetWindowLong(hwnd, DWL_USER, lParam);

			// Get the dropdown combo box
			hcb = GetDlgItem(hwnd, IDC_MIXERLIST);

			// Find out how many mixer devices there are
			cMixerDevs = mixerGetNumDevs();

			// For each mixer device...
			for (uMsg = 0; uMsg < cMixerDevs; uMsg++)
			{
				// Retrieve its name, and add it to the dropdown list of mixer devices
				mixerGetDevCaps(uMsg, &mixercaps, sizeof(MIXERCAPS));
				ComboBox_AddString(hcb, &mixercaps.szPname[0]);

				// If this device is the one that we've currently got open...
				if ((UINT)lParam == uMsg)
				{
					HWND	hedit;

					// ...display info about it in the multi-line edit control
					hedit = GetDlgItem(hwnd, IDC_MIXERCAPS);

					showMixerCaps(hedit, lParam, &mixercaps);
				}
			}

			// Highlight the currently opened mixer's name in our dropdown list
			ComboBox_SetCurSel(hcb, lParam);

			// Success
			return(TRUE);
		}

		// User is manipulating one of the buttons or dropdown listbox
		case WM_COMMAND:
		{
			// Which control is he manipulating?
			switch (GET_WM_COMMAND_ID(wParam, lParam))
			{
				// The OK button
				case IDOK:
				{
					// Get the currently selected Mixer's ID, and return
					// that to the main window's procedure after we close
					// the IDD_MIXERS dialog
					if ((lParam = ComboBox_GetCurSel(GetDlgItem(hwnd, IDC_MIXERLIST))) != CB_ERR)
						goto endit;

					// NOTE: If an error, fall through to cancel
				}

				case IDCANCEL:
				{
					// Return the Mixer ID that was initially set when the
					// IDD_MIXERS dialog first opened.
					lParam = (UINT)GetWindowLong(hwnd, DWL_USER);

endit:				EndDialog(hwnd, lParam);
					break;
				}

				// The user is manipulating the dropdown listbox
				case IDC_MIXERLIST:
				{
					// What action is he taking?
					switch (GET_WM_COMMAND_CMD(wParam, lParam))
					{
						// The user selected a new Mixer to be the "currently open mixer"
						case CBN_SELCHANGE:
						{
							MIXERCAPS	mixercaps;
							HWND		hedit;

							// Get the selected Mixer's ID # (it's the same as its index in
							// the listbox. For example, the first Mixer has an ID of 0)
							if ((lParam = ComboBox_GetCurSel(GET_WM_COMMAND_HWND(wParam, lParam))) != CB_ERR)
							{
								// Get info about that Mixer into a MIXERCAPS struct
								mixerGetDevCaps(lParam, &mixercaps, sizeof(MIXERCAPS));

								// Display that info in the edit box
								hedit = GetDlgItem(hwnd, IDC_MIXERCAPS);
								showMixerCaps(hedit, lParam, &mixercaps);
							}

//							break;
						}
					}
				}
			}

			return(TRUE);
		}
	}

	return(FALSE);
}
