//--------------------------------------------------------------------------;
//
//	malines.c
//
// Manages the DLG_LINEINFO dialog. This dialog displays info about
// the selected audio component (line) of the currently opened Mixer
// device, such as its ID #, component type, etc. (The user selects the
// desired component via the main window's Tree full of audio component
// names). This info is displayed in a listbox.
//
// This dialog is opened when the user selects the "View -> Component Info..."
// menu item on the main window.
//==========================================================================;

#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>
#include <commctrl.h>

#include "resource.h"
#include "mixapp.h"





// Character length limit upon the string that getComponentType() will return
#define MIXER_COMPONENT_NAMELEN	18





const TCHAR		gszMixerErr4[] = TEXT("Error %lu getting info upon Mixer!");
const TCHAR		gszLineErr2[] = TEXT("Error %lu getting info on Component %lu!");





// ************************** getComponentType() **************************
// Gets a string that describes the type of audio line for the specified
// mixer line.
//
// NOTE: This must not return a string longer than MIXER_COMPONENT_NAMELEN.
//
//	ARGS:
//		MIXERLINE *mixerLine:	MIXERLINE struct filled-in with information
//								about the desired line.
//
//	RETURNS:
//		Pointer to the string.

LPTSTR getComponentType(MIXERLINE *mixerLine)
{
	// A destination component (ie, not a source)?
	if (0 == (MIXERLINE_LINEF_SOURCE & mixerLine->fdwLine))
	{
		// Determine which type of destination this is
		switch (mixerLine->dwComponentType)
		{
			case MIXERLINE_COMPONENTTYPE_DST_DIGITAL:
				return("Digital");		/* For example, SPDIF OUT connector */

			case MIXERLINE_COMPONENTTYPE_DST_LINE:
				return("Line Level");	/* For example, LINE OUT jack, which would typically by used to output to an external analog tape recorder or analog mixer */

			case MIXERLINE_COMPONENTTYPE_DST_MONITOR:
				return("Monitor");		/* For example, secondary monitors (ie, not main speakers, as below) */

			case MIXERLINE_COMPONENTTYPE_DST_SPEAKERS:
				return("Speakers");		/* Speaker jack */

			case MIXERLINE_COMPONENTTYPE_DST_HEADPHONES:
				return("Headphones");	/* Headphone jack */

			case MIXERLINE_COMPONENTTYPE_DST_TELEPHONE:
				return("Telephone");	/* For example, to daisy-chain a telephone to an analog modem's PHONE jack */

			case MIXERLINE_COMPONENTTYPE_DST_WAVEIN:
				return("Wave Input");	/* The card's ADC */

			case MIXERLINE_COMPONENTTYPE_DST_VOICEIN:
				return("Voice Recognition");

			case MIXERLINE_COMPONENTTYPE_DST_UNDEFINED:
				return("Undefined");	/* If none of the others above are applicable */
		}
	}
	else
	{
		// Determine which type of source this is
		switch (mixerLine->dwComponentType)
		{
			case MIXERLINE_COMPONENTTYPE_SRC_DIGITAL:
				return("Digital");		/* For example, SPDIF IN */

			case MIXERLINE_COMPONENTTYPE_SRC_LINE:
				return("Line Level");	/* Typically, Line input if there is a separate Mic jack */

			case MIXERLINE_COMPONENTTYPE_SRC_MICROPHONE:
				return("Microphone");	/* Typically Mic In (but also used for combination of Mic/Line in) */

			case MIXERLINE_COMPONENTTYPE_SRC_SYNTHESIZER:
				return("Synthesizer");	/* Typically, if the card contains a synth capable of playing MIDI. This would be the audio out of that built-in synth */

			case MIXERLINE_COMPONENTTYPE_SRC_COMPACTDISC:
				return("CD Audio");		/* For the audio out of an internal CDROM to play through the sound card. Usually this is an internal connection inside the computer between sound card and CDROM */

			case MIXERLINE_COMPONENTTYPE_SRC_TELEPHONE:
				return("Telephone");	/* For example, for a telephone line's audio to be piped through the computer's speakers */

			case MIXERLINE_COMPONENTTYPE_SRC_PCSPEAKER:
				return("PC Speaker");;	/* Typically, to allow sound, that normally goes to the computer's built-in speaker, to instead be routed through the card's audio output */

			case MIXERLINE_COMPONENTTYPE_SRC_WAVEOUT:
				return("Wave Out");		/* Wave playback (ie, this is the card's DAC) */

			case MIXERLINE_COMPONENTTYPE_SRC_AUXILIARY:
				return("Auxiliary");	/* An aux jack connected to the computer's speakers, or which can be recorded by the ADC */

			case MIXERLINE_COMPONENTTYPE_SRC_ANALOG:
				return("Analog");

			case MIXERLINE_COMPONENTTYPE_SRC_UNDEFINED:
				return("Undefined");	/* If none of the others above are applicable */
		}
	}

	// Uh oh. Must be some sort of type that was added to the Mixer API after
	// our program was written
	return("UNKNOWN TYPE!!!");
}





// *************************** redrawLineInfo() ***************************
// Displays information (in the main window) for the currently selected
// line (of the currently open Mixer device, such as its ID #, line type,
// etc. (The user selects the desired line via the main window's Tree full
// of audio line names). This info is displayed in a multiline edit box.
//
// NOTE: The global 'CurrMixerLine' MIXERLINE struct must be filled in with
// info about the currently selected line.
//
// RETURNS:
//		TRUE if successful. FALSE if an error.

BOOL redrawLineInfo(void)
{
	HWND		hedit;

	if ((hedit = GetDlgItem(MainWindow, IDC_MAIN_LINEINFO)))
	{
		// Clear edit control
		MEditPrintF(hedit, NULL);

		// Display the ID # of this line (must be unique
		MEditPrintF(hedit, "%.08Xh", CurrMixerLine.dwLineID);

		// Display the various flags
		MEditPrintF(hedit, "~%.08lXh", CurrMixerLine.fdwLine);
		if (MIXERLINE_LINEF_DISCONNECTED & CurrMixerLine.fdwLine)
			MEditPrintF(hedit, "~, disconnected");
		if (MIXERLINE_LINEF_ACTIVE & CurrMixerLine.fdwLine)
			MEditPrintF(hedit, "~, active");
		if (~(MIXERLINE_LINEF_SOURCE | MIXERLINE_LINEF_DISCONNECTED | MIXERLINE_LINEF_ACTIVE) & CurrMixerLine.fdwLine)
			MEditPrintF(hedit, "~, *INVALID FLAGS*");
		MEditPrintF(hedit, "");

		// Display the User field
		MEditPrintF(hedit, "%.08lXh", CurrMixerLine.dwUser);

		// Display how many channels
		MEditPrintF(hedit, "%lu", CurrMixerLine.cChannels);

		// Display the short name of this line
		MEditPrintF(hedit, "'%s'", (LPTSTR)CurrMixerLine.szShortName);

		// Display Target type
		switch (CurrMixerLine.Target.dwType)
		{
			case MIXERLINE_TARGETTYPE_WAVEOUT:
			{
				MEditPrintF(hedit, "%s", "WAVEOUT");
				break;
			}

			case MIXERLINE_TARGETTYPE_WAVEIN:
			{
				MEditPrintF(hedit, "%s", "WAVEIN");
				break;
			}

			case MIXERLINE_TARGETTYPE_MIDIOUT:
			{
				MEditPrintF(hedit, "%s", "MIDIOUT");
				break;
			}

			case MIXERLINE_TARGETTYPE_MIDIIN:
			{
				MEditPrintF(hedit, "%s", "MIDIIN");
				break;
			}

			case MIXERLINE_TARGETTYPE_AUX:
			{
				MEditPrintF(hedit, "%s", "AUX");
				break;
			}

			default:
			{
				MEditPrintF(hedit, "%s", "undefined");
			}
		}
	}

	return(TRUE);
}





// ************************* redrawAudioLines() *************************
// Fills the main window's Tree control with information about all of
// the audio lines in the currently open mixer device.
//
// RETURNS:
//		0 if success, -1 if memory error, or error return from one of the
//		Mixer APIs.

long redrawAudioLines(void)
{
	MIXERLINE		mixerLine;
	LPTSTR			ptr;
	MIXERCAPS		mixercaps;
	HWND *			hwndTree;
	MMRESULT		err;
	UINT			u, v, cConnections;
	TCHAR			buffer[MIXER_LONG_NAME_CHARS + MIXER_COMPONENT_NAMELEN];

	// Get Tree control
	if ((hwndTree = GetDlgItem(MainWindow, IDC_MAIN_LINELIST)))
	{
		// Get info upon currently open Mixer
		if ((err = mixerGetDevCaps((UINT)MixerHandle, &mixercaps, sizeof(MIXERCAPS))))
		{
			doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, gszMixerErr4, err);
			return(err);
		}

		// Hide and clear the Tree
		SetWindowRedraw(hwndTree, FALSE);
		TreeView_DeleteAllItems(hwndTree);

		// For each destination line...
		for (u = 0; u < mixercaps.cDestinations; u++)
		{
			TV_INSERTSTRUCT	insertStruct;

			// ...get info on that destination line
			mixerLine.cbStruct = sizeof(MIXERLINE);
			mixerLine.dwDestination = u;

			if ((err = mixerGetLineInfo((HMIXEROBJ)MixerHandle, &mixerLine, MIXER_GETLINEINFOF_DESTINATION)))
			{
				// An error. Notify user and ask him if he wants to continue !!!
				doMsgBox(MainWindow, MB_OK|MB_ICONEXCLAMATION, gszLineErr2, err, u);
				continue;
			}

			// Initialize TV_INSERTSTRUCT struct for inserting an item in the Tree
			insertStruct.hInsertAfter = TVI_LAST;
			insertStruct.item.mask = TVIF_STATE|TVIF_TEXT|TVIF_PARAM;
			insertStruct.item.stateMask = TVIS_EXPANDED|TVIS_FOCUSED|TVIS_SELECTED;
	
			// Put destination line as a root in the Tree
			insertStruct.hParent = 0;

			// Get the type of line as a mnuemonic name, for example, "Speakers"
			ptr = getComponentType(&mixerLine);
			lstrcpy(&buffer[0], ptr);
			ptr = &buffer[0] + lstrlen(&buffer[0]);

			// Append the line's name in parenthesis (unless it's the same
			// as the line type)
			if (lstrcmpi(&buffer[0], &mixerLine.szName[0]))
			{
				*(ptr)++ = ' ';
				*(ptr)++ = '(';
				lstrcpy(ptr, &mixerLine.szName[0]);
				ptr += lstrlen(&mixerLine.szName[0]);
				*(ptr)++ = ')';
				*ptr = 0;
			}

			// Set this string as the Tree item's text
			insertStruct.item.pszText = &buffer[0];
			insertStruct.item.cchTextMax = ptr - &buffer[0];

			insertStruct.item.state = TVIS_EXPANDED;

			// Is this the currently selected line? (ie, By default, the first line)
			if (!u)
			{
				// Highlight the first line by default
				insertStruct.item.state = TVIS_EXPANDED|TVIS_FOCUSED;

				// Copy the MIXERLINE to our global
				CopyMemory(&CurrMixerLine, &mixerLine, sizeof(MIXERLINE));

				// Show info about this line
				redrawLineInfo();

				// List its parameters (ie, controls) in the list box
				redrawParameters();
			}

			// Save the ID # of this line as the lparam field of the TV_ITEM
			insertStruct.item.lParam = mixerLine.dwLineID;

			// Add line to the Tree, and also save the item handle
			// as the Parent for the leaves below
			insertStruct.hParent = TreeView_InsertItem(hwndTree, &insertStruct);

			/////////////////////////////////////////////////////////////////////
			// Iterate all of the source lines attached to this destination.
			// Make them leaves of the root above

			// None of these are selected
			insertStruct.item.state = 0;

			// For each source line that can be connected to this destination...
			cConnections = (UINT)mixerLine.cConnections;
			for (v = 0; v < cConnections; v++)
			{
				// ...get info upon that source line
				mixerLine.cbStruct = sizeof(MIXERLINE);
				mixerLine.dwDestination = u;
				mixerLine.dwSource = v;

				if ((err = mixerGetLineInfo((HMIXEROBJ)MixerHandle, &mixerLine, MIXER_GETLINEINFOF_SOURCE)))
				{
					// An error
					doMsgBox(MainWindow, MB_OK | MB_ICONEXCLAMATION, gszLineErr2, err, v);
					continue;
				}

				// Get the type of line as a mnuemonic name, for example, "Speakers"
				ptr = getComponentType(&mixerLine);
				lstrcpy(&buffer[0], ptr);
				ptr = &buffer[0] + lstrlen(&buffer[0]);

				// Append the line's name in parenthesis (unless it's the same as the line type)
				if (lstrcmpi(&buffer[0], &mixerLine.szName[0]))
				{
					*(ptr)++ = ' ';
					*(ptr)++ = '(';
					lstrcpy(ptr, &mixerLine.szName[0]);
					ptr += lstrlen(&mixerLine.szName[0]);
					*(ptr)++ = ')';
					*ptr = 0;
				}

				// Set this string as the Tree item's text
				insertStruct.item.pszText = &buffer[0];
				insertStruct.item.cchTextMax = ptr - &buffer[0];

				// Save the ID # of this line as the lparam field of the TV_ITEM
				insertStruct.item.lParam = mixerLine.dwLineID;

				// Insert item in Tree as a leaf
				TreeView_InsertItem(hwndTree, &insertStruct);
			}
		}
	}

	// Allow the Tree to be redrawn
	SetWindowRedraw(hwndTree, TRUE);

	return(0);
}





// ************************* MixAppLineChange() **************************
// Called by main window's precedure when it needs to refresh the list
// of audio lines (perhaps because the operating system notified the
// window precedure that some line in our currently open mixer has
// changed).
//
// ARGS:
//		HWND hwnd:		Handle to main window.
//
//		HMIXER hmx:		Handle to our open Mixer upon which a
//						line has changed.
//
//		DWORD dwLineID:	ID # of the line that changed.
//
//	RETURNS:

LRESULT MixAppLineChange(HWND hwnd, HMIXER hmx, DWORD dwLineID)
{
	// Are we currently displaying the Control Settings dialog? If so, we may need to
	// notify that dialog to update its contents too
	if (gfDisplayingControl &&

		// Is the line that has changed the same line as our currently selected line?
		// If so, we'll need to update the main window
		(dwLineID == CurrMixerLine.dwLineID) &&

		ghdlgControl)
	{
		// Send the Control Settings dialog a MM_MIXM_LINE_CHANGE message so that its
		// dialog procedure will update the info displayed in that dialog too
		SendMessage(ghdlgControl, MM_MIXM_LINE_CHANGE, (WPARAM)hmx, dwLineID);
	}

	// Refresh the listing of audio lines in the main window
	redrawAudioLines();

	// Return zero because we handled the message
	return(0);
}
