// MfcMidiFile.cpp : implementation file
// This contains most all of the relevant stuff concerning reading a MIDI file using MIDIFILE.DLL

#include <sys/types.h>
#include <sys/stat.h>
#include "stdafx.h"
#include "MfcRead.h"
#include "MfcReadDlg.h"
#include "MfcMidiFile.h"
#include "..\midifile.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif





// An Instance of MfcMidiFile
MfcMidiFile		MyCMidiFile;





// A text description of the various text-based Meta-Events (types 0x01 to 0x0F)
char Types[10][14] = {
	"Unknown Text",
	"Text event  ",
	"Copyright   ",
	"Track Name  ",
	"Instrument  ",
	"Lyric       ",
	"Marker      ",
	"Cue Point   ",
	"Device Name ",
	"Pgm Name    ",
};





// For Key Signature Meta-Event
unsigned char Keys[15][3] = { "Cb", "Gb", "Db", "Ab", "Eb", "Bb", " F", " C", " G", " D", " A", " E", " B", "F#", "C#" };





// **************************** LoadMidi() ***************************
// Called when user clicks on the "Load" button.

void MfcMidiFile::LoadMidi(const char * Filename)
{
	long			result;
	char			buf[80];

	// Store a pointer to the filename in MIDIFILE's Handle field
	MidiFile.Handle = (HANDLE)Filename;

	// Here we would set MidiFile.Flags if we wanted something other than the default

	// Set the DLL Variable MidiApp
	MidiApp = (MIDIFILE *)this;

	// Tell MIDIFILE.DLL to read in the file, calling my callback functions
	result = MidiReadFile2(&MidiFile);

	// Print out error message. (NOTE: For 0, DLL returns a "Successful MIDI file load" message).
	if (result)
	{
		// See if it's one of the DLL's errors. If not the return will be 0 */
		if (!MidiGetErr(&MidiFile, (unsigned char *)&buf[0], 80, result))
		{
			// Must be an error that one of my callbacks returned
			strcpy(&buf[0], "My callback returned error #");
			_itoa(result, &buf[28], 10);
		}

		// Display the error message
		AfxMessageBox((LPCTSTR)&buf[0], MB_OK|MB_ICONEXCLAMATION, (unsigned long)result);
	}
}





// **************************** FormatTime() *******************************
// * Formats the time that the event occurs upon so that we can display it.
// * This time is referenced from 0 (ie, as opposed to the previous event in
// * the track as is done with delta-times in the MIDI file). The MIDIFILE DLL
// * automatically maintains the MIDIFILE's Time field, updating it for the
// * current event.
// *************************************************************************

void MfcMidiFile::FormatTime(char * Buffer)
{
	// Format the time
	sprintf(Buffer, "%-12u", MidiFile.Time);
}





// ***************************** AddToList() *******************************
// * Adds the passed string to the Main window's Listbox. If the listbox has
// * 50 strings, then delete the first string to make room for this string.
// * In that way, we don't end up adding thousands of strings to the listbox.
// *************************************************************************

void MfcMidiFile::AddToList(char * Buffer)
{
	CListBox *	hwndList;

	// Get Listbox
	hwndList = (CListBox *)theApp.m_pMainWnd->GetDlgItem(IDC_MIDI);

	// Don't allow more than 50 items
	if (hwndList->GetCount() > 50)
	{
		hwndList->DeleteString(0);
	}

	// Add the string. We won't bother to error check here, as this is a trivial app
	hwndList->AddString((LPCTSTR)Buffer);
}





// ***************************** OnStartMThd() ******************************
// * This is called by MIDIFILE.DLL after the DLL encounters and loads the MThd
// * chunk (ie, at the head of the MIDI file). We simply print out the info
// * that the DLL has loaded into the MIDIFILE structure. At this point, the
// * valid MIDIFILE fields set by the DLL include Handle (if you let the DLL
// * open the MIDI file), FileSize, Format, NumTracks, Division, ChunkSize
// * (minus the 6 loaded bytes of a standard MThd), and TrackNum = -1 (ie, we
// * haven't encountered an MTrk yet).
// **************************************************************************

long APIENTRY MfcMidiFile::OnStartMThd()
{
	char buffer[120];

	// Print the MThd info
	sprintf(&buffer[0], "MThd Format=%d, # of Tracks=%d, Division=%d", MidiFile.Format, MidiFile.NumTracks, MidiFile.Division);
	AddToList(&buffer[0]);

	// Return 0 to indicate no error, and let the DLL continue loading the MIDI file. It will likely
	// next encounter an MTrk chunk and then call our OnStartMTrk callback.
	return(0);
}





// ****************************** OnStartMTrk() *****************************
// * This is called by MIDIFILE.DLL after the DLL encounters and loads the Mtrk
// * header (ie, not the data in the chunk -- just the 8-byte IFF header). We
// * simply print out an indication that this is the start of the next MTrk,
// * and the track number.
// **************************************************************************

long APIENTRY MfcMidiFile::OnStartMTrk()
{
	char buffer[120];

	// Print heading
	sprintf(&buffer[0], "==================== Track #%d ====================", MidiFile.TrackNum);
	AddToList(&buffer[0]);

	// NOTE: Here we could set the MidiFile.Time field to point to some buffer (whose size is at least
	// equal to MidiFile.ChunkSize). In this case, when we return, the DLL will load the entire MTrk
	// data into that buffer and then call our OnStandardEvt callback once. Alternately, if we want
	// the DLL to load one event at a time into the MIDIFILE structure, calling one of our callbacks
	// after each loaded event, then we leave the Time field as 0. So which one of our callbacks gets
	// called for each event? That depends upon the event. (ie, a Meta-Tempo event would result in the
	// DLL calling OnMetaTempo, a standard MIDI Voice message would result in OnStandardEvt getting
	// called, a System Exclusive message would result in OnSysexEvt getting called, etc).

	// Return 0 to indicate no error, and let the DLL continue loading the MIDI file. It will next
	// start loading each event in the track, calling one of our callbacks after each event is loaded
	// into the MIDIFILE
	return(0);
}





// **************************** OnUnknownChunk() *****************************
// * This is called by MIDIFILE.DLL when it encounters a chunk that isn't an MTrk
// * or MThd. We simply print out info about this. NOTES: The ID and ChunkSize are
// * in the MIDIFILE. If TrackNum is -1, then this was encountered after the MThd,
// * but before any MTrk chunks. Otherwise, TrackNum is the MTrk number that this
// * chunk follows.
// ***************************************************************************

long APIENTRY MfcMidiFile::OnUnknownChunk()
{
	register char *				ptr;
	register unsigned char		i;
	char						str[6];
	char						buffer[120];

	// Normally, we would inspect the ID to see if it's something that we recognize. If so, we
	// would read in the chunk with 1 or more calls to MidiReadBytes until ChunkSize is 0.
	// (ie, We could read in the entire chunk with 1 call, or many calls). Note that MidiReadBytes
	// automatically decrements ChunkSize by the number of bytes that we ask it to read. We
	// should never try to read more bytes than ChunkSize. On the other hand, we could read less
	// bytes than ChunkSize. Upon return, the DLL would skip any bytes that we didn't read from
	// that chunk.

	// Copy ID to str[]
	ptr = (char *)&MidiFile.ID;
	for (i=0; i<4; i++)
	{
		str[i] = *(ptr)++;
	}
	str[4] = 0;

	AddToList("******************** Unknown ********************");
	sprintf(&buffer[0], "     ID = %s    ChunkSize=%lu", &str[0], MidiFile.ChunkSize);
	AddToList(&buffer[0]);

	return(0);
}





// ****************************** OnMetaText() ******************************
// * This is called by MIDIFILE.DLL when it encounters a variable length
// * Meta-Event within an MTrk chunk. We figure out which of the several
// * "type of events" it is, and simply print out info about that event. NOTE:
// * The DLL will skip any bytes that we don't read in of this event.
// **************************************************************************

long APIENTRY MfcMidiFile::OnMetaText()
{
	unsigned char	i, p;
	char			chr[60];
	long			result;
	char *			ptr;
	char			buffer[260];

	// Format the event's time
	FormatTime(&buffer[0]);

	// If a text-based event, print what kind of event
	if ( MidiFile.Status < 0x10 )
	{
		if (MidiFile.Status > 9) MidiFile.Status = 0;	// I only know about 9 of the possible 15
		sprintf(&buffer[12], "%s| len=%-6lu|", &Types[MidiFile.Status][0], MidiFile.EventSize);
	}

	// A Proprietary event, or some Meta event that we don't know about (because the MIDI spec doesn't define it)
	else
	{
		switch (MidiFile.Status)
		{
			case 0x7F:
				sprintf(&buffer[12], "Proprietary | len=%-6lu", MidiFile.EventSize);
				break;

			default:
				sprintf(&buffer[12], "Unknown Meta| Type = 0x%02x, len=%lu", MidiFile.Status, MidiFile.EventSize);
		}
	}

	AddToList(&buffer[0]);

	// Print out data bytes as ascii chars (enclosed between < and >) one byte at a time, 16 chars to a line.
	// (And also print out each byte as a hexadecimal value)
	i=0;
	while (MidiFile.EventSize)
	{
		if (!i)
		{
			strcpy(&buffer[0], "         <");
			ptr = &buffer[10];
		}

		// NOTE: MidiReadBytes() decrements EventSize by the number of bytes read
		if ( (result = MidiReadBytes(&MidiFile, (unsigned char *)&chr[i], 1)) ) return(result);

		sprintf(ptr,(isprint(chr[i])||isspace(chr[i])) ? "%c" : "." , chr[i]);
		ptr++;

		if (i++ == 15)
		{
			strcpy(ptr, ">    ");
			ptr += 5;

			for (i=0; i<16; i++)
			{
				sprintf(ptr, "%02x ", chr[i]);
				ptr += 3;
			}
			
			i=0;

			AddToList(&buffer[0]);
		}
	}

	if (i)
	{
		sprintf(ptr++, ">");

		for (p=20-i; p; p--)
		{
			*(ptr)++ = ' ';
		}
		for (p=0; p<i; p++)
		{
			sprintf(ptr, "%02x ", chr[p]);
			ptr += 3;
		}

		AddToList(&buffer[0]);
	}

	return(0);
}





// ******************************* OnSysexEvt() *******************************
// * This is called by MIDIFILE.DLL when it encounters a MIDI SYSEX or ESCAPE
// * (ie, usually REALTIME or COMMON) event within an MTrk chunk. We simply print
// * out info about that event.
// *
// * Note that 0xF7 is used both as SYSEX CONTINUE and ESCAPE. Which one it is
// * depends upon what preceded it. When 0xF7 is used as a SYSEX CONTINUE, it
// * must follow an 0xF0 event that doesn't end with 0xF7, and there can't be
// * intervening Meta-Events or MIDI messages with Status < 0xF0 inbetween. Of
// * course, it IS legal to have an ESCAPE event, (ie, REALTIME) inbetween the
// * 0xF0 and SYSEX CONTINUE events. The DLL helps differentiate between SYSEX
// * CONTINUE and ESCAPE by setting the MIDISYSEX flag when a 0xF0 is encountered.
// * If this flag is not set when you receive a 0xF7 event, then it is definitely
// * an ESCAPE. If MIDISYSEX flag is set, then a 0xF7 could be either SYSEX
// * CONTINUE or an ESCAPE. You can check the first loaded byte; if it is < 0x80
// * or it's 0xF7, then you've got a SYSEX CONTINUE. Whenever you have an event
// * that ends in 0xF7 while the MIDISYSEX is set, then you should clear the
// * MIDISYSEX flag this is the end of a stream of SYSEX packets.
// *
// * NOTE: The DLL will skip any bytes that we don't read in of this event.
// ***************************************************************************

long APIENTRY MfcMidiFile::OnSysexEvt()
{
	unsigned char	chr;
	long			result;
	char			buffer[260];

	// Format the event's time
	FormatTime(&buffer[0]);

	// Load the first byte
	if ( (result = MidiReadBytes(&MidiFile, &chr, 1)) ) return(result);

	// Did we encounter a SYSEX Status (0xF0)? If not, then this can't possibly be a SYSEX event.
	// (ie, It must be an ESCAPE)
	if ( MidiFile.Flags & MIDISYSEX )
	{
		// If the first byte > 0x7F but not 0xF7, then we've really got an ESCAPE
		if (chr > 0x7F && chr != 0xF7) goto escd;

		// OK, this really is a System exclusive message

		// NOTE: Normally, we would allocate some memory to contain the SYSEX message, and
		// read it in via MidiReadBytes. We'd check the last loaded byte of this event, and if
		// a 0xF7, then clear MIDISYSEX. Note that, by simply returning, the DLL will skip any
	    // bytes that we haven't read of this event.

		// Seek to and read the last byte
		chr = 0;
		if (MidiFile.EventSize)
		{
			MidiSeek(&MidiFile, MidiFile.EventSize-1);
			if ( (result = MidiReadBytes(&MidiFile, &chr, 1)) ) return(result);
		}

		// Is this a SYSEX CONTINUE message (ie, starts with 0xF7)?
		if (MidiFile.Status == 0xF7)
		{
			// If last char = 0xF7, then this is the end of the SYSEX stream (ie, last packet)
			if (chr == 0xF7)
			{
				sprintf(&buffer[12], "Last Packet | len=%-6lu|",(MidiFile.EventSize)+2);

				// Clear MIDISYSEX flag
				MidiFile.Flags &= ~MIDISYSEX;
			}

			// Just another packet. There will be more of them coming up
			else
			{
				sprintf(&buffer[12], "Packet %-5ld| len=%-6lu|", Packet++, (MidiFile.EventSize)+2);
			}
		}

		// Must be an ordinary SYSEX message (ie, starting with 0xF0)
		else
		{
			// If last char = 0xF7, then this is a full SYSEX message, rather than the first of
			// a series of packets (ie, more SYSEX CONTINUE events to follow)
			if (chr == 0xF7)
			{
				sprintf(&buffer[12], "Sysex 0xF0  | len=%-6lu|",(MidiFile.EventSize)+3); // +3 to also include the 0xF0 status which
																						// is normally considered part of the SYSEX
				// Clear MIDISYSEX flag
				MidiFile.Flags &= ~MIDISYSEX;
			}

			// This is just the first packet. There will be more coming up
			else
			{
				Packet = 1;
				sprintf(&buffer[12], "First Packet| len=%-6lu|",(MidiFile.EventSize)+3);
			}
		}
	}

	// Must be an escaped event such as MIDI REALTIME or COMMON.
	else
	{
escd:
		sprintf(&buffer[12], "Escape 0x%02x | len=%-6lu|", chr, MidiFile.EventSize+1);
	}

	AddToList(&buffer[0]);

	// NOTE: The DLL will skip any bytes of this event that we don't read in, so let's just
    // ignore the rest of this event even if there are more bytes to it.

	return(0);
}





// ***************************** OnStandardEvt() *****************************
// * This is called by MIDIFILE.DLL after it encounters and loads a MIDI
// * message of Status < 0xF0 (ie, "Voice" message) within an MTrk chunk.
// ***************************************************************************

long APIENTRY MfcMidiFile::OnStandardEvt()
{
	char			buffer[260];
	unsigned char	chan;

	// Get MIDI channel for this event
	chan = MidiFile.Status & 0x0F;

	// Format the event's time
	FormatTime(&buffer[0]);

	// Print info about the MIDI event
	switch (MidiFile.Status & 0xF0)
	{
		case 0x80:
			sprintf(&buffer[12], "Note off    | chan=%2u   | pitch= %-3u | vol=%u", chan, MidiFile.Data[0], MidiFile.Data[1]);
			break;

		case 0x90:
			sprintf(&buffer[12], "Note on     | chan=%2u   | pitch= %-3u | vol=%u", chan, MidiFile.Data[0], MidiFile.Data[1]);
			break;

		case 0xA0:
			sprintf(&buffer[12], "Aftertouch  | chan=%2u   | pitch= %-3u | press=%u", chan, MidiFile.Data[0], MidiFile.Data[1]);
			break;

		case 0xB0:
			sprintf(&buffer[12], "Controller  | chan=%2u   | contr #%-3u | value=%u", chan, MidiFile.Data[0], MidiFile.Data[1]);
			break;

		case 0xC0:
			sprintf(&buffer[12], "Program     | chan=%2u   | pgm #=%u", chan, MidiFile.Data[0]);
			break;

		case 0xD0:
			sprintf(&buffer[12], "Poly Press  | chan=%2u   | press= %-3u", chan, MidiFile.Data[0]);
			break;

		case 0xE0:
			sprintf(&buffer[12], "Pitch Wheel | chan=%2u   | LSB=%u MSB=%u", chan, MidiFile.Data[0], MidiFile.Data[1]);
	}

	AddToList(&buffer[0]);

	return(0);
}





// ******************************* OnMetaSeqNum() ****************************
// * This is called by MIDIFILE.DLL after it encounters and loads a Sequence
// * Number Meta-Event within an MTrk chunk. The event is loaded into our
// * MIDIFILE structure, but note how we redeclare it as a METASEQ structure
// * to more easily access it (ie, easily grab the Sequence Number by accessing
// * the SeqNum field of the METASEQ, which is really just the Data[2] and
// * Data[3] fields of the MIDIFILE).
// **************************************************************************

long APIENTRY MfcMidiFile::OnMetaSeqNum()
{
	METASEQ *	mf;
	char		buffer[260];

	FormatTime(&buffer[0]);
	
	mf = (METASEQ *)&MidiFile;
	sprintf(&buffer[12], "Seq # = %-4u|", mf->SeqNum);

	AddToList(&buffer[0]);

	return(0);
}





// **************************** OnMetaTimeSig() *****************************
// * This is called by MIDIFILE.DLL after it loads a Time Signature
// * Meta-Event within an MTrk chunk.
// **************************************************************************

long APIENTRY MfcMidiFile::OnMetaTimeSig()
{
	METATIME *	mf;
	char		buffer[260];

	FormatTime(&buffer[0]);

	mf = (METATIME *)&MidiFile;
	sprintf(&buffer[12], "Time sig    | %2d/%-7lu| MIDI-clocks/click=%u | 32nds/quarter=%u",
			mf->Nom, mf->Denom, mf->Clocks, mf->_32nds);

	AddToList(&buffer[0]);

	return(0);
}





// ***************************** OnMetaKeySig() ******************************
// * This is called by MIDIFILE.DLL after it loads a Key Signature
// * Meta-Event within an MTrk chunk.
// ***************************************************************************

long APIENTRY MfcMidiFile::OnMetaKeySig()
{
	METAKEY *	mf;
	char		buffer[260];

	FormatTime(&buffer[0]);

	mf = (METAKEY *)&MidiFile;
	sprintf(&buffer[12], "Key sig     | %s %-7s|", &Keys[mf->Key+7][0], mf->Minor ? "Minor" : "Major");

	AddToList(&buffer[0]);

	return(0);
}





// ******************************* OnMetaTempo() *****************************
// * This is called by MIDIFILE.DLL after it loads a Tempo Meta-Event
// * within an MTrk chunk.
// ***************************************************************************

long APIENTRY MfcMidiFile::OnMetaTempo()
{
	METATEMPO *	mf;
	char		buffer[260];

	FormatTime(&buffer[0]);

	mf = (METATEMPO *)&MidiFile;
	sprintf(&buffer[12], "Tempo       | BPM=%-6u| micros/quarter=%lu", mf->TempoBPM, mf->Tempo);

	AddToList(&buffer[0]);

	return(0);
}





// ******************************* OnMetaSMPTE() ******************************
// * This is called by MIDIFILE.DLL after it loads a SMPTE Meta-Event within
// * an MTrk chunk.
// ****************************************************************************

long APIENTRY MfcMidiFile::OnMetaSMPTE()
{
	METASMPTE *	mf;
	char		buffer[260];

	FormatTime(&buffer[0]);

	mf = (METASMPTE *)&MidiFile;
	sprintf(&buffer[12], "SMPTE       | hour=%u | min=%u | sec=%u | frame=%u | subs=%u",
			mf->Hours, mf->Minutes, mf->Seconds, mf->Frames, mf->SubFrames);

	AddToList(&buffer[0]);

	return(0);
}





// ****************************** OnMetaEOT() *******************************
// * This is called by MIDIFILE.DLL after it loads an End of Track
// * Meta-Event within an MTrk chunk.
// **************************************************************************

long APIENTRY MfcMidiFile::OnMetaEOT()
{
	char		buffer[260];
	FormatTime(&buffer[0]);
	strcpy(&buffer[12], "End of track");

	AddToList(&buffer[0]);

	return(0);
}





// ******************************* OnMetaPort() *******************************
// * This is called by MIDIFILE.DLL after it loads a Port Number Meta-Event
// * within an MTrk chunk.
// ****************************************************************************

long APIENTRY MfcMidiFile::OnMetaPort()
{
	METAPORT *	mf;
	char		buffer[260];

	FormatTime(&buffer[0]);

	mf = (METAPORT *)&MidiFile;
	sprintf(&buffer[12], "Port Number | Port Number = %u", mf->PortNumber);

	AddToList(&buffer[0]);

	return(0);
}





// ******************************* OnMetaChan() *******************************
// * This is called by MIDIFILE.DLL after it loads a MIDI Channel Number
// * Meta-Event within an MTrk chunk.
// ****************************************************************************

long APIENTRY MfcMidiFile::OnMetaChan()
{
	METACHAN *	mf;
	char		buffer[260];

	FormatTime(&buffer[0]);

	mf = (METACHAN *)&MidiFile;
	sprintf(&buffer[12], "Chan Number | Chan Number = %u", mf->ChanNumber);

	AddToList(&buffer[0]);

	return(0);
}




// **************************** MfcMidiFile() **************************
// MfcMidiFile constructor. Calls the CMidiFile constructor, and then
// initializes the MIDICALL fields with our callback function addresses.

MfcMidiFile::MfcMidiFile() : CMidiFile()
{
#ifdef BUFFERED_IO
	// I'll do my own buffered Open, Read, Seek, and Close
	ReadWriteMidi = (long (__stdcall CMidiFile::*)(unsigned char *,unsigned long))OnReadWriteMidi;
	OpenMidi = (long (__stdcall CMidiFile::*)(void))OnOpenMidi;
	SeekMidi = (long (__stdcall CMidiFile::*)(long))OnSeekMidi;
	CloseMidi = (long (__stdcall CMidiFile::*)(void))OnCloseMidi;
#endif
#ifdef RAMFILE_IO
	// I'll do my own Open, Read, Seek, and Close reading from RAM
	ReadWriteMidi = (long (__stdcall CMidiFile::*)(unsigned char *,unsigned long))OnReadWriteMidi;
	OpenMidi = (long (__stdcall CMidiFile::*)(void))OnOpenMidi;
	SeekMidi = (long (__stdcall CMidiFile::*)(long))OnSeekMidi;
	CloseMidi = (long (__stdcall CMidiFile::*)(void))OnCloseMidi;
#endif
	// Store Callback function ptrs for MIDIFILE.DLL
	StartMThd = (long (__stdcall CMidiFile::*)(void))OnStartMThd;
	StartMTrk = (long (__stdcall CMidiFile::*)(void))OnStartMTrk;
	UnknownChunk = (long (__stdcall CMidiFile::*)(void))OnUnknownChunk;
	MetaText = (long (__stdcall CMidiFile::*)(void))OnMetaText;
	SysexEvt = (long (__stdcall CMidiFile::*)(void))OnSysexEvt;
	StandardEvt = (long (__stdcall CMidiFile::*)(void))OnStandardEvt;
	MetaSeqNum = (long (__stdcall CMidiFile::*)(void))OnMetaSeqNum;
	MetaTimeSig = (long (__stdcall CMidiFile::*)(void))OnMetaTimeSig;
	MetaKeySig = (long (__stdcall CMidiFile::*)(void))OnMetaKeySig;
	MetaTempo = (long (__stdcall CMidiFile::*)(void))OnMetaTempo;
	MetaSMPTE = (long (__stdcall CMidiFile::*)(void))OnMetaSMPTE;
	MetaEOT = (long (__stdcall CMidiFile::*)(void))OnMetaEOT;
//	MetaPort = (long (__stdcall CMidiFile::*)(void))OnMetaPort;	// If we want the DLL to convert Device Name Meta-Events to Port Number Meta-Events
	MetaPort = 0;												// If we don't want the DLL to convert Device Name Meta-Events
	MetaChan = (long (__stdcall CMidiFile::*)(void))OnMetaChan;
}





// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// What follows are just some extra callbacks you can add for some optional
// features; to add buffered file I/O, or to read a MIDI file in RAM.

#ifdef BUFFERED_IO

// **************************** OnOpenMidi() **************************
// Demonstrates adding a callback whereby we open the MIDI file using
// standard C file I/O, fopen(), for buffered I/O, rather than letting
// the DLL open the MIDI file. We also add OnReadWriteMidi, OnSeekMidi,
// and OnCloseMidi callbacks to manage our own file I/O instead of
// relying upon the DLL to read and seek bytes. Note that we could have
// opened the file back in LoadMidi before we called MidiReadFile(), but
// we'd still need an OpenMidi callback which did nothing more than return
// 0.

long APIENTRY MfcMidiFile::OnOpenMidi()
{
	struct _stat buffer;

	// Retrieve the filename ptr which we stored in the MIDIFILE handle field, and open the file
	if(!(MidiFile.Handle  = (FILE *)fopen((const char *)MidiFile.Handle, "rb" )))
	{
		// If we wanted to force MidiReadFile() to return some error code (to LoadMidi) that we could
		// distinguish from an error that the DLL returns, we could return some negative number here.
		// The DLL does not return any negative error numbers via MidiReadFile(), but our callback can
		// do that. Nevertheless, I'll just return an appropriate error code that the DLL would return
		// if it couldn't open a file, and thus, we'll simply display the DLL's error message. Note that
		// this will abort the read operation on return.
		return(MIDIERRFILE);
	}

	// We must also set MidiFile.FileSize
	if (_fstat((int)MidiFile.Handle, &buffer)) return(MIDIERRINFO);
	MidiFile.FileSize = buffer.st_size;

	// Success
	return(0);
}






// ************************* OnReadWriteMidi() *************************
// Demonstrates adding a callback whereby we read bytes from the MIDI
// file using standard C file I/O, fread(), for buffered I/O, rather than
// letting the DLL read from the MIDI file.

long APIENTRY MfcMidiFile::OnReadWriteMidi(unsigned char * Buffer, unsigned long Count)
{
	if (fread((void *)Buffer, sizeof(char), Count, (FILE *)MidiFile.Handle) != Count)
	{
		// If we wanted to force MidiReadFile() to return some error code (to LoadMidi) that we could
		// distinguish from an error that the DLL returns, we could return some negative number here (other
		// than -1, which this particular callback is not allowed to return). The DLL does not return any
		// negative error numbers via MidiReadFile(), but our callback can do that. Nevertheless, I'll just
		// return an appropriate error code that the DLL would return if it couldn't read from a file, and
		// thus, we'll simply display the DLL's error message. Note that this will abort the read operation
		// on return.
		return(MIDIERRREAD);
	}

	// Success
	return(0);
}





// **************************** OnSeekMidi() ****************************
// Demonstrates adding a callback whereby we seek within the MIDI
// file using standard C file I/O, fseek(), for buffered I/O, rather
// than letting the DLL seek within the MIDI file.

long APIENTRY MfcMidiFile::OnSeekMidi(long Offset)
{
	if (fseek((FILE *)MidiFile.Handle, Offset, SEEK_CUR))
	{
		// If we wanted to force MidiReadFile() to return some error code (to LoadMidi) that we could
		// distinguish from an error that the DLL returns, we could return some negative number here.
		// The DLL does not return any negative error numbers via MidiReadFile(), but our callback can
		// do that. Nevertheless, I'll just return an appropriate error code that the DLL would return
		// if it couldn't seek, and thus, we'll simply display the DLL's error message. Note that
		// this will abort the read operation on return.
		return(MIDIERRSEEK);
	}

	// Success
	return(0);
}





// **************************** OnCloseMidi() **************************
// Demonstrates adding a callback whereby we close the MIDI file using
// standard C file I/O, fclose(), for buffered I/O, rather than letting
// the DLL close the MIDI file. Note that we don't have to close the
// file here if we want it to be left open when MidiReadFile returns
// back in LoadMidi. But we'd still need a CloseMidi callback which did
// nothing.

long APIENTRY MfcMidiFile::OnCloseMidi()
{
	// Close the file. Note that this never gets called if my OnOpenMidi callback returned an error.
	// But, if any other subsequent errors occur, this gets called before the read operation
	// aborts. So, I can always be assured that the file handle gets closed, even if an error
	// occurs
	fclose((FILE *)MidiFile.Handle);

	// Success. (Actually, we don't need a return for this callback, but it makes it easier to setup the
	// MIDICALL structure)
	return(0);
}
#endif





#ifdef RAMFILE_IO

unsigned char * RAM_Pointer;

// Let's create a MIDI file right here in our program's data section.
unsigned char RAM_File[] = {'M','T','h','d', 0, 0, 0, 6, 0, 0, 0, 1, 0, 96,
 'M','T','r','k', 0, 0, 0, 27, 
 0, 0xFF, 0, 2, 0, 1,
 0, 0xFF, 3, 8, 'M','y',' ','T','r','a','c','k',
 0, 0xFF, 0x21, 1, 0,
 96, 0xFF, 0x2F, 0};

// **************************** OnOpenMidi() **************************
// Demonstrates adding a callback whereby we setup to parse a MIDI file
// that is in RAM, rather than letting the DLL open a MIDI file on disk.
// We also add OnReadWriteMidi, OnSeekMidi, and OnCloseMidi callbacks to
// manage our parsing of the file in RAM instead of relying upon the DLL
// to read and seek to disk.

long APIENTRY MfcMidiFile::OnOpenMidi()
{
	// Initialize a global pointer to the start of our MIDI file in RAM. Note that
	// we could have done this back in LoadMidi before calling MidiReadFile, but
	// we'd still need an OpenMidi callback which did nothing but return 0.
	RAM_Pointer = &RAM_File[0];

	// We must also set MidiFile.FileSize
	MidiFile.FileSize = sizeof(RAM_File);

	// Success
	return(0);
}






// ************************* OnReadWriteMidi() *************************
// Demonstrates adding a callback whereby we setup to parse a MIDI file
// that is in RAM, rather than letting the DLL read from a MIDI file on disk.

long APIENTRY MfcMidiFile::OnReadWriteMidi(unsigned char * Buffer, unsigned long Count)
{
	// Copy the requested bytes into the DLL supplied buffer
	memcpy(Buffer, RAM_Pointer, Count);
	
	// Update our RAM pointer
	RAM_Pointer += Count;
	
	// Success
	return(0);
}





// **************************** OnSeekMidi() ****************************
// Demonstrates adding a callback whereby we setup to parse a MIDI file
// that is in RAM, rather than letting the DLL read from a MIDI file on disk.

long APIENTRY MfcMidiFile::OnSeekMidi(long Offset)
{
	// Adjust our RAM pointer by the Offset
	RAM_Pointer += Offset;
	
	// Success
	return(0);
}





// **************************** OnCloseMidi() **************************
// Demonstrates adding a callback whereby we setup to parse a MIDI file
// that is in RAM, rather than letting the DLL read from a MIDI file on disk.

long APIENTRY MfcMidiFile::OnCloseMidi()
{
	// This callback doesn't need to do anything, but we still need to supply it

	// Success. (Actually, we don't need a return for this callback, but it makes it easier to setup the
	// MIDICALL structure)
	return(0);
}
#endif
