/* ===========================================================================
 * MfRead.c
 *
 * Reads in and displays information about a MIDI file.
 * =========================================================================
 */

#include <windows.h>
#include <stdio.h>
#include "..\midifile.h"

/* Uncomment this define if you want standard C buffered file I/O */
/* #define BUFFERED_IO 1 */

/* Uncomment this define if you want an example of reading a MIDI file in RAM */
/* #define RAMFILE_IO 1 */

/* function definitions */
void initfuncs();





/* Need a MIDICALL structure for the DLL */
MIDICALL cb;





/* Need a MIDIFILE structure for the DLL */
MIDIFILE Mfs;





/* A text description of the various text-based Meta-Events (types 0x01 to 0x0F) */
char Types[10][14] = {
	"Unknown Text",
	"Text event  ",
	"Copyright   ",
	"Track Name  ",
	"Instrument  ",
	"Lyric       ",
	"Marker      ",
	"Cue Point   ",
	"Device Name ",
	"Pgm Name    ",
};

/* For Key Signature Meta-Event */
unsigned char Keys[15][3] = { "Cb", "Gb", "Db", "Ab", "Eb", "Bb", " F", " C", " G", " D", " A", " E", " B", "F#", "C#" };

/* For Sysex */
unsigned long Packet;

/* For compressed listing (ie, don't list individual events in MTrk */
unsigned char Compress=0;
unsigned long Counts[27];

unsigned char * Strptrs[] = { "Note Off", "Note On", "Aftertouch", "Controller", "Program", "Channel Pressure",
			"Pitch Wheel", "System Exclusive", "Escaped", "Unknown Text",
			"Text", "Copyright", "Track Name", "Instrument Name", "Lyric",
			"Marker", "Cue Point", "Device Name", "Pgm Name",
			"Proprietary", "Unknown Meta", "Key Signature",
			"Tempo", "Time Signature", "SMPTE", "Port Number", "Chan Number"};

unsigned char * Language[] = { "English", "Other" };





/********************************** main() **********************************
 * Program entry point. Calls the MIDIFILE.DLL function to read in a MIDI file.
 * The callbacks do all of the real work of displaying info about the file.
 ****************************************************************************/

void main(int argc, char *argv[], char *envp[])
{
	long			result;
	unsigned char *	ptr;
	unsigned char	buf[80];

#ifndef RAMFILE_IO
	/* If no filename arg supplied by user, exit with usage info */
	if ( argc < 2 )
	{
		printf("This program displays the contents of a MIDI (sequencer)\nfile.\n\n");
		printf("Syntax: MFREAD.EXE [filename] /I\n");
		printf("    where /I means list info about the MTrk but not each event\r\n");
		exit(1);
	}

	/* See if he wants compressed listing */
	if (argc > 2)
	{
		if (strncmp(argv[2],"/I",2) ||strncmp(argv[2],"/i",2) )
		{
			Compress=1;
		}
	}
#endif

	/* Print out the DLL version/language */
	ptr = MidiGetVers();
	if ((result = *(ptr+1))) result = 1;
	printf("MIDIFILE DLL version #%ld, language: %s\n\n", *ptr, Language[result]);

	/* Initialize the pointers to our callback functions (ie, for the MIDIFILE.DLL to call) */
	initfuncs();

#ifndef RAMFILE_IO
	/* Store pointer to the Filename */
	Mfs.Handle = (HANDLE)argv[1];
#endif

	/* Set the Flags */
	Mfs.Flags = 0;

	/* Tell MIDIFILE.DLL to read in the file, calling my callback functions */
	result = MidiReadFile(&Mfs);

	/* Print out error message. NOTE: For 0, DLL returns a "Successful MIDI file load" message.
	Also note that DLL returns the length even though we ignore it. */
	MidiGetErr(&Mfs, &buf[0], 80, result);
	printf(&buf[0]);

	exit(0);
}





/******************************** prtime() *******************************
 * Prints out the time that the event occurs upon. This time is referenced
 * from 0 (ie, as opposed to the previous event in the track as is done
 * with delta-times in the MIDI file). The MIDIFILE DLL automatically
 * maintains the MIDIFILE's Time field, updating it for the current event.
 *************************************************************************/

void prtime(MIDIFILE * mf)
{
	if (!Compress) printf("%8ld |",mf->Time);
}





/******************************** startMThd() ******************************
 * This is called by MIDIFILE.DLL after the DLL encounters and loads the MThd
 * chunk (ie, at the head of the MIDI file). We simply print out the info
 * that the DLL has loaded into the MIDIFILE structure. At this point, the
 * valid MIDIFILE fields set by the DLL include Handle (if you let the DLL
 * open the MIDI file), FileSize, Format, NumTracks, Division, ChunkSize
 * (minus the 6 loaded bytes of a standard MThd), and TrackNum = -1 (ie, we
 * haven't encountered an MTrk yet).
 **************************************************************************/

long APIENTRY startMThd(MIDIFILE * mf)
{
	/* Print the MThd info */
	printf("MThd Format=%d, # of Tracks=%d, Division=%d\r\n",mf->Format,mf->NumTracks,mf->Division);

	/* Return 0 to indicate no error, and let the DLL continue loading the MIDI file. It will likely
	   next encounter an MTrk chunk and then call our startMTrk callback */
	return(0);
}





/******************************** startMTrk() *****************************
 * This is called by MIDIFILE.DLL after the DLL encounters and loads the Mtrk
 * header (ie, not the data in the chunk -- just the 8-byte IFF header). We
 * simply print out an indication that this is the start of the next MTrk,
 * and the track number.
 **************************************************************************/

long APIENTRY startMTrk(MIDIFILE * mf)
{
	unsigned short i;

	/* Print heading */
	printf("\r\n==================== Track #%d ====================\r\n   %s      Event\r\n", mf->TrackNum, (Compress) ? "Total" : "Time" );

	/* If compressing info on display, init array for this track */
	if (Compress)
	{
		for (i=0; i<27; i++)
		{
			Counts[i] = 0;
		}
	}

	// NOTE: Here we could set the MidiFile.Time field to point to some buffer (whose size is at least
	// equal to MidiFile.ChunkSize). In this case, when we return, the DLL will load the entire MTrk
	// data into that buffer and then call our standardEvt callback once. Alternately, if we want
	// the DLL to load one event at a time into the MIDIFILE structure, calling one of our callbacks
	// after each loaded event, then we leave the Time field as 0. So which one of our callbacks gets
	// called for each event? That depends upon the event. (ie, a Meta-Tempo event would result in the
	// DLL calling metatempo, a standard MIDI Voice message would result in standardEvt getting
	// called, a System Exclusive message would result in sysexEvt getting called, etc).

	// Return 0 to indicate no error, and let the DLL continue loading the MIDI file. It will next
	// start loading each event in the track, calling one of our callbacks after each event is loaded
	// into the MIDIFILE
	return(0);
}





/****************************** standardEvt() ******************************
 * This is called by MIDIFILE.DLL after it encounters and loads a MIDI
 * message of Status < 0xF0 (ie, "Voice" message) within an MTrk chunk.
 ***************************************************************************/

long APIENTRY standardEvt(MIDIFILE * mf)
{
	/* Get MIDI channel for this event */
	unsigned char chan = mf->Status & 0x0F;

	if (!Compress)
	{
		/* Print out the event's time */
		prtime((MIDIFILE *)mf);

		switch ( mf->Status & 0xF0 )
		{
			case 0x80:
				printf("Note off    | chan=%2d   | pitch= %-3d | vol=%d\r\n", chan, mf->Data[0], mf->Data[1]);
				break;

			case 0x90:
				printf("Note on     | chan=%2d   | pitch= %-3d | vol=%d\r\n", chan, mf->Data[0], mf->Data[1]);
				break;

			case 0xA0:
				printf("Aftertouch  | chan=%2d   | pitch= %-3d | press=%d\r\n", chan, mf->Data[0], mf->Data[1]);
				break;

			case 0xB0:
				printf("Controller  | chan=%2d   | contr #%-3d | value=%d\r\n", chan, mf->Data[0], mf->Data[1]);
				break;

			case 0xC0:
				printf("Program     | chan=%2d   | pgm #=%d\r\n", chan, mf->Data[0]);
				break;

			case 0xD0:
				printf("Poly Press  | chan=%2d   | press= %-3d\r\n", chan, mf->Data[0]);
				break;

			case 0xE0:
				printf("Pitch Wheel | chan=%2d   | LSB=%d MSB=%d\r\n", chan, mf->Data[0], mf->Data[1]);
		}
	}
	else
	{
		switch ( mf->Status & 0xF0 )
		{
			case 0x80:
				Counts[0]+=1;
				break;

			case 0x90:
				Counts[1]+=1;
				break;

			case 0xA0:
				Counts[2]+=1;
				break;

			case 0xB0:
				Counts[3]+=1;
				break;

			case 0xC0:
				Counts[4]+=1;
				break;

			case 0xD0:
				Counts[5]+=1;
				break;

			case 0xE0:
				Counts[6]+=1;
		}
	}
	return(0);
}





/******************************** sysexEvt() ********************************
 * This is called by MIDIFILE.DLL when it encounters a MIDI SYSEX or ESCAPE
 * (ie, usually REALTIME or COMMON) event within an MTrk chunk. We simply print
 * out info about that event.
 *
 * Note that 0xF7 is used both as SYSEX CONTINUE and ESCAPE. Which one it is
 * depends upon what preceded it. When 0xF7 is used as a SYSEX CONTINUE, it
 * must follow an 0xF0 event that doesn't end with 0xF7, and there can't be
 * intervening Meta-Events or MIDI messages with Status < 0xF0 inbetween. Of
 * course, it IS legal to have an ESCAPE event, (ie, REALTIME) inbetween the
 * 0xF0 and SYSEX CONTINUE events. The DLL helps differentiate between SYSEX
 * CONTINUE and ESCAPE by setting the MIDISYSEX flag when a 0xF0 is encountered.
 * If this flag is not set when you receive a 0xF7 event, then it is definitely
 * an ESCAPE. If MIDISYSEX flag is set, then a 0xF7 could be either SYSEX
 * CONTINUE or an ESCAPE. You can check the first loaded byte; if it is < 0x80
 * or it's 0xF7, then you've got a SYSEX CONTINUE. Whenever you have an event
 * that ends in 0xF7 while the MIDISYSEX is set, then you should clear the
 * MIDISYSEX flag this is the end of a stream of SYSEX packets.
 *
 * NOTE: The DLL will skip any bytes that we don't read in of this event.
 ***************************************************************************/

long APIENTRY sysexEvt(MIDIFILE * mf)
{
	unsigned char	chr;
	long			result;

	prtime((MIDIFILE *)mf);

	/* Load the first byte */
	if ( (result =MidiReadBytes(mf, &chr, 1)) ) return(result);

	/* Did we encounter a SYSEX (0xF0)? If not, then this can't possibly be a SYSEX event.
	(ie, It must be an ESCAPE) */
	if ( mf->Flags & MIDISYSEX )
	{
		/* If the first byte > 0x7F but not 0xF7, then we've really got an ESCAPE */
		if (chr > 0x7F && chr != 0xF7) goto escd;

		/* OK, this really is a System exclusive message */

		/* NOTE: Normally, we would allocate some memory to contain the SYSEX message, and
		   read it in via MidiReadBytes. We'd check the last loaded byte of this event, and if
		   a 0xF7, then clear MIDISYSEX. Note that, by simply returning, the DLL will skip any
	       bytes that we haven't read of this event */

		/* Seek to and read the last byte */
		chr=0;
		if (mf->EventSize)
		{
			MidiSeek(mf, mf->EventSize-1);
			if ( (result =MidiReadBytes(mf, &chr, 1)) ) return(result);
		}

		/* Is this a SYSEX CONTINUE message (ie, starts with 0xF7)? */
		if (mf->Status == 0xF7)
		{
			/* If last char = 0xF7, then this is the end of the SYSEX stream (ie, last packet) */
			if (chr == 0xF7)
			{
				if (!Compress)
					printf("Last Packet | len=%-6lu|\r\n",(mf->EventSize)+2);

				// Clear MIDISYSEX flag
				mf->Flags &= ~MIDISYSEX;

			}

			/* Just another packet. There will be more of them coming up */
			else
			{
				if (!Compress)
					printf("Packet %-5lu| len=%-6lu|\r\n", Packet++, (mf->EventSize)+2);
			}
		}

		/* Must be an ordinary SYSEX message (ie, starting with 0xF0) */
		else
		{
			if (!Compress)
			{
				/* If last char = 0xF7, then this is a full SYSEX message, rather than the first of
				a series of packets (ie, more SYSEX CONTINUE events to follow) */
				if (chr == 0xF7)
				{
					printf("Sysex 0xF0  | len=%-6lu|\r\n",(mf->EventSize)+3); /* +3 to also include the 0xF0 status which
									      is normally considered part of the SYSEX */

					/* Clear MIDISYSEX flag */
					mf->Flags &= ~MIDISYSEX;
				}

				/* This is just the first packet. There will be more coming up */
				else
				{
					Packet = 1;
					printf("First Packet| len=%-6lu|\r\n",(mf->EventSize)+3);
				}
			}
			else
				Counts[7] += 1;
		}
	}

	/* Must be an escaped event such as MIDI REALTIME or COMMON. */
	else
	{
escd:
		if (!Compress)
			printf("Escape 0x%02x | len=%-6lu|\r\n", chr, mf->EventSize+1);
		else
			Counts[8] += 1;

		/* NOTE: The DLL will skip any bytes of this event that we don't read in, so let's just
	     ignore the rest of this event even if there are more bytes to it. */
	}

	return(0);
}





/******************************** metatext() ******************************
 * This is called by MIDIFILE.DLL when it encounters a variable length
 * Meta-Event within an MTrk chunk. We figure out which of the several
 * "type of events" it is, and simply print out info about that event. NOTE:
 * The DLL will skip any bytes that we don't read in of this event.
 **************************************************************************/

long APIENTRY metatext(MIDIFILE * mf)
{
	unsigned short	i, p;
	unsigned char	chr[60];
	long			result;

	/* Print the event's time */
	prtime((MIDIFILE *)mf);

	/* If a text-based event, print what kind of event */
	if ( mf->Status < 0x10 )
	{
		if (mf->Status > 9) mf->Status=0;	/* I only know about 9 of the possible 15 */
		if (!Compress)
			printf("%s| len=%-6ld|", &Types[mf->Status][0], mf->EventSize);
		else
		{
			Counts[9+mf->Status]+=1;
		}
	}

	/* A Proprietary event, or some Meta event that we don't know about (because the MIDI spec doesn't define it) */
	else
	{
		switch (mf->Status)
		{
			case 0x7F:
				if (!Compress)
					printf("Proprietary | len=%-6ld|", mf->EventSize);
				else
					Counts[19]+=1;
				break;

			default:
				if (!Compress)
					printf("Unknown Meta| Type = 0x%02x, len=%ld", mf->Status, mf->EventSize);
				else
					Counts[20]+=1;
		}
	}

	/* Print out data bytes as ascii chars (enclosed between < and >) one byte at a time, 16 chars to a line.
	   (And also print out each byte as a hexadecimal value) */
	if (!Compress)
	{
		i=0;
		while (mf->EventSize)
		{
			if (!i) printf("\r\n         <");

			/* NOTE: MidiReadBytes() decrements EventSize by the number of bytes read */
			if ( (result =MidiReadBytes(mf, &chr[i], 1)) ) return(result);

			printf( (isprint(chr[i])||isspace(chr[i])) ? "%c" : "." , chr[i]);

			if (i++ == 15)
			{
				printf(">    ");
				for (i=0; i<16; i++)
				{
					printf("%02x ", chr[i]);
				}
				i=0;
			}
		}

		if (i)
		{
			printf(">");
			for (p=20-i; p; p--)
			{
				printf(" ");
			}
			for (p=0; p<i; p++)
			{
				printf("%02x ", chr[p]);
			}
			printf("\r\n");
		}
	}

	return(0);
}





/******************************* metaseqnum() *****************************
 * This is called by MIDIFILE.DLL after it encounters and loads a Sequence
 * Number Meta-Event within an MTrk chunk. The event is loaded into our
 * MIDIFILE structure, but note how we redeclare it as a METASEQ structure
 * to more easily access it (ie, easily grab the Sequence Number by accessing
 * the SeqNum field of the METASEQ, which is really just the Data[2] and
 * Data[3] fields of the MIDIFILE).
 **************************************************************************/

long APIENTRY metaseqnum(METASEQ * mf)
{
	prtime((MIDIFILE *)mf);
	printf("Seq # = %-4d|\r\n", mf->SeqNum);
	return(0);
}





/******************************** metaend() *******************************
 * This is called by MIDIFILE.DLL after it loads an End of Track
 * Meta-Event within an MTrk chunk.
 **************************************************************************/

long APIENTRY metaend(METAEND * mf)
{
	unsigned short i;

	if (!Compress)
	{
		prtime((MIDIFILE *)mf);
		printf("End of track\r\n");
	}

	/* If compressing info on display, print out the counts now that we're at the end of track */
	else
	{
		for (i=0; i<27; i++)
		{
			printf("%-10ld %s events.\r\n", Counts[i], Strptrs[i]);
		}
	}

	return(0);
}




/******************************** metakey() ********************************
 * This is called by MIDIFILE.DLL after it loads a Key Signature
 * Meta-Event within an MTrk chunk.
 ***************************************************************************/

long APIENTRY metakey(METAKEY * mf)
{
	prtime((MIDIFILE *)mf);

	if (!Compress)
		printf("Key sig     | %s %-7s|\r\n", &Keys[mf->Key+7][0], mf->Minor ? "Minor" : "Major");
	else
		Counts[21]+=1;

	return(0);
}





/******************************** metatempo() ******************************
 * This is called by MIDIFILE.DLL after it loads a Tempo Meta-Event
 * within an MTrk chunk.
 ***************************************************************************/

long APIENTRY metatempo(METATEMPO * mf)
{
	prtime((MIDIFILE *)mf);

	if (!Compress)
		printf("Tempo       | BPM=%-6d| micros/quarter=%ld \r\n", mf->TempoBPM, mf->Tempo);
	else
		Counts[22]+=1;

	return(0);
}





/******************************** metatime() *******************************
 * This is called by MIDIFILE.DLL after it loads a Time Signature
 * Meta-Event within an MTrk chunk.
 **************************************************************************/

long APIENTRY metatime(METATIME * mf)
{
	prtime((MIDIFILE *)mf);

	if (!Compress)
		printf("Time sig    | %2d/%-7d| MIDI-clocks/click=%d | 32nds/quarter=%d\r\n",
				mf->Nom, mf->Denom, mf->Clocks, mf->_32nds);
	else
		Counts[23]+=1;

	return(0);
}





/******************************** metasmpte() ********************************
 * This is called by MIDIFILE.DLL after it loads a SMPTE Meta-Event within
 * an MTrk chunk.
 ****************************************************************************/

long APIENTRY metasmpte(METASMPTE * mf)
{
	prtime((MIDIFILE *)mf);

	if (!Compress)
		printf("SMPTE       | hour=%d | min=%d | sec=%d | frame=%d | subs=%d\r\n",
				mf->Hours, mf->Minutes, mf->Seconds, mf->Frames, mf->SubFrames);
	else
		Counts[24]+=1;

	return(0);
}





/********************************* metaport() *******************************
 * This is called by MIDIFILE.DLL after it loads a Port Number Meta-Event
 * within an MTrk chunk.
 ****************************************************************************/

long APIENTRY metaport(METAPORT * mf)
{
	prtime((MIDIFILE *)mf);

	if (!Compress)
		printf("Port Number | Port=%u\r\n", mf->PortNumber);
	else
		Counts[25]+=1;

	return(0);
}





/********************************* metachan() *******************************
 * This is called by MIDIFILE.DLL after it loads a MIDI Channel Number Meta-Event
 * within an MTrk chunk.
 ****************************************************************************/

long APIENTRY metachan(METACHAN * mf)
{
	prtime((MIDIFILE *)mf);

	if (!Compress)
		printf("Chan Number | Chan=%u\r\n", mf->ChanNumber);
	else
		Counts[26]+=1;

	return(0);
}





/********************************* unknown() ********************************
 * This is called by MIDIFILE.DLL when it encounters a chunk that isn't an MTrk
 * or MThd. We simply print out info about this. NOTES: The ID and ChunkSize are
 * in the MIDIFILE. If TrackNum is -1, then this was encountered after the MThd,
 * but before any MTrk chunks. Otherwise, TrackNum is the MTrk number that this
 * chunk follows.
 ***************************************************************************/

long APIENTRY unknown(MIDIFILE * mf)
{
	register unsigned char *	ptr = (unsigned char *)&mf->ID;
	register unsigned short		i;
	unsigned char				str[6];

	/* Normally, we would inspect the ID to see if it's something that we recognize. If so, we
	would read in the chunk with 1 or more calls to MidiReadBytes until ChunkSize is 0.
	(ie, We could read in the entire chunk with 1 call, or many calls). Note that MidiReadBytes
	automatically decrements ChunkSize by the number of bytes that we ask it to read. We
	should never try to read more bytes than ChunkSize. On the other hand, we could read less
	bytes than ChunkSize. Upon return, the DLL would skip any bytes that we didn't read from
	that chunk. */

	/* Copy ID to str[] */
	for (i=0; i<4; i++)
	{
		str[i] = *(ptr)++;
	}
	str[4] = 0;

	printf("\r\n******************** Unknown ********************\r\n");
	printf("     ID = %s    ChunkSize=%ld\r\n", &str[0], mf->ChunkSize);

	return(0);
}






/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
 * What follows are just some extra callbacks you can add for some optional
 * features; to add buffered file I/O, or to read a MIDI file in RAM.
 */

#ifdef BUFFERED_IO

/******************************* midi_open() *******************************
 * Demonstrates adding a callback whereby we open the MIDI file using
 * standard C file I/O, fopen(), for buffered I/O, rather than letting
 * the DLL open the MIDI file. We also add midi_read, midi_seek, and
 * midi_close callbacks to manage our own file I/O instead of relying upon
 * the DLL to read and seek bytes. Note that we could have opened the file
 * back in main() before we called MidiReadFile(), but we'd still need an
 * OpenMidi callback which did nothing more than return 0.
 **************************************************************************/

long APIENTRY midi_open(MIDIFILE * mf)
{
	struct _stat buffer;

	/* Retrieve the filename ptr which we stored in the MIDIFILE handle field, and open the file */
	if(!(mf->Handle  = (FILE *)fopen((const char *)mf->Handle, "rb")))
	{
		/* If we wanted to force MidiReadFile() to return some error code (to main) that we could
		   distinguish from an error that the DLL returns, we could return some negative number here.
		   The DLL does not return any negative error numbers via MidiReadFile(), but our callback can
		   do that. Nevertheless, I'll just return an appropriate error code that the DLL would return
		   if it couldn't open a file, and thus, we'll simply display the DLL's error message. Note that
		   this will abort the read operation on return */
		return(MIDIERRFILE);
	}

	/* We must also set MidiFile.FileSize */
	if (_fstat((int)mf->Handle, &buffer)) return(MIDIERRINFO);
	mf->FileSize = buffer.st_size;

	/* Success */
	return(0);
}





/****************************** midi_close() ********************************
 * Demonstrates adding a callback whereby we close the MIDI file using
 * standard C file I/O, fclose(), for buffered I/O, rather than letting
 * the DLL close the MIDI file. Note that we don't have to close the
 * file here if we want it to be left open when MidiReadFile returns
 * back to main(). But we'd still need a CloseMidi callback which did
 * nothing.
 ***************************************************************************/

long APIENTRY midi_close(MIDIFILE * mf)
{
	/* Close the file. Note that this never gets called if my MidiOpen callback returned an error.
	But, if any other subsequent errors occur, this gets called before the read operation
	aborts. So, I can always be assured that the file handle gets closed, even if an error
	occurs. */
	fclose((FILE *)mf->Handle);

	/* Success. (Actually, we don't need a return for this callback, but it makes it easier to setup the
	   CALLBACK structure) */
	return(0);
}





/******************************* midi_seek() ******************************
 * Demonstrates adding a callback whereby we seek within the MIDI
 * file using standard C file I/O, fseek(), for buffered I/O, rather
 * than letting the DLL seek within the MIDI file.
 **************************************************************************/

long APIENTRY midi_seek(MIDIFILE * mf, long amt, unsigned long type)
{
	if (fseek((FILE *)MidiFile.Handle, Offset, SEEK_CUR))
	{
		/* If we wanted to force MidiReadFile() to return some error code (to main) that we could
		   distinguish from an error that the DLL returns, we could return some negative number here.
		   The DLL does not return any negative error numbers via MidiReadFile(), but our callback can
		   do that. Nevertheless, I'll just return an appropriate error code that the DLL would return
		   if it couldn't seek, and thus, we'll simply display the DLL's error message. Note that
		   this will abort the read operation on return */
		return(MIDIERRSEEK);
	}

	/* Success */
	return(0);
}





/******************************* midi_read() *********************************
 * Demonstrates adding a callback whereby we read bytes from the MIDI
 * file using standard C file I/O, fread(), for buffered I/O, rather than
 * letting the DLL read from the MIDI file.
 **************************************************************************/

long APIENTRY midi_read(MIDIFILE * mf, unsigned char * buffer, unsigned long count)
{
	if (fread((void *)Buffer, sizeof(char), Count, (FILE *)mf->Handle) != Count)
	{
		/* If we wanted to force MidiReadFile() to return some error code (to main) that we could
		   distinguish from an error that the DLL returns, we could return some negative number here (other
		   than -1, which this particular callback is not allowed to return). The DLL does not return any
		   negative error numbers via MidiReadFile(), but our callback can do that. Nevertheless, I'll just
		   return an appropriate error code that the DLL would return if it couldn't read from a file, and
		   thus, we'll simply display the DLL's error message. Note that this will abort the read operation
		   on return */
		return(MIDIERRREAD);
	}

	/* Success */
	return(0);
}

#endif





#ifdef RAMFILE_IO

unsigned char * RAM_Pointer;

/* Let's create a MIDI file right here in our program's data section */
unsigned char RAM_File[] = {'M','T','h','d', 0, 0, 0, 6, 0, 0, 0, 1, 0, 96,
 'M','T','r','k', 0, 0, 0, 27, 
 0, 0xFF, 0, 2, 0, 1,
 0, 0xFF, 3, 8, 'M','y',' ','T','r','a','c','k',
 0, 0xFF, 0x21, 0, 0,
 96, 0xFF, 0x2F, 0};

/****************************** midi_open() *****************************
 * Demonstrates adding a callback whereby we setup to parse a MIDI file
 * that is in RAM, rather than letting the DLL open a MIDI file on disk.
 * We also add OnReadWriteMidi, OnSeekMidi, and OnCloseMidi callbacks to
 * manage our parsing of the file in RAM instead of relying upon the DLL
 * to read and seek to disk.
 *************************************************************************/

long APIENTRY midi_open(MIDIFILE * mf)
{
	/* Initialize a global pointer to the start of our MIDI file in RAM. Note that
	   we could have done this back in main() before calling MidiReadFile, but
	   we'd still need an OpenMidi callback which did nothing but return 0 */
	RAM_Pointer = &RAM_File[0];

	/* We must also set MidiFile.FileSize */
	mf->FileSize = sizeof(RAM_File);

	/* Success */
	return(0);
}





/******************************* midi_read() *****************************
 * Demonstrates adding a callback whereby we setup to parse a MIDI file
 * that is in RAM, rather than letting the DLL read from a MIDI file on disk.
 *************************************************************************/

long APIENTRY midi_read(MIDIFILE * mf, unsigned char * Buffer, unsigned long Count)
{
	/* Copy the requested bytes into the DLL supplied buffer */
	memcpy(Buffer, RAM_Pointer, Count);
	
	/* Update our RAM pointer */
	RAM_Pointer += Count;
	
	/* Success */
	return(0);
}





/******************************* midi_seek() *****************************
 * Demonstrates adding a callback whereby we setup to parse a MIDI file
 * that is in RAM, rather than letting the DLL read from a MIDI file on disk.
 *************************************************************************/

long APIENTRY midi_seek(MIDIFILE * mf, long Offset)
{
	/* Adjust our RAM pointer by the Offset */
	RAM_Pointer += Offset;
	
	/* Success */
	return(0);
}





/****************************** midi_close() *****************************
 * Demonstrates adding a callback whereby we setup to parse a MIDI file
 * that is in RAM, rather than letting the DLL read from a MIDI file on disk.
 *************************************************************************/

long APIENTRY midi_close(MIDIFILE * mf)
{
	/* This callback doesn't need to do anything, but we still need to supply it */

	/* Success. (Actually, we don't need a return for this callback, but it makes it easier to setup the
	   CALLBACK structure) */
	return(0);
}
#endif





/********************************* initfuncs() ********************************
 * This initializes the CALLBACK structure containing pointers to functions.
 * These are used by MIDIFILE.DLL to call our callback routines while reading
 * the MIDI file. It also initializes a few fields of MIDIFILE structure.
 **************************************************************************/

void initfuncs()
{
	/* Place CALLBACK into MIDIFILE */
	Mfs.MidiCalls = &cb;

#ifdef BUFFERED_IO
	/* I'll do my own buffered Open, Read, Seek, and Close */
	cb.OpenMidi = midi_open;
	cb.ReadWriteMidi = midi_read;
	cb.SeekMidi = midi_seek;
	cb.CloseMidi = midi_close;
#endif
#ifdef RAMFILE_IO
	/* I'll do my own Open, Read, Seek, and Close with a RAM file */
	cb.OpenMidi = midi_open;
	cb.ReadWriteMidi = midi_read;
	cb.SeekMidi = midi_seek;
	cb.CloseMidi = midi_close;
#endif

#ifndef BUFFERED_IO
#ifndef RAMFILE_IO
	/* Let the DLL Open, Read, Seek, and Close the MIDI file */
	cb.OpenMidi = cb.ReadWriteMidi = cb.SeekMidi = cb.CloseMidi = 0;
#endif
#endif

	cb.UnknownChunk = unknown;
	cb.StartMThd = startMThd;
	cb.StartMTrk = startMTrk;
	cb.StandardEvt = standardEvt;
	cb.SysexEvt = sysexEvt;
	cb.MetaSMPTE = metasmpte;
	cb.MetaTimeSig = metatime;
	cb.MetaTempo = metatempo;
	cb.MetaKeySig = metakey;
	cb.MetaSeqNum = metaseqnum;
	cb.MetaText = metatext;
	cb.MetaEOT = metaend;
	cb.MetaPort = metaport;
	cb.MetaChan = metachan;
}
