/* 
You must link with winmm.lib. If using Visual C++, go to Build->Settings. Flip to the Link page,
and add winmm.lib to the library/object modules.

This app shows how to use the MMIO (RIFF) File Parsing functions. It opens, reads, and displays
info about the file C:\Windows\Chord.wav.
*/

#include <windows.h>
#include <stdio.h>
#include <conio.h>
#include <mmsystem.h>
#include <malloc.h>




/* Name of the WAVE file */
TCHAR		WaveName[] = "C:\\WINDOWS\\CHORD.WAV";

/* WAVEFORMATEX structure for reading in the WAVE fmt chunk */
WAVEFORMATEX WaveFormat;

/* Size of waveform data */
DWORD		WaveDataSize;

// Display strings
const char PCM[] = "PCM (uncompressed)";
const char IBM[] = "Adaptive Differential";
const char MuLaw[] = "IBM mu-law";
const char ALaw[] = "IBM a-law";
const char Mono[] = "Mono";
const char Stereo[] = "Stereo";





// ************************** PrintWaveInfo() *****************************
// Prints info on the WAVE.

void PrintWaveInfo() 
{
	register char *	ptr;
	char			buffer[20];

	/* Display wave size */
	printf("Wave size = %u\n", WaveDataSize);

	/* Display Format */
	switch (WaveFormat.wFormatTag)
	{
		case 1:
			ptr = (char *)&PCM[0];
			break;

		case 0x0101:
			ptr = (char *)MuLaw[0];
			break;

		case 0x0102:
			ptr = (char *)ALaw[0];
			break;

		case 0x0103:
			ptr = (char *)&IBM[0];
			break;

		default:
			_ultoa(WaveFormat.wFormatTag, &buffer[0], 10);
			ptr = &buffer[0];
	}
	printf("Format = %s\n", ptr);
	
	/* Display bit resolution */
	printf("Resolution = %u bit\n", WaveFormat.wBitsPerSample);

	/* Display number of chans */
	switch (WaveFormat.nChannels)
	{
		case 1:
			ptr = (char *)&Mono[0];
			break;

		case 2:
			ptr = (char *)&Stereo[0];
			break;

		default:
			_ultoa(WaveFormat.nChannels, &buffer[0], 10);
			ptr = &buffer[0];
	}
	printf("Number of chans = %s\n", ptr);

	/* Display sample rate */
	printf("Sample rate = %u Hz\n", WaveFormat.nSamplesPerSec);
}





/* ******************************** main() ******************************** */

int main(int argc, char **argv)
{
	HMMIO       hmmio;              /* file handle for open file */
	MMCKINFO    mmckinfoParent;		/* parent chunk information structure */
	MMCKINFO    mmckinfoSubchunk;	/* subchunk information structure */
	
	/*	Open the file for reading with buffered I/O. Let windows use its default internal buffer */
	if ((hmmio = mmioOpen(&WaveName[0], 0, MMIO_READ|MMIO_ALLOCBUF)))
	{
		/*	Tell Windows to locate a WAVE FileType chunk header somewhere in the file.
			This marks the start of any embedded WAVE format within the file */
		mmckinfoParent.fccType = mmioFOURCC('W', 'A', 'V', 'E'); 
		if (mmioDescend(hmmio, (LPMMCKINFO)&mmckinfoParent, 0, MMIO_FINDRIFF)) 
		{
			/* Oops! No embedded WAVE format within this file */
			printf("This file doesn't contain a WAVE!\n");

			/* Close the file and exit with error */
out:		mmioClose(hmmio, 0);
			return(-2);
		} 
 
		/* Tell Windows to locate the WAVE's "fmt " chunk, and read in its size field */
		mmckinfoSubchunk.ckid = mmioFOURCC('f', 'm', 't', ' '); 
		if (mmioDescend(hmmio, &mmckinfoSubchunk, &mmckinfoParent, MMIO_FINDCHUNK)) 
		{
			/* Oops! The required fmt chunk was not found! */
			printf("Required fmt chunk was not found!\n");
			goto out;
		}
 
		/* Tell Windows to read in the "fmt " chunk into our WAVEFORMATEX structure */
		if (mmioRead(hmmio, (HPSTR)&WaveFormat, mmckinfoSubchunk.cksize) != (LRESULT)mmckinfoSubchunk.cksize)
		{
			/* Oops! */
			printf("An erroring reading the fmt chunk!\n");
			goto out;
		}
 
		/*	Ascend out of the "fmt " subchunk. If you plan to parse any other chunks in the file, you need to
			"ascend" out of any chunk that you've mmioDescend()'ed into */
		mmioAscend(hmmio, &mmckinfoSubchunk, 0); 

		/*	Tell Windows to locate the data chunk. Upon return, the file
			pointer will be ready to read in the actual waveform data within
			the data chunk */
		mmckinfoSubchunk.ckid = mmioFOURCC('d', 'a', 't', 'a'); 
		if (mmioDescend(hmmio, &mmckinfoSubchunk, &mmckinfoParent, MMIO_FINDCHUNK)) 
		{
			/* Oops! */
			printf("An erroring reading the data chunk!\n");
			goto out;
		}
 
		/* Get the size of the data chunk (ie, the size of the waveform data) */
		WaveDataSize = mmckinfoSubchunk.cksize;

		/* Close the file */
		mmioClose(hmmio, 0);

		/* Print info about the wave */
		PrintWaveInfo();

		/* Success */
		return(0);
	}

	/* Oops! */
	printf("Can't find the WAVE file!\n");

	return(-1);
}
