/* PgmNum.c
 *
 * An example of using GENMIDI.DLL.
 *
 * This example shows how to call the DLL function MidiGetPgmName() to retrieve
 * the program name for a given program number. When running this app, supply a
 * number from 1 to 128 inclusive. The program will print the respective program
 * name for that program number. For example, program #1 in a GM bank is "Grand Piano".
 *
 * Typing a number of 0 results in all 128 patch names (of the first bank in
 * the first instrument) being displayed.
 */

#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include "..\genmidi.h"

/* ********************************* main() ********************************* */

void main(int argc, char *argv[])
{
	unsigned char * bankPtr;
	unsigned char * insPtr;
	unsigned char * ptr;
	unsigned long	pgm;
	unsigned short	bankNum;
	unsigned char	i, insNum;

	/* Check if any parameters were given, if not, display some help info */
	if (argc < 2)
	{
		_cputs("Displays the Program Name for the specified program number.\r\n\nSyntax is '");
		_cputs(argv[0]);
		_cputs( " pgm'.\r\n\twhere pgm is a program number from 1 to 128 inclusive.\r\n");
		_cputs( "\tAssumes the first instrument and first bank in the DLL database.\r\n");
		_cputs( "\tSpecifying a number of 0 displays all Program Names.\r\n\n");
		_cputs("Version 1.0\r\n");
		exit(-1);
	}

	/* Get the program number (as a binary value) */
	pgm = atoi(argv[1]);

	/* Make sure that it's not > 128 */
	if (pgm > 128)
	{
		_cputs("ERROR: MIDI Program numbers must be 1 to 128 inclusive.\r\n");
		exit(-2);
	}

	/* Indicate that we're accessing the database, but don't lock out other programs */
	if (!MidiLock(MIDILOCK_ACCESS, MIDILOCK_WAITFOREVER))
	{
		/* Get name and number of the first instrument */
		insPtr = 0;
		if ((insNum = (unsigned char)MidiNextInstrument(&insPtr)) == 0xFF)
		{
			MidiUnlock();
			_cputs("ERROR: No Instruments in DLL database!\r\n");
			exit(-3);
		}

		/* Get name and number of first bank in first instrument */
		bankPtr = 0;
		bankNum = (unsigned short)MidiNextBank(&bankPtr, insPtr);
		if (bankNum == 0xFFFF)
		{
			MidiUnlock();
			_cputs("ERROR: No Banks in first Instrument!\r\n");
			exit(-4);
		}

		/* See if he wants all Names displayed. If so, print 4 upon each line */
		if (!pgm)
		{
			i=0;

			/* Start with first program in this bank */
			insPtr = 0;

			/* Get pointer to next program name */
			while ((insNum = MidiNextPgm(&ptr, bankPtr, &insPtr)) != 0xFF)
			{
				/* Wrap around? */
				if (!((i++)%4)) printf("\r\n");

				/* Print program # and name */
				printf("%3u %-15s", insNum, ptr);
			}
		}

		/* Print just the 1 Name */
		else
		{
			/* MIDI Program numbers actually must be 0 to 127, although the user normally
			   considers the first program to be 1 rather than 0. Adjust for MidiGetPgmName() */
			insPtr = MidiGetPgmName((unsigned char)(pgm-1), bankNum, insNum);
			if (insPtr)
				printf("%u %s\r\n", pgm, insPtr);
			else
				_cputs("This Program number is not used in the first Instrument's first Bank.\r\n");
		}

		/* Indicate that we're done accessing the database */
		MidiUnlock();
	}

	exit(0);
}
