/*
 * You must link with winmm.lib. If using Visual C++, go to Build->Settings. Flip to the
 * Link page, and add winmm.lib to the library/object modules.
 *
 * This example records a WAVE file named TEST.WAVE using the Wave Audio device (ie, High level
 * Digital Audio API). It demonstrates both the both the string and command message interfaces.
 */

#include <windows.h>
#include <stdio.h>
#include <conio.h>
#include <mmsystem.h>


/* Comment out the below line to build an example using mciSendString() instead  of mciSendCommand() */
#define SHOW_MESSAGE_INTERFACE



/*********************** PrintWaveErrorMsg() **************************
 * Retrieves and displays an error message for the passed Wave Out error
 * number. It does this using mciGetErrorString().
 *************************************************************************/

void PrintWaveErrorMsg(DWORD err, TCHAR * str)
{
#define BUFFERSIZE 128
	char	buffer[BUFFERSIZE];
	
	printf("ERROR 0x%08X: %s\r\n", err, str);
	if (mciGetErrorString(err, &buffer[0], sizeof(buffer)))
	{
		printf("%s\r\n", &buffer[0]);
	}
	else
	{
		printf("0x%08X returned!\r\n", err);
	}
}





/* Declare a union of the various structures we need to pass to mciSendCommand()
 * for the various commands
 */
#ifdef SHOW_MESSAGE_INTERFACE
union _WAVEPARAMS
{
	MCI_WAVE_OPEN_PARMS	WaveOpenParms;
	MCI_RECORD_PARMS	WaveRecordParms;
	MCI_SAVE_PARMS		WaveSaveParms;
	MCI_WAVE_SET_PARMS	WaveSetParms;
} WAVEPARAMS;
#endif

/* ******************************** main() ******************************** */

int main(int argc, char **argv)
{
	DWORD				err;
#ifdef SHOW_MESSAGE_INTERFACE
	union _WAVEPARAMS	params;
	MCIDEVICEID			devID;

	/*	Specify the Wave Audio device as the type of device	*/
	params.WaveOpenParms.lpstrDeviceType = (LPCSTR)MCI_DEVTYPE_WAVEFORM_AUDIO;

	/* We're recording a new file, so the name is "" */
	params.WaveOpenParms.lpstrElementName = "";

	/* Open the default Wave Audio device, and wait for the operation to complete */
	if ((err = mciSendCommand(0, MCI_OPEN, MCI_WAIT|MCI_OPEN_ELEMENT|MCI_OPEN_TYPE|MCI_OPEN_TYPE_ID, (DWORD)(LPVOID)&params.WaveOpenParms)))
	{
		/* Error */
		PrintWaveErrorMsg(err, "mciSendCommand() did not open the Wave Audio device!");

		/* Exit the program */
		return(-1);
	}

	/* The device opened successfully. params.WaveOpenParams.wDeviceID now contains the device ID */
	devID = params.WaveOpenParms.wDeviceID;

	/* Set its record parameters to 8bit, 11Khz, all channels,
	 * and wait for this operation to complete
	 */
	ZeroMemory(&params.WaveSetParms, sizeof(MCI_WAVE_SET_PARMS));
/*	params.WaveSetParms.dwAudio = 0; */
/*	params.WaveSetParms.wOutput = 0; */						/* Use Device 0 */
	params.WaveSetParms.wFormatTag = WAVE_FORMAT_PCM;		/* Uncompressed format */
	params.WaveSetParms.nChannels = 2;						/* Stereo */
	params.WaveSetParms.nSamplesPerSec = 11025;				/* 11 KHz same rate */
	params.WaveSetParms.wBitsPerSample = 8;					/* 8-bit */
	params.WaveSetParms.nBlockAlign = ((params.WaveSetParms.wBitsPerSample)/8) * params.WaveSetParms.nChannels;
	params.WaveSetParms.nAvgBytesPerSec = params.WaveSetParms.nBlockAlign * params.WaveSetParms.nSamplesPerSec;

   if ((err = mciSendCommand(devID, MCI_SET, MCI_WAIT |
							MCI_WAVE_SET_FORMATTAG|
							MCI_WAVE_SET_BITSPERSAMPLE |
							MCI_WAVE_SET_CHANNELS |
							MCI_WAVE_SET_SAMPLESPERSEC |
							MCI_WAVE_SET_AVGBYTESPERSEC |
							MCI_WAVE_SET_BLOCKALIGN,
							(DWORD)(LPVOID)&params.WaveSetParms)))
	{
		/* Error */
		PrintWaveErrorMsg(err, "mciSendCommand() did not set parameters!");
	}
	else
	{
		/* Record a 10 second waveform, and wait for this operation to complete */
		params.WaveRecordParms.dwFrom = 0;
		params.WaveRecordParms.dwTo = 10 * 1000;
		if ((err = mciSendCommand(devID, MCI_RECORD, MCI_WAIT|MCI_FROM|MCI_TO|MCI_RECORD_OVERWRITE, (DWORD)(LPVOID)&params.WaveRecordParms)))
		{
			/* Error */
			PrintWaveErrorMsg(err, "mciSendCommand() did not record the waveform!");
		}
		else
		{
			/* Save the data to a file named TEST.WAV, and wait for the operation to complete */
			params.WaveSaveParms.lpfilename = "test.wav";
			if (err = mciSendCommand(devID, MCI_SAVE, MCI_WAIT|MCI_SAVE_FILE, (DWORD)(LPVOID)&params.WaveSaveParms))
			{
				/* Error */
				PrintWaveErrorMsg(err, "mciSendCommand() did not save the waveform!");
		    }
		}
	}

	/* Close the device, and wait for this operation to complete */
	mciSendCommand(devID, MCI_CLOSE, MCI_WAIT, (DWORD)(LPVOID)&params.WaveOpenParms);

#else

	/* =================================================================== */

	/* Now do the same thing as above, but this time, let's use the string
	 * interface of mciSendString(), rather than mciSendCommand().
	 */

	/* Open the wave device to record new data */
	if ((err = mciSendString("open new type waveaudio alias mysound wait", 0, 0, 0)))
	{
		/* Error */
		PrintWaveErrorMsg(err, "mciSendString() did not open the Wave Audio device!");

		/* Exit the program */
		return(-1);
	}

	/* The device opened successfully. we now use the alias "mysound" to reference the device */

	/* Set its record parameters to 8bit, 11Khz, mono, and wait for this operation to complete */
	if ((err = mciSendString("set mysound format tag pcm bitspersample 8 channels 1 samplespersec 11025 wait", 0, 0, 0)))
	{
		/* Error */
		PrintWaveErrorMsg(err, "mciSendString() did not set the parameters!");
	}
	else
	{
		/* Start recording the wave data, but don't wait for operation to finish */
		if ((err = mciSendString("record mysound from 0 overwrite", 0, 0, 0)))
		{
			/* Error */
			PrintWaveErrorMsg(err, "mciSendString() did not record!");
		}

		else
		{
			/* Wait for user to stop recording */
			printf("Press ENTER key to stop recording...\n");
			getchar();

			/* Stop the recording */
			if ((err = mciSendString("stop mysound wait", 0, 0, 0)))
			{
				/* Error */
				PrintWaveErrorMsg(err, "mciSendString() did not stop recording!");
			}

			else
			{
				/* Save the wave data, and wait for operation to finish */
				if ((err = mciSendString("save mysound test.wav wait", 0, 0, 0)))
				{
					/* Error */
					PrintWaveErrorMsg(err, "mciSendString() did not save the data!");
				}
			}
		}
	}

	/* Close the device, and wait for this operation to complete */
	mciSendString("close mysound wait", 0, 0, 0);
#endif

	return(0);
}