/*
* codelns
* Bob Crispen (crispen@hiwaay.net)
*
* Called:
*   % codelns *
*   % codelns `find . -type f -print`
*
* Attempts to identify source code files for Ada, C, C++, and FORTRAN,
* shell scripts, and binary files.  Calls everything else it can't
* figure out "other".  Counts source lines of code, comments and
* statements in source files; counts lines only in shell scripts and
* "other" files; doesn't count anything in binary files.
*
* All comments are counted once.  That is, multi-line comments in C
* are counted as one comment, in-line comments are counted as well as
* separate-line comments.
*
* A "statement" in Fortran is a non-comment, non-blank line without
* a continuation character in column 6.
*
* A "statement" in C is a semicolon, except semicolons found in comments
* or in character or string literals.  "for" statements count as two
* statements.  Curly braces don't count.
*
* A "statement" in Ada is a semicolon, except semicolons found in
* comments, in character or string literals, or in subprogram
* parameter lists.
*
* Uncompilable Ada, C or Fortran files may produce erroneous results.
*
* XXX The way this program figures out what type the file is is
* really, really ugly.  You'll probably want to customize it for your
* system.
*
* This software is distributed according to the terms of the Gnu
* General Public License.
*/
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#ifndef bzero
#define bzero(x,y) memset((x),0,(y))
#endif

#define NUM_LANG 8
char *filetypes[NUM_LANG] =
    {"Ada", "C", "C++", "Header", "FORTRAN", "Script", "Binary", "Other"};

void printLine();
int isBlankLine (char *buf);
int find_filetype (char *filename, FILE *srcfile);
void count_ada_file (FILE *srcfile);
void count_fortran_file (FILE *srcfile);
void count_c_file (FILE *srcfile);

#define ADA	0
#define C	1
#define CPLUS   2
#define HDR	3
#define FORTRAN 4
#define SCRIPT	5
#define BINARY	6
#define OTHER	7

#define FALSE	(0)
#define TRUE	(!0)

/*
* Comment this out if you want to count semicolons in Ada subprogram
* parameter lists.  Usually you don't
*/
#define DONT_COUNT_PARAMETERS
#define BUFSIZE 10000

static int lines,comments,stmts;

main (int argc, char **argv)
{
    int filetype;
    int total_files[NUM_LANG];
    char filename[300];
    FILE *srcfile;
    int filecount;
    int c, i;
    char buf[BUFSIZE];
    int bufptr = 0;
    int total_lines[NUM_LANG];
    int total_comments[NUM_LANG];
    int total_stmts[NUM_LANG];
    int grand_total_lines = 0;
    int grand_total_comments = 0;
    int grand_total_stmts = 0;
    int grand_total_files = 0;


    if (argc<2) {
	fprintf (stderr, "Usage: %s file...\n", argv[0]);
	exit(1);
    }

    for (i=0; i<NUM_LANG; i++) {
	total_files[i] = 0;
	total_lines[i] = 0;
	total_comments[i] = 0;
	total_stmts[i] = 0;
    }

    /*
    * Print header
    */
    printLine();
    printf("File                                              Language  Lines  Cmnts Stmnts\n");
    printLine();

    /*
    * Loop through each file named in the command line
    */
    for (filecount=1; filecount<argc; filecount++) {
	lines = 0;
	stmts = 0;
	comments = 0;
	strcpy (filename, argv[filecount]);
	if ((srcfile = fopen (filename, "r")) <= 0) {
	    fprintf (stderr, "%s: can't open %s\n", argv[0], filename);
	    perror ("Reason");
	} else {
	    filetype = find_filetype(filename, srcfile);
	    /*
	    * If this is an object file then the number of lines in the
	    * file is meaningless
	    */
	    if (filetype == BINARY)
		lines = 0;
	    /*
	    * Now go do the counts
	    */
	    if (filetype != BINARY) {
		rewind (srcfile);
		switch (filetype) {
		    case ADA:
			count_ada_file (srcfile);
			break;
		    case FORTRAN:
			count_fortran_file (srcfile);
			break;
		    case C:
		    case CPLUS:
		    case HDR:
			count_c_file (srcfile);
			break;
		    default:
			break;
		}
	    }
	    fclose(srcfile);
	    total_lines[filetype] += lines;
	    total_comments[filetype] += comments;
	    total_stmts[filetype] += stmts;
	    grand_total_lines += lines;
	    grand_total_comments += comments;
	    grand_total_stmts += stmts;
	}
	/*
	* Print the line for this file
	*/
	total_files[filetype]++;
	grand_total_files++;

	if (strlen(filename) < 50)
	    printf ("%-50.50s", filename);
	else {
	    for (i=strlen(filename)-46;
		((i<strlen(filename)) && (filename[i] != '/')); i++);
	    if (i >= strlen(filename))
		i = strlen(filename)-46;
	    printf ("...%-47.47s", &filename[i]);
	}

	switch (filetype) {
	    case C:
	    case CPLUS:
	    case HDR:
	    case ADA:
	    case FORTRAN:
		printf ("%-8.8s%7d%7d%7d\n",
		    filetypes[filetype], lines, comments, stmts);
		break;
	    case BINARY:
		printf ("%-8.8s     --     --     --\n",
		    filetypes[filetype]);
		break;
	    case SCRIPT:
	    case OTHER:
		printf ("%-8.8s%7d     --     --\n",
		    filetypes[filetype], lines);
		break;
	}
    }

    /*
    * Print the summary lines
    */
    if (grand_total_files > 1) {
	printLine();
	printf ("***TOTALS***\n");
	if (total_files[ADA])
	    printf ("%-8.8s  Files: %5d                                    %7d%7d%7d\n",
		filetypes[ADA], total_files[ADA],
		total_lines[ADA], total_comments[ADA], total_stmts[ADA]);
	if (total_files[C])
	    printf ("%-8.8s  Files: %5d                                    %7d%7d%7d\n",
		filetypes[C], total_files[C],
		total_lines[C], total_comments[C], total_stmts[C]);
	if (total_files[CPLUS])
	    printf ("%-8.8s  Files: %5d                                    %7d%7d%7d\n",
		filetypes[CPLUS], total_files[CPLUS],
		total_lines[CPLUS], total_comments[CPLUS], total_stmts[CPLUS]);
	if (total_files[HDR])
	    printf ("%-8.8s  Files: %5d                                    %7d%7d%7d\n",
		filetypes[HDR], total_files[HDR],
		total_lines[HDR], total_comments[HDR], total_stmts[HDR]);
	if (total_files[FORTRAN])
	    printf ("%-8.8s  Files: %5d                                    %7d%7d%7d\n",
		filetypes[FORTRAN], total_files[FORTRAN],
		total_lines[FORTRAN], total_comments[FORTRAN],
		total_stmts[FORTRAN]);
	if (total_files[SCRIPT])
	    printf ("%-8.8s  Files: %5d                                    %7d     --     --\n",
		filetypes[SCRIPT], total_files[SCRIPT], total_lines[SCRIPT]);
	if (total_files[BINARY])
	    printf ("%-8.8s  Files: %5d                                         --     --     --\n",
		filetypes[BINARY], total_files[BINARY]);
	if (total_files[OTHER])
	    printf ("%-8.8s  Files: %5d                                    %7d     --     --\n",
		filetypes[OTHER], total_files[OTHER], total_lines[OTHER]);

	printf ("%-8.8s  Files: %5d                                    %7d%7d%7d\n",
	    "TOTAL", grand_total_files, grand_total_lines,
	    grand_total_comments, grand_total_stmts);
    }
    printLine();
    exit (0);
}

/*
 * Print a line of dashes
 */
void printLine()
{
    int i;
    for (i=0; i<80;i++)
	putc ('-', stdout);
    putc ('\n', stdout);
}

/*
 * Is this a blank line?
 */
int isBlankLine (char *buf)
{
    int i;
    for (i=0; buf[i] != '\0'; i++) {
	if (!isspace(buf[i]))
	    return(FALSE);
    }
    return(TRUE);
}

/*
 * Find out what kind of file the current file is, and if it's not
 * a binary file, count the lines while you're at it.
 */
int find_filetype (char *filename, FILE *srcfile)
{
    int c, i;
    char buf[BUFSIZE];
    int bufptr;
    char lcfilename[BUFSIZE];

    bufptr=0;
    while ((c=getc(srcfile)) != EOF) {
	buf[bufptr++] = c;
	/*
	* These tests will catch some object files and count the rest
	*/
	if (bufptr >= BUFSIZE) {
	    return BINARY;
	}
	if (c > '\177') {
	    return BINARY;
	}
	if ((c != '\n') && (c != '\t') && (c != '\v') && (c != '\f') &&
	    (c != '\r') && (c < ' ')) {
	    return BINARY;
	}
	/*
	* Deal with one line at a time in the buffer
	*/
	if (c == '\n') {
	    lines++;
	    buf[bufptr] = '\0';
	    /*
	    * Check for a shell script
	    */
	    if ((lines == 1) && (buf[0] == '#')) {
		if ((buf[1] == '!') ||
		    ((buf[1] == 'c') && (buf[2] == 's')
		    && (buf[3] == 'h')))
		return (SCRIPT);
	    }
	    bufptr = 0;
	}
    }
    /*
    * OK, the file hasn't revealed itself to us yet from its internal
    * structure.  Now let's cheat and look at the filenames.
    *
    * XXX This really just looks at a subset of names, and
    * you'll probably want to tweak it for your system.
    */

    /* Convert filename to lower-case to simplify comparisons */
    bzero(lcfilename, 1000);
    for (i=0; (lcfilename[i] = tolower(filename[i])) != '\0'; i++);

    --i;
    if (lcfilename[i-1] == '.') {
	if (lcfilename[i] == 'a')
	    return BINARY;       /* But on some systems, it's Ada */
	if (lcfilename[i] == 'c')
	    return C;
	if (lcfilename[i] == 'h')
	    return HDR;
	if (lcfilename[i] == 'f')
	    return FORTRAN;
	if (lcfilename[i] == 'o')
	    return BINARY;
	} else {
	if (lcfilename[i-3] == '.') {
	    if (!strcmp(&lcfilename[i-2], "bat"))
	        return SCRIPT;
	    if (!strcmp(&lcfilename[i-2], "ads"))
	        return ADA;
	    if (!strcmp(&lcfilename[i-2], "adb"))
	        return ADA;
	    if (!strcmp(&lcfilename[i-2], "exe"))
	        return BINARY;
	    if (!strcmp(&lcfilename[i-2], "cpp"))
		return CPLUS;
	} else {
	    if (!strcmp(lcfilename, "makefile"))
	        return SCRIPT;
	    if (!strcmp(lcfilename, "a.out"))
	        return BINARY;
	}
    }
    return OTHER;
}

/*
 * Count statements and comments in an Ada file
 */
void count_ada_file (FILE *srcfile)
{
    char buf[BUFSIZE];
    int c, i, bufptr=0;
    int paren_level = 0;
    int in_string = 0;

    while ((c = getc(srcfile)) != EOF) {
	buf[bufptr++] = c;
	if (c == '\n') {
	    buf[bufptr] = '\0';
	    in_string = 0;
	    for (i=0; i<bufptr; i++) {
		if ((buf[i] == '-') && (buf[i+1] == '-')) {
		    comments++;
		    break;
		}
		if ((buf[i] == '\047') && (buf[i+2] == '\047'))
		    i += 3;
		if (buf[i] == '"')
		    in_string = !in_string;
#ifdef DONT_COUNT_PARAMETERS
		if (!in_string) {
		    if (buf[i] == '(')
			paren_level++;
		    if (buf[i] == ')')
			paren_level--;
		}
#endif
		if ((!in_string) && (paren_level == 0)) {
		    if (buf[i] == ';')
			stmts++;
		}
	    }
	bufptr = 0;
	}
    }
}

/*
 * Count statements and comments in an FORTRAN file
 */
void count_fortran_file (FILE *srcfile)
{
    char buf[BUFSIZE];
    int c, i, bufptr=0;

    while ((c = getc(srcfile)) != EOF) {
	buf[bufptr++] = c;
	if (c == '\n') {
	    buf[bufptr] = '\0';
	    if (buf[0] == 'C') {
		comments++;
	    } else {
		if ((!isBlankLine(buf)) &&
		    ((isspace(buf[5])) || (buf[0] == '\t')))
		    stmts++;
	    }
	    bufptr = 0;
	}
    }
}

/*
* Count statements and comments in C, C++ or header file
* Cute algorithm uses a bunch of states instead of a buffer
*/
void count_c_file (FILE *srcfile)
{
    int c;
    int comment_start = FALSE;
    int comment_end = FALSE;
    int in_comment = FALSE;
    int in_cplus_comment = FALSE;
    int in_literal = FALSE;
    int in_string = FALSE;

    while ((c = getc(srcfile)) != EOF) {
	if (in_comment) {
	    if (comment_end) {
		if (c == '/') {
		    in_comment = FALSE;
		    comment_start = FALSE;
		    comment_end = FALSE;
		}
	    } else {
		comment_end = (c == '*');
	    }
	} else if (in_cplus_comment) {
	    if (c == '\n') {
		in_cplus_comment = FALSE;
		comment_start = FALSE;
		comment_end = FALSE;
	    }
	} else {
	    if (comment_start) {
		if (c == '*') {
		    in_comment = TRUE;
		    comment_start = FALSE;
		    comment_end = FALSE;
		    comments++;
		} else if (c == '/') {
		    in_cplus_comment = TRUE;
		    comment_start = FALSE;
		    comment_end = FALSE;
		    comments++;
		}
	    } else {
		comment_start = (c == '/');
	    }
	    if (c == '\047')
		in_literal = !in_literal;
	    if (!in_literal) {
		if (c == '"')
		    in_string = !in_string;
	    }
	    if (!in_literal && !in_string) {
		if (c == ';') {
		    stmts++;
		}
	    }
	}
    }
}
