/*
 * numpoints
 * by Bob Crispen
 * crispen@hiwaay.net
 *
 * Called: numpoints infile outfile
 *
 * Not all CAD programs are perfect.  Some export VRML that has
 * missing faces.  If you've got a beautiful VRML tool that will
 * patch up your faces automatically, no problem.  But if you
 * have to patch by hand, you need to know the numbers of the
 * vertices.  Until now that meant trial and error.
 *
 * If you stick a VRML 2.0 Shape node (the whole thing, including
 * IndexedFaceSet including the coord field and the coordIndex
 * field into a file (the infile) and run numpoints, naming
 * an outfile (e.g. numpoints foo foo.wrl), it will generate
 * a viewable VRML world that has Text nodes for each of the
 * numbers.
 *
 * Hint: The scale factors here are completely arbitrary.
 * On a recent project I used an outer scale of 100 and inner
 * scales of 0.005 0.005 0.005 -- any good text editor will
 * let you do a block change.  The scale you end up using will
 * depend on the object and how close the points are to one another.
 *
 * Hint: You may want to play with the transparency of your Shape 
 * node (which is included verbatim in the outfile) to make it
 * easier to see the numbers.
 * 
 * Hint: Some browsers (we won't mention any names) make the
 * Examine mode virtually useless unless your object is centered
 * at the origin.  At the bottom of your outfile, you'll see
 * max, min, and center X, Y, and Z values.  You can put their
 * negatives in a translation field in the outermost Transform.
 * Don't forget to multiply by the scale factor.
 *
 */
#include <stdio.h>
#include <string.h>

#ifdef MSDOS
#define OUTARG "wb"
#else
#define OUTARG "w"
#endif

main (int argc, char **argv)
{
    FILE *infile;
    FILE *outfile;
    char buf[1000];
    float coord[3];
    int c, bufptr=0;
    char *token, *tokptr;
    int inpoints = 0;
    int coordptr = 0;
    int lines=0;
    float minx = 99999999.9;
    float maxx = -99999999.9;
    float miny = 99999999.9;
    float maxy = -99999999.9;
    float minz = 99999999.9;
    float maxz = -99999999.9;

    if (argc != 3) {
	fprintf (stderr, "%s: infile outfile\n", argv[0]);
	exit(-1);
    }
    if ((infile = fopen(argv[1], "r")) == NULL) {
	fprintf (stderr, "can't open %s\n", argv[1]);
	exit(-1);
    }
    if ((outfile = fopen(argv[2], OUTARG)) == NULL) {
	fprintf (stderr, "can't open %s\n", argv[2]);
	fclose (infile);
	exit(-1);
    }
    fprintf (outfile, "#VRML V2.0 utf8\n");
    fprintf (outfile, "NavigationInfo {\n");
    fprintf (outfile, "  type [ \"EXAMINE\", \"ANY\" ]\n");
    fprintf (outfile, "}\n");
    fprintf (outfile, "Viewpoint {\n");
    fprintf (outfile, "  position 0 0 10\n");
    fprintf (outfile, "  description \"Entry\"\n");
    fprintf (outfile, "}\n");
    fprintf (outfile, "Transform {\n");
    fprintf (outfile, "  scale 10 10 10\n");
    fprintf (outfile, "  children [\n");
    
    while ((c=getc(infile)) != EOF) {
	buf[bufptr++] = c;
	if (c == '\n') {
	    buf[bufptr] = '\0';
	    tokptr = buf;
	    while((token = strtok(tokptr, " \n\r\t\0")) != NULL) {
		tokptr = NULL;
		if (inpoints) {
		    if (token[0] == ']') {
			inpoints = 0;
		    } else {
			if (token[0] != '[') {
			    sscanf(token, "%f", &coord[coordptr]);
			    if (++coordptr == 3) {
				coordptr=0;
				fprintf (outfile, "    Transform {\n");
				fprintf (outfile, "      translation %f %f %f\n",
				    coord[0], coord[1], coord[2]);
				fprintf (outfile, "      children [\n");
				fprintf (outfile, "\tTransform {\n");
				fprintf (outfile, "\t  scale 0.01 0.01 0.01\n");
				fprintf (outfile, "\t  children [\n");
				fprintf (outfile, "\t    Billboard {\n");
				fprintf (outfile, "\t      axisOfRotation 0 0 0\n");
				fprintf (outfile, "\t      children [\n");
				fprintf (outfile, "\t\tShape {\n");
				fprintf (outfile, "\t\t  geometry Text { string \"%d\" }\n", lines++);
				fprintf (outfile, "\t\t}\n");
				fprintf (outfile, "\t      ]\n");
				fprintf (outfile, "\t    }\n");
				fprintf (outfile, "\t  ]\n");
				fprintf (outfile, "\t}\n");
				fprintf (outfile, "      ]\n");
				fprintf (outfile, "    }\n\n");
				if (coord[0] < minx) minx=coord[0];
				if (coord[0] > maxx) maxx=coord[0];
				if (coord[1] < miny) miny=coord[1];
				if (coord[1] > maxy) maxy=coord[1];
				if (coord[2] < minz) minz=coord[2];
				if (coord[2] > maxz) maxz=coord[2];
			    }
			}
		    }
		}
		if (!strcmp(token, "point")) {
		    inpoints = 1;
		}
	    }
	    bufptr=0;
	}
    }
    rewind (infile);
    while ((c=getc(infile)) != EOF)
	putc(c, outfile);
    fprintf (outfile, "  ]\n");
    fprintf (outfile, "}\n");
    fprintf (outfile, "# X Min = %f\tMax = %f\tCenter = %f\n",
	 minx, maxx, (minx + maxx) * 0.5);
    fprintf (outfile, "# Y Min = %f\tMax = %f\tCenter = %f\n",
	 miny, maxy, (miny + maxy) * 0.5);
    fprintf (outfile, "# Z Min = %f\tMax = %f\tCenter = %f\n",
	 minz, maxz, (minz + maxz) * 0.5);

    fclose(outfile);
    fclose(infile);
}
