/*
* vpp
* by Bob Crispen
* crispen@hiwaay.net
*
* Called: see usage() or execute "vpp -?"
*
* Purpose: Prettyprints a VRML file
*
* This program is subject to the Gnu General Public License.
* See ftp://prep.ai.mit.edu/pub/gnu/ for details.
*
* Hints:
*
* Version: 1.0, 11 December 1997
*
* Known bugs:
*
* Planned Improvements:
*	(1) Vpp works on one line at a time.  It would be nice
*	    if it could combine lines and extend lines that are
*	    too long.
*
*	(2) Vpp currently ignores commas.  It would be nice if
*	    it could (optionally) remove commas and put commas
*	    only between MFNode elements.
*
*	(3) We could (optionally) insert blank lines between
*	    elements of a given type.  Haven't thought this
*	    through yet.
*	    
*/

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#ifdef MSDOS
#define WRITEMODE "wb"
#else
#define WRITEMODE "w"
#endif

#define MAXBUFSIZE 1000
#define BLANK " \t\r\n"

main(int argc, char **argv)
{
    FILE *infile;
    FILE *outfile;
    int argn = 1;
    char infilename[400];
    char outfilename[400];
    int c, i, j, l;
    char buf[MAXBUFSIZE];
    char buf2[MAXBUFSIZE];
    int bufptr=0;
    int lineptr=0;
    char *token;
    char *pbuf;
    int indent = 2;
    int indentlevel = 0;
    int notabs = 0;
    int tabstop = 8;
    int incomment = 0;
    int inquote = 0;

    infile = stdin;
    outfile = stdout;
    strcpy(infilename, "stdin");
    strcpy(outfilename, "stdout");

    /* Get user args */
    while (argn < argc && argv[argn][0] == '-' && argv[argn][1] != '\0') {
	if (isdigit(argv[argn][1])) {
	    sscanf(&argv[argn][1], "%d", &indent);
	    /* Account for "--" from old DOS habits */
	    if (indent < 0)
		indent = -indent;
	    ++argn;
	    continue;
	}
	if (!strncmp(argv[argn], "-notabs", 2)) {
	    notabs = 1;
	    ++argn;
	    continue;
	}
	if (!strncmp(argv[argn], "-tabstop", 2)) {
	    sscanf(argv[++argn], "%d", &tabstop);
	    ++argn;
	    continue;
	}
	usage(argv[0]);
    }

    if (argn != argc) {
	strcpy(infilename, argv[argn]);
	if((infile = fopen(argv[argn++], "r")) == NULL) {
	    fprintf(stderr, "%s: can't open %s\n", argv[0], argv[argn-1]);
	    exit(1);
	}
    }
    if (argn != argc) {
	strcpy(outfilename, argv[argn]);
	if ((outfile = fopen(argv[argn++], WRITEMODE)) == NULL) {
	    fprintf(stderr, "%s: can't open %s\n", argv[0], argv[argn-1]);
	    exit(1);
	}
    }
    if (argn != argc) usage(argv[0]);

    /* Process input file */
    while ((c=getc(infile)) != EOF) {
	buf[bufptr++] = c;
	if (bufptr >= MAXBUFSIZE-1) {
	    fprintf(stderr, "Line %d too long\n", lineptr+1);
	    fclose(outfile);
	    fclose(infile);
	    exit(-1);
	}
	/* Process one line at a time */
	if (c == '\n') {
	    buf[bufptr]='\0';
	    lineptr++;
	    pbuf=buf;
	    memset(buf2, 0, sizeof(buf2));
	    /* Start off by putting a single space between each token */
	    while (token=strtok(pbuf, BLANK)) {
		if (!pbuf)
		    strcat(buf2, " ");
		pbuf=NULL;
		strcat (buf2, token);
	    }
	    /* Emit a character at a time */
	    l=strlen(buf2);
	    incomment = 0;
	    /* Handle right brackets and braces that start lines */
	    if ((buf2[0] == '}') || (buf2[0] == ']')) {
		indentlevel--;
		do_indent(indent, indentlevel, notabs, tabstop, outfile);
		indentlevel++;
	    } else {
		do_indent(indent, indentlevel, notabs, tabstop, outfile);
	    }
	    for (i=0; i<l; i++) {
		c = buf2[i];
		/* Take care of comments and quotes first */
		if (c == '#')
		    incomment = 1;
		if (incomment) {
		    putc(c, outfile);
		    continue;
		}
		if (c == '"')
		    !inquote;
		if (inquote) {
		    putc(c, outfile);
		    continue;
		}
		/* Unjam trailing chars */
		if ((c == '}') || (c == ']')) {
		    if ((i > 0) && (!isspace(buf2[i-1]))) {
			putc(' ', outfile);
		    }
		    indentlevel--;
		}
		putc(c, outfile);
		if ((c == '{') || (c == '[')) {
		    indentlevel++;
		    if ((i+1 < l) && (!isspace(buf2[i+1])))
			putc(' ', outfile);
		}
	    }
	    putc('\n', outfile);
	    bufptr=0;
	}
    }
    fclose(outfile);
    fclose(infile);
    exit(0);
}

do_indent(int indent, int indentlevel, int notabs, int tabstop, FILE *outfile)
{
    int i;
    int spaces;
    int tabs;

    spaces = indentlevel*indent;
    if (notabs) {
	tabs = 0;
    } else {
	if (tabstop) {
	    tabs = spaces/tabstop;
	    spaces %= tabstop;
	} else {
	    tabs = indentlevel;
	    spaces = 0;
	}
    }
    for (i=0; i<tabs; i++)
	putc('\t', outfile);
    for (i=0; i<spaces; i++)
	putc(' ', outfile);
}

usage(char *name)
{
    fprintf(stderr, "usage: %s [args] [infile] [outfile]\n", name);
    fprintf(stderr, "  Where args are any of the following:\n");
    fprintf(stderr, "  -n :  number of spaces per indent level (default: 2)\n");
    fprintf(stderr, "  -t[abstop] n : replace every n spaces with a tab (default: 8)\n");
    fprintf(stderr, "  -n[otabs] : use only spaces, no tab characters (default: use tabs)\n");
    fprintf(stderr, "Version 1.0, 11 December 1997\n");
    exit(-1);
}
