//
// foxynews.c -- batch-process newsreader and attachment extractor
//
// Version: See version.h
//
// Copyright (C) 1999-2000 by Bob Crispen <crispen@hiwaay.net> doing
//   business as Foxy Software
//
// Heavily adapted from checkmail.c by Lewis Napper
//   http://www.sockaddr.com/
//
// LICENSE:
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// Documentation:
//   The license is contained in /doc/copying-2.0.txt
//   Instructions are contained in /doc/readme.txt
// 
// Compiling notes:
//   Link with wsock32.lib
//   Resource file: foxynews.rc, requires resource.h
//   Most of the non-Windows stuff is in newslib.c, however there are
//     still some Windows calls in newslib.c
//   Compiles under lcc.  No guarantees for other systems.  Because
//     it uses Winsock in asynchronous mode, it can't run on Unix systems.
//
// Known deficiencies:
//  (1) Doesn't yet reassemble files that have been split and posted
//      in multiple articles
//  (2) There are way too many global variables.

#include <windows.h>
#include <string.h>
#include <stdio.h>
#include <winsock.h>
#include <richedit.h>

#include "foxynews.h"
#include "resource.h"

// Global Variables
static char szAppName[] = "FoxyNews";
int	gnAppState;			// Application State
int	gnStateCmd;			// State command
char	gszServerName[BUFSIZE];		// News Host Name
char	gszNewsgroup[BUFSIZE];		// Newsgroup name
char	gszSearchString[BUFSIZE];	// Search String
char	gszLoginName[SHORTBUFSIZE];	// Login name	
char	gszPassword[SHORTBUFSIZE];	// Password
HANDLE	hndlGetHost;			// Handle for WSAAsyncGetHostByName()
char	bufHostEnt[MAXGETHOSTSTRUCT];	// HostEnt for WSAAsyncGetHostByName()
SOCKET	sockNews;			// The socket
char	gszTemp[BUFSIZE];		// Scratch buffer for sprintf()
HWND	hwndStatusBar;			// Status bar
HWND	hwndRichEdit;			// Window for displaying headers
HWND	hwndProgress;			// Progress bar handle
int	aWidths[3];			// Width of each part of the status bar
BOOL	gfArticleText;			// User wants to get article text
BOOL	gfBinaryAttachments;		// User wants to get binary attachments
BOOL	gfURLs;				// Extract URLs from articles
float	x_proportion;			// Ratio of DB units to pixels on X
float	y_proportion;			// ditto on Y
int	edit_border_x;			// Space to the R of dialog box
int	edit_border_y;			// Space below dialog box
unsigned long int ipAddress;		// IP address, only from inet_addr
BOOL	gfPastDays;			// Select past days
char	thisDir[BUFSIZE];		// Current news host directory
HINSTANCE ghInstance;			// Current instance handle
BOOL	gfMakeLog;			// Make log of images extracted
BOOL	fMakeLog;
BOOL	gfDontSaveDups;			// Don't save duplicate files
BOOL	fDontSaveDups;
BOOL	gfLoginReq;			// Login required

// Window coordinates in dialog box units
#define EDIT_LEFT 128
#define EDIT_TOP 30
#define EDIT_WIDTH 268
#define EDIT_HEIGHT 220
#define EDIT_RIGHT (EDIT_LEFT + EDIT_WIDTH)
#define EDIT_BOTTOM (EDIT_TOP + EDIT_HEIGHT)
#define NEWSGROUP_LEFT 168
#define NEWSGROUP_TOP 0
#define NEWSGROUP_WIDTH 228
#define NEWSGROUP_HEIGHT 100
#define NEWSHOST_LEFT 6
#define NEWSHOST_TOP 8
#define NEWSHOST_WIDTH 113
#define NEWSHOST_HEIGHT 100
#define DIALOG_TOP 10
#define DIALOG_LEFT 10
#define DIALOG_BOTTOM (266 + DIALOG_TOP)
#define DIALOG_RIGHT (400 + DIALOG_LEFT)
#define PROGRESS_WIDTH 98

//
// WinMain() - Entry Point
//
int WINAPI WinMain(
    HINSTANCE	hinstCurrent,
    HINSTANCE	hinstPrevious,
    LPSTR	szCmdLine,
    int		iCmdShow)
{
    int nReturnCode;
    WSADATA wsaData;
    WORD wVersionRequired = MAKEWORD(1,1);

    ghInstance = hinstCurrent;
    InitCommonControls();
    LoadLibrary("riched32.dll");

    // Initialize the WinSock DLL
    nReturnCode = WSAStartup(wVersionRequired, &wsaData);
    if (nReturnCode != 0 )
    {
	MessageBox(NULL,"Error on WSAStartup()",
		    "FoxyNews", MB_OK);
	return 1;
    }

    // Confirm that the version requested is available.
    if (wsaData.wVersion != wVersionRequired) {
	MessageBox(NULL,
	    "Wrong WinSock Version",
	    "FoxyNews", MB_OK);
	WSACleanup();
	return 1;
    }

    // Use a dialog box for our main window
    DialogBox(
	hinstCurrent,
	MAKEINTRESOURCE(IDD_DIALOG_MAIN),
	NULL,
	MainDialogProc);

    // Release WinSock DLL
    WSACleanup();
    return 0;
}


//
// MainDialogProc() - Main Window Procedure
//
LRESULT CALLBACK MainDialogProc(
    HWND hwndDlg,
    UINT msg,
    WPARAM wParam,
    LPARAM lParam)
{
    static WNDCLASSEX wndclass;

    BOOL fRet = FALSE;
    RECT thisControl;
    RECT dialogRect;
    RECT dialogRectInDialogUnits;
    HWND hwndDays;
    HWND hwndSpinDays;
    HWND hwndShowAll;
    HWND hwndPastDays;
    HWND hwndNewshost;
    HWND hwndNewsgroup;
    LOGFONT logfont;
    HFONT hFont;

    switch(msg)
    {
	case WM_INITDIALOG:
	    {
		// Clear out our buffers
		bzero(gszNewsgroup, BUFSIZE);
		bzero(gszServerName, BUFSIZE);
		bzero(gszSearchString, BUFSIZE);
		bzero(gszLoginName, SHORTBUFSIZE);
		bzero(gszPassword, SHORTBUFSIZE);

		// Default flags
		gfMakeLog = TRUE;
		gfDontSaveDups = FALSE;
		gfLoginReq = FALSE;
		gfURLs = FALSE;

		// Set up the icon for the window
		bzero(&wndclass, sizeof(wndclass));
		wndclass.hInstance = ghInstance;
		wndclass.hIcon = LoadIcon(wndclass.hInstance, szAppName);
		wndclass.hIconSm = LoadIcon(wndclass.hInstance, szAppName);
		SendMessage(hwndDlg, WM_SETICON, 1, (LPARAM)wndclass.hIconSm);

		// Dimensions of the dialog box in dialog box units
		// -- copied from the RC file
		dialogRectInDialogUnits.top = DIALOG_TOP;
		dialogRectInDialogUnits.bottom = DIALOG_BOTTOM;
		dialogRectInDialogUnits.left = DIALOG_LEFT;
		dialogRectInDialogUnits.right = DIALOG_RIGHT;

		// Copy and convert
		dialogRect.top = dialogRectInDialogUnits.top;
		dialogRect.bottom = dialogRectInDialogUnits.bottom;
		dialogRect.left = dialogRectInDialogUnits.left;
		dialogRect.right = dialogRectInDialogUnits.right;
		MapDialogRect(hwndDlg, &dialogRect);
		
		// Compute the conversion factor
		x_proportion =
		    (float)(dialogRect.right - dialogRect.left) /
		    (float)(dialogRectInDialogUnits.right -
		    dialogRectInDialogUnits.left);
		y_proportion =
		    (float)(dialogRect.bottom - dialogRect.top) /
		    (float)(dialogRectInDialogUnits.bottom -
		    dialogRectInDialogUnits.top);

		// Compute the borders on the bottom and right of the
		// edit window
		edit_border_x = (DIALOG_RIGHT - DIALOG_LEFT - EDIT_RIGHT) *
		    x_proportion;
		edit_border_y = (DIALOG_BOTTOM - DIALOG_TOP - EDIT_BOTTOM) *
		    y_proportion;

		// Status bar
		hwndStatusBar = CreateStatusWindow(
		    WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | CCS_BOTTOM |
		    SBARS_SIZEGRIP,
		    "Ready", hwndDlg, IDC_STATUSBAR);
		GetWindowRect(hwndStatusBar, &thisControl);
		aWidths[0] = PROGRESS_WIDTH * x_proportion;
		aWidths[1] =
		    (thisControl.right - thisControl.left + aWidths[0])/2;
		aWidths[2] = -1;
		SendMessage(hwndStatusBar, SB_SETPARTS, 3, (LPARAM)aWidths);

		// Initial status bar messages
		Display("Winsock initialized\r\n");
		Display3("Ready");

		// Progress bar is a child of the status bar
		hwndProgress = CreateWindowEx (
		    WS_EX_CLIENTEDGE,
		    PROGRESS_CLASS,
		    "",
		    WS_CHILD | WS_VISIBLE,
		    0,
		    2,
		    PROGRESS_WIDTH * x_proportion,
		    thisControl.bottom - thisControl.top - 2,
		    hwndStatusBar, 
		    (HMENU)IDC_PROGRESS, 
		    NULL,
		    NULL);
		SendMessage(hwndProgress, PBM_SETRANGE, 0, MAKELPARAM(0, 100));

		// Display window for article names
		hwndRichEdit = CreateWindowEx(
		    WS_EX_CLIENTEDGE,
		    "RICHEDIT",
		    "",
		    WS_CHILD | WS_VISIBLE | ES_MULTILINE | ES_SUNKEN |
		    ES_SAVESEL | WS_HSCROLL | WS_VSCROLL,
		    EDIT_LEFT * x_proportion,
		    EDIT_TOP * y_proportion,
		    EDIT_WIDTH * x_proportion,
		    EDIT_HEIGHT * y_proportion,
		    hwndDlg,
		    (HMENU)IDC_EDIT2,
		    NULL,
		    NULL);

		// Set the font for the window that displays headers
		GetObject(GetStockObject(ANSI_VAR_FONT), sizeof(LOGFONT),
		    (PSTR)&logfont);
		hFont = CreateFontIndirect(&logfont);
		SendMessage(hwndRichEdit, WM_SETFONT, (WPARAM)hFont, 0);

		// Initialize the radio buttons to "Show All"
		hwndShowAll = GetDlgItem(hwndDlg, ID_SHOWALL);
		hwndPastDays = GetDlgItem(hwndDlg, ID_PASTDAYS);
		CheckRadioButton(hwndDlg, ID_SHOWALL, ID_PASTDAYS, ID_SHOWALL);

		// Set up the spin control and its text window buddy
		// for number of days
		hwndDays = GetDlgItem(hwndDlg, ID_DAYS);
		hwndSpinDays = CreateWindowEx (
		  0L,
		  UPDOWN_CLASS,
		  "",
		  WS_CHILD | WS_BORDER | WS_VISIBLE | UDS_WRAP | UDS_ARROWKEYS |
		  UDS_ALIGNRIGHT | UDS_SETBUDDYINT,
		  0, 0, 10, 10,
		  hwndDlg,
		  (HMENU)ID_SPINDAYS,
		  NULL,
		  NULL);
		SendMessage(hwndSpinDays, UDM_SETBUDDY,
		    (LONG)hwndDays, 0L);
		SendMessage(hwndSpinDays, UDM_SETRANGE, 0L,
		    (LPARAM) MAKELONG((short)10, (short)0));
		SetDlgItemInt (hwndDlg, ID_DAYS, 1, FALSE);

		// Create combo box for news host
		hwndNewshost = CreateWindowEx (
		    WS_EX_CLIENTEDGE,
		    "COMBOBOX",
		    "",
		    WS_CHILD | WS_VISIBLE | CBS_DROPDOWN | WS_VSCROLL |
		    WS_HSCROLL | CBS_AUTOHSCROLL | CBS_SORT,
		    NEWSHOST_LEFT * x_proportion,
		    NEWSHOST_TOP * y_proportion,
		    NEWSHOST_WIDTH * x_proportion,
		    NEWSHOST_HEIGHT * y_proportion,
		    hwndDlg,
		    (HMENU) IDC_SERVERNAME,
		    NULL,
		    NULL);
		SendMessage(hwndNewshost, WM_SETFONT, (WPARAM)hFont, 0);

		// Copy the news host directory names to the combo box
		fill_in_news_host_list(hwndDlg);

		// Create combo box for newsgroup but don't initialize its
		// drop down list
		hwndNewsgroup = CreateWindowEx (
		    WS_EX_CLIENTEDGE,
		    "COMBOBOX",
		    "",
		    WS_CHILD | WS_VISIBLE | CBS_DROPDOWN | WS_VSCROLL |
		    WS_HSCROLL | CBS_AUTOHSCROLL | CBS_SORT,
		    NEWSGROUP_LEFT * x_proportion,
		    NEWSGROUP_TOP * y_proportion,
		    NEWSGROUP_WIDTH * x_proportion,
		    NEWSGROUP_HEIGHT * y_proportion,
		    hwndDlg,
		    (HMENU) IDC_NEWSGROUP,
		    NULL,
		    NULL);
		SendMessage(hwndNewsgroup, WM_SETFONT, (WPARAM)hFont, 0);
		// fill_in_newsgroup_list(hwndDlg);

		// Select download binary attachments, but not article text
		gfArticleText = FALSE;
		gfBinaryAttachments = TRUE;
		SendDlgItemMessage(hwndDlg, ID_ATTACHMENTS, BM_SETCHECK, 1, 0);
	    }
	    break;

	case WM_SIZE:
	    {
		int parentWidth = LOWORD(lParam);
		int parentHeight = HIWORD(lParam);
		int x, y, myWidth, myHeight;
		HWND hwndEdit2;

		// Resize the status bar
		GetWindowRect(hwndStatusBar, &thisControl);
		myHeight = thisControl.bottom - thisControl.top;
		x = 0;
		y = parentHeight - myHeight;
		myWidth = parentWidth;
		MoveWindow(hwndStatusBar, x, y, myWidth, myHeight, TRUE);
		aWidths[0] = PROGRESS_WIDTH * x_proportion;
		aWidths[1] = (parentWidth + aWidths[0])/2;
		aWidths[2] = -1;
		SendMessage(hwndStatusBar, SB_SETPARTS, 3, (LPARAM)aWidths);

		// Resize the edit window
		hwndEdit2 = GetDlgItem(hwndDlg, IDC_EDIT2);
		x = EDIT_LEFT * x_proportion;
		y = EDIT_TOP * y_proportion;
		myWidth = parentWidth - x - edit_border_x;
		myHeight = parentHeight - y - edit_border_y;
		MoveWindow(hwndEdit2, x, y, myWidth, myHeight, TRUE);

		// Resize the newsgroup window
		hwndNewsgroup = GetDlgItem(hwndDlg, IDC_NEWSGROUP);
		GetWindowRect(hwndNewsgroup, &thisControl);
		x = NEWSGROUP_LEFT * x_proportion;
		y = NEWSGROUP_TOP * y_proportion;
		myWidth = parentWidth - x - edit_border_x;
		myHeight = thisControl.bottom - thisControl.top;
		MoveWindow(hwndNewsgroup, x, y, myWidth, myHeight, TRUE);
	    }
	    break;

	// "F1"
	case WM_HELP:
	    WinHelp(hwndDlg, "doc/Foxynews.hlp", HELP_FINDER, 0);
	    return 0;

	case WM_COMMAND:
	    switch(wParam)
	    {
		/**************** "AUTOMATIC" THINGS ******************/

		// When the user clicks on the number of days' articles to be
		// selected, the user probably wants to display or download
		// by number of days.
		case MAKEWPARAM(ID_DAYS, EN_SETFOCUS):
		    CheckRadioButton(hwndDlg, ID_SHOWALL, ID_PASTDAYS,
			ID_PASTDAYS);
		    gfPastDays = TRUE;
		    break;

		// When the mouse focus goes to the newsgroup list,
		// the news server may have changed since we looked last,
		// so invalidate the present list and rebuild it.
		case MAKEWPARAM(IDC_NEWSGROUP, CBN_SETFOCUS):
		    fill_in_newsgroup_list(hwndDlg);
		    break;

		/***************** DIALOG BOX CONTROLS ****************/

		// "Login Required
		case ID_LOGIN_REQ:
		    gfLoginReq = !gfLoginReq;
		    if (!gfLoginReq) {
			bzero(gszLoginName, SHORTBUFSIZE);
			bzero(gszPassword, SHORTBUFSIZE);
		    }
		    break;

		// "Article Text"
		case ID_ARTICLETEXT:
		    gfArticleText = !gfArticleText;
		    break;

		// "Binary Attachments"
		case ID_ATTACHMENTS:
		    gfBinaryAttachments = !gfBinaryAttachments;
		    break;

		// "Urls from Articles"
		case ID_URLS:
		    gfURLs = !gfURLs;
		    break;

		// "Show Headers: All"
		case ID_SHOWALL:
		    gfPastDays = FALSE;
		    break;

		// "Show Headers: Past n days"
		case ID_PASTDAYS:
		    gfPastDays = TRUE;
		    break;

		// "Show Headers"
		case ID_SHOWHDRS:
		    if (gfLoginReq && (gszLoginName[0] == '\0')) {
			DialogBox(
			    wndclass.hInstance, MAKEINTRESOURCE(IDD_LOGINBOX),
			    hwndDlg, LoginDlgProc);
		    }
		    start_socket_connection(hwndDlg, SHOW_HEADERS);
		    fill_in_news_host_list(hwndDlg);
		    fRet = TRUE;
		    break;

		// "Get Attachments"
		case ID_CHECKNEWS:
		    if (gfLoginReq && (gszLoginName[0] == '\0')) {
			DialogBox(
			    wndclass.hInstance, MAKEINTRESOURCE(IDD_LOGINBOX),
			    hwndDlg, LoginDlgProc);
		    }
		    start_socket_connection(hwndDlg, GET_ATTACHMENTS);
		    fill_in_news_host_list(hwndDlg);
		    fRet = TRUE;
		    break;

		// "Stop"
		case ID_STOP:
		    if (gnAppState)
			signal_quit(TRUE);
		    break;

		// "Quit" or "x"
		case ID_QUIT:
		case IDCANCEL:
		    if (gnAppState)
			signal_quit(TRUE);
		    PostQuitMessage(0);
		    fRet = TRUE;
		    break;

		/******************** MENU ITEMS **********************/

		// "Help/Help Topics" -- call Winhelp
		case WM_HELP:
		    WinHelp(hwndDlg, "doc/Foxynews.hlp", HELP_FINDER, 0);
		    return 0;

		// "Help/About" -- show the popup
		case IDM_ABOUT:
		    DialogBox(
			wndclass.hInstance, MAKEINTRESOURCE(IDD_ABOUTBOX),
			hwndDlg, AboutDlgProc);
		    return 0;

		// "List/Newsgroups" -- show the popup
		case IDM_SHOW:
		    DialogBox(
			wndclass.hInstance, MAKEINTRESOURCE(IDD_SHOWBOX),
			hwndDlg, ShowDlgProc);
		    fill_in_news_host_list(hwndDlg);
		    return 0;

		// Command kicked up to us by "List/Newsgroups" dialog box
		case ID_LIST_NEWSGROUPS:
		    if (gfLoginReq && (gszLoginName[0] == '\0')) {
			DialogBox(
			    wndclass.hInstance, MAKEINTRESOURCE(IDD_LOGINBOX),
			    hwndDlg, LoginDlgProc);
		    }
		    start_socket_connection(hwndDlg, LIST_NEWSGROUPS);
		    fill_in_news_host_list(hwndDlg);
		    fRet = TRUE;
		    break;

		// "Options/Preferences" -- show the popup
		case IDM_PREFS:
		    DialogBox(
			wndclass.hInstance, MAKEINTRESOURCE(IDD_PREFSBOX),
			hwndDlg, PrefsDlgProc);
		    return 0;

		// Not implemented yet
		case IDM_BYSUBJECT:
		case IDM_BYARTICLENUMBER:
		case IDM_BYDATE:
		    MessageBox(hwndDlg,
			"Sorry, not implemented yet",
			"FoxyNews", MB_OK);
		    break;
	    }
	    break;

	    // Asynchronous gethostbyname() return
	    case SM_GETHOST:
		HandleGetHostMsg(hwndDlg, wParam, lParam);
		fRet = TRUE;
		break;

	    // Asynchronous messages
	    case SM_ASYNC:
		HandleAsyncMsg(hwndDlg, wParam, lParam);
		fRet = TRUE;
		break;
    }
    return fRet;
}

//
// HandleGetHostMsg()
// Called when WSAAsyncGetHostByName() completes
//
void HandleGetHostMsg(HWND hwndDlg, WPARAM wParam, LPARAM lParam)
{
    SOCKADDR_IN	saServ;		// Socket address for Internet
    LPHOSTENT	lpHostEnt;	// Pointer to host entry
    LPSERVENT	lpServEnt;	// Pointer to server entry
    int		nRet;		// Return code

    if ((HANDLE)wParam != hndlGetHost)
	return;

    // Check error code
    nRet = WSAGETASYNCERROR(lParam);
    if (nRet) {
	if (nRet == WSAHOST_NOT_FOUND)
	    sprintf(gszTemp, "Host %s not found", gszServerName);
	else
	    sprintf(gszTemp, "WSAAsyncGetHostByName() error: %d", nRet);
	MessageBox(hwndDlg,
	    gszTemp,
	    "GetHostByName", MB_OK);
	EnableButtons(TRUE);
	return;
    }

    // Server found OK, bufHostEnt contains server info
    Display("Calling socket()...");

    // Create a socket
    sockNews = socket(AF_INET, SOCK_STREAM, 0);
    if (sockNews == INVALID_SOCKET)
    {
	Display("Could not create a socket\r\n");
	EnableButtons(TRUE);
	return;
    }
    Display("Calling socket()...OK\r\n");

    // Make socket non-blocking and register for asynchronous notification
    if (WSAAsyncSelect(sockNews, hwndDlg, SM_ASYNC,
		FD_CONNECT|FD_READ|FD_WRITE|FD_CLOSE)) {
	Display("WSAAsyncSelect() failed\r\n");
	EnableButtons(TRUE);
	return;
    }

    // Try to resolve the port number with getservbyname()
    // Simpler synchronous version used since it almost always
    // completes quickly.
    Display("getservbyname()...");
    lpServEnt = getservbyname("nntp", "tcp");
    Display("getservbyname...OK");

    if (lpServEnt == NULL) {
	// Fill in with well-known port
	saServ.sin_port = htons(119);
	Display("getservbyname()...failed. Using port 119\r\n");
    } else {
	// Use port returned in servent
	saServ.sin_port = lpServEnt->s_port;
    }
    // Fill in the server address structure
    saServ.sin_family = AF_INET;
    lpHostEnt = (LPHOSTENT)bufHostEnt;
    saServ.sin_addr = *((LPIN_ADDR)*lpHostEnt->h_addr_list);

    // Connect the socket
    Display("Calling connect()...");
    nRet = connect(sockNews, (LPSOCKADDR)&saServ, sizeof(SOCKADDR_IN));
    if (nRet == SOCKET_ERROR) {
	if (WSAGetLastError() != WSAEWOULDBLOCK) {
	    Display("Error connecting\r\n");
	    EnableButtons(TRUE);
	    return;
	}
    }
    Display("Calling connect()...OK.  Waiting for news server reply.\r\n");
    gnAppState = CONNECTING;
    start_state_machine(hwndDlg);
}

// Handle asynchronous messages
void HandleAsyncMsg(HWND hwndDlg, WPARAM wParam, LPARAM lParam)
{
    int retry;

    switch(WSAGETSELECTEVENT(lParam))
    {
	case FD_CONNECT:
	    break;

	case FD_WRITE:
	    break;

	case FD_READ:
	    // Run our state machine
	    gnAppState = state_machine();
	    if (gnAppState == LOGIN_NOT_REQUIRED) {
		gfLoginReq = FALSE;
		bzero(gszLoginName, SHORTBUFSIZE);
		bzero(gszPassword, SHORTBUFSIZE);
		SendDlgItemMessage(hwndDlg, ID_LOGIN_REQ, BM_SETCHECK, 0, 0);
		gnAppState = QUITTING;
		MessageBox(hwndDlg,
		    "This server doesn't require a login",
		    "FoxyNews",
		    MB_OK | MB_ICONINFORMATION);
	    }
	    if (gnAppState == AUTHORIZATION_FAILED) {
		if (gfLoginReq) {
		    sprintf(gszTemp,
			"Do you want to try a different login name and password?");
		} else {
		    sprintf(gszTemp,
			"Do you want to try a login name and password?");
		}
		retry = MessageBox (hwndDlg,
		    gszTemp,
		    "Authorization Error",
		    MB_YESNO | MB_ICONQUESTION);
		bzero(gszLoginName, SHORTBUFSIZE);
		bzero(gszPassword, SHORTBUFSIZE);
		if (retry == IDYES) {
		    gfLoginReq = TRUE;
		    SendDlgItemMessage(
			hwndDlg, ID_LOGIN_REQ, BM_SETCHECK, 1, 0);
		} else {
		    gfLoginReq = FALSE;
		    SendDlgItemMessage(
			hwndDlg, ID_LOGIN_REQ, BM_SETCHECK, 0, 0);
		}
		gnAppState = QUITTING;
	    }
	    if (gnAppState == QUITTING)
		do_reset(hwndDlg);
	    break;

	case FD_CLOSE:
	    Display("FD_CLOSE notification received\r\n");
	    EnableButtons(TRUE);
	    break;
    }
}

// Handle messages to "About" dialog
static BOOL CALLBACK AboutDlgProc (
    HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    switch (iMsg) {
	case WM_INITDIALOG:
	    return TRUE;

	case WM_COMMAND:
	    switch (LOWORD (wParam)) {
		case IDOK:
		case IDCANCEL:
		    EndDialog (hDlg, 0);
		    return TRUE;
	    }
	    break;
    }
    return FALSE;
}

// Handle messages to "Preferences" dialog
static BOOL CALLBACK PrefsDlgProc (
    HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    switch (iMsg) {
	case WM_INITDIALOG:
	    fMakeLog = gfMakeLog;
	    fDontSaveDups = gfDontSaveDups;
	    if (gfMakeLog) {
		SendDlgItemMessage(hDlg, ID_MAKELOG, BM_SETCHECK, 1, 0);
	    }
	    if (gfDontSaveDups) {
		SendDlgItemMessage(hDlg, ID_DONTSAVEDUPS, BM_SETCHECK, 1, 0);
	    }
	    return TRUE;

	case WM_COMMAND:
	    switch (LOWORD (wParam)) {
		case ID_MAKELOG:
		    fMakeLog = !fMakeLog;
		    return TRUE;
		case ID_DONTSAVEDUPS:
		    fDontSaveDups = !fDontSaveDups;
		    return TRUE;
		case IDOK:
		    gfMakeLog = fMakeLog;
		    gfDontSaveDups = fDontSaveDups;
		    // Fall through
		case IDCANCEL:
		    EndDialog (hDlg, 0);
		    return TRUE;
	    }
	    break;
    }
    return FALSE;
}

// Handle messages to "Login" dialog
static BOOL CALLBACK LoginDlgProc (
    HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    switch (iMsg) {
	case WM_INITDIALOG:
	    SetDlgItemText(hDlg, IDC_LOGIN_NAME, gszLoginName);
	    SetDlgItemText(hDlg, IDC_PASSWORD, gszPassword);
	    return TRUE;

	case WM_COMMAND:
	    switch (LOWORD (wParam)) {
		case IDOK:
		    if (!GetDlgItemText(hDlg, IDC_LOGIN_NAME,
			gszLoginName, sizeof(gszLoginName))) {
			MessageBox(hDlg,
			    "Please enter a login name",
			    "Login", MB_OK);
			return FALSE;
		    }
		    if (!GetDlgItemText(hDlg, IDC_PASSWORD,
			gszPassword, sizeof(gszPassword))) {
			MessageBox(hDlg,
			    "Please enter a password",
			    "Login", MB_OK);
			return FALSE;
		    }
		    EndDialog (hDlg, 0);
		    return TRUE;
		case IDCANCEL:
		    EndDialog (hDlg, 0);
		    return TRUE;
	    }
	    break;
    }
    return FALSE;
}

// Handle "Show" dialog -- everything is in the resource file
static BOOL CALLBACK ShowDlgProc (
    HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
    HWND hwndParent;

    switch (iMsg) {
	case WM_INITDIALOG:
	    return TRUE;

	case WM_COMMAND:
	    switch (LOWORD (wParam)) {
		case IDOK:
		    hwndParent = GetParent(hDlg);
		    EndDialog (hDlg, 0);
		    SendMessage(hwndParent, WM_COMMAND, ID_LIST_NEWSGROUPS, 0);
		    return TRUE;
		case IDCANCEL:
		    EndDialog (hDlg, 0);
		    return TRUE;
	    }
	    break;
    }
    return FALSE;
}

// Shut down anything that's pending and get ready to receive a new user input
void do_reset(HWND hwndDlg)
{
    if (gnAppState != QUITTING) {
	signal_quit(TRUE);
    }
    gnAppState = QUITTING;
    EnableButtons(TRUE);
}

// Rebuild the list of newsgroups in the drop-down window
void fill_in_newsgroup_list(HWND hwndDlg)
{
    WIN32_FIND_DATA FileData;
    HANDLE hSearch;
    DWORD dwAttrs;
    char thisNewsgroup[BUFSIZE];
    BOOL fFinished = FALSE;
    BOOL fFirst = TRUE;
    HWND hwndNewshost;
    HWND hwndNewsgroup;

    bzero(thisDir, BUFSIZE);
    hwndNewshost = GetDlgItem(hwndDlg, IDC_SERVERNAME);
    hwndNewsgroup = GetDlgItem(hwndDlg, IDC_NEWSGROUP);
    SendMessage(hwndNewsgroup, CB_RESETCONTENT, 0, 0);
    if (!GetDlgItemText(hwndDlg, IDC_SERVERNAME, thisDir, BUFSIZE)) {
	SetFocus(hwndNewshost);
	MessageBox(hwndDlg,
	    "Please enter the name of a news server",
	    "FoxyNews", MB_OK);
	return;
    }
    strcat(thisDir, "\\*.*");
    // Copy the directory names to the combo box
    hSearch = FindFirstFile(thisDir, &FileData);
    fFinished = FALSE;
    if (hSearch != INVALID_HANDLE_VALUE) {
	while (!fFinished) {
	    dwAttrs = GetFileAttributes(FileData.cFileName);
	    if (strstr(FileData.cFileName, ".log")) {
		bzero(thisNewsgroup, BUFSIZE);
		strcpy(thisNewsgroup, FileData.cFileName);
		thisNewsgroup[strlen(thisNewsgroup)-4] = '\0';
		SendMessage(hwndNewsgroup, CB_ADDSTRING,
		    0, (LPARAM)thisNewsgroup);
		if (fFirst) {
		    SetDlgItemText(hwndDlg, IDC_NEWSGROUP, thisNewsgroup);
		    fFirst = FALSE;
		}
	    }
	    if (!FindNextFile(hSearch, &FileData)) {
		if (GetLastError() == ERROR_NO_MORE_FILES) {
		    fFinished = TRUE;
		} else {
		    MessageBox(hwndDlg,
			"Can't get next file",
			"Filesystem error.", MB_OK);
		    return;
		}
	    }
	}
	/* Close search handle. */
	if (!FindClose(hSearch)) {
	    MessageBox(hwndDlg,
		"Can't close file handle",
		"Filesystem error.", MB_OK);
	}
    }
    if (fFirst) {
	SendMessage(hwndNewsgroup, CB_ADDSTRING, 0,
	    (LPARAM)"<Please enter the name of a newsgroup>");
    }
}

// Rebuild the list of news servers in the drop-down window
void fill_in_news_host_list(HWND hwndDlg)
{
    WIN32_FIND_DATA FileData;
    HANDLE hSearch;
    DWORD dwAttrs;
    BOOL fFinished = FALSE;
    BOOL fFirst = TRUE;
    HWND hwndNewshost;
    char szPresentNewsHost[BUFSIZE];
    BOOL fNotEmpty;

    fNotEmpty = GetDlgItemText(hwndDlg, IDC_SERVERNAME,
	szPresentNewsHost, sizeof(gszServerName));
    hwndNewshost = GetDlgItem(hwndDlg, IDC_SERVERNAME);
    SendMessage(hwndNewshost, CB_RESETCONTENT, 0, 0);
    hSearch = FindFirstFile("*.*", &FileData);
    if (hSearch != INVALID_HANDLE_VALUE) {
	while (!fFinished) {
	    dwAttrs = GetFileAttributes(FileData.cFileName);
	    if (dwAttrs & FILE_ATTRIBUTE_DIRECTORY) {
		if ((strcmp(FileData.cFileName, ".")) &&
		    (strcmp(FileData.cFileName, "..")) &&
		    (strcmp(FileData.cFileName, "src")) &&
		    (strcmp(FileData.cFileName, "doc"))) {
		    SendMessage(hwndNewshost, CB_ADDSTRING,
			0, (LPARAM)FileData.cFileName);
		    if (fFirst) {
			if (fNotEmpty) {
			    SetDlgItemText(hwndDlg, IDC_SERVERNAME,
				szPresentNewsHost);
			} else {
			    SetDlgItemText(hwndDlg, IDC_SERVERNAME,
				FileData.cFileName);
			}
			fFirst = FALSE;
		    }
		}
	    }
	    if (!FindNextFile(hSearch, &FileData)) {
		if (GetLastError() == ERROR_NO_MORE_FILES) {
		    fFinished = TRUE;
		} else {
		    MessageBox(hwndDlg,
			"Can't get next file",
			"Filesystem error.", MB_OK);
		    return;
		}
	    }
	}
	if (!FindClose(hSearch)) {
	    MessageBox(hwndDlg,
		"Can't close file handle",
		"Filesystem error.", MB_OK);
	}
    }
    if (fFirst) {
	SendMessage(hwndNewshost, CB_ADDSTRING, 0,
	    (LPARAM)"<Enter a news server name>");
    }
}

//
// Connect to the server
//
void start_socket_connection(HWND hwndDlg, int nStateCmd)
{
    gnStateCmd = nStateCmd;

    // See if the user has checked enough of the right boxes to
    // enable us to proceed
    switch (nStateCmd) {
	case LIST_NEWSGROUPS:
	    break;
	case SHOW_HEADERS:
	    if (!GetDlgItemText(hwndDlg, IDC_NEWSGROUP,
		gszNewsgroup, sizeof(gszNewsgroup))) {
		MessageBox(hwndDlg,
		    "Please enter a newsgroup",
		    "FoxyNews", MB_OK);
		return;
	    }
	    break;
	case GET_ATTACHMENTS:
	    if (!GetDlgItemText(hwndDlg, IDC_NEWSGROUP,
		gszNewsgroup, sizeof(gszNewsgroup))) {
		MessageBox(hwndDlg,
		    "Please enter a newsgroup",
		    "FoxyNews", MB_OK);
		return;
	    }
	    if (!GetDlgItemText(hwndDlg, IDC_SEARCHSTRING,
		gszSearchString, sizeof(gszSearchString))) {
		MessageBox(hwndDlg,
		    "Please enter a search string",
		    "FoxyNews", MB_OK);
		return;
	    }
	    if (!gfArticleText && !gfBinaryAttachments && !gfURLs) {
		MessageBox(hwndDlg,
		    "You must check one of the check boxes",
		    "NNTP Info", MB_OK);
		return;
	    }
	    break;
    }

    // Get the news server name from the control
    if (!GetDlgItemText(hwndDlg, IDC_SERVERNAME,
	gszServerName, sizeof(gszServerName))) {
	MessageBox(hwndDlg,
	    "Please enter the name of a news server",
	    "FoxyNews", MB_OK);
	return;
    }


    Display3("Connecting");
    sprintf(gszTemp, "Looking up %s...\r\n", gszServerName);
    Display(gszTemp);

    // First see if the user entered an IP address
    ipAddress = inet_addr(gszServerName);
    if (ipAddress == INADDR_NONE) {

	// Request a lookup of the host name using the asynchronous
	// version of gethostbyname()
	// SM_GETHOST message will be posted to this window when it completes.
	hndlGetHost = WSAAsyncGetHostByName(
		hwndDlg,
		SM_GETHOST,
		gszServerName,
		bufHostEnt,
		MAXGETHOSTSTRUCT);
    } else {
	hndlGetHost = WSAAsyncGetHostByAddr(
		hwndDlg,
		SM_GETHOST,
		(char *)&ipAddress,
		sizeof(unsigned long int),
		AF_INET,
		bufHostEnt,
		MAXGETHOSTSTRUCT);
    }

    if (hndlGetHost == 0) {
	MessageBox(hwndDlg,
	    "Error initiating "
	    "WSAAsyncGetHostByName()",
	    "FoxyNews", MB_OK);
    } else {
	EnableButtons(FALSE);
    }
}

//
// Start up the state machine
//
void start_state_machine(HWND hwndDlg)
{
    int nDaysPast;
    BOOL bErr;

    signal_quit(FALSE);

    switch (gnStateCmd) {

	case LIST_NEWSGROUPS:
	    init_state_machine_data(
		hwndDlg,
		hwndProgress,
		sockNews,
		gszServerName,
		"No newsgroup",
		"No search string",
		gszLoginName,
		gszPassword,
		LIST_NEWSGROUPS,
		0,
		TRUE,
		FALSE,
		gfPastDays,
		FALSE,
		FALSE,
		gfURLs,
		gfMakeLog,
		gfDontSaveDups);
	    break;

	case SHOW_HEADERS:
	    if (gfPastDays) {
		nDaysPast = GetDlgItemInt(hwndDlg, ID_DAYS, &bErr, FALSE);
	    } else {
		nDaysPast = 999;
	    }
	    init_state_machine_data(
		hwndDlg,
		hwndProgress,
		sockNews,
		gszServerName,
		gszNewsgroup,
		"No search string",
		gszLoginName,
		gszPassword,
		SHOW_HEADERS,
		nDaysPast,
		FALSE,
		TRUE,
		gfPastDays,
		FALSE,
		FALSE,
		gfURLs,
		gfMakeLog,
		gfDontSaveDups);
	    break;

	case GET_ATTACHMENTS:
	    if (gfPastDays) {
		nDaysPast = GetDlgItemInt(hwndDlg, ID_DAYS, &bErr, FALSE);
	    } else {
		nDaysPast = 999;
	    }
	    init_state_machine_data(
		hwndDlg,
		hwndProgress,
		sockNews,
		gszServerName,
		gszNewsgroup,
		gszSearchString,
		gszLoginName,
		gszPassword,
		GET_ATTACHMENTS,
		nDaysPast,
		FALSE,
		FALSE,
		gfPastDays,
		gfArticleText,
		gfBinaryAttachments,
		gfURLs,
		gfMakeLog,
		gfDontSaveDups);
	    break;
    }
}
